/**
 * @class PMUI
 * Base class PMUI
 * @singleton
 */
var PMUI.util.PMUI = {};

PMUI.version = '0.1.1';
PMUI.isCtrl = false;
PMUI.isShift = false;
PMUI.activeCanvas = null; 
PMUI.currentContextMenu = null;

$(document).keydown(function (e) {
    
    if (PMUI.activeCanvas) {
        switch (e.which) {
            case 16: // SHIFT KEY
                PMUI.isShift = true;
                break;
            case 17: // CTRL KEY
                PMUI.isCtrl = true;
                break;
            case 116: // F5 KEY
                e.preventDefault();
                window.location.reload(true);
                break;
            case 37:
                // Left
                if (!PMUI.activeCanvas.currentLabel) {
                    e.preventDefault();
                    PMUI.activeCanvas.moveElements(PMUI.activeCanvas, 'LEFT');
                }
                break;
            case 38:
                // Top
                if (!PMUI.activeCanvas.currentLabel) {
                    e.preventDefault();
                    PMUI.activeCanvas.moveElements(PMUI.activeCanvas, 'TOP');
                }
                break;
            case 39:
                // Right
                if (!PMUI.activeCanvas.currentLabel) {
                    e.preventDefault();
                    PMUI.activeCanvas.moveElements(PMUI.activeCanvas, 'RIGHT');
                }
                break;
            case 40:
                // Bottom
                if (!PMUI.activeCanvas.currentLabel) {
                    e.preventDefault();
                    PMUI.activeCanvas.moveElements(PMUI.activeCanvas, 'BOTTOM');
                }
                break;
            case 67:    // char 'c'
                if (!PMUI.activeCanvas.currentLabel && PMUI.isCtrl) {
                    if (PMUI.activeCanvas.copyAndPaste) {
                        e.preventDefault();
                        PMUI.activeCanvas.copy();
                    }

                }
                break;
            case 86:    // char 'v'
                if (!PMUI.activeCanvas.currentLabel && PMUI.isCtrl) {
                    if (PMUI.activeCanvas.copyAndPaste) {
                        e.preventDefault();
                        PMUI.activeCanvas.paste();
                    }
                }
                break;
            case 90:    // char 'z'
                if (PMUI.isCtrl && !PMUI.activeCanvas.readOnly) {
                    if (PMUI.isShift) {
                        // ctrl + shift + z (redo)
                        PMUI.activeCanvas.redo();
                        e.preventDefault();
                    } else {
                        // ctrl + z (undo)
                        PMUI.activeCanvas.undo();
                        e.preventDefault();
                    }
                }
                break;
        }
    }
}).keypress(function (e) {

}).keyup(function (e) {
    var current;
    e.preventDefault();
    switch (e.which) {
        case 8:  //BACKSPACE
            if (PMUI.isCtrl) {
                if (PMUI.activeCanvas && !PMUI.activeCanvas.currentLabel) {
                    PMUI.activeCanvas.removeElements();
                }
            }
            break;
        case 13:    // ENTER
            if (PMUI.activeCanvas && PMUI.activeCanvas.currentLabel) {
                PMUI.activeCanvas.currentLabel.loseFocus();
            }
            break;
        case 46: // DELETE KEY
            if (PMUI.activeCanvas && !PMUI.activeCanvas.currentLabel) {
                PMUI.activeCanvas.removeElements();
            }
            break;
        case 16: // SHIFT KEY
            PMUI.isShift = false;
            break;
        case 17: //CTRL KEY
            PMUI.isCtrl = false;
            break;
        case 113: //F2 KEY
            if (PMUI.activeCanvas &&
                PMUI.activeCanvas.getCurrentSelection().getLast() !== null) {
                //Run other code here when the element
                // 'CurElement' is deleted
                current = PMUI.activeCanvas.getCurrentSelection().getLast();
                if (current !== undefined && current.label.html !== null) {
                    $(current.label.html).dblclick();
                    $(current.label.text.html).focus();
                }
            }
            break;
    }
});


/**
 * Extends the PMUI namespace with the given `path` and making a pointer
 * from `path` to the given `class` (note that the `path`'s last token will be the pointer visible from outside
 * the definition of the class).
 *
 *      // e.g.
 *      // let's define a class inside an anonymous function
 *      // so that the global scope is not polluted
 *      (function () {
 *          var Canvas = function () {...};
 *
 *          // let's extend the namespace
 *          PMUI.extendNamespace('PMUI.core.Panel', Canvas);
 *
 *      }());
 *
 *      // now PMDraw.draw.Canvas is a pointer to the class defined above
 *
 *  Another example:
 *
 *      // let's define a class inside an anonymous function
 *      // so that the global scope is not polluted
 *      (function () {
 *          var Shape = function () {...};
 *
 *          // let's extend the namespace
 *          PMUI.extendNamespace('PMUI.draw.RandomName', Shape);
 *
 *      }());
 *
 *      // now PMUI.draw.RandomName is a pointer to the class Shape
 *      // note that this class can only be accessed through this pointer
 *
 * @param {string} path
 * @param {Object} newClass
 * @return {Object} The argument `newClass`
 */
PMUI.extendNamespace = function (path, newClass) {
    var current,
        pathArray,
        extension,
        i;

    if (arguments.length !== 2) {
        throw new Error("PMUI.extendNamespace(): method needs 2 arguments");
    }

    pathArray = path.split('.');
    if (pathArray[0] === 'PMUI') {
        pathArray = pathArray.slice(1);
    }
    current = PMUI;

    // create the 'path' namespace
    for (i = 0; i < pathArray.length - 1; i += 1) {
        extension = pathArray[i];
        if (typeof current[extension] === 'undefined') {
            current[extension] = {};
        }
        current = current[extension];
    }

    extension = pathArray[pathArray.length - 1];
    if (current[extension]) {
        //console.log("PMUI.extendNamespace(): Warning! overriding the class '" + pathArray.join('.') + "'");
    }
    current[extension] = newClass;
    return newClass;
};
/**
 * Checks if `path` (a string separated with dots) is a valid path inside the `from` object if provided otherwise
 * checks if `path` is a valid path inside the {@link PMUI} object,
 * if so then returns a pointer to the object which is the last token of the string
 *
 *      // e.g
 *      validPath('PMDraw.event.Keyboard.modifiers.alt');    // returns a pointer to alt
 *      validPath('modifiers.alt', PMUI.event.Keyboard);   // returns a pointer to alt
 *
 * @param {string} path
 * @param {Object} [from]
 * @return {Object}
 */
PMUI.validPath = function (path, from) {
    var pathArray = path.split('.'),
        current,
        extension,
        i;
    if (!from) {
        if (pathArray[0] === 'PMUI') {
            pathArray = pathArray.slice(1);
        }
        current = PMUI;
    } else {
        current = from;
    }
    for (i = 0; i < pathArray.length; i += 1) {
        extension = pathArray[i];
        if (!current[extension]) {
            return null;
        }
        current = current[extension];
    }
    return current;
};

/**
 * Creates an object whose [[Prototype]] link points to an object's prototype (the object is gathered using the
 * argument `path` and it's the last token in the string), since `subClass` is given it will also mimic the
 * creation of the property `constructor` and a pointer to its parent called `superclass`:
 *
 *      // constructor pointer
 *      subClass.prototype.constructor === subClass       // true
 *
 *      // let's assume that superClass is the last token in the string 'path'
 *      subClass.superclass === superClass         // true
 *
 * An example of use:
 *
 *      (function () {
 *          var Core = function () {...};
 *
 *          // extending the namespace
 *          PMDraw.extendNamespace('PMDraw.draw.Core', Core);
 *
 *      }());
 *
 *      (function () {
 *          var BehavioralElement = function () {...};
 *
 *          // this class inherits from PMDraw.draw.Core
 *          PMDraw.inheritFrom('PMDraw.draw.Core', BehavioralElement);
 *
 *          // extending the namespace
 *          PMDraw.extendNamespace('PMDraw.draw.BehavioralElement', BehavioralElement);
 *
 *      }());
 *
 * @param {string} path
 * @param {Object} subClass
 * @return {Object}
 */
PMUI.inheritFrom = function (path, subClass) {
    var current,
        extension,
        pathArray,
        i,
        prototype;

    if (arguments.length !== 2) {
        throw new Error("PMUI.inheritFrom(): method needs 2 arguments");
    }

    // function used to create an object whose [[Prototype]] link
    // points to `object`
    function clone(object) {
        var F = function () {};
        F.prototype = object;
        return new F();
    }

    pathArray = path.split('.');
    if (pathArray[0] === 'PMUI') {
        pathArray = pathArray.slice(1);
    }
    current = PMUI;

    // find that class the 'path' namespace
    for (i = 0; i < pathArray.length; i += 1) {
        extension = pathArray[i];
        if (typeof current[extension] === 'undefined') {
            throw new Error("PMUI.inheritFrom(): object " + extension + " not found, full path was " + path);
        }
        current = current[extension];
    }

    prototype = clone(current.prototype);

    prototype.constructor = subClass;
    subClass.prototype = prototype;
    subClass.superclass = current;
};

/**
 * Generates 32-digits alphanumeric unique IDs
 * @return {String} Alphanumeric 32-char unique string
 */
PMUI.generateUniqueId = function () {
    var rand = function (min, max) {
            // Returns a random number
            //
            // version: 1109.2015
            // discuss at: http://phpjs.org/functions/rand
            // +   original by: Leslie Hoare
            // +   bugfixed by: Onno Marsman
            // %          note 1: See the commented out code below for a
            // version which will work with our experimental
            // (though probably unnecessary) srand() function)
            // *     example 1: rand(1, 1);
            // *     returns 1: 1

            // fix for jsLint
            // from: var argc = arguments.length;
            if (typeof min === "undefined") {
                min = 0;
            }
            if (typeof max === "undefined") {
                max = 999999999;
            }
            return Math.floor(Math.random() * (max - min + 1)) + min;
        },
        uniqid = function (prefix, more_entropy) {
            // +   original by: Kevin van Zonneveld (http://kevin.vanzonneveld.net)
            // +    revised by: Kankrelune (http://www.webfaktory.info/)
            // %        note 1: Uses an internal counter (in php_js global) to avoid collision
            // *     example 1: uniqid();
            // *     returns 1: 'a30285b160c14'
            // *     example 2: uniqid('foo');
            // *     returns 2: 'fooa30285b1cd361'
            // *     example 3: uniqid('bar', true);
            // *     returns 3: 'bara20285b23dfd1.31879087'
            if (typeof prefix === 'undefined') {
                prefix = "";
            }

            var retId,
                formatSeed = function (seed, reqWidth) {
                    var tempString = "",
                        i;
                    seed = parseInt(seed, 10).toString(16); // to hex str
                    if (reqWidth < seed.length) { // so long we split
                        return seed.slice(seed.length - reqWidth);
                    }
                    if (reqWidth > seed.length) { // so short we pad
                        // jsLint fix
                        tempString = "";
                        for (i = 0; i < 1 + (reqWidth - seed.length); i += 1) {
                            tempString += "0";
                        }
                        return tempString + seed;
                    }
                    return seed;
                };

            // BEGIN REDUNDANT
            if (!this.php_js) {
                this.php_js = {};
            }
            // END REDUNDANT
            if (!this.php_js.uniqidSeed) { // init seed with big random int
                this.php_js.uniqidSeed = Math.floor(Math.random() * 0x75bcd15);
            }
            this.php_js.uniqidSeed += 1;

            retId = prefix; // start with prefix, add current milliseconds hex string
            retId += formatSeed(parseInt(new Date().getTime() / 1000, 10), 8);
            retId += formatSeed(this.php_js.uniqidSeed, 5); // add seed hex string
            if (more_entropy) {
                // for more entropy we add a float lower to 10
                retId += (Math.random() * 10).toFixed(8).toString();
            }

            return retId;
        },
        sUID;

    do {
        sUID = uniqid(rand(0, 999999999), true);
        sUID = sUID.replace('.', '0');
    } while (sUID.length !== 32);

    return sUID;
};

/**
 * Creates and returns a HTML element
 * @param  {String} type The type for the element to be created, for example: div, span, p
 * @return {HTMLElement}    An HTML element
 */
PMUI.createHTMLElement = function(type) {
    return document.createElement(type);
};

/**
 * Calculates the text width usign a font family
 * @param {String} text The text which width will be calculated
 * @param {String} font The font family and size (expressed as the 'font' css properties)
 * to be used to calculate the width 
 * @return {Number}
 */
PMUI.calculateWidth = function (text, font) {
    var f = font || '12px arial',
        $o = $(this.createHTMLElement('div')), w;
        $o.text(text)
            .css({'position': 'absolute', 'float': 'left', 'white-space': 'nowrap', 'visibility': 'hidden', 'font': f})
            .appendTo($('body'));
        w = $o.width();

    $o.remove();

    return w;
};

PMUI.emToPx = function(emUnits, context) {
    var testDiv = PMUI.createHTMLElement('div'), theWidth, $div;

    testDiv.style.margin = '0';
    testDiv.style.padding = '0';
    testDiv.style.position = 'absolute';
    testDiv.style.display = 'inline-block';
    testDiv.style.backgroundColor = 'none';
    testDiv.style.width = emUnits + "em";
    context = PMUI.isHTMLElement(context) ? context : document.body;
    if(context !== document.body && !jQuery(context).parents('body').length) {
        throw new Error("The context must be an HTML element appended to the DOM.");
    }
    context.appendChild(testDiv);
    $div = jQuery(testDiv);
    theWidth = $div.outerWidth();
    $div.remove();
    return theWidth;
};

/**
 * Get PMUI Version.
 * @return {String}
 */
PMUI.getVersion = function () {
    return this.version;
};

/**
 * Trigger events defined in the element
 * @param  {PMUI.core.Element} el Element associated with the event
 * @param  {String} eventName Event Name or alias
 * @param  {Object} scope  Calling scope for the event
 */
PMUI.triggerEvent = function(el, eventName, scope) {
    var scopeTrigger = scope || this;
    if (el instanceof PMUI.core.Element) {
        if (el.events[eventName] instanceof PMUI.event.Event){
            el.events[eventName].handler.call(scopeTrigger);
        } else {
            throw new Error('Event name is not registered int this element');    
        }
    } else {
        throw new Error('Current Element is not able to trigger events');
    }
};

/**
 * Sets the active canvas.
 * @param {PMUI.draw.Canvas} canvas
 * @chainable
 */
PMUI.setActiveCanvas = function (canvas) {
    PMUI.activeCanvas = canvas;
    return this;
};
/**
 * Gets the active canvas
 * @return {PMUI.draw.Canvas}
 */
PMUI.getActiveCanvas = function () {
    return PMUI.activeCanvas;
};

/**
 * Converts the coordinates `xCoord` and `yCoord` (assuming that xCoord and yCoord are pageCoordinates)
 * or the page coordinates gathered from the object `e` if there is no `xCoord` or `yCoord` to
 * `shape` coordinates, this new coordinate also considers the scroll done in the canvas
 *
 *      // e.g.
 *      // Let's assume that:
 *      // the canvas coordinates are [100, 100] and that it has no scroll
 *      // the shape coordinates are [100, 100] (inside the canvas)
 *      // e is an object containing page.X = 300, page.Y = 300
 *      Utils.pageCoordinatesToShapeCoordinates(shape, e)  // new Point(100, 100) respect to the shape
 *
 *
 * @param {Object} shape
 * @param {Object} e
 * @param {number} [xCoord]
 * @param {number} [yCoord]
 * @return {PMUI.util.Point} a point relative to the canvas
 */
PMUI.pageCoordinatesToShapeCoordinates = function (shape, e, xCoord, yCoord, customShape) {
    var coordinates,
        x = (!xCoord) ? e.pageX : xCoord,
        y = (!yCoord) ? e.pageY : yCoord,
        orgX = (!xCoord) ? e.pageX : xCoord,
        orgY = (!yCoord) ? e.pageY : yCoord,
        canvas = shape.getCanvas();
    x += canvas.getLeftScroll() - shape.getAbsoluteX() - canvas.getX();
    y += canvas.getTopScroll() - shape.getAbsoluteY() - canvas.getY();
    if (orgX === x) {
        x -= parseInt(customShape.width / 2, 10);
    } 
    if (orgY === y) {
        y -= parseInt(customShape.height / 2, 10);
    }
    coordinates = new PMUI.util.Point(x, y);
    return coordinates;
};
/**
 * Validates the parameters that represents the coordinates for create a new shape
 * @param  {Object} canvas      Represents the current canvas for the project
 * @param  {Object} e           Represents the event that triggered on the event
 * @param  {Object} customShape Represents the object that will create a new element on the canvas
 * @return {Boolean} flag       After the review of the coordinates the method return true if the component
 * will create on the canvas and false if not possible create it.
 */
PMUI.validCoordinatedToCreate = function (canvas, e, customShape) {
    return function (canvas, e, customShape) {
        return true;
    };
};
/**
 * Converts the coordinates of the `shape` to page coordinates, this method
 * also considers the scroll of the canvas in the calculation
 *
 *      // e.g.
 *      // Let's assume that:
 *      // the canvas coordinates are [100, 100] and that it has no scroll
 *      // the shape coordinates are [100, 100] (inside the canvas)
 *      Utils.getPointRelativeToPage(shape)     // new Point(200, 200) respect to the page
 *
 * @param {Object} shape
 * @return {PMUI.util.Point} a point relative to the page
 */
PMUI.getPointRelativeToPage = function (shape) {
    var canvas = shape.getCanvas(),
        x = shape.absoluteX + canvas.getX() - canvas.getLeftScroll() +
            shape.zoomWidth / 2,
        y = shape.absoluteY + canvas.getY() - canvas.getTopScroll() +
            shape.zoomHeight / 2;
    return new PMUI.point.Point(x, y);
};

PMUI.json2xml = function (o) {
    var len,
        toXML,
        safeXMLValue,
        a;
    
    len = function (o) {
        var n = 0,
            a;
        for (a in o) {
            if (o.hasOwnProperty(a)) {
                n += 1;    
            }
        }
        return n;
    };

    toXML = function (tag, o) {
        var a,
            b,
            doc = '<' + tag,
            sw = false,
            i;

        if (typeof o === 'undefined' || o === null) {
            doc += '/>';
            return doc;
        }
        if (typeof o !== 'object') {
            doc += '>' + safeXMLValue(o) + '</' + tag + '>';
            return doc;
        }
        if (o.constructor === Object) {
            for (a in o) {
                if (o.hasOwnProperty(a)) {
                    if (a.charAt(0) === '@') {
                        if (typeof o[a] !== 'object') {
                            doc += ' ' + a.substring(1) + '="' + o[a] + '"';
                            delete o[a];
                        } else {
                            throw new Error((typeof o[a])
                                    + ' being attribute is not supported.');
                        }
                    }
                }
            }
            if (len(o) === 0) {
                doc += '/>';
                sw = true;
            } else {
                doc += '>';
            }
            if (sw) {
                return doc;
            }
            if (typeof o['#text'] !== 'undefined') {
                if (typeof o['#text'] !== 'object') {
                    doc += o['#text'];
                    delete o['#text'];
                } else {
                    throw new Error((typeof o['#text'])
                            + ' being #text is not supported.');
                }
            }
            for (b in o) {
                if (o.hasOwnProperty(b)) {
                    if (o[b].constructor === Array) {
                        for (i = 0; i < o[b].length; i++) {
                            if (typeof o[b][i] !== 'object'
                                    || o[b][i].constructor === Object) {
                                doc += toXML(b, o[b][i]);
                            } else {
                                throw new Error((typeof o[b][i])
                                        + ' is not supported.');
                            }
                        }
                    } else if (o[b].constructor === Object
                            || typeof o[b] !== 'object') {
                        doc += toXML(b, o[b]);
                    } else {
                        throw new Error((typeof o[b]) + ' is not supported.');
                    }
                }
            }
            doc += '</' + tag + '>';
            return doc;
        }

        
    };

    safeXMLValue = function (value) {
        var s = value.toString();
        s = s.replace(/\&/g, '&amp;');
        s = s.replace(/\"/g, '&quot;');
        s = s.replace(/</g, '&lt;');
        s = s.replace(/>/g, '&gt;');
        return s;
    };

    if (typeof o === 'object' && o.constructor === Object && len(o) === 1) {
        for (a in o) {
            if (o.hasOwnProperty(a)){
                return toXML(a, o[a]);
            }
        }
    }

};

PMUI.linkToPMUIObject = function(element, pmuiObject) {
    if(this.isHTMLElement(element) && pmuiObject instanceof PMUI.core.Element) {
        jQuery(element).data('pmui', pmuiObject);
    } else {
        throw new Error("PMUI.linkToPMUIObject(): the first parameter must be a HTMLElement and the second one must be "
            + "an instance of PMUI.core.Element.");
    }
};

PMUI.getPMUIObject = function(element) {
    if(this.isHTMLElement(element)) {
        return jQuery(element).data("pmui");
    }
    throw new Error("PMUI.getPMUIObject(): the parameter must be a HTMLElement.");
};

PMUI.isHTMLElement = function(obj) {
    try {
        //Using W3 DOM2 (works for FF, Opera and Chrom)
        return obj instanceof HTMLElement;
    }
    catch(e){
        //Browsers not supporting W3 DOM2 don't have HTMLElement and
        //an exception is thrown and we end up here. Testing some
        //properties that all elements have. (works on IE7)
        return (typeof obj==="object") &&
            (obj.nodeType===1) && (typeof obj.style === "object") &&
            (typeof obj.ownerDocument ==="object");
    }
};
PMUI.removeCurrentMenu = function() {
    if (PMUI.currentContextMenu && PMUI.currentContextMenu.displayed) {
            PMUI.currentContextMenu.hide();
            PMUI.currentContextMenu = null;
    }
};
PMUI.init = function () {
    String.prototype.translate = PMUI.lang.I18N.translate;
    String.prototype.translateContext = PMUI.lang.I18N.translateContext; 
    return this;
};

PMUI.loadLanguage = function (data, lang, loaded){
    PMUI.lang.I18N.loadLanguage(data, lang, loaded);
    return this;
};

PMUI.setCurrentLanguage = function (lang){
    PMUI.lang.I18N.setCurrentLanguage(lang);
    return this;
};

PMUI.setDefaultLanguage = function (lang){
    PMUI.lang.I18N.setDefaultLanguage(lang);
    return this;
};

PMUI.convertDateFormat = function(dateInput) {
    //TODO Implement this method considering locate options
    return dateInput;
};

PMUI.castValue = function (value, format) {
    try {
        switch(format) {
            case 'string':
                out = String(value);
                break;
            case 'number':
                    if(value!=''){
                        out = Number(value);
                    } else {
                        out = 'NaN';
                    }
                break;
            case 'date':
                out = PMUI.convertDateFormat(value);
                break;
            case 'boolean':
                out = Boolean(value);
                break;
            default:
                out = value;
        }
        return out;
    } catch (e) {
        throw new Error("The value cannot be showed in '" + format + "' format");
    }
};

if (typeof exports !== "undefined") {
    module.exports = PMUI;
}

(function () {
    /**
     * @class PMUI.util.ArrayList
     * Construct a List similar to Java's ArrayList that encapsulates methods for
     * making a list that supports operations like get, insert and others.
     *
     *      some examples:
     *      var item,
     *          arrayList = new ArrayList();
     *      arrayList.getSize()                 // 0
     *      arrayList.insert({                  // insert an object
     *          id: 100,
     *          width: 100,
     *          height: 100
     *      });
     *      arrayList.getSize();                // 1
     *      arrayList.asArray();                // [{id : 100, ...}]
     *      item = arrayList.find('id', 100);   // finds the first element with an id that equals 100
     *      arrayList.remove(item);             // remove item from the arrayList
     *      arrayList.getSize();                // 0
     *      arrayList.isEmpty();                // true because the arrayList has no elements
     *
     * @constructor Returns an instance of the class ArrayList
     */
    var ArrayList = function () {
        /**
         * The elements of the arrayList
         * @property {Array}
         * @private
         */
        var elements = [],
            /**
             * The size of the array
             * @property {number} [size=0]
             * @private
             */
            size = 0,
            index,
            i;
        return {

            /**
             * The ID of this ArrayList is generated using the function Math.random
             * @property {number} id
             */
            id: Math.random(),
            /**
             * Gets an element in the specified index or undefined if the index
             * is not present in the array
             * @param {number} index
             * @returns {Object / undefined}
             */
            get : function (index) {
                return elements[index];
            },
            /**
             * Inserts an element at the end of the list
             * @param {Object} item
             * @chainable
             */
            insert : function (item) {
                elements[size] = item;
                size += 1;
                return this;
            },
            /**
             * Inserts an element in a specific position
             * @param {Object} item
             * @chainable
             */
            insertAt: function(item, index) {
                elements.splice(index, 0, item);
                size = elements.length;
                return this;
            },
            /**
             * Removes an item from the list
             * @param {Object} item
             * @return {boolean}
             */
            remove : function (item) {
                index = this.indexOf(item);
                if (index === -1) {
                    return false;
                }
                //swap(elements[index], elements[size-1]);
                size -= 1;
                elements.splice(index, 1);
                return true;
            },
            /**
             * Gets the length of the list
             * @return {number}
             */
            getSize : function () {
                return size;
            },
            /**
             * Returns true if the list is empty
             * @returns {boolean}
             */
            isEmpty : function () {
                return size === 0;
            },
            /**
             * Returns the first occurrence of an element, if the element is not
             * contained in the list then returns -1
             * @param {Object} item
             * @return {number}
             */
            indexOf : function (item) {
                for (i = 0; i < size; i += 1) {
                    if (item === elements[i]) {
                        return i;
                    }
                }
                return -1;
            },
            /**
             * Returns the the first object of the list that has the
             * specified attribute with the specified value
             * if the object is not found it returns undefined
             * @param {string} attribute
             * @param {string} value
             * @return {Object / undefined}
             */
            find : function (attribute, value) {
                var i,
                    current;
                for (i = 0; i < elements.length; i += 1) {
                    current = elements[i];
                    if (current[attribute] === value) {
                        return current;
                    }
                }
                return undefined;
            },

            /**
             * Returns true if the list contains the item and false otherwise
             * @param {Object} item
             * @return {boolean}
             */
            contains : function (item) {
                if (this.indexOf(item) !== -1) {
                    return true;
                }
                return false;
            },
            /**
             * Sorts the list using compFunction if possible, if no compFunction
             * is passed as an parameter then a default sorting method will be used. This default method will sort in 
             * ascending order.
             * @param {Function} [compFunction] The criteria function used to find out the position for the elements in
             * the array list. This function will receive two parameters, each one will be an element from the array 
             * list, the function will compare those elements and it must return:
             *
             * - 1, if the first element must be before the second element.
             * - -1, if the second element must be before the first element.
             * - 0, if the current situation doesn't met any of the two situations above. In this case both elements 
             * can be evaluated as they had the same value. For example, in an array list of numbers, when you are 
             * trying to apply a lineal sorting (ascending/descending) in a array list of numbers, if the array sorting
             * function finds two elements with the value 3 they should be evaluated returning 0, since both values are
             * the same.
             *
             * IMPORTANT NOTE: for a correct performance the sent parameter must return at least two of the values 
             * listed above, if it doesn't the function can produce an infinite loop and thus an error.
             * @return {boolean}
             */
            sort : function (compFunction) {
                var compFunction = compFunction || function(a, b) {
                        if(a < b) {
                            return 1;
                        } else if(a > b) {
                            return -1;
                        } else {
                            return 0;
                        }
                    }, swap = function (items, firstIndex, secondIndex){
                        var temp = items[firstIndex];
                        items[firstIndex] = items[secondIndex];
                        items[secondIndex] = temp;
                    }, partition = function(items, left, right) {
                        var pivot = items[Math.floor((right + left) / 2)],
                            i = left,
                            j = right;
                        while (i <= j) {
                            while (compFunction(items[i], pivot) > 0) {
                                i++;
                            }
                            while (compFunction(items[j], pivot) < 0) {
                                j--;
                            }
                            if (i <= j) {
                                swap(items, i, j);
                                i++;
                                j--;
                            }
                        }
                        return i;
                    }, quickSort = function (items, left, right) {
                        var index;
                        if (items.length > 1) {
                            index = partition(items, left, right);
                            if (left < index - 1) {
                                quickSort(items, left, index - 1);
                            }
                            if (index < right) {
                                quickSort(items, index, right);
                            }
                        }
                        return items;
                    };

                return quickSort(elements, 0, size - 1);
            },
            /**
             * Returns the list as an array
             * @return {Array}
             */
            asArray : function () {
                return elements.slice(0);
            },
            /**
             * Swaps the position of two elements
             * @chainable
             */
            swap: function(index1, index2) {
                var aux;
                if(index1 < size && index1 >=0 && index2 < size && index2 >= 0) {
                    aux = elements[index1];
                    elements[index1] = elements[index2];
                    elements[index2] = aux;
                }
                return this;
            },
            /**
             * Returns the first element of the list
             * @return {Object}
             */
            getFirst : function () {
                return elements[0];
            },
            /**
             * Returns the last element of the list
             * @return {Object}
             */
            getLast : function () {
                return elements[size - 1];
            },

            /**
             * Returns the last element of the list and deletes it from the list
             * @return {Object}
             */
            popLast : function () {
                var lastElement;
                size -= 1;
                lastElement = elements[size];
                elements.splice(size, 1);
                return lastElement;
            },
            /**
             * Returns an array with the objects that determine the minimum size
             * the container should have
             * The array values are in this order TOP, RIGHT, BOTTOM AND LEFT
             * @return {Array}
             */
            getDimensionLimit : function () {
                var result = [100000, -1, -1, 100000],
                    objects = [undefined, undefined, undefined, undefined];
                //number of pixels we want the inner shapes to be
                //apart from the border

                for (i = 0; i < size; i += 1) {
                    if (result[0] > elements[i].y) {
                        result[0] = elements[i].y;
                        objects[0] = elements[i];

                    }
                    if (result[1] < elements[i].x + elements[i].width) {
                        result[1] = elements[i].x + elements[i].width;
                        objects[1] = elements[i];
                    }
                    if (result[2] < elements[i].y + elements[i].height) {
                        result[2] = elements[i].y + elements[i].height;
                        objects[2] = elements[i];
                    }
                    if (result[3] > elements[i].x) {
                        result[3] = elements[i].x;
                        objects[3] = elements[i];
                    }
                }
                return result;
            },
            /**
             * Clears the content of the arrayList
             * @chainable
             */
            clear : function () {
                if (size !== 0) {
                    elements = [];
                    size = 0;
                }
                return this;
            },
            /**
             * Sets the elements for the object.
             * @param {Array|null} items Array with the items to set.
             * @chainable
             */
            set: function(items) {
                if(!(items === null || jQuery.isArray(items))) {
                    throw new Error("set(): The parameter must be an array or null.");
                }
                elements = (items && items.slice(0)) || [];
                size = elements.length;
                return this;
            }
        };
    };

    PMUI.extendNamespace('PMUI.util.ArrayList', ArrayList);

    if (typeof exports !== "undefined") {
        module.exports = ArrayList;
    }

}());

(function () {
    /**
     * @class PMUI.util.Style
     * Class that represent the style of a an object.
     *
     *      // i.e
     *      // Let's assume that 'shape' is a CustomShape
     *      var style = new Style({
     *          cssClasses: [
     *              'sprite-class', 'marker-class', ...
     *          ],
     *          cssProperties: {
     *              border: 1px solid black,
     *              background-color: grey,
     *              ...
     *          },
     *          belongsTo: shape
     *      })
     *
     * @constructor Creates a new instance of this class
     * @param {Object} options
     * @cfg {Array} [cssClasses=[]] the classes that `this.belongsTo` has
     * @cfg {Object} [cssProperties={}] the css properties that `this.belongsTo` has
     * @cfg {Object} [belongsTo=null] a pointer to the owner of this instance
     */
    var Style = function (options) {

        /**
         * JSON Object used to map each of the css properties of the object,
         * this object has the same syntax as the object passed to jQuery.css()
         *      cssProperties: {
         *          background-color: [value],
         *          border: [value],
         *          ...
         *      }
         * @property {Object}
         */
        this.cssProperties = null;

        /**
         * Array of all the classes of this object
         *      cssClasses = [
         *          'class_1',
         *          'class_2',
         *          ...
         *      ]
         * @property {Array}
         */
        this.cssClasses = null;

        /**
         * Pointer to the object to whom this style belongs to
         * @property {Object}
         */
        this.belongsTo = null;


        Style.prototype.initObject.call(this, options);
    };


    /**
     * The type of this class
     * @property {String}
     */
    Style.prototype.type = "Style";

    /**
     * Constant for the max z-index
     * @property {number} [MAX_ZINDEX=100]
     */
    Style.MAX_ZINDEX = 100;

    /**
     * Instance initializer which uses options to extend the config options to
     * initialize the instance
     * @private
     * @param {Object} options
     */
    Style.prototype.initObject = function (options) {
        var defaults = {cssClasses: [],cssProperties: {}, belongsTo: null};
        jQuery.extend(true, defaults, options);
        this.cssClasses = defaults.cssClasses;
        this.cssProperties = defaults.cssProperties;
        this.belongsTo = defaults.belongsTo;
    };

    /**
     * Applies cssProperties and cssClasses to `this.belongsTo`
     * @chainable
     */
    Style.prototype.applyStyle = function () {

        if (!this.belongsTo.html) {
            throw new Error("applyStyle(): can't apply style to an" +
                " object with no html.");
        }

        var i, t, class_i;

        // apply the cssProperties
        jQuery(this.belongsTo.html).css(this.cssProperties);

        //adding default classes
        t = this.belongsTo.type.toLowerCase();
        if (this.cssClasses.indexOf('pmui-' + t) === -1) {
            this.cssClasses.unshift('pmui-' + t);
        }
        if (this.cssClasses.indexOf('pmui') === -1) {
            this.cssClasses.unshift('pmui');
        }

        // apply saved classes
        for (i = 0; i < this.cssClasses.length; i += 1) {
            class_i = this.cssClasses[i];
            if (!$(this.belongsTo.html).hasClass(class_i)) {
                jQuery(this.belongsTo.html).addClass(class_i);
            }
        }
        return this;
    };
    Style.prototype.unapplyStyle = function() {
        var i, t, property;
        if(!this.belongsTo.html) {
            throw new Error("unapplyStyle(): can't unapply style to an object with no html.");
        }
        t = this.belongsTo.type.toLowerCase();
        jQuery(this.belongsTo.html).removeClass("pmui-" + t);
        for(property in this.cssProperties) {
            jQuery(this.belongsTo.html).css(property, "");
        }
    };

    /**
     * Extends the property `cssProperties` with a new object and also applies those new properties
     * @param {Object} properties
     * @chainable
     */
    Style.prototype.addProperties = function (properties) {
        jQuery.extend(true, this.cssProperties, properties);
        jQuery(this.belongsTo && this.belongsTo.html).css(properties);
        return this;
    };

    /**
     * Gets a property from `this.cssProperties` using jQuery or `window.getComputedStyle()`
     * @param {String} property
     * @return {String}
     */
    Style.prototype.getProperty = function (property) {
        return this.cssProperties[property] ||
            jQuery(this.belongsTo.html).css(property) ||
                (this.belongsTo.html && window.getComputedStyle(this.belongsTo.html, null)
                .getPropertyValue(property)) || "";
    };
    /**
     * Returns all the style's css properties set explicitly.
     * @return {Object} An object literal with the properties.
     */
    Style.prototype.getProperties = function() {
        return this.cssProperties;
    };
    /**
     * Removes ´properties´ from the ´this.cssProperties´, also disables those properties from
     * the HTMLElement
     * @param {Array} properties An array in which each element is th name of the cssProperty to be removed.
     * @chainable
     */
    Style.prototype.removeProperties = function (properties) {
        var property,
            i;
        for (i = 0; i < properties.length; i += 1) {
            property = properties[i];
            if (this.cssProperties.hasOwnProperty(property)) { // JS Code Convention
                jQuery(this.belongsTo.html).css(property, "");   // reset inline style
                delete this.cssProperties[property];
            }
        }
        return this;
    };
    /**
     * Removes all properties from the object.
     * @chainable
     */
    Style.prototype.removeAllProperties = function() {
        var key;
        if(this.belongsTo) {
            for(key in this.cssProperties) {
                jQuery(this.belongsTo.html).css(key, "");
            }   
        }
        this.cssProperties = {};
        return this;
    };
    /**
     * Adds new classes to ´this.cssClasses´ array
     * @param {Array} cssClasses
     * @chainable
     */
    Style.prototype.addClasses = function (cssClasses) {
        var i,
            cssClass;
        if (cssClasses && cssClasses instanceof Array) {
            for (i = 0; i < cssClasses.length; i += 1) {
                cssClass = cssClasses[i];
                if (typeof cssClass === "string") {
                    if (this.cssClasses.indexOf(cssClass) === -1) {
                        this.cssClasses.push(cssClass);
                        jQuery(this.belongsTo && this.belongsTo.html).addClass(cssClass);
                    }
                } else {
                    throw new Error("addClasses(): array element is not of type string");
                }
            }
        } else {
            throw new Error("addClasses(): parameter must be of type Array");
        }
        return this;
    };

    /**
     * Removes classes from ´this.cssClasses´ array, also removes those classes from
     * the HTMLElement
     * @param {Array} cssClasses
     * @chainable
     */
    Style.prototype.removeClasses = function (cssClasses) {

        var i,
            index,
            cssClass;
        if (cssClasses && cssClasses instanceof Array) {
            for (i = 0; i < cssClasses.length; i += 1) {
                cssClass = cssClasses[i];
                if (typeof cssClass === "string") {
                    index = this.cssClasses.indexOf(cssClass);
                    if (index !== -1) {
                        jQuery(this.belongsTo.html).removeClass(this.cssClasses[index]);
                        this.cssClasses.splice(index, 1);
                    }
                } else {
                    throw new Error("removeClasses(): array element is not of " +
                        "type string");
                }
            }
        } else {
            throw new Error("removeClasses(): parameter must be of type Array");
        }
        return this;
    };
    /**
     * Removes all the classes from ´this.cssClasses´ array
     * @param {Array} cssClasses
     * @chainable
     */
    Style.prototype.removeAllClasses = function () {
        while(this.cssClasses.length) {
            jQuery(this.belongsTo && this.belongsTo.html).removeClass(this.cssClasses.pop());   
        }
        return this;
    };

    /**
     * Checks if the class is a class stored in ´this.cssClasses´
     * @param cssClass
     * @return {boolean}
     */
    Style.prototype.containsClass = function (cssClass) {
        return this.cssClasses.indexOf(cssClass) !== -1;
    };

    /**
     * Returns an array with all the classes of ´this.belongsTo´
     * @return {Array}
     */
    Style.prototype.getClasses = function () {
        return this.cssClasses.slice(0);
    };

    /**
     * Clears all the css properties and classes.
     * @chainable
     */
    Style.prototype.clear = function() {
        return this.removeAllClasses().removeAllProperties();
    }

    /**
     * Serializes this instance
     * @return {Object}
     * @return {Array} return.cssClasses
     */
    Style.prototype.stringify = function () {
        return {
            cssClasses: this.cssClasses
    //        cssProperties: this.cssProperties
        };
    };

    // Declarations created to instantiate in NodeJS environment
    if (typeof exports !== "undefined") {
        module.exports = Style;
    }

    PMUI.extendNamespace('PMUI.util.Style', Style);

}());
(function () {
    /**
     * @class PMUI.util.Factory
     * This class encapsulate the way to construct object using the product definition inside
     *
     * @constructor
     * This method creates a new instance of this object
     * @param {Object} settings Constructor setiings
     */
    var Factory = function (settings) {
        /**
         * Defines the products can be make by the factory
         * @type {Object}
         */
        this.products = null;
        
        /**
         * Defines the default product to make
         * @type {String}
         */
        this.defaultProduct = null;
        Factory.prototype.init.call(this, settings);
    };

    /**
     * Defines the object type
     * @type {String}
     */
    Factory.prototype.type = "Factory";

    /**
     * Defines the object family
     * @type {String}
     */
    Factory.prototype.family = "Factory";
    /**
     * Initializes the object with the default values
     * @param  {Object} options Contructor options
     */
    Factory.prototype.init = function (options) {
        var defaults;
        if (!options) {
            options = {};
        }
        defaults = {
            defaultProduct: options.defaultProduct || "element",
            products: options.products || {"element": PMUI.core.Element}
        };
        this.setDefaultProduct(defaults.defaultProduct)
            .setProducts(defaults.products);
    };

    /**
     * Sets the default product property
     * @param {String} def Default value
     */
    Factory.prototype.setDefaultProduct = function (def) {
        this.defaultProduct = def;
        return this;
    };

    /**
     * Sets the product object
     * @param {Object} products Products object
     */
    Factory.prototype.setProducts = function (products) {
        this.products = products;
        return this;
    };
    /**
     * Removes a product from the factory.
     * @param  {String|Object} product The product to be removed, it can be:
     * 
     *  - a string: the product referenced with that pmType will be removed from factory.
     *  - a object: the constructor, all the products with that constructor will be removed.
     * @return {[type]}         [description]
     */
    Factory.prototype.removeProduct = function(product) {
        var key, products = this.products;
        if(typeof product === 'string') {
            delete products[product];
        } else {
            for(key in products) {
                if(products[key] === product) {
                    delete products[key];
                }
            }
        }
        return this;
    };
    /**
     * Removes all the productrs from the factory.
     * @chainable
     */
    Factory.prototype.clearProducts = function() {
        var key;
        for(key in this.products) {
            this.removeProduct(key);
        }
        return this;
    };
    /**
     * Register a new product into the products object
     * @param  {String} name  Product type
     * @param  {Object} classObj Product Class 
     * @chainable
     */
    Factory.prototype.register = function (name, classObj) {
        var aux = this.products || {};
        aux[name] = classObj;
        this.products = aux;
        return this;
    };  

    /**
     * Retuns a new instance (product)
     * @param  {String} type    Product type
     * @param  {Object} options Settings object
     * @return {Object}         Instance of the object
     */
    Factory.prototype.build = function (type, options) {
        var Constructor, 
            instance;
        if (this.isValidName(type)){
            Constructor = this.products[type];
            instance = new Constructor(options);
        } else {
            throw new Error('The type "' + type + '" has not valid constructor or is undefined.');
        }
        return instance;
    };

    /**
     * Retuns true if the type is valid into the products object
     * @param  {String}  name Product Name
     * @return {Boolean}   
     */
    Factory.prototype.isValidName = function (name) {
        var test = this.products[name];
        return !!test;
    };

    /**
     * Returns true if the class input is instance of one class into the product object
     * @param  {Object}  className [description]
     * @return {Boolean}           [description]
     */
    Factory.prototype.isValidClass = function (className) {
        var valid = false;
        jQuery.each(this.products, function(type, classProduct){
            if (className instanceof classProduct){
                valid = true;
            }
        });
        return valid;
    };

    
    /**
     * Comprobes the obj in and returns the instance of the object
     * @param  {Object} obj Input can be an isntance or an object with the pmType property or simply a JSON
     * @return {Object}     Returns an instance made from the products object.
     */
    Factory.prototype.make = function (obj) {
        var product,
            productType = obj.pmType || '';
        if (this.isValidClass(obj)){
            product = obj;
        } else if (this.isValidName(productType)) {
            product = this.build.call(this, productType, obj);
        } else {
            product = this.build.call(this, this.defaultProduct, obj);
        }
        return product;
    };

    //Create a namespace for Factory class
    PMUI.extendNamespace('PMUI.util.Factory', Factory);

    // Publish to NodeJS environment
    if (typeof exports !== 'undefined'){
        module.exports = Factory;
    }

}());
(function () {
    /**
     * @class PMUI.util.Color
     * This class holds the representation and operations of RGBa representation of color,
     * it's very useful if we want to save color constants as an instance and later get the representation
     * in CSS.
     *
     *      //e.g.
     *      var color = new PMUI.util.Color(
     *          128,    // red
     *          128,    // green
     *          128,    // blue
     *          1       // opacity
     *      )
     *
     * @constructor Creates an instance of this class.
     * @param {number} red
     * @param {number} green
     * @param {number} blue
     * @param {number} opacity
     * @return {PMUI.util.Color}
     */
    var Color = function (red, green, blue, opacity) {
        /**
         * Red value of the RGB Color
         * @property {number} [red=0]
         */
        this.red = (!red) ? 0 : red;
        /**
         * Green value of the RGB Color
         * @property {number} [green=0]
         */
        this.green = (!green) ? 0 : green;
        /**
         * Blue value of the RGB Color
         * @property {number} [blue=0]
         */
        this.blue = (!blue) ? 0 : blue;
        /**
         * Opacity of the RGB Color
         * @property {number} [opacity=1]
         */
        this.opacity = (!opacity) ? 1 : opacity;
    };

    /**
     * Type of this class
     * @property {String}
     */
    Color.prototype.type = "Color";

    /**
     * Constant for the color grey
     * @property {PMUI.util.Color} [GREY=new Color(192, 192, 192, 1)]
     */
    Color.GREY = new Color(192, 192, 192, 1);

    /**
     * Returns the red value of the RGB Color
     * @returns {number}
     */
    Color.prototype.getRed = function () {
        return this.red;
    };

    /**
     * Returns the green value of the RGB Color
     * @returns {number}
     */
    Color.prototype.getGreen = function () {
        return this.green;
    };

    /**
     * Returns the blue value of the RGB Color
     * @returns {number}
     */
    Color.prototype.getBlue = function () {
        return this.blue;
    };

    /**
     * Returns the opacity of the RGB Color
     * @returns {number}
     */
    Color.prototype.getOpacity = function () {
        return this.opacity;
    };

    /**
     * Sets the red value of the RGB Color
     * @param {number} newRed
     * @chainable
     */
    Color.prototype.setRed = function (newRed) {
        if (typeof newRed === "number" && newRed >= 0 && newRed <= 255) {
            this.red = newRed;
        }
        return this;
    };

    /**
     * Sets the green value of the RGB Color
     * @param {number} newRed
     * @chainable
     */
    Color.prototype.setGreen = function (newGreen) {
        if (typeof newGreen === "number" && newGreen >= 0 && newGreen <= 255) {
            this.green = newGreen;
        }
        return this;
    };

    /**
     * Sets the blue value of the RGB Color
     * @param {number} newBlue
     * @chainable
     */
    Color.prototype.setBlue = function (newBlue) {
        if (typeof newBlue === "number" && newBlue >= 0 && newBlue <= 255) {
            this.blue = newBlue;
        }
        return this;
    };

    /**
     * Sets the opacity of the RGB Color
     * @param {number} newOpacity
     * @chainable
     */
    Color.prototype.setOpacity = function (newOpacity) {
        if (typeof newOpacity === "number" && newOpacity >= 0 && newOpacity <= 255) {
            this.opacity = newOpacity;
        }
        return this;
    };

    /**
     * Returns the css representation of the RGB color
     *      //e.g.
     *      var color = new PMUI.util.Color(10, 20, 30, 0.1);
     *      color.getCSS();         // "rgba(10, 20, 30, 0.1)"
     * @returns {String}
     */
    Color.prototype.getCSS = function () {
        var css = "rgba(" + this.red + "," + this.green + "," + this.blue +
            "," + this.opacity + ")";
        return css;
    };

    PMUI.extendNamespace('PMUI.util.Color', Color);
}());

(function () {
    /**
     * @class PMUI.util.Point
     * Class to represent points in the PMDraw library
     *
     *        // e.g.
     *        var p = new PMUI.util.Point(100, 100);
     *
     * @constructor Creates an instance of this class
     * @param {number} xCoordinate x-coordinate of the point
     * @param {number} yCoordinate y-coordinate of the point
     * @return {PMUI.util.Point}
     */
    var Point = function (xCoordinate, yCoordinate) {
        /**
         * x coordinate of the point in the plane
         */
        this.x = xCoordinate;
        /**
         * y coordinate of the point in the plane
         */
        this.y = yCoordinate;
    };

    /**
     * Type of this class
     * @property {String}
     */
    Point.prototype.type = "Point";

    /**
     * Returns the X coordinate
     * @property {number}
     **/
    Point.prototype.getX = function () {
        return this.x;
    };

    /**
     * Returns the Y coordinate
     * @property {number}
     **/
    Point.prototype.getY = function () {
        return this.y;
    };

    /**
     * Adds `other` point to `this` point and returns a new point with those coordinates.
     *
     *      // e.g.
     *      var p1 = new PMUI.util.Point(3, 5),
     *          p2 = new PMUI.util.Point(2, 3);
     *      p1.add(p2);     // new Point(5, 8)
     *
     * @param {PMUI.util.Point} other Point to be added to the current point
     * @returns {PMUI.util.Point}
     */
    Point.prototype.add = function (other) {
        return new Point(this.x + other.x, this.y + other.y);
    };

    /**
     * Subtracts the other point to the one that called the function.
     *
     *      // e.g.
     *      var p1 = new PMUI.util.Point(3, 5),
     *          p2 = new PMUI.util.Point(2, 3);
     *      p1.subtract(p2);     // new Point(1, 2)
     *
     * @param {PMUI.util.Point} other Point to be added to the current point
     * @returns {PMUI.util.Point}
     */
    Point.prototype.subtract = function (other) {
        return new Point(this.x - other.x, this.y - other.y);
    };

    /**
     * Multiplies the point with a scalar k.
     *
     *      // e.g.
     *      var p1 = new PMUI.util.Point(3, 5),
     *          k = 3;
     *      p1.multiply(k);     // new Point(9, 15)
     *
     * @param {number} k
     * @return {PMUI.util.Point}
     */
    Point.prototype.multiply = function (k) {
        return new Point(this.x * k, this.y * k);
    };

    /**
     * Determine if the points are equal.
     *
     *      // e.g.
     *      var p1 = new PMUI.util.Point(3, 5),
     *          p2 = new PMUI.util.Point(2, 3),
     *          p3 = new PMUI.util.Point(3, 5);
     *      p1.equals(p2);     // false
     *      p1.equals(p3);     // true
     *      p1.equals(p1);     // true
     *
     * @param {PMUI.util.Point} other Point to be compared with the current point
     * @returns {boolean}
     */
    Point.prototype.equals = function (other) {
        return (Math.abs(this.x - other.x) < PMUI.draw.Geometry.eps) &&
                (Math.abs(this.y - other.y) < PMUI.draw.Geometry.eps);
    };

    /**
     * Determine the distance between two Points
     *
     *      // e.g.
     *      // distance = sqrt(pow(x1 - x2, 2) + pow(y1 - y2, 2))
     *      var p1 = new PMUI.util.Point(3, 5),
     *          p2 = new PMUI.util.Point(2, 3);
     *      p1.getDistance(p2);         // sqrt(1 + 4)
     *
     * @param {PMUI.util.Point} other Point to be calculated from current point
     * @returns {number}
     **/
    Point.prototype.getDistance = function (other) {
        return Math.sqrt(
            (this.x - other.x) * (this.x - other.x) +
                (this.y - other.y) * (this.y - other.y)
        );
    };

    /**
     * Determine the squared distance between two Points
     *
     *      // e.g.
     *      // distance = sqrt(pow(x1 - x2, 2) + pow(y1 - y2, 2))
     *      // but since it's the squared distance then
     *      // distance = pow(distance, 2)
     *      var p1 = new PMUI.util.Point(3, 5),
     *          p2 = new PMUI.util.Point(2, 3);
     *      p1.getSquaredDistance(p2);         // (1 + 4)
     *
     * @param {PMUI.util.Point} other Point to be calculated from current point
     * @returns {number}
     **/
    Point.prototype.getSquaredDistance = function (other) {
        return (this.x - other.x) * (this.x - other.x) +
            (this.y - other.y) * (this.y - other.y);
    };

    /**
     * Determine the manhattan distance between two Points
     *
     *      // e.g.
     *      var p1 = new PMUI.util.Point(3, 5),
     *          p2 = new PMUI.util.Point(2, 3);
     *      p1.getManhattanDistance(p2);         // (1 + 2)
     *
     * @param {PMUI.util.Point} other Point to be calculated from current point
     * @returns {number}
     **/
    Point.prototype.getManhattanDistance = function (other) {
        return Math.abs(this.x - other.x) + Math.abs(this.y - other.y);
    };

    /**
     * Makes a clone of this
     *
     *      // e.g.
     *      var p1 = new PMUI.util.Point(3, 5),
     *          cloneP1;
     *      cloneP1 = p1.clone();       // cloneP1 is Point(3, 5)
     *
     * @returns {PMUI.util.Point} This point
     */
    Point.prototype.clone = function () {
        return new Point(this.x, this.y);
    };

    // Declarations created to instantiate in NodeJS environment
    if (typeof exports !== 'undefined') {
        module.exports = Point;
    }

    // extend namespace
    PMUI.extendNamespace('PMUI.util.Point', Point);

}());

(function () {
    /**
     * @class PMUI.draw.Graphics
     * Class Graphics is the HTMLElement drawing engine used to draw lines (as HTMLElement divs),
     * arcs (using HTMLElement divs) and ovals.
     * Currently some elements in the designer are completely represented with the drawing engine:
     *
     * - Connection => a set of segments (regular, segmented, dotted)
     * - Port => oval
     * - Intersection => arc
     *
     * Some important notes:
     *
     * - Currently this class acts as an interface between the library **PMDraw** and the HTMLElement
     * drawing engine wz_jsGraphics
     * - The drawing engine constructor needs the HTMLElement where it will go as a parameter **(this
     * HTMLElement must exist in the DOM)**
     *
     * @constructor
     * Creates an instance of this class (currently it's an interface of the wz_graphics framework)
     * @param {Object} html This parameter can be either an id, or an html object
     */
    var Graphics = function (html) {

        if (!html) {
            return null;
        }

        /**
         * Create an instance of the class JSGraphics (this.graphics is an interface)
         * @type {Object}
         */
        this.graphics = new JSGraphics(html);

        /**
         * Creates an instance of the class Color (color black)
         * @type {PMUI.util.Color}
         */
        this.color = new PMUI.util.Color(0, 0, 0);
    };

    /**
     * Draws a line of a given type between two points.
     *
     * @param {number} x1
     * @param {number} y1
     * @param {number} x2
     * @param {number} y2
     * @param {string} type the type of line we wish to draw
     * @param {PMUI.util.Color} color The color of the segment
     * @param {number} segLength the segment length for segmented and segmentdot type of line
     * @param {number} spaceLength the space length for segmented and segmentdot type of line
     * @param {boolean} [doNotErasePrevious] if set to true then it won't clear the elements drew with this instance
     */
    Graphics.prototype.drawLine = function (x1, y1, x2, y2, type, color, segLength, spaceLength, doNotErasePrevious) {

        if (!doNotErasePrevious) {
            this.graphics.clear();
        }

        if (!type) {
            type = "regular";
        }
        switch (type) {
            case "dotted":
                this.graphics.setStroke(-1);
                break;
            case "segmented":
                this.graphics.setStroke(1);
                this.graphics.drawLine = this.makeSegmentedLine;
                break;
            case "segmentdot":
                this.graphics.setStroke(1);
                this.graphics.drawLine = this.makeSegmentDotLine;
                break;
            default:
                this.graphics.setStroke(1);
        }

        this.graphics.setColor(color.getCSS());
        this.graphics.drawLine(x1, y1, x2, y2, segLength, spaceLength);
        this.graphics.paint();

    };

///**
// * NOTE: Unused definition
// * Initializes the graphics variable with a new html container
// * @param {HTMLElement} html
// * @returns {Graphics}
// */
//Graphics.prototype.initGraphics = function (html) {
//    if (html) {
//        this.graphics = new PMUI.draw.JSGraphics(html);
//    }
//    return this;
//};

    /**
     * Returns the color that was being used for drawing
     * @returns {PMUI.util.Color}
     */
    Graphics.prototype.getColor = function () {
        return this.color;
    };

    /**
     * Sets the color to be used for drawing
     * @param newColor
     * @chainable
     */
    Graphics.prototype.setColor = function (newColor) {
        if (newColor.type === "Color") {
            this.color = newColor;
        }
        return this;
    };

    /**
     * This function will make a segmented line between two points in the same axis,
     * if points in different axis are provided the method would simple return
     *
     * @param {number} x1
     * @param {number} y1
     * @param {number} x2
     * @param {number} y2
     * @param {number} segmentLength the segment length for segmented and
     * segmentdot type of line
     * @param {number} spLength the space length for segmented and
     * segmentdot type of line
     */
    Graphics.prototype.makeSegmentedLine = function (x1, y1, x2, y2, segmentLength, spLength) {
        var dx,
            dy,
            aux,
            segLength = 4,
            spaceLength = 3,
            diff = 0,
            x,
            y;

        //not same axis so just return
        if ((x2 !== x1 && y2 !== y1)) {
            return;
        }

        if (x2 === x1) {
            //same point just return
            if (y2 === y1) {
                return;
            }
            dx = 0;
            //swap
            if (y2 < y1) {
                aux = y2;
                y2 = y1;
                y1 = aux;
            }
            dy = diff = y2 - y1;
        } else {
            dy = 0;
            if (x2 < x1) {
                aux = x2;
                x2 = x1;
                x1 = aux;
            }
            dx = diff = x2 - x1;
        }

        x = x1;
        y = y1;

        if (diff < 7) {
            segLength = 2;
            spaceLength = 1;
        }

        segLength = (!segmentLength) ? segLength : segmentLength;
        spaceLength = (!spLength) ? spaceLength : spLength;

        if (dy === 0) {
            while (dx > 0) {
                if (dx >= segLength) {
                    this._mkDiv(x, y, segLength, 1);
                    x += segLength + spaceLength;
                    dx -= (segLength + spaceLength);
                } else {
                    this._mkDiv(x, y, dx, 1);
                    dx = 0;
                }
            }
        } else {
            while (dy > 0) {
                if (dy >= segLength) {
                    this._mkDiv(x, y, 1, segLength);
                    y += segLength + spaceLength;
                    dy -= (segLength + spaceLength);
                } else {
                    this._mkDiv(x, y, 1, dy);
                    dy = 0;
                }
            }
        }
    };

    /**
     * This function will make a segment between two points in the same axis with
     * the following structure segment-dot-segment
     * if points in different axis are provided the function will simply return.
     *
     * @param {number} x1
     * @param {number} y1
     * @param {number} x2
     * @param {number} y2
     * @param {number} segmentLength the segment length for segmented and
     * segmentdot type of line
     * @param {number} spLength the space length for segmented and
     * segmentdot type of line
     */

    Graphics.prototype.makeSegmentDotLine = function (x1, y1, x2, y2, segmentLength, spLength) {
        var dx,
            dy,
            aux,
            segLength = 7,
            spaceLength = 4,
        //in case its necessary dot Length
            dotLength = 1,
            diff = 0,
            x,
            y;

        //not same axis so just return
        if ((x2 !== x1 && y2 !== y1)) {
            return;
        }


        if (x2 === x1) {
            //same point just return
            if (y2 === y1) {
                return;
            }
            dx = 0;
            //swap
            if (y2 < y1) {
                aux = y2;
                y2 = y1;
                y1 = aux;
            }
            dy = y2 - y1;
            diff = dy;
        } else {
            dy = 0;
            if (x2 < x1) {
                aux = x2;
                x2 = x1;
                x1 = aux;
            }
            dx = x2 - x1;
            diff = dx;
        }

        x = x1;
        y = y1;

        segLength = (!segmentLength) ? segLength : segmentLength;
        spaceLength = (!spLength) ? spaceLength : spLength;

        if (dy === 0) {
            while (dx > 0) {
                if (dx >= segLength) {
                    this._mkDiv(x, y, segLength, 1);
                    dx -= (segLength + spaceLength);
                    x += segLength + spaceLength;
                    if (dx > 0) {
                        //we need to implement this if the dot length would be
                        // different than one
                        //if(dx < dotLength){
                        //  this._mkDiv(x,y,dx,1);
                        //  dx  = 0; continue;
                        //}
                        this._mkDiv(x, y, dotLength, 1);
                        dx -= (dotLength + spaceLength);
                        x += dotLength + spaceLength;

                    }
                } else {
                    this._mkDiv(x, y, dx, 1);
                    dx = 0;
                }
            }
        } else {
            while (dy > 0) {
                if (dy >= segLength) {
                    this._mkDiv(x, y, 1, segLength);
                    dy -= (segLength + spaceLength);
                    y += segLength + spaceLength;
                    if (dy > 0) {
                        //we need to implement this if the dot length would be
                        // different than one
                        //if(dy < dotLength){
                        //  this._mkDiv(x,y,1,dy);
                        //  dy  = 0; continue;
                        //}
                        this._mkDiv(x, y, 1, dotLength);
                        dy -= (dotLength + spaceLength);
                        y += dotLength + spaceLength;

                    }
                } else {
                    this._mkDiv(x, y, 1, dy);
                    dy = 0;
                }
            }
        }

    };
    /**
     * Draws an arc with the center `[cx, cy]`, with a radius equal to `radius` from `startAngle` to `endAngle`
     * and drawing a line every `step` steps.
     * Logic:
     *
     * 1. Let's assume that we have a circle with center `[cx, cy]` and with a radius equal to `radius`
     * 2. We want to draw only a portion of the circle (from `startAngle` to `endAngle`)
     * 3. Given any angle of the circle `0 <= angle < 360` we can get its `x` and `y` coordinates using
     *    Pythagoras triangle rectangle laws.
     *      - We know that `hyp^2 = dx^2 + dy^2` and that `hyp = radius`
     *      - We know that `cos(angle) = dx / radius` so `dx = radius * cos(angle)`
     *      - We know that `sin(angle) = dy / radius` so `dx = radius * cos(angle)`
     * 4. Finally let's use the given center of the circle to move the triangle
     *
     * @param {number} cx
     * @param {number} cy
     * @param {number} radius
     * @param {number} startAngle
     * @param {number} endAngle
     * @param {number} step
     */
    Graphics.prototype.drawArc = function (cx, cy, radius, startAngle, endAngle, step) {
        var x,
            y,
            angle = startAngle;

        if (!step) {
            step = 10;
        }

        while (Math.abs(angle - endAngle) > 1e-5) {
            angle = (angle + step) % 360;
            x = cx + radius * Math.cos(angle * PMUI.draw.Geometry.pi / 180.0);
            y = cy + radius * Math.sin(angle * PMUI.draw.Geometry.pi / 180.0);
            this.graphics.drawLine(x, y, x, y);
        }
    };

    PMUI.extendNamespace('PMUI.draw.Graphics', Graphics);
}());

(function () {
    /**
     * @class PMUI.draw.Geometry
     * A little object that encapsulates most geometry functions used in the designer, most of the examples
     * are in 'spec/draw/geometry.spec.js'
     *
     * @singleton
     */
    var Geometry = {
        /**
         * The number pi
         * @property {number} [pi=Math.acos(-1)]
         */
        pi : Math.acos(-1),
        /**
         * Epsilon used for the correctness in comparison of float numbers
         * @property {number} [eps=1e-8]
         */
        eps : 1e-8,
        /**
         * Calculates the cross product of 2-dimensional vectors
         * @param {PMUI.util.Point} p1
         * @param {PMUI.util.Point} p2
         * @return {number}
         */
        cross : function (p1, p2) {
            return p1.x * p2.y - p1.y * p2.x;
        },
        /**
         * Calculates the SIGNED area of a parallelogram given three points, these three points are the points
         * that conforms the triangle that is half of the parallelogram, so. the area of the triangle
         * defined with these points is half the returned number (this method can return negative values)
         *
         *      // e.g.
         *      var p1 = new PMUI.util.Point(0, 0),
         *          p2 = new PMUI.util.Point(0, 1),
         *          p3 = new PMUI.util.Point(1, 0),
         *          parallelogramArea,
         *          triangleArea;
         *
         *      parallelogramArea = Geometry.area(p1, p2, p3)   // -1 (area of the parallelogram)
         *      triangleArea = parallelogramArea / 2            // -0.5 (area of the triangle)
         *
         * @param {PMUI.util.Point} p1
         * @param {PMUI.util.Point} p2
         * @param {PMUI.util.Point} p3
         * @return {number}
         */
        area : function (p1, p2, p3) {
            var auxP2 = p2.clone(),
                auxP3 = p3.clone();
            return this.cross(auxP2.subtract(p1), auxP3.subtract(p1));
        },
        /**
         * Determines if the point P is on segment AB
         * @param {PMUI.util.Point} P
         * @param {PMUI.util.Point} A
         * @param {PMUI.util.Point} B
         * @return {boolean}
         */
        onSegment : function (P, A, B) {
            return (Math.abs(this.area(A, B, P)) < this.eps &&
                P.x >= Math.min(A.x, B.x) && P.x <= Math.max(A.x, B.x) &&
                P.y >= Math.min(A.y, B.y) && P.y <= Math.max(A.y, B.y));
        },
        /**
         * Checks if two perpendicular segments intersect, if so it returns the intersection point,
         * (this method only allows the perpendicular segment to be parallel to the x and y axis)
         * @param {PMUI.util.Point} A
         * @param {PMUI.util.Point} B
         * @param {PMUI.util.Point} C
         * @param {PMUI.util.Point} D
         * @return {Object}
         */
        perpendicularSegmentIntersection : function (A, B, C, D) {
            var clone,
                returnValue = null;

            // swap the segments if possible
            if (A.x > B.x || A.y > B.y) {
                clone = A.clone();
                A = B.clone();
                B = clone;
            }

            if (C.x > D.x || C.y > D.y) {
                clone = C.clone();
                C = D.clone();
                D = clone;
            }

            if (A.x === B.x) {
                if (C.y === D.y && C.x < A.x && A.x < D.x &&
                            A.y < C.y && C.y < B.y) {
                    returnValue = new PMUI.util.Point(A.x, C.y);
                }
            } else if (A.y === B.y) {
                if (C.x === D.x && A.x < C.x && C.x < B.x &&
                        C.y < A.y && A.y < D.y) {
                    returnValue = new PMUI.util.Point(C.x, A.y);
                }
            }
            return returnValue;
        },
        /**
         * Determines if segment AB intersects with segment CD (won't check infinite intersections),
         * if `strict` is set to `true` then it'll consider the case when one end of a segment is right in the
         * other segment
         * @param {PMUI.util.Point} A
         * @param {PMUI.util.Point} B
         * @param {PMUI.util.Point} C
         * @param {PMUI.util.Point} D
         * @param {boolean} [strict]
         * @return {boolean}
         */
        segmentIntersection : function (A, B, C, D, strict) {

            var area1 = this.area(C, D, A),
                area2 = this.area(C, D, B),
                area3 = this.area(A, B, C),
                area4 = this.area(A, B, D),
                returnValue;
            if (((area1 > 0 && area2 < 0) || (area1 < 0 && area2 > 0)) &&
                    ((area3 > 0 && area4 < 0) || (area3 < 0 && area4 > 0))) {
                return true;
            }

            returnValue = false;
            if (strict) {
                if (area1 === 0 && this.onSegment(A, C, D)) {
                    returnValue = true;
                } else if (area2 === 0 && this.onSegment(B, C, D)) {
                    returnValue = true;
                } else if (area3 === 0 && this.onSegment(C, A, B)) {
                    returnValue = true;
                } else if (area4 === 0 && this.onSegment(D, A, B)) {
                    returnValue = true;
                }
            }
            return returnValue;
        },
        /**
         * Checks if two segments intersect, if so it returns the intersection point
         * @param {PMUI.util.Point} A
         * @param {PMUI.util.Point} B
         * @param {PMUI.util.Point} C
         * @param {PMUI.util.Point} D
         * @return {PMUI.util.Point}
         */
        segmentIntersectionPoint: function (A, B, C, D) {
            return A.add((B.subtract(A))
                .multiply(this.cross(C.subtract(A), D.subtract(A)) /
                    this.cross(B.subtract(A), D.subtract(C))));
        },
        /**
         * Determines whether a point is in a given rectangle or not given its
         * upperLeft and bottomRight corner (consider that a rectangle is turned in the y-axis)
         * @param {PMUI.util.Point} point
         * @param {PMUI.util.Point} upperLeft
         * @param {PMUI.util.Point} bottomRight
         * @return {boolean}
         */
        pointInRectangle : function (point, upperLeft, bottomRight) {
            return (point.x >= upperLeft.x && point.x <= bottomRight.x &&
                point.y >= upperLeft.y && point.y <= bottomRight.y);
        },
        /**
         * Determines whether a point is in a circle or not given its center and
         * radius
         * @param {PMUI.util.Point} point
         * @param {PMUI.util.Point} center
         * @param {number} radius
         * @returns {boolean}
         */
        pointInCircle : function (point, center, radius) {
            return center.getDistance(point) <= radius;
        },
        /**
         * Determine whether a point is inside a rhombus or not given its center
         * and its points in clockwise order
         * @param {PMUI.util.Point} point
         * @param {Array} rhombus
         * @param {PMUI.util.Point} center
         * @return {boolean}
         */
        pointInRhombus : function (point, rhombus, center) {
            var i,
                j = rhombus.length - 1;
            for (i = 0; i < rhombus.length; j = i, i += 1) {
                if (this.segmentIntersection(center, point,
                        rhombus[j], rhombus[i], true) &&
                        this.onSegment(point, rhombus[j], rhombus[i]) === false) {
                    return false;
                }
            }
            return true;
        }
    };

    // Declarations created to instantiate in NodeJS environment
    if (typeof exports !== 'undefined') {
        module.exports = Geometry;
    }

    // extend the namespace
    PMUI.extendNamespace('PMUI.draw.Geometry', Geometry);
}());

(function () {
    /**
     * @class PMUI.event.Action
     * Handles the action for buttons, menues, toolbars
     *
     * Example:
     * 
     *      var action = new PMUI.event.Action({
     *          text: 'Save Document',
     *          icon: 'pm-icon-save',
     *          disabled: false,
     *          handler: function () {
     *              alert('Action has been called!');
     *          }
     *      });
     *
     * Using action as parameter to create another objects:
     *
     *      var button = new PMUI.ui.Button(action);
     *
     *      var form = new PMUI.form.Form({
     *          ...
     *          buttons: [
     *              action,
     *              {
     *                  text: 'Refresh',
     *                  handler: function() {
     *                      obj.refresh()  //this object is fake used for sample purposes only
     *                  }
     *              }
     *          ]
     *          ...
     *      });
     *
     * Actions are created to handle user functionality you can extend or use over several places or components
     *
     * 
     * @constructor
     * Creates a new instance of object
     * @param {Object} options
     */
    var Action = function (options) {
        /**
         * Defines the text of the action
         * @type {String}
         */
        this.actionText = null;
        /**
         * Defines the icon asssociated with the action
         * @type {String}
         */
        this.actionIcon = null;
        /**
         * Defines the state of the action
         * @type {Boolean}
         */
        this.disabled = false;
        /**
         * Defines the action to be executed
         * @type {Function}
         */
        this.handler = null;

        Action.prototype.init.call(this, options);
    };

    /**
     * Defines the object's type
     * @type {String}
     */
    Action.prototype.type = 'Action';
    /**
     * Defines the object's family
     * @type {String}
     */
    Action.prototype.family = 'Action';

    /**
     * Defines the action flag
     * @type {Boolean}
     */
    Action.prototype.isAction = true;

    /**
     * @private
     * Initializes the object with the default values
     * @param  {Object} options Constructor options
     */
    Action.prototype.init = function (options) {
        var defaults;
        defaults = {
            icon : null,
            text : null,
            disabled: false,
            handler: function () {}
        };
        jQuery.extend(true, defaults, options);
        this.setActionIcon(defaults.icon)
            .setActionText(defaults.text)
            .setDisable(defaults.disabled)
            .setHandler(defaults.handler);
    };

    /**
     * Sets the action Icon
     * @param {String} icon Icon URL or class
     */
    Action.prototype.setActionIcon = function (icon) {
        this.actionIcon = icon;
        return this;
    };

    /**
     * Sets the action's text
     * @param {String} text Actions's name
     */
    Action.prototype.setActionText = function (text) {
        this.actionText = text;
        return this; 
    };

    /**
     * Sets the disabled value
     * @param {Boolean} value 
     */
    Action.prototype.setDisable = function (value) {
        if (typeof(value)==='boolean') {
            this.disabled = value;
        }
        return this;
    };

    /**
     * Returns the icon url or class
     * @return {String} [description]
     */
    Action.prototype.getIcon = function () {
        return this.actionIcon;
    };

    /**
     * Returns the action name
     * @return {String} [description]
     */
    Action.prototype.getText = function () {
        return this.actionText;
    };

    /**
     * Sets the handler function
     * @param {Function} fn
     */
    Action.prototype.setHandler = function (fn) {
        if (typeof(fn) === 'function') {
            this.handler = fn;
        }
        return this;
    };

    /**
     * Enables the action functionality
     */
    Action.prototype.enable = function () {
        this.setDisable(false);
    };

    /**
     * Disables the action functionality
     */
    Action.prototype.disable = function () {
        this.setDisable(true);
    };

    /**
     * Returns if the action is enabled
     * @return {Boolean}
     */
    Action.prototype.isEnabled = function () {
        return !this.disabled;
    };

    /**
     * Executes the action if is enabled and has a valid handler
     */
    Action.prototype.execute = function () {
        if (!this.disabled && typeof this.handler === 'function') {
            this.handler();
        }
    };


    if (typeof exports !== "undefined") {
        module.exports = Action;
    }

    PMUI.extendNamespace('PMUI.event.Action', Action );

}());
(function (){
    /**
     * @class PMUI.core.Base
     * Base class for any other class in PMUI, this class is an abstract so it shouldn't be instantiated.
     * @param {Object} settings All the settings the the class requires
     *
     * Usage example:
     *
     *      var anObject = new PMUI.core.Base({
     *          id: "myUniqueId"
     *      });
     *
     * @constructor
     * Creates a new instance of the class 'Base'.
     * @param {Object} [options={}] The config options for the object.
     *
     * @cfg {String} [id=[an autogenerated unique id]] The unique identifier for the object. If it is not specified 
     * then one is created an assigned to it.
     */
    var Base = function(settings) {
        /**
         * Unique ID
         * @type {String}
         */
        this.id = null;
        Base.prototype.init.call(this, settings);
    };
    /**
     * Class type
     * @type {String}
     */
    Base.prototype.type = 'Base';
    /**
     * Class family
     * @type {String}
     */
    Base.prototype.family = 'Core';
    /**
     * Initialize the object
     * @param  {Object} settings Settings for Base Class
     * @return {String}          the generated id
     * @private
     */
    Base.prototype.init = function(settings) {
        var defaults = {
            id: 'pmui-' + PMUI.generateUniqueId()
        };

        jQuery.extend(true, defaults, settings);

        this.setID(defaults.id);
    };
    /**
     * Set the id property for the object
     * @param {String} id
     * @chainable
     */
    Base.prototype.setID = function(id) {
        this.id = id;
        return this;
    };
    /**
     * Returns the id property from the object
     * @return {String} [description]
     */
    Base.prototype.getID = function() {
        return this.id;
    };
    /**
     * Returns the object's type
     * @return {String}
     */
    Base.prototype.getType = function () {
        return this.type;
    };
    /**
     * Return's the object's family name
     * @return {String} 
     */
    Base.prototype.getFamily = function() {
        return this.family;
    };

    // Declarations created to instantiate in NodeJS environment
    if (typeof exports !== "undefined") {
        module.exports = Base;
    }

    PMUI.extendNamespace('PMUI.core.Base', Base);
}());
(function () {
    /**
     * @class PMUI.core.Element
     * Base class to handle objects that have a visual representation (HTML representation).
     * @extends PMUI.core.Base
     *
     * Usage example:
     *
     *      @example
     *      var element;
     *
     *      element = new PMUI.core.Element({
     *          x: 10,
     *          y: 20,
     *          positionMode: "absolute",
     *          width: 200,
     *          height: 200,
     *          style: {
     *              cssProperties: {
     *                  "background-color": "red"
     *              },
     *              cssClasses: [".square"]
     *          }
     *      });
     *
     *      document.body.appendChild(element.getHTML());
     *
     * @constructor
     * Create a new instace of the class 'Element'
     * @param {Object} options
     *
     * @cfg {String} [elementTag="div"] The html tag for the object's HTML element.
     * @cfg {String} [positionMode="relative"] The css position for the object's HTML element. It can take one of the 
     * following values "static", "absolute", "fixed", "relative", "inherit" or "". Please read about the css "position" 
     * property to know about how each value impacts in the html.
     * @cfg {Object} [style={cssProperties:{}, cssClasses:[]}] A JSON object which will contain the css properties for 
     * the object's html.
     * 
     * This object can have optionally two properties:
     *
     * - cssProperties: another JSON object in which each property can be a css property (if the css property has any 
     * special character like "-" enclose the property name between quotes), the value can be any valid css value for 
     * the property.
     * - cssClasses: An array where each element is a String, this String is the css class name.
     * 
     * @cfg {Number} [x=0] The position coordinate in the x axis for the object's HTML, please note that the visual 
     * position will depend on the {@link PMUI.core.Element#cfg-positionMode positionMode option}.
     * @cfg {Number} [y=0] The position coordinate in the y axis for the object's HTML, please note that the visual 
     * position will depend on the {@link PMUI.core.Element#cfg-positionMode positionMode option}.
     * @cfg {Number|String} [width="auto"] The width for the object's HTML, it can be a Number or an String. In the 
     * latter case the value must be "auto" or "inherit" or have one of the following formats: 
     *
     * - "##px".
     * - "##%".
     * - "##em"
     *
     * Note that ## is a number.
     * @cfg {Number|String} [height="auto"] The height for the object's HTML, the values that can take are the same 
     * specified in the {@link PMUI.core.Element#cfg-width}.
     * @cfg {Number|String} [zOrder="auto"] The position in the z axis for the object's HTML, it can be a Number or a 
     * String, in the latter case the value only can be "auto" or "inherit".
     * @cfg {Boolean} [visible=true] If the object will be visible or not.
     * @cfg {Number} [proportion=1] A numeric value that specifies the final width of the object's HTML in case to be 
     * contained by a {@link PMUI.core.Container Container} which is applying a {@link PMUI.layout.Layout Layout}.
     * @cfg {String} [display=""] The display property specifies the type of box used for an HTML element. 
     * find out more about the css display property
     * @cfg {Function} [onBeforeContextMenu=null] description here.
     */
    var Element = function(settings) {
        Element.superclass.call(this, settings);
        /**
         * HTML element.
         * @type {HTMLElement}
         * @readonly
         */
        this.html = null;
        /**
         * @property {Object|PMUI.util.Style} [style={
                cssProperties: {},
                cssClasses: []
            }]
         * A {@link PMUI.util.Style Style} object or a JSON object with the settings to create a new 
         * {@link PMUI.util.Style Style} object for the current's HTML element.
         * @readonly
         */
        this.style = null;
        /**
         * X coordinate for the HTML element.
         * @type {Number}
         * @readonly
         */
        this.x = null;
        /**
         * @property {Number} [y=0]
         * Y Coordinate for the HTML element.
         * @readonly
         */
        this.y = null;
        /**
         * Width for the HTML element, it can be a number or a string with the following format: 
         ##px when ## is a number.
         * @type {Number|String}
         * @readonly
         */
        this.width = null;
        /**
         * Height for the HTML element, it can be a number or a string with the following format: 
         ##px when ## is a number.
         * @type {Number|String}
         * @readonly
         */
        this.height = null;
        /**
         * A boolean value that indicates if the HTML element is visible or not.
         * @type {Boolean}
         * @readonly
         */
        this.visible = null;
        /**
         * A Number that indicates the HTML element's position on the Z axis.
         * @type {Number}
         * @readonly
         */
        this.zOrder = null;
        /**
         * @property {String} [elementTag="div"]
         * Tag name for the element to be created, it defaults to "div".
         * @readonly
         */
        this.elementTag = null;
        /**
         * @property {String} [positionMode="relative"]
         * Position for the object's html element, it must be a valid value for the "position" CSS property.
         * @readonly
         */
        this.positionMode = null;
        /**
         * Defines the proportion of the html.
         * @type {Number}
         */
        this.proportion = null;
        /**
         * @property {String} [display=""] The display mode for the element. It is set by the 
         * {@link #cfg-display display} config option and the {@link #method-setDisplay setDisplay} method.
         * @readonly
         */
        this.display = null;
        /**
         * Defines the menu component for the element
         * @type {PMUI.ui.Menu}
         */
        this.menu = null;
        /**
         * Defines an object to handle/register events.
         * @type {Object}
         */
        this.events = {};
        /**
         * @property {Boolean} eventsDefined If the events were defined or not
         */
        this.eventsDefined = false;
        /**
         * [onContextMenu description]
         * @type {Boolean}
         */
        this.onContextMenu = false;
        /**
         * [onBeforeContextMenu description]
         * @type {Boolean}
         */
        this.onBeforeContextMenu = false;
        /**
         * The element's parent Container.
         * @type {PMUI.core.Element}
         */
        this.parent = null;
        /**
         * The HTML area in which some activities, actions can be performed (like display the context menu).
         * @type {HTMLElement}
         * @private
         */
        this.actionHTML = null;

        Element.prototype.init.call(this, settings);
    };

    PMUI.inheritFrom('PMUI.core.Base', Element);
    /**
     * The Object's type.
     * @type {String}
     */
    Element.prototype.type = 'Element';
    /**
     * Initialize the object.
     * @param  {Object} settings A JSON object with the settings to create the new class instance.
     * @private
     */
    Element.prototype.init = function(settings) {
        var defaults = {
            elementTag: "div",
            positionMode: "relative",
            menu: null,
            style: {
                cssProperties: {},
                cssClasses: []
            },
            x: 0,
            y: 0,
            width: "auto", 
            height: "auto",
            zOrder: "auto",
            display: "",
            visible: true,
            proportion: 1,
            onContextMenu: null,
            onBeforeContextMenu: null
        };

        jQuery.extend(true, defaults, settings);

        this.setElementTag(defaults.elementTag)
            .setStyle(defaults.style)
            .setPositionMode(defaults.positionMode)
            .setDisplay(defaults.display)
            .setX(defaults.x)
            .setY(defaults.y)
            .setWidth(defaults.width)
            .setHeight(defaults.height)
            .setZOrder(defaults.zOrder)
            .setVisible(defaults.visible)
            .setProportion(defaults.proportion)
            .setContextMenu(defaults.menu)
            .setOnBeforeContextMenu(defaults.onBeforeContextMenu)
            .setOnContextMenuHandler(defaults.onContextMenu);
    };
    /**
     * [setOnBeforeContextMenu description]
     */
    Element.prototype.setOnBeforeContextMenu = function(handler) {
        if(!(handler === null || typeof handler === 'function')) {
            throw new Error("setOnBeforeContextMenu(): The parameter must be a function or null.");
        }
        this.onBeforeContextMenu = handler;
        return this;
    };
    /**
     * [setOnContextMenuHandler description]
     * @param {[type]} handler [description]
     */
    Element.prototype.setOnContextMenuHandler = function(handler) {
        if(typeof handler === 'function' || null) {
            this.onContextMenu = handler;
        }
        return this;
    };
    /**
     * Sets the display mode for the Element.
     * @param {String} display It can take one of the following values:
     *
     * - ""
     * - "block"
     * - "inline"
     * - "inline-block"
     * - "none"
     *
     * @chainable
     */
    Element.prototype.setDisplay = function(display) {
        if(display === "" || display === 'block' || display === 'inline' 
            || display === 'inline-block' || display === 'none') {
            this.display = display;
            this.applyStyle();
        } else {
            throw new Error('The setDisplay() method only accepts one od the following options: ' + 
                ' "", "block", "inline", "inline-block", "none"');
        }

        return this;
    };
    /**
     * Sets the position mode for the Element.
     * @param {String} position It can take one ot the following values:
     *
     * - "static"
     * - "asolute"
     * - "fixed"
     * - "relative"
     * - "inherit"
     *
     * @chainable
     */
    Element.prototype.setPositionMode = function(position) {
        if(position === 'static' || position === 'absolute' || position === 'fixed' || position === 'relative' || 
            position === 'inherit' || position === "") {
            this.positionMode = position;
            this.applyStyle();
        } else {
            throw new Error('The setPosition() method only accepts one of the following options:' + 
                ' "static", "absolute", "fixed", "relative", "inherit" or an empty string.');
        }

        return this;
    };
    /**
     * Set the HTML tag for the HTML element to be created, note that it'll only work when its html property
      is still not set.
     * @param {String} tag a HTML tag
     * @chainable
     */
    Element.prototype.setElementTag = function(tag) {
        if(!this.html && typeof tag === 'string') {
            this.elementTag = tag;
        }

        return this;
    };
    /**
     * [addCSSClasses description]
     * @param {[type]} classes [description]
     */
    Element.prototype.addCSSClasses = function(classes) {
        this.style.addClasses(classes);
        return this;
    };
    /**
     * [addCSSProperties description]
     * @param {[type]} properties [description]
     */
    Element.prototype.addCSSProperties = function(properties) {
        this.style.addProperties(properties);
        return this;
    };
    /**
     * [removeCSSClasses description]
     * @param  {[type]} classes [description]
     * @return {[type]}         [description]
     */
    Element.prototype.removeCSSClasses = function(classes) {
        this.style.removeClasses(classes);
        return this;
    };
    /**
     * [removeCSSProperties description]
     * @param  {[type]} properties [description]
     * @return {[type]}            [description]
     */
    Element.prototype.removeCSSProperties = function(properties) {
        this.style.removeProperties(properties);
        return this;
    };
    /**
     * Set the style properties for the HTML element.
     * @param {Object|null} style an JSON structure with attributes cssProperties (another object) and cssClasses 
     * (array). It also can be null, in this case no custom styles will be applied to the the element.
     * @chainable
     */
    Element.prototype.setStyle = function(style) {
        style = style || {};
        if(this.style && this.html) {
            this.style.unapplyStyle();
        }
        if(style instanceof PMUI.util.Style) {
            this.style = style;
            style.belongsTo = this;
        } else if(typeof style === 'object') {
            style.belongsTo = this;
            this.style = new PMUI.util.Style(style);
        }
        this.applyStyle();
        return this;
    };
    /**
     * Set the x position coordinate for the HTML element.
     * @chainable
     * @param {Number} x
     * @chainable
     */
    Element.prototype.setX = function(x) {
        if(typeof x === 'number') {
            this.x = x;
        } else if(/^\d+(\.\d+)?px$/.test(x)) {
            this.x = parseInt(x, 10);
        } else {
            throw new Error('setX: x param is not a number');
        }
        this.style.addProperties({left: this.x});

        return this;
    };
    /**
     * Returns the x position coordinate for the HTML element
     * @return {Number}
     */
    Element.prototype.getX = function() {
        return this.x;
    };
    /**
     * Set the y position coordinate for the HTML element.
     * @param {Number} y
     * @chainable
     */
    Element.prototype.setY = function(y) {
        if(typeof y === 'number') {
            this.y = y;
        } else if(/^\d+(\.\d+)?px$/.test(y)) {
            this.y = parseInt(y, 10);
        } else {
            throw new Error('setY: y param is not a number');
        }
        this.style.addProperties({top: this.y});
        return this;
    };
    /**
     * Returns the y position coordinate for the HTML element
     * @return {Number}
     */
    Element.prototype.getY = function() {
        return this.y;
    };
    /**
     * Set the width for the HTML element
     * @param {Number|String} width height it can be a number or a string.
      In case of using a String you only can use 'auto' or ##px or ##% or ##em when ## is a number.
     * @chainable
     */
    Element.prototype.setWidth = function(width) {
        if(typeof width === 'number') {
            this.width = width;
        } else if(/^\d+(\.\d+)?px$/.test(width)) {
            this.width = parseInt(width, 10);
        } else if(/^\d+(\.\d+)?%$/.test(width)) {
            this.width = width;
        } else if(/^\d+(\.\d+)?em$/.test(width)) {
            this.width = width;
        } else if(width === 'auto') {
            this.width = width;
        } else {
            throw new Error('setWidth: width param is not a number');
        }
        this.style.addProperties({width: this.width});
        return this;
    };
    /**
     * Returns the HTML element's width
     * @return {Number}
     */
    Element.prototype.getWidth = function() {
        return this.width;
    };
    /**
     * Set the height for the HTML element
     * @param {Number|String} height it can be a number or a string.
      In case of using a String you only can use 'auto' or 'inherit' or ##px or ##% or ##em when ## is a number.
     * @chainable
     */
    Element.prototype.setHeight = function(height) {
        if(typeof height === 'number') {
            this.height = height;   
        } else if(/^\d+(\.\d+)?px$/.test(height)) {
            this.height = parseInt(height, 10);
        } else if(/^\d+(\.\d+)?%$/.test(height)) {
            this.height = height;
        } else if(/^\d+(\.\d+)?em$/.test(height)) {
            this.height = height;
        } else if(height === 'auto' || height === 'inherit') {
            this.height = height;
        } else {
            throw new Error('setHeight: height param is not a number');
        }
        this.style.addProperties({height: this.height});
        return this;
    };
    /**
     * Returns the HTML element's height.
     * @return {Number}
     */
    Element.prototype.getHeight = function() {
        return this.height;
    };
    /**
     * Set the position index om the z axis for the object's HTML element.
     * @param {Number|String} zOrder it can be a Number or a String.
      In case of using a String you only can use 'auto' or 'inherit'.
     * @chainable
     */
    Element.prototype.setZOrder = function(zOrder) {
        if(typeof zOrder === 'number') {
            this.zOrder = parseInt(zOrder, 10);
        } else if(zOrder === 'auto' || zOrder === 'inherit') {
            this.zOrder = zOrder;
        } else {
            throw new Error('setZOrder: zOrder param is not a number'); 
        }
        if(this.html) {
            this.style.addProperties({"z-index": this.zOrder});
        }
        return this;
    };
    /**
     * Returns the HTML element's zOrder.
     * @return {Number}
     */
    Element.prototype.getZOrder = function() {
        return this.zOrder;
    };
    /**
     * Set if the HTML element is visible or not.
     * @param {Boolean} visible
     * @chainable
     */
    Element.prototype.setVisible = function (visible) {
        visible = !!visible;
        this.visible = visible;
        if(this.html) {
            this.style.addProperties({"display": this.visible ? this.display : "none"});
        }        

        return this;
    };
    /**
     * Returns a boolean value that indicates if the HTML element is visible or not.
     * @return {Boolean} [description]
     */
    Element.prototype.isVisible = function() {
        return this.visible;
    };
    /**
     * Set the HTML element's position.
     * @param {Object} position an object with x and y properties, both of them are Numbers
     * @chainable
     */
    Element.prototype.setPosition = function(position) {
        this.setX(position.x);
        this.setY(position.y);

        return this;
    };
    /**
     * Returns and object which contains the x and y positions from the HTML element.
     * @return {Object}
     */
    Element.prototype.getPosition = function() {
        return {
            x: this.getX(),
            y: this.getY()
        };
    };
    /**
     * Set the HTML element's dimension (width and height).
     * @param {Object} dimension
     * @chainable
     */
    Element.prototype.setDimension = function(dimension) {
        this.setWidth(dimension.width);
        this.setHeight(dimension.height);

        return this;
    };
    /**
     * Returns the HTML element's dimension.
     * @return {Object} and object with width and height properties.
     */
    Element.prototype.getDimension = function() {
        return {
            width: this.getWidth(),
            height: this.getHeight()
        };
    };
    /**
     * Applies the css classes and ptoperties to the element's html which the object is related.
     * @chainable
     */
    Element.prototype.applyStyle = function() {
        if(this.html) {
            this.style.applyStyle();

            this.style.addProperties({
                display: this.visible ? this.display: "none",
                position: this.positionMode,
                left: this.x,
                top: this.y,
                width: this.width,
                height: this.height,
                zIndex: this.zOrder
            });
        }
        return this;
    };
    /**
     * Creates the object's HTML element.
     * @return {HTMLElement} returns a HTML element
     */
    Element.prototype.createHTML = function() {
        var html;
        if(this.html) {
            return this.html;
        }        
        html = PMUI.createHTMLElement(this.elementTag || "div");
        html.id = this.id;
        PMUI.linkToPMUIObject(html, this);
        this.html = html;
        this.actionHTML = html;
        this.applyStyle();
        if(this.eventsDefined) {
            this.defineEvents();
        }
        return this.html;
    };
    /**
     * Returns the object's HTML element, if it doesn't exist it's created then it is returned.
     * @return {HTMLElement}
     */
    Element.prototype.getHTML = function() {
        if(!this.html) {
            this.html = this.createHTML();
        }
        return this.html;
    };

    /**
     * Set the proportion of the html element.
     * @param {Number} p Proportion value.
     * @chainable
     */
    Element.prototype.setProportion = function (p) {
        this.proportion = p;
        return this;
    };

    /**
     * Creates and Instanciate an event.
     * @param {String} type  Event type.
     * @param {String} [alias] Registered Name.
     * @return {PMUI.event.Event}
     */
    Element.prototype.addEvent = function (type, alias) {
        var factory = new PMUI.event.EventFactory(),
            newEvent = factory.make(type),
            registerName = alias || PMUI.generateUniqueId();
        this.events[registerName] = newEvent;
        return newEvent;
    };

    /**
     * @abstract
     * Defines the events associated with the element.
     * @chainable
     * @abstract
     */
    Element.prototype.defineEvents = function () {
        var that = this;
        this.removeEvents();
        this.addEvent('contextmenu').listen(this.actionHTML, function(e) {
            if(typeof that.onBeforeContextMenu === 'function') {
                that.onBeforeContextMenu(that);
            }
            if(typeof that.onContextMenu === 'function') {
                that.onContextMenu(that);
            }
            if(that.menu) {
                e.stopPropagation();
                e.preventDefault();
                that.menu.show(e.pageX, e.pageY);
            }
        });
        this.eventsDefined = true;
        return this;
    };
    /**
     * Remove a specific event attached to the object.
     * @param  {String} identifier The id or alias used in the moment of event addition.
     * @chainable
     */
    Element.prototype.removeEvent = function(identifier) {
        if(this.events[identifier]) {
            this.events[identifier].unlisten();
            delete this.events[identifier];
        }
        return this;
    };
    /**
     * Removes events attached to the object.
     * @chainable
     */
    Element.prototype.removeEvents = function () {
        var k;
        for(k in this.events) {
            this.removeEvent(k);
        };
        return this;
    };
    /**
     * Calculates the width size of the container.
     *
     * @param {String} text
     * @param {String} font
     * @chainable
     */
    Element.prototype.calculateWidth = function (text, font) {
        //TODO Improve the div creation (maybe we can use a singleton for this)
        var f = font || '12px arial',
            $o = $(PMUI.createHTMLElement('div')), w;
            $o.text(text)
                .css({
                    'position': 'absolute', 
                    'float': 'left', 
                    'white-space': 'nowrap', 
                    'visibility': 'hidden', 
                    'font': f
                })
                .appendTo($('body'));
            w = $o.width();

        $o.remove();

        return w;
    };
    /**
     * Sets the context menu for the component and the menu will be contained on the {@link PMUI.core.Element#menu menu} property.
     * 
     * If the settings parameter is not a valid json the {@link PMUI.core.Element#menu menu} property will be an simple empty object.
     *
     * The example below shows the appropriate way to send a basic settings parameter.
     *
     *      component = new Class();
     *      component.setContextMenu({
     *          id: "id", //optional
     *          items: [
     *              {
     *                  text: "Option 1"
     *              },
     *              {
     *                  text: "Option 2"
     *              }
     *          ]
     *      });
     *
     * For more details about the context menu, the documentation is on the {@link PMUI.ui.Menu Menu} class.
     * 
     * @param {Object} settings Configuration or config options for the context {@link PMUI.ui.Menu menu}.
     * @return {Object} this Represents the component of the class.
     */
    Element.prototype.setContextMenu = function (menuItems) {
        var contextMenu;
        this.removeContextMenu();
        if(menuItems !== null) {
            if(jQuery.isArray(menuItems)) {
                contextMenu = new PMUI.menu.Menu({items: menuItems});
            } else if(menuItems instanceof PMUI.menu.Menu) {
                contextMenu = menuItems;
            } else if (typeof menuItems === 'object') {
                contextMenu = new PMUI.menu.Menu(menuItems);
            }
            if(contextMenu) {
                this.menu = contextMenu.setTargetElement(this);
            }
        }
        return this;
    };
    /**
     * [removeContextMenu description]
     * @return {[type]} [description]
     */
    Element.prototype.removeContextMenu = function() {
        if(this.menu) {
            this.menu.setTargetElement(null);
            this.menu = null;    
        }
        return this;
    };
    /**
     * Returns the elemnent's parent container.
     * @return {PMUI.core.Element|null} The parent container.
     */
    Element.prototype.getParent = function() {
        return this.parent;
    };
    /**
     * Shows the context menu for the element provided that the element contains an object that represent a
     * {@link PMUI.ui.Menu Menu}. 
     * 
     * If the {@link PMUI.core.Element#menu menu} property is an empty object the context menu will
     * never appear on the page.
     *
     * By default the context menu will appear after to do right click on the element.
     */
    Element.prototype.showContextMenu = function () {
        if(this.menu) {
            this.menu.show();   
        }
        return this;
    };    
    /**
     * Return true if the object's events were defined, otherwise it returns false.
     * @return {Boolean}
     */
    Element.prototype.eventsAreDefined = function() {
        return this.eventsDefined;
    };
    // Declarations created to instantiate in NodeJS environment
    if (typeof exports !== "undefined") {
        module.exports = Element;
    }

    PMUI.extendNamespace('PMUI.core.Element', Element);

}());
(function () {
    /**
     * @class PMUI.core.Container
     * Handles HTML elements that have a container behavior (they can be contain {Element} objects)
     * @abstract
     * @extend PMUI.core.Element  
     *  * Usage example (only for subclasses since this is an abstract class):
     *
     *      @example
     *       //Remember, this is an abstract class so it shouldn't be instantiate,
     *       //anyway we are instantiating it just for this example
     *       c = new PMUI.core.Container({
     *            x: 10,
     *            y: 20,
     *            positionMode: "absolute",
     *            width: 300,
     *            height: 100,
     *            style: {
     *                cssProperties: {
     *                    "background-color": "yellow"
     *                },
     *                cssClasses: [".square"]
     *            },
     *            items : [
     *               new PMUI.core.Element({
     *                    y : 0,
     *                    width: 100,
     *                    height: 100,
     *                    style: {
     *                        cssProperties: {
     *                            "background-color": "red"
     *                        },
     *                        cssClasses: [".square"]
     *                    }
     *               }),
     *               new PMUI.core.Element({
     *                    y : 100,
     *                    width: 100,
     *                    height: 100,
     *                    style: {
     *                        cssProperties: {
     *                            "background-color": "yellow"
     *                        },
     *                        cssClasses: [".square"]
     *                    }
     *               })
     *            ]
     *        });
     *        document.body.appendChild(c.getHTML());
     *            
     * @constructor 
     * Creates a new instance
     * @param {Object} settings
     *
     * @cfg {Array} [items=[]] An array with the items to be added to tyhe container. If the 
     * {@link PMUI.core.Container#cfg-dataItems dataItems config option} is  specified then this config option will be 
     * ignored.
     *
     * Each array's item can be:
     *
     *  - A JSON object with the config options required by an {@link PMUI.core.Element Element object}.
     *  - An instance of the {@link PMUI.core.Element Element class}.
     *
     * @cfg {String} [behavior='nobehavior'] A string that specifies the behavior for the container and its items.
     * The accepted values are the same ones that the specified ones for the 
     * {@link PMUI.core.Container#method-setBehavior setBehavior() method}.
     *
     * @cfg {Array} [dataItems=null] An array in which each element is the data for the items that will be contained by 
     * the container. If this config option is specfied the {@link PMUI.core.Container#cfg-items items config option} 
     * will be ignored.
     *
     * @cfg {Function} [onDrop=null] The callback function to be executed everytime a draggable item is dropped on the 
     * container. This callback can return a boolean value or not return anything. If it returns "false" then the drop 
     * action will be cancelled. For info about the parameters sent to the callback function please read the 
     * {@link PMUI.core.Container#event-onDrop onDrop} event documentation.
     * 
     * @cfg {Function} [onSort=null] The callback function to be executed everytime a container's item order changes.
     * For info about the parameters sent to the callback function please read the 
     * {@link PMUI.core.Container#event-onSort onSort} event documentation.
     *
     * @cfg {String} [sortableItems=null] Specifies which items inside the element should be sortable in case a 
     * sort behavior is applied. It must be a jQuery selector.
     *
     * @cfg {Boolean} [disabled=false] If the interactive functions like the drag, drop, sort behaviors will be 
     * disabled or not.
     *
     * @cfg {Function|null} [onBeforeDrop=null] The callback to be executed before a draggable item that was dropped 
     * on the container is added.
     *
     * @cfg {Function|null} [onDropOut=null] The callback to be executed everytime a draggable item is drop out from 
     * the container to another one. To know the parameters sent to the callback function read the documentation for
     * the {@link #event-onDropOut onDropOut} event.
     *
     * @cfg {Function|null} [onDragStart=null] The callback function to be executed everytime any of its draggable items 
     * starts to be dragged. For info about the parameters sent to the callback function read the documentation for
     * the {@link #event-onDropOut onDropOut} event.
     *
     * @cfg {Function|null} [onAdd=null] The callback function be executed eveytime the {@link #event-onAdd onAdd event}.
     */
    var Container = function (settings) {
        Container.superclass.call(this, settings);
        /**
         * @property {PMUI.util.ArrayList} items
         * An ArrayList object that contains all the child Items
         */
        this.items = null;
        /**
         * @property {PMUI.util.Factory} factory
         * X
         */
        this.factory = null;
        /**
         * @property {PMUI.behavior.ContainerItemBehavior} behaviorObject The container's behavior object.
         * @private
         */
        this.behaviorObject = null;
        /**
         * @property {String} behavior The behavior's pmType being used by the object.
         * @readonly
         */
        this.behavior = null;
        /**
         * @property {HTMLElement} containmentArea The HTML area in which the items HTML will be append
         * @readonly
         */
        this.containmentArea = null;
        /**
         * @event onDragStart
         * Fired when the dragging is starting on any ot its items.
         * @param {PMUI.core.Container} container The container from which the draggable is starting to be dragged.
         * @param {PMUI.core.Element} draggableItem The item thais starting to be dragged.
         */
        this.onDragStart = null;
        /**
         * @event onDrop
         * Fired everytime a draggable item is dropped on the container.
         * @param {PMUI.core.Container} container The container on which the draggable item was dropped.
         * @param {PMUI.core.Element} draggableItem The item that was dropped on the container.
         * @param {Number} index The order index in which the new element was dropped.
         */
        this.onDrop = null;
        /**
         * @event onDropOut
         * Fired when a draggable item is drop out from the container to another one.
         * @param {PMUI.core.Element} draggable The draggable PMUI object involved in the action.
         * @param {PMUI.core.Container} origin The container the draggable item was drag from.
         * @param {PMUI.core.Container} destiny The container the draggable was dropped on.
         */
        this.onDropOut = null;
        /**
         * @event
         * Fired before an item is added to the container by dropping.
         * @param {PMUI.core.Element} draggableItem The container.
         * @param {PMUI.core.Element} draggableItem The item that was dropped and is ready to be appended to the 
         * container.
         * @param {Number} index The new element's order index in which it will be added.
         */
        this.onBeforeDrop = null;
        /**
         * @event onSort
         * Fired everytime a container's item order changes.
         * @param {PMUI.core.Container} container The container on which the sort occur.
         * @param {PMUI.core.Element} orderedElement The element which was ordered.
         * @param {Number} index The new element's order index.
         */
        this.onSort = null;
        /**
         * Specifies which items inside the element should be sortable in case a drag/sort behavior is applied.
         * @type {String}
         */
        this.sortableItems = null;
        /**
         * If the container is disabled or not. Set by the {@link #cfg-disabled disabled} config option and the 
         * {@link #method-disable disable()} method.
         * @type {Boolean}
         * @readonly
         */
        this.disabled = null;
        /**
         * If the container's behavior is disabled or not. Set by the {@link #cfg-disabledBehavio disabledBehavior} config option and the 
         * {@link #method-disableBehavior disableBehavior()} method.
         * @type {Boolean}
         */
        this.disabledBehavior = null;
        /**
         * A flag that indicates if actually a massive action is being performed.
         * @private
         * @type {Boolean}
         */
        this.massiveAction = false;
        /**
         * @event onAdd
         * Fired everytime an item is added to the object.
         * @param {PMUI.core.Container} container The container in which the item was added to.
         * @param {PMUI.core.Element} item The item that was added.
         * @param {Number} index The index in which the item was added.
         */
        this.onAdd = null;
        /**
         * @property {String} dragDropBehaviorScope A draggable/droppable container with the same scope value as 
         * another droppable/draggable container's item will be accepted by the behavior.
         */
        this.dragDropBehaviorScope = null;

        this.onSortStart = null;

        /**
         * Restricts dragging/sorting from starting unless the mousedown occurs on the specified element(s). Only
         * elements that descend from the draggable element are permitted.
         * @type {String}
         */
        this.dragHandler = null;
        Container.prototype.init.call(this, settings);
    };

    PMUI.inheritFrom('PMUI.core.Element', Container);

    Container.prototype.type = 'Container';
    /**
     * Initializes the object.
     * @param  {Object} settings A JSON object with the config options.
     * @private
     */
    Container.prototype.init = function (settings) {
        var defaults = {
            items: [],
            behavior: 'nobehavior',
            dataItems: null,
            onDragStart: null,
            onBeforeDrop: null,
            onDrop: null,
            onDropOut: null,
            onSort: null,
            sortableItems: '> *',
            disabled: false,
            onAdd: null,
            onSortStart : null,
            dragDropBehaviorScope: 'pmui-containeritem-behavior',
            dragHandler: null
        };

        jQuery.extend(true, defaults, settings);

        this.items = new PMUI.util.ArrayList(); 

        this.setFactory(defaults.factory)
            .setOnAddHandler(defaults.onAdd);

        if(jQuery.isArray(defaults.dataItems)) {
            this.setDataItems(defaults.dataItems);
        } else {
            this.setItems(defaults.items);
        }
            
        this.setSortableItems(defaults.sortableItems)
            .setDragHandler(defaults.dragHandler)
            .setDragDropBehaviorScope(defaults.dragDropBehaviorScope)
            .setBehavior(defaults.behavior)
            .setOnDragStartHandler(defaults.onDragStart)
            .setOnBeforeDropHandler(defaults.onBeforeDrop)
            .setOnDropHandler(defaults.onDrop)
            .setOnDropOutHandler(defaults.onDropOut)
            .setOnSortHandler(defaults.onSort);
        this.setOnSortStartHandler(defaults.onSortStart);
        if(this.disabled) {
            this.disable();
        } else {
            this.enable();
        }
    };
    /**
     * If specified, restricts dragging/sorting from starting unless the mousedown occurs on the specified element(s). 
     * Only elements that descend from the draggable element are permitted.
     * @param {String} handler The selector for the handler.
     * @chainable
     */
    Container.prototype.setDragHandler = function(handler) {
        if(!(typeof handler === 'string' || handler === null)) {
            throw new Error("The parameter must be a string or null");
        }
        this.dragHandler = handler;
        this.setBehavior(this.behavior);
        return this;
    };
    /**
     * Sets the {@link #property-dragDropBehaviorScope dragDropBehaviorScope} property.
     * @param {String} scope Any string that represents a scope.
     * @chainable
     */
    Container.prototype.setDragDropBehaviorScope = function(scope) {
        if(typeof scope !== 'string') {
            throw new Error("setDragDropBehaviorScope(): The parameter must be a string.");
        }
        this.dragDropBehaviorScope = scope;
        this.setBehavior(this.behavior);
        return this;
    };
    /**
     * Sets the handler to be executed everytime the {@link #event-onAdd onAdd} event is fired.
     * @param {Function|null} handler The handler to execute, it also can be null.
     * @chainable
     */
    Container.prototype.setOnAddHandler = function(handler) {
        if(!(handler === null || typeof handler === 'function')) {
            throw new Error('setOnAddHandler(): The parameter ust be a function or null.');
        }
        this.onAdd = handler;
        return this;
    };
    /**
     * Sets the callback function to be executed everytime any of its draggable items starts to be dragged. For info 
     * about the parameters sent to the callback function read the documentation for
     * the {@link #event-onDropOut onDropOut} event.
     * @param {Function|null} handler It can be a function or null.
     */
    Container.prototype.setOnDragStartHandler = function(handler) {
        if(!(typeof handler === 'function' || handler === null)) {
            throw new Error('setOnDragStartHandler(): the parameter must be a function or null.');
        }
        this.onDragStart = handler;
        return this;
    };
    /**
     * Sets the callback function to be executed when a draggable item is dragged from the container and dropped on 
     * another container. This callback can return a boolean value or not return anything. If it returns "false" then 
     * the drop action will be cancelled. For info about the parameters sent to the callback function please read the 
     * {@link PMUI.core.Container#event-onDrop onDrop} event documentation.
     * @param {Function|null} handler
     * @chainable
     */
    Container.prototype.setOnDropOutHandler = function(handler) {
        if(!(typeof handler === 'function' || handler === null)) {
            throw new Error('setOnDropOutHandler(): the parameter must be a function or null.');
        }
        this.onDropOut = handler;
        return this;
    };
    /**
     * Disables the container's behavior.
     * @chainable
     */
    Container.prototype.disableBehavior = function() {
        this.disabledBehavior = true;
        this.behaviorObject.disable();
        return this;
    };
    /**
     * Enables the container's behavior.
     * @chainable
     */
    Container.prototype.enableBehavior = function() {
        this.disabledBehavior = false;
        this.behaviorObject.enable();
        this.disabled = false;
        this.style.removeClasses(['pmui-disabled']);
        return this;
    };
    /**
     * Disables interactive functions like the drag, drop, sort behaviors.
     * @chainable
     */
    Container.prototype.disable = function() {
        this.disabled = true;
        this.style.addClasses(['pmui-disabled']);
        return this.disableBehavior();
    };
    /**
     * Enables interactive functions like the drag, drop, sort behaviors.
     * @chainable
     */
    Container.prototype.enable = function() {
        this.disabled = false;
        this.style.removeClasses(['pmui-disabled']);
        return this.enableBehavior();
    };
    /**
     * Specifies which items inside the element should be sortable in case a drag/sort behavior is applied.
     * @param {String} selector A jQuery selector.
     * @chainable
     */
    Container.prototype.setSortableItems = function(selector) {
        this.sortableItems = selector;
        return this.setBehavior(this.behavior);
    };
    /**
     * Sets the callback function to be executed everytime a container's item order changes.
     * @param {Function} handler It can be a function or null. For info about the parameters sent to the callback 
     * function please read the {@link PMUI.core.Container#event-onSort onSort} event documentation.
     */
    Container.prototype.setOnSortHandler = function(handler) {
        if(handler !== null && typeof handler !== 'function') {
            throw new Error('setOnSortHandler(): the parameter must be a function or null.');    
        }
        this.onSort = handler;
        return this;
    };

    Container.prototype.setOnSortStartHandler = function(handler) {
        if(handler !== null && typeof handler !== 'function') {
            throw new Error('setOnSortStartHandler(): the parameter must be a function or null.');    
        }
        this.onSortStart = handler;
        return this;
    };
    /**
     * Sets the callback function to be executed before a draggable item is added to the container by dropping.
     * @param {Function} handler It can be a function or null. For info about the parameters sent to the callback 
     * function please read the {@link PMUI.core.Container#event-onBeforeDrop onBeforeDrop} event documentation.
     */
    Container.prototype.setOnBeforeDropHandler = function(handler) {
        if(handler !== null && typeof handler !== 'function') {
            throw new Error('setOnDropHandler(): the parameter must be a function or null.');    
        }
        this.onBeforeDrop = handler;
        return this;
    };
    /**
     * Sets the callback function to be executed everytime a draggable item is dropped on the container.
     * @param {Function} handler It can be a function or null. For info about the parameters sent to the callback 
     * function please read the {@link PMUI.core.Container#event-onDrop onDrop} event documentation.
     */
    Container.prototype.setOnDropHandler = function(handler) {
        if(handler !== null && typeof handler !== 'function') {
            throw new Error('setOnDropHandler(): the parameter must be a function or null.');    
        }
        this.onDrop = handler;
        return this;
    };
    /**
     * Adds a new item but using raw data to create it. 
     * @param {Object} dataItem A JSON object with the data to create the new object.
     * @param {Number} [index] The position index in which the data item will be inserted.
     */
    Container.prototype.addDataItem = function(dataItem, index) {
        var args, i;
        if(typeof dataItem !== 'object') {
            throw new Error('addDataItem(): The parameter must be an object.');
        }
        args = [{
            data: dataItem
        }];
        for(i = 1; i < arguments.length; i += 1) {
            args.push(arguments[i]);
        }
        this.addItem.apply(this, args);
        return this;
    };
    /**
     * Sets the data toi be used to create every item to be contained by the object.
     * @param {Array} dataItems An array in which each element is an object with the data for each of the items.
     */
    Container.prototype.setDataItems = function(dataItems) {
        var i;
        if (jQuery.isArray(dataItems)) {
            this.clearItems();
            this.massiveAction = true;
            for (i = 0; i < dataItems.length; i += 1) {
                this.addDataItem(dataItems[i]);
            }
            this.massiveAction = false;
            this.paintItems();
        }

        return this;
    };
    /**
     * Returns the html element in which the items are appended.
     * @return {HTMLElement}
     */
    Container.prototype.getContainmentArea = function() {
        //this.createHTML();
        this.getHTML();
        return this.containmentArea;
    };
    /**
     * Returns all the container's items the behavior will be applied on.
     * @return {Array} An array of items.
     * @private
     */
    Container.prototype.getBehavioralItems = function() {
        return this.getItems();
    };
    /**
     * Enables/disables one, two, all of the following behaviors:
     * 
     * - "drag", allows to drag items from the object, also them can be dropped on another containers that have enables 
     * the "drop" behavior.
     * - "drop", allows to drop draggable items from another container on the object.
     * - "sort", allows reorder the items in the object. Combined with "drop" behavior makes the object capable to 
     * receive objects (drop) in a specific position.
     * @param {String} behavior The behavior to be set, the accepted values are:
     * 
     * - "drag", enables the dragging functionality.
     * - "dragclone", enables the clone-dragging functionality.
     * - "drop", enables the dropping functionality.
     * - "sort", enabled the sorting functionality.
     * - "dragdrop", enables both dragging and dropping functionality.
     * - "dragsort", enables both dragging and sorting functionality.
     * - "dropsort", enables both dropping and sorting functionality.
     * - "dragdropsort", enables the three behaviors: dragging, dropping and sorting.
     * - "nobehavior", disables all the behaviors.
     * @chainable
     */
    Container.prototype.setBehavior = function(behavior) {
        var obj, behaviorObject, factory;
        if(typeof behavior === 'string') {
            factory = new PMUI.behavior.ContainerItemBehaviorFactory();
            obj = {
                pmType: behavior,
                targetObject: this,
                scope: this.dragDropBehaviorScope,
                handle: this.dragHandler
            };
            if(this.behaviorObject) {
                this.behaviorObject.detachBehavior();
            }
            behaviorObject = factory.make(obj);
            if(behaviorObject) {
                this.behaviorObject = behaviorObject;
                this.behavior = behavior;
            }
            if(this.html) {
                this.behaviorObject.attachBehavior();
            }
        }

        return this;
    };
    /**
     * Sets the "factory" property with a {PMUI.util.Factory} object
     * @param {Object} factory
     * @chainable
     * @private
     */
    Container.prototype.setFactory = function (factory) {
        if (factory instanceof PMUI.util.Factory){
            this.factory = factory;
        } else {
            this.factory = new PMUI.util.Factory(factory);
        }
        return this;
    };

    /**
     * Removes a child element from the object
     * @param  {PMUI.core.Element|String|Number} item It can be a string (id of the child to remove), 
     a number (index of the child to remove) or a {Element} object.
     * @chainable
     */
    Container.prototype.removeItem = function (item) {
        var itemToRemove;
        //if (item instanceof PMUI.item.ContainableItem) {
        if (item instanceof PMUI.core.Element) {
            itemToRemove = item;
        } else {
            if (typeof item === 'string') {
                itemToRemove = this.items.find("id", item.id);
            } else if (typeof item === 'number') {
                itemToRemove = this.items.get(item);
            }
        }
        if (itemToRemove) {
            //this.data.removeItem(itemToBeRemove.getData());
            itemToRemove.parent = null;
            if(this.html) {
                jQuery(itemToRemove.html).detach();    
            }
            this.items.remove(itemToRemove);
        }
        if(!this.items.getSize()) {
            this.style.addClasses(['pmui-empty']);
        }
        if(this.massiveAction) {
            return this;   
        }
        return this.setBehavior(this.behavior);
    };
    /**
     * Removes all the child items
     * @chainable
     */
    Container.prototype.clearItems = function () {
        this.massiveAction = true;
        while (this.items.getSize() > 0) {
            this.removeItem(0);
        }
        this.setBehavior(this.behavior);
        this.massiveAction = false;
        return this;
    };
    /**
     * Returns true if the item, used as the unique method parameter, is a direct child of the current Container object,
      otherwise returns false
     * @param  {PMUI.core.Element}  item The target child object
     * @return {Boolean}
     */
    Container.prototype.isDirectParentOf = function (item) {
        return this.items.indexOf(item) >= 0;
    };
    /**
     * Returns true if the supplied argument can be accepted as an item of the container.
     * @param  {PMUI.core.Element} obj The element to be evaluated.
     * @return {Boolean}
     */
    Container.prototype.canContain = function(obj) {
        return obj instanceof PMUI.core.Element ? this.factory.isValidClass(obj) : false;
    };
    /**
     * Paint a container's item (puts its HTML into the container's HTML).
     * @param  {PMUI.core.Element} item The item to add.
     * @chainable
     * @private
     */
    Container.prototype.paintItem = function(item, index) {
        var referenceItem;
        if(this.html) {
            if(index !== null && index !== undefined) {
                referenceItem = this.items.get(index + 1);
                if(referenceItem) {
                    this.containmentArea.insertBefore(item.getHTML(), referenceItem.getHTML());
                } else {
                    this.containmentArea.appendChild(item.getHTML());        
                }
            } else {
                this.containmentArea.appendChild(item.getHTML());
            }
            if(this.eventsDefined) {
                item.defineEvents();
            }
            this.setBehavior(this.behavior);
        }
        return this;
    };
    /**
     * Paint all the container items.
     * @chainable
     * @private
     */
    Container.prototype.paintItems = function() {
        var items, i;
        if(this.containmentArea) {
            items = this.items.asArray()
            for(i = 0; i < items.length; i++) {
                this.paintItem(items[i]);
            }    
        }
        return this;
    };
    /**
     * Adds an child item to the object.
     * @param {PMUI.core.Element|Object} item It can be one of the following data types:
     * - {PMUI.core.Element} the object to add.
     * - {Object} a JSON object with the settings for the Container to be added.
     * @param {Number} [index] An index in which the item will be added.
     * @chainable
     */
    Container.prototype.addItem = function (item, index) {
        var itemToBeAdded;
        if (this.factory) {
            itemToBeAdded = this.factory.make(item);
        }
        if (itemToBeAdded && !this.isDirectParentOf(itemToBeAdded)) {
            //itemToBeAdded.setParent(this);
            if(index !== null && index !== undefined) {
                this.items.insertAt(itemToBeAdded, index);
            } else {
                this.items.insert(itemToBeAdded);    
            }
            itemToBeAdded.parent = this;
            //this.data.addItem(itemToBeAdded.getData());
            if(!this.massiveAction) {
                this.paintItem(itemToBeAdded, index);
            }
            if(typeof this.onAdd === 'function') {
                index = typeof index === 'number' ? index : this.items.getSize() - 1;
                this.onAdd(this, this.getItem(index), index);
            }
        }
        return this;
    };
    /**
     * Clear all the object's current child items and add new ones
     * @param {Array} items An array where each element can be a {Element} object or a JSON object 
     where is specified the setting for the child item to be added
     * @chainable
     */
    Container.prototype.setItems = function (items) {
        var i;
        if (jQuery.isArray(items)) {
            this.clearItems();
            this.massiveAction = true;
            for (i = 0; i < items.length; i += 1) {
                this.addItem(items[i]);
            }
            this.massiveAction = false;
            this.paintItems();
        }

        return this;
    };
    /**
     * Returns an array with all the child elements
     * @return {Array}
     */
    Container.prototype.getItems = function () {
        return this.items.asArray();
    };
    /**
     * Returns one single item based on the id or index position
     * @param  {String|Number} id If the parameter is a string then 
     it will be take as the id for the element to find and return, 
     but if the element is a Number it returns the object iwth that 
     index position
     * @return {PMUI.core.Element}
     */
    Container.prototype.getItem = function (i) {
        var t;
        if(typeof i === 'number') {
            t = this.items.get(i);
        } else if(typeof i === 'string') {
            t = this.items.find('id', i);
        } else if(this.items.indexOf(i) >= 0) {
            t = i;
        }
        return t;
    };
    /**
     * Returns the index for the specified item.
     * @param  {PMUI.core.Element|String} item It can be:
     * 
     * - {@link PMUI.core.Element}
     * - String, it will be interpreted as the items's id.
     * @return {Number}
     */
    Container.prototype.getItemIndex = function(item) {
        var evaluatedItem;

        if(item instanceof PMUI.core.Element) {
            evaluatedItem = item;
        } else if(typeof item === 'number') {
            evaluatedItem = this.items.get(item);
        } else if(typeof item === 'string') {
            evaluatedItem = this.items.find("id", item);
        }

        return this.items.indexOf(evaluatedItem);
    };
    /**
     * Creates the object's HTML element
     * @return {HTMLElement}
     */
    Container.prototype.createHTML = function () {
        if (this.html) {
            return this.html;
        }
        Container.superclass.prototype.createHTML.call(this);
        this.containmentArea = this.html;
        this.paintItems();
        this.style.applyStyle();
        this.setBehavior(this.behavior);

        return this.html;
    };
    /**
     * Moves an object's item from a position to another.
     * @param  {PMUI.core.Element} item The item to move.
     * @param  {Number} index The index position in which the item will be moved.
     * @chainable
     */
    Container.prototype.moveItem = function(item, index) {
        var items = this.items, currentIndex,
            referenceObject;

        item = this.getItem(item);

        if(item instanceof PMUI.core.Element && this.isDirectParentOf(item)) {
            currentIndex = items.indexOf(item);
            items = items.asArray();
            referenceObject = this.items.get(index + (currentIndex < index ? 1 : 0));
            item = items.splice(currentIndex, 1)[0];
            this.items.remove(item);
            this.items.insertAt(item, index);
            if(this.html) {
                if(index === this.items.getSize() - 1) {
                    this.containmentArea.appendChild(item.getHTML());
                } else {
                    this.containmentArea.insertBefore(item.getHTML(), referenceObject.getHTML());
                }
            }
        }
        return this;
    };
    /**
     * Returns the function to be executed when an accepted draggable is dragged over the droppable area.
     * @return {Function}
     */
    Container.prototype.onDragOver = function() {
        return function() {};
    };
    /**
     * Returns the function to be executed when the items order has been changed.
     * @return {Function}
     * @private
     */
    Container.prototype.onSortingChange = function() {
        var that = this;
        return function(e, ui) {
            var itemHTML = ui.item.get(0), item = PMUI.getPMUIObject(itemHTML),
                newIndex;
            newIndex = jQuery(that.getContainmentArea()).find('>*').index(itemHTML);
            
            that.moveItem(item, newIndex);
            if(typeof that.onSort === 'function') {
                that.onSort(that, item, newIndex);
            }
        };
    };
    /**
     * Handler to be executed everytime the mouse enter a position over any container's draggable/sortable item.
     * @param {PMUI.core.Element} draggable The draggable PMUI object.
     * @chainable
     */
    Container.prototype.onDraggableMouseOver = function(draggable) {
        return this;
    };
    /**
     * Handler to be executed everytime the mouse leaves a position over any container's draggable/sortable item.
     * @param {PMUI.core.Element} draggable The draggable PMUI object.
     * @chainable
     */
    Container.prototype.onDraggableMouseOut = function(draggable) {
        return this;
    };
    /**
     * Executes children events defined
     * @return {PMUI.core.Container}
     * @chainable
     */
    Container.prototype.defineEvents = function () {
        var j,
            children,
            that = this;
        Container.superclass.prototype.defineEvents.call(this);
        if (that.items.getSize() > 0) {
            children = that.getItems();
            for (j = 0; j < children.length; j++) {
                children[j].defineEvents();
            }
        }
        return this;
    };
    
    PMUI.extendNamespace('PMUI.core.Container', Container);

    // Publish to NodeJS environment
    if (typeof exports !== 'undefined') {
        module.exports = Container;
    }

}());
(function (){
    /**
     * @class  PMUI.core.Item
     * @extends PMUI.core.Container
     * Create a items container
     *
     * @constructor
     * Creates a new instance of the class
     * @param {Object} options Contructor object
     */
    var Item = function (options) {
        Item.superclass.call(this, options);

        /**
         * Parent object associated to this item
         * @type {Object}
         */
        this.parent = null;
        Item.prototype.init.call(this, options);
    };

    PMUI.inheritFrom('PMUI.core.Container', Item);

    /**
     * Defines the object's type
     * @type {String}
     */
    Item.prototype.type = "Item";

    /**
     * Defines the object's family
     * @type {String}
     */
    Item.prototype.family = "Item";

    /**
     * @private
     * Initializes the object with default values
     * @param  {Object} options 
     */
    Item.prototype.init = function (options) {
        var defaults = {
            parent: null
        };
        jQuery.extend(true, defaults, options);
        this.setParent(defaults.parent);
    };

    /**
     * Sets the parent object of the current item
     * @param {Object} parent Parent object
     */
    Item.prototype.setParent = function (parent) {
        this.parent = parent;
        return this;
    };

    /**
     * Returns the parent object pointer
     * @return {Object}
     */
    Item.prototype.getParent = function () {
        return this.parent;
    };
    /**
     * Removes the item from its parent container.
     * @chainable
     */
    Item.prototype.remove = function() {
        if(this.parent) {
            this.parent.removeItem(this);
        }

        return this;
    };

    PMUI.extendNamespace('PMUI.core.Item', Item);

    if (typeof exports !== 'undefined') {
        module.exports = Item;
    }
}());
(function () {
    /**
     * @class PMUI.core.Panel
     * Handles panels to be inserted into containers, it is composed by three main elements: header, body and footer
     * @abstract
     * @extends {PMUI.core.Container}
     * 
     *       //Remember, this is an abstract class so it shouldn't be instantiate,
     *       //anyway we are instantiating it just for this example
     *        var p1, p2, panel;
     *        p1 = new PMUI.core.Panel({
     *            height : 140,
     *            width : 90,
     *            display : 'inline-block',
     *            style : {
     *                cssProperties : {
     *                    'background-color':'red'
     *                }
     *            }
     *        });
     *        p2 = new PMUI.core.Panel({
     *            height : 140,
     *            width : 90,
     *            display : 'inline-block',
     *            style : {
     *                cssProperties : {
     *                    'background-color':'yellow'
     *                }
     *            }
     *        });
     *        panel = new PMUI.core.Panel({
     *            height : 150,
     *            width : 200,
     *            display : 'block',
     *            items : [
     *                p1,
     *                p2
     *            ],
     *            layout : 'vbox',
     *            padding : "5px",
     *            borderWidth : "1px",
     *            border : true
     *        });
     *        document.body.appendChild(panel.getHTML());
     *           
     * @constructor 
     * Creates a new instacne of the object
     * @param {Object} settings
     *
     * @cfg {Object|PMUI.core.Panel|null} [panel=null] The panel to be appended into the current Panel. It can be an 
     * object literal with the {@link PMUI.core.Panel config options for Panel} or an instance of 
     * {@link PMUI.core.Panel Panel} or null.
     * @cfg {PMUI.core.Container} [parent=null] The panel's parent element.
     * @cfg {String|PMUI.layout.Layout} [layout='box'] The layout to be applied to the panel, the layout determines the
     * render position for the panel's items. The possible layouts are:
     *
     * - box: this applies no layout to the panel. To set the layout to box if you use a string as value for the config
     * option use "box", in the other case, if you use an object, use an instance of {@link PMUI.layout.Box Box}.
     * - hbox: this applies a layout that renders the panel's items side by side using the items's 
     * {@link PMUI.core.Element#property-propotion proportion property} to calculate the width for each item. To set
     * this layout to the panel, if you are using a string as the value for the config option, use "hbox", in the other
     * case, if you are using an object use an instance of {@link PMUI.layout.HBox HBox}.
     * - vbox: this applies a layout that renders the panel's items in vertical position using the items's
     * {@link PMUI.core.Element#property-propotion proportion property} to calculate the height for each item. To set
     * this layout to the panel, if you are using a string as the value for the config option, use "vbox", in the other
     * case, if you are using an object use an instance of {@link PMUI.layout.VBox VBox}.
     * @cfg {String|Number} [padding=""] Sets the css padding property for the panel.  It can be a number or and string
     * with the format ##px or an empty string. If you use an empty string as the parameter, the padding will be set 
     * dinamically from the used css file.
     * @cfg {String|Number} [borderWidth=""] Sets the border width for the panel. It can be a number or a string with 
     * the format '##px' or an empty string. If you use an empty string as the parameter, the border width will be set
     * dinamically from the used css file.
     */
    var Panel = function(settings) {
        Panel.superclass.call(this, settings);
        /**
         * The child {Panel} object
         * @type {PMUI.core.Panel}
         */
        this.panel = null;
        /**
         * The Panel's parent object
         * @type {PMUI.core.Container}
         */
        this.parent = null;
        /**
         * A {Layout} object which handles the position layout for the object's direct child elements
         * @property {PMUI.layout.Layout}
         */
        this.layout = null;
        /**
         * @property {String} [topPadding=""] 
         * The value for the css padding-top property to be applied to the object's html.
         * @readonly
         */
        this.topPadding = null;
        /**
         * @property {String} [leftPadding=""] 
         * The value for the css padding-left property to be applied to the object's html.
         * @readonly
         */
        this.leftPadding = null;
        /**
         * @property {String} [bottomPadding=""] 
         * The value for the css padding-bottom property to be applied to the object's html.
         * @readonly
         */
        this.bottomPadding = null;
        /**
         * @property {String} [rightPadding=""] 
         * The value for the css padding-right property to be applied to the object's html.
         * @readonly
         */
        this.rightPadding = null;
        /**
         * @property {String} [borderWidth=""] A string with the format ##px (where ## is a number). 
         * It determines the panel's border width.
         * @readonly
         */
        this.borderWidth = null;
        /**
         * @property {Boolean} [border=false] If the border is being shown.
         * @readonly
         */
        this.border = null;
        Panel.prototype.init.call(this, settings);
    };

    PMUI.inheritFrom('PMUI.core.Container', Panel);

    /**
     * Defines the object's type
     * @type {String}
     */
    Panel.prototype.type = 'Panel';

    /**
     * @private
     * Initializes the object with default values
     * @param  {Object} options 
     */
    Panel.prototype.init = function(settings) {
        var defaults = {
            panel: null,
            parent: null,
            layout: 'box',
            padding: "",
            borderWidth: ""
        };

        jQuery.extend(true, defaults, settings);

        this.setPanel(defaults.panel)
            .setParent(defaults.parent)
            .setLayout(defaults.layout)
            .setPadding(defaults.padding)
            .setBorderWidth(defaults.borderWidth);

    };
    /**
     * Applies the css classes and ptoperties to the element's html which the object is related
     * @chainable
     */
    Panel.prototype.applyStyle = function() {
        if(this.html) {
            this.style.applyStyle();

            this.style.addProperties({
                display: this.visible ? this.display: "none",
                position: this.positionMode,
                left: this.x,
                top: this.y,
                width: this.width,
                height: this.height,
                zIndex: this.zOrder,
                "padding-top": this.topPadding,
                "padding-right": this.rightPadding,
                "padding-bottom": this.bottomPadding,
                "padding-left": this.leftPadding,
                "border-width": this.borderWidth,
                "border-style": this.borderWidth !== "" ? "solid" : "",
                "box-sizing": "border-box"
            });
        }
        return this;
    };
    /**
     * Sets the border width for the panel
     * @param {String|Number} width It can be a number or a string with the format '##px' or an empty string.
     * If you use an empty string as the parameter, the border width will be set dinamically from the used css file.
     */
    Panel.prototype.setBorderWidth = function(width) {
        var error = false;
        if(typeof width === 'number') {
            this.borderWidth = width + "px";
        } else if(typeof width === 'string') {
            width = jQuery.trim(width);
            if(/^\d+(\.\d+)?px$/.test(width) || width === "") {
                this.borderWidth = width;
            } else {
                error = true;
            }
        } else {
            error = false;
        }

        if(error) {
            throw new Error("setBorderWidth(): invalid parameter.");
        } else {
            this.applyStyle();
            if(this.layout) {
                this.layout.applyLayout();
            }
        }

        return this;
    };
    /**
     * Returns the border width from the object
     * @return {String}
     */
    Panel.prototype.getBorderWidth = function() {
        return this.borderWidth || this.style.getProperty("border-width");
    };
    /**
     * Sets the padding values for the panel.
     * @param {String|Number} padding It can be a number or and string with the format ##px or an empty string. 
     * If you use an empty string as the parameter, the padding will be set dinamically from the used css file.
     */
    Panel.prototype.setPadding = function(padding) {
        var error = false;
        if(typeof padding === 'string') {
            padding = jQuery.trim(padding);
            padding = padding.split(/\s+/).join(" ");
            if(/^(\d+(\.\d+)?(px|%)\s)+$/.test(padding + " ")) {
                padding = padding.match(/\d+(\.\d+)?(px|%)/g);
                switch(padding.length) {
                    case 4:
                        this.topPadding = padding[0];
                        this.rightPadding = padding[1];
                        this.bottomPadding = padding[2];
                        this.leftPadding = padding[3];
                        break;
                    case 3:
                        this.topPadding = padding[0];
                        this.rightPadding = this.leftPadding = padding[1];
                        this.bottomPadding = padding[2];
                        break;
                    case 2: 
                        this.topPadding = this.bottomPadding = padding[0];
                        this.rightPadding = this.leftPadding = padding[1];
                        break;
                    case 1:
                        this.topPadding = this.rightPadding = this.bottomPadding = this.leftPadding = padding[0];
                        break;
                }
            } else if(padding === "") {
                this.topPadding = this.rightPadding = this.bottomPadding = this.leftPadding = "";
            } else {
                error = true;
            }
        } else if(typeof padding === 'number') {
            this.topPadding = this.rightPadding = this.bottomPadding = this.leftPadding = padding + "px";
        } else {
            error = true;
        }

        if(error) {
            throw new Error("setPadding(): Invalid parameter.");   
        } else {
            this.applyStyle();
            if(this.layout) {
                this.layout.applyLayout();
            }
        }

        return this;
    };
    /**
     * Returns an array with the padding values in the following order: [top, right, bottom, left].
     * @return {Array}
     */
    Panel.prototype.getPadding = function() {
        return [
            this.topPadding || this.style.getProperty("padding-top"), 
            this.rightPadding || this.style.getProperty("padding-right"), 
            this.bottomPadding || this.style.getProperty("padding-bottom"), 
            this.leftPadding || this.style.getProperty("padding-left")
        ];
    };
    /**
     * Sets the panel object
     * @param {Object} panel Panel object
     */
    Panel.prototype.setPanel = function(panel) {
        if(panel) {
            if(panel instanceof Panel) {
                this.panel = panel;
            } else if(typeof panel === 'object') {
                this.panel = new Panel(panel);
            }

            if(this.html) {
                jQuery(this.html).empty().append(panel.getHTML());
            }
        }

        return this;
    };
    /**
     * Returns the object's html usable width.
     * @return {Number|String} It can return a Number or an String with a css property like "auto".
     */
    Panel.prototype.getUsableWidth = function() {
        var padding = this.getPadding(), border = parseInt(this.getBorderWidth(), 10) * 2;
        if(isNaN(this.getWidth())) {
            return this.getWidth();
        }
        return this.getWidth() - (parseInt(padding[1], 10) || 0) - (parseInt(padding[3], 10) || 0) - (border || 0);
    };
    /**
     * Returns the object's html usable height
     * @return {Number|String} It can return a Number or an String with a css property like "auto".
     */
    Panel.prototype.getUsableHeight = function() {
        var padding = this.getPadding(), border = parseInt(this.getBorderWidth(), 10) * 2;
        if(isNaN(this.getHeight())) {
            return this.getHeight();
        }
        return this.getHeight() - (parseInt(padding[0], 10) || 0) - (parseInt(padding[2], 10) || 0) - (border || 0);
    };
    /**
     * Sets the parent object
     * @param {Object} parent
     */
    Panel.prototype.setParent = function(parent) {
        this.parent = parent;
        return this;
    };

    /**
     * Sets the layout for the panel.
     * @param {Object|String} layout Layout object
     */
    Panel.prototype.setLayout = function(layout) {
        var factory = new PMUI.layout.LayoutFactory();
        if(!(typeof layout === "string")) {
            throw new Error ("setLayout(): Property values sent an invalid, expected 'box' , 'vbox' or 'hbox'");
        }
        this.layout = factory.make(layout);
        this.layout.setContainer(this);
        if (this.html) {
            this.layout.applyLayout();
        }
        return this;
    };

    /**
     * Add an item to the panel.
     * @param {PMUI.core.Element|Object} item.
     * @param {Number} [index] The index position in which the item will be appended.
     * It can be a valid JSON object or an object that inherits from {@link PMUI.core.Element Element}.
     * @chainable
     */
    Panel.prototype.addItem = function(item, index) {
        Panel.superclass.prototype.addItem.call(this, item, index);
        if(this.layout) {
            this.layout.applyLayout();
        }

        return this;
    };
    /**
     * Sets the width for the Panel's HTML element
     * @param {Number|String} width It can be a number or a string.
     * In case of using a String you only can use 'auto' or 'inherit' or ##px or ##% or ##em when ## is a number
     * @chainable
     */
    Panel.prototype.setWidth = function(width) {
        Panel.superclass.prototype.setWidth.call(this, width);
        if(this.layout) {
            this.layout.applyLayout();
        }

        return this;
    };
    /**
     * Sets the height for the Panel's HTML element
     * @param {Number|String} height It can be a number or a string.
     * In case of using a String you only can use 'auto' or 'inherit' or ##px or ##% or ##em when ## is a number
     * @chainable
     */
    Panel.prototype.setHeight = function(height) {
        Panel.superclass.prototype.setHeight.call(this, height);
        if(this.layout) {
            this.layout.applyLayout();
        }
        return this;
    };
    /**
     * Returns the object pointed to the parent
     * @return {Object}
     */
    Panel.prototype.getParent = function() {
        return this.parent;
    };
    /**
     * Creates the HTML element for the Panel
     * @return {HTMLElement} 
     */
    Panel.prototype.createHTML = function() {
        if(this.html) {
            return this.html;
        }

        Panel.superclass.prototype.createHTML.call(this);
        this.applyStyle();
        if(this.layout) {
            this.layout.applyLayout();
        }

        return this.html;
    };

    // Declarations created to instantiate in NodeJS environment
    if (typeof exports !== "undefined") {
        module.exports = Panel;
    }

    PMUI.extendNamespace('PMUI.core.Panel', Panel);
}());
(function(){
    /**
     * @class PMUI.control.Control
     * Class that encapsulates the basic bahavior for a form control.
     * Since this class is abstract it shouldn't be instantiated.
     * @extends PMUI.core.Element
     * @abstract
     *
     * Usage example (only for subclasses since this is an abstract class):
     *
     *      //Remember, this is an abstract class so it shouldn't be instantiate,
     *      //anyway we are instantiating it just for this example
     *      var myControl = new PMUI.control.Control({
     *          name: "phrase",
     *          value: "Sometimes the sun goes down!",
     *          disabled: false,
     *          onChange: function(currentValue, previousValue) {
     *              console.log(this.name + " has changed!");
     *              console.log("Its previous value was: " + previousValue);
     *              console.log("Its current value is: " + currentValue);
     *          }
     *      });
     * 
     * @constructor
     * While it is true that this class must not be instantiated, 
     * it is useful to mention the settings parameter for the constructor 
     * function (which will be used for the non abstract subclasses).
     * The parameter for the constructor will be a JSON object whose properties are specified in the
     * {@link PMUI.control.Control#cfg-disabled Config Options section}.
     * 
     * @cfg {String} [name=The object's ID] The name for the control. If it is not specified then the object's id is 
     * used.
     * @cfg {String} [value=""] The initial value to be set to the control.
     * @cfg {PMUI.form.Field} [field=null] The parent {@link PMUI.form.Field Field} object for the control.
     * @cfg {Boolean} [disabled=false] A boolean value which determines if the control will be enabled or not.
     * @cfg {Function} [onChange=null] A callback function to be invoked when the control changes. For info about the 
     * received parameters please read the {@link PMUI.control.Control#event-onChange onChange event} section.
     * @cfg {Function} [onBeforeChange=null] A callback function to be invoked after the control's value changes. The
     * function can return false to avoid the value changing. For info about the received parameter please read the 
     * {@link #event-onBeforeChange onBeforeChange event} documentation.
     */
    var Control = function(settings) {
        Control.superclass.call(this, settings);
        /**
         * @property {String} [name=id] The control's name, it defaults to null.
         * @readonly
         */
        this.name = null;
        /**
         * @property {String} [value=""] The control's value.
         * @readonly
         */
        this.value = null;
        /**
         * @property {PMUI.form.Field} [field=null] The {@link PMUI.form.Field Field} 
         object the current object belongs to.
         * @readonly
         */
        this.field = null;
        /**
         * @property {Boolean} [disabled=false] If the control is disabled or not.
         * @readonly
         */
        this.disabled = null;
        /**
         * @event onChange
         * Fired when the control's value is changed.
         * @param {String} newValue The new control's value.
         * @param {String} prevValue The previous control's value.
         */
        this.onChange = null;
        /**
         * @event onBeforeChange
         * Fired after the control's value changes.
         * @param {String} newValue The new control's value.
         * @param {String} prevValue The previous vontrol's value.
         */
        this.onBeforeChange = null;
        Control.prototype.init.call(this, settings);
    };

    PMUI.inheritFrom('PMUI.core.Element', Control);

    Control.prototype.type = "Control";
    /**
     * Initialize the object.
     * @param  {Object} settings The config options.
     */
    Control.prototype.init = function(settings) {
        var defaults = {
            name: this.id, 
            value: "",
            field: null, 
            disabled: false,
            onBeforeChange: null,
            onChange: null
        };

        $.extend(true, defaults, settings);

        this.setName(defaults.name)
            .setValue(defaults.value)
            .setField(defaults.field)
            .disable(defaults.disabled)
            .setOnBeforeChangeHandler(defaults.onBeforeChange)
            .setOnChangeHandler(defaults.onChange);
    };
    /**
     * Sets the callback function to be executed when the {@link #event-onBeforeChange onBeforeChange event} fires.
     * @param {Function|null} handler
     * @chainable
     */
    Control.prototype.setOnBeforeChangeHandler = function(handler) {
        if(!(handler === null || typeof handler === 'function')) {
            throw new Error('setOnBeforeChangeHandler(): The parameter must be a function or null.');
        }
        this.onBeforeChange = handler;
        return this;
    };
    /**
     * Sets the control's name.
     * @param {String} name
     * @chainable
     */
    Control.prototype.setName = function(name) {
        if(typeof name === "string" || typeof name === "number") {
            this.name = name.toString();
            if(this.html) {
                this.html.setAttribute("name", name);
            }
        } else {
            throw new Error("The setName() method only accepts string or number values");
        }

        return this;
    };
    /**
     * Returns the control's name.
     * @return {String}
     */
    Control.prototype.getName = function() {
        return this.name;
    };
    /**
     * Sets the control's value.
     * @param {String} value
     * @chainable
     */
    Control.prototype.setValue = function(value) {
        if(typeof value !== 'undefined') {
            this.value = value.toString();
        } else {
            throw new Error("setValue(): a parameter is required.");
        }

        return this;
    };
    /**
     * Returns the control's value.
     * @return {String}
     */
    Control.prototype.getValue = function() {
        return this.value;
    };
    /**
     * Sets the control's parent {@link PMUI.form.Field Field} object.
     * @param {PMUI.form.Field}
     * @chainable
     */
    Control.prototype.setField = function(field) {
        if(field instanceof PMUI.form.Field) {
            this.field = field;
        }

        return this;
    };
    /**
     * Returns the control's parent {@link PMUI.form.Field Field} object.
     * @return {PMUI.form.Field}
     */
    Control.prototype.getField = function() {
        return this.field;
    };
    /**
     * Disables/enables the control.
     * @param {Boolean} disable If the value is evaluated as true then the control is disabled, 
     otherwise the control is enabled.
     * @chainable
     */
    Control.prototype.disable = function(disable) {
        this.disabled = !!disable;
        return this;
    };
    /**
     * Returns true if the control is enabled, if it don't then it returns false
     * @return {Boolean}
     */
    Control.prototype.isEnabled = function() {
        return !this.disabled;
    };
    /**
     * Sets the callback function which will be executed everytime the control changes (using the interface).
     *
     * The callback function will receive two parameters:
     *
     * - first argument: the current control's value.
     * - second argument: the previous control's value.
     *
     * 
     *          //Remember, this is an abstract class so it shouldn't be instantiate,
     *          //anyway we are instantiating it just for this example
     *          var myControl = new PMUI.control.Control({
     *              name: "phrase",
     *              value: "Sometimes the sun goes down!",
     *              disabled: false
     *          });
     *
     *          myControl.setOnChangeHandler(function(currentValue, previousValue) {
     *              console.log(this.name + " has changed!");
     *              console.log("Its previous value was: " + previousValue);
     *              console.log("Its current value is: " + currentValue);
     *          });
     *      
     * @chainable
     */
    Control.prototype.setOnChangeHandler = function(handler) {
        if(typeof handler === 'function') {
            this.onChange = handler;
        }

        return this;
    };
    /**
     * Returns the control's value directly from the control's HTML element.
     * 
     * This method is used internally by the object, so in most of the cases you won't need to invocated. 
     * To get the control's value please use the {@link PMUI.control.Control#getValue getValue()} method.
     *
     * Since this is an abstract method, it must be implemented in its non-abstract subclasses
     * @return {String}
     * @abstract
     */
    Control.prototype.getValueFromRawElement = function() {
        throw new Error("Calling getValueFromRawElement() from PMUI.control.Control: this is an abstract method!");
    };
    /**
     * A method which is called everytime the control changes.
     * 
     * This method is used internally by the object, so in most of the cases you won't need to invocated. 
     * To execute instructions when the control changes, please use the 
     {@link PMUI.control.Control#setOnChangeHandler setOnChangeHandler()} method.
     * 
     * @chainable
     * @private
     */
    Control.prototype.onChangeHandler = function() {
        var prevValue = this.value, newValue = this.getValueFromRawElement(), resCallback;
        if(typeof this.onBeforeChange === 'function' && newValue !== prevValue) {
            resCallback = this.onBeforeChange(newValue, prevValue);
            if(resCallback === false) {
                newValue = prevValue;
                this.setValue(newValue);
            }
        }
        this.value = newValue;
        if(typeof this.onChange === 'function' && this.value !== prevValue) {
            this.onChange(this.value, prevValue);
        }

        return this;
    };
    /**
     * Attach the event listeners for the control's HTML element.
     * 
     * Since this is an abstract method, it must be implemented in its non-abstract subclasses
     * @abstract
     * @chainable
     */
    Control.prototype.attachListeners = function() {
        throw new Error("Calling attachListeners() from PMUI.control.Control: this is an abstract method!");
    };
    /**
     * Creates the HTML element for the control.
     * 
     * Since this is an abstract method, it must be implemented in its non-abstract subclasses
     * @return {HTMLElement}
     * @abstract
     */
    Control.prototype.createHTML = function() {
        Control.superclass.prototype.createHTML.call(this);
        return this.html;
    };
    /**
     * Returns the HTML element for the control
     * @return {HTMLElement}
     */
    Control.prototype.getHTML = function() {
        if(!this.html) {
            this.html = this.createHTML();
        }

        return this.html;
    };
    /**
     * @method setFocus
     * set the focus on control
     * @chainable
     */
    Control.prototype.setFocus = function () {
        if (this.html){
            this.html.focus();
        } 
    };

    PMUI.extendNamespace('PMUI.control.Control', Control);

    // Declarations created to instantiate in NodeJS environment
    if (typeof exports !== "undefined") {
        module.exports = Control;
    }
}());
(function() {
    /**
     * @class PMUI.control.HTMLControl
     * Class that encapsulates the HTML native control's behavior.
     * Since this class is abstract it shouldn't be instantiated.
     * @extends PMUI.control.Control
     * @abstract
     *
     * @constructor
     * While it is true that this class must not be instantiated, 
     * it is useful to mention that the settings parameter for the constructor function
     * has the same structure that the one for the superclass 
     * (for more info see the {@link PMUI.control.Control#constructor constructor} method for Control class).
     * @param {Object} [settings=null] A JSON object with the config options.
     */
    var HTMLControl = function(settings) {
        HTMLControl.superclass.call(this, settings);
        /**
         * @property {String} [elementTag='input'] The tag for the HTML element to be created.
         * @private
         */
        this.elementTag = 'input';
    };

    PMUI.inheritFrom('PMUI.control.Control', HTMLControl);

    HTMLControl.prototype.type = "HTMLControl";

    /**
     * Sets the value for the control
     * @param {String} value
     * @chainable
     */
    HTMLControl.prototype.setValue = function(value) {
        HTMLControl.superclass.prototype.setValue.call(this, value);
        if(this.html && this.html.value !== this.value) {
            this.html.value = this.value;           
        }

        return this;
    };
    
    /**
     * Disables/enables the control
     * @param {Boolean} disable If the value is evaluated as true then the control 
     is disabled, otherwise the control is enabled.
     * @chainable
     */
    HTMLControl.prototype.disable = function(disable) {
        HTMLControl.superclass.prototype.disable.call(this, disable);
        if(this.html) {
            this.html.disabled = this.disabled;
        }

        return this;
    };
    /**
     * Returns the control's value directly from the control's HTML element.
     * 
     * This method is used internally by the object, so in most of the cases you won't need to invocated. 
     * To get the control's value please use the {@link PMUI.control.Control#getValue getValue()} method.
     * @return {String}
     */
    HTMLControl.prototype.getValueFromRawElement = function() {
        return this.html.value;
    };
    /**
     * Attach the event listeners for the control's HTML element
     * @chainable
     */
    HTMLControl.prototype.defineEvents = function() {
        var that = this;
        HTMLControl.superclass.prototype.defineEvents.call(this);
        if(this.html) {
            this.addEvent('blur').listen(this.html, function() {
                that.onChangeHandler();
            });
        }

        return this;
    };
    /**
     * Creates the HTML element for the control.
     * @return {HTMLElement}
     */
    HTMLControl.prototype.createHTML = function() {
        if(this.html) {
            return this.html;
        }
        HTMLControl.superclass.prototype.createHTML.call(this);
        this.setName(this.name)
            .setValue(this.value)
            .disable(this.disabled);

        return this.html;
    };

    PMUI.extendNamespace('PMUI.control.HTMLControl', HTMLControl);

    // Declarations created to instantiate in NodeJS environment
    if (typeof exports !== "undefined") {
        module.exports = HTMLControl;
    }
}());
(function(){
    /**
     * @class PMUI.control.TextControl
     * Class for handle the HTML native text input control.
     * @extends PMUI.control.HTMLControl
     *
     * Quick usage example:
     *
     *      @example
     *      myTextbox = new PMUI.control.TextControl({
     *          name: "my_name",
     *          value: "John Doe",
     *          maxLength: 12,
     *          placeholder: "insert your name here",
     *          disabled: false,
     *          onChange: function(currentValue, previousValue) {
     *              if(previousValue !== "") {
     *                  alert("Your name is not \"" + previousValue + "\" anymore.\nNow it's \"" + currentValue + "\"");
     *              } else {
     *                  alert("Now your name is " + currentValue);
     *              }
     *          }
     *      });
     *      
     *      document.body.appendChild(myTextbox.getHTML());
     *
     * @constructor
     * Creates a new instance of the TextControl object.
     * @param {Object} [settings=null] A JSON object with the config options.
     *
     * @cfg {String} [placeholder=""] A string to be used as the control's placeholder.
     * @cfg {Number} [maxLength=524288] A number which specifies the maximum character length the control can accept.
     * @cfg {Function} [onKeyUp=null] A callback function to be called every time a pressed key is released on the 
     * control. For info about the callback parameters please read the 
     * {@link PMUI.control.TextControl#event-onKeyUp onKeyUp event} section.
     */
    var TextControl = function(settings) {
        TextControl.superclass.call(this, settings);
        /**
         * @property {String} [placeholder=""]
         * The control's placeholder (the text to be shown inside the control when there is not any text in it).
         * @readonly
         */
        this.placeholder = null;
        /**
         * @property {Number} [maxLength=524288] The maximum character length the control accepts.
         * @readonly
         */
        this.maxLength = null;
        /**
         * @event onKeyUp
         * Fired when a pressed key is released.
         * @param {Object} event The event object.
         */
        this.onKeyUp = null;
        TextControl.prototype.init.call(this, settings);
    };

    PMUI.inheritFrom('PMUI.control.HTMLControl', TextControl);

    TextControl.prototype.type  = "TextControl";
    /**
     * Initializes the object.
     * @param  {Object} [settings=null] A JSON object with the config options.
     * @private
     */
    TextControl.prototype.init = function(settings) {
        var defaults = {
            placeholder: "",
            maxLength: 524288,
            onKeyUp: null,
            height : 30
        };

        $.extend(true, defaults, settings);

        this.setPlaceholder(defaults.placeholder)
            .setMaxLength(defaults.maxLength)
            .setOnKeyUpHandler(defaults.onKeyUp)
            .setHeight(defaults.height);

    };
    /**
     * Sets the callback function to be called when the keyup event occurs.
     * @chainable
     */
    TextControl.prototype.setOnKeyUpHandler = function(handler) {
        if(typeof handler === 'function') {
            this.onKeyUp = handler;
        }

        return this;
    };
    /**
     * Sets the placeholder test to show in the control when there's not ant value in it.
     * @param {String} placeholder
     * @chainable
     */
    TextControl.prototype.setPlaceholder = function(placeholder) {
        if(typeof placeholder === 'string') {
            this.placeholder = placeholder;
            if(this.html) {
                this.html.placeholder = placeholder;
            }
        }

        return this;
    };
    /**
     * Sets the maximun character number to be accepted in the control.
     * @param {Number} maxLength The number must be an integer. 
     * If the value is minor or equal to 0 then the maxLength property is set to the default (524288).
     * @chainable>
     */
    TextControl.prototype.setMaxLength = function(maxLength) {
        if(typeof maxLength === 'number' && maxLength % 1 === 0) {
            this.maxLength = maxLength;
            if(this.html) {
                this.html.maxLength = maxLength > 0 ? maxLength : 524288;
            }
        } else {
            throw new Error("method setMaxLength() only accepts integer values.");
        }

        return this;
    };
    /**
     * Set the events for the object.
     * @chainable
     */
    TextControl.prototype.defineEvents = function() {
        var that = this;
        TextControl.superclass.prototype.defineEvents.call(this);
        if(this.html) {
            this.addEvent('keyup').listen(this.html, function(e) {
                if(typeof that.onKeyUp === 'function') {
                    that.onKeyUp(e);
                }
            });
        }
        return this;
    };
    /**
     * Creates the HTML element for the object
     * @return {HTMLElement}
     */
    TextControl.prototype.createHTML = function() {
        TextControl.superclass.prototype.createHTML.call(this);
        this.html.type = "text";
        this.setPlaceholder(this.placeholder)
            .setMaxLength(this.maxLength)
            .setReadOnly(this.readonly);

        return this.html;
    };
    /**
     * Sets if the control will be enabled for read only.
     * @param {Boolean}
     * @chainable
     */
    TextControl.prototype.setReadOnly = function(readonly) {
        if(typeof readonly !== 'undefined') {
            this.readonly = !!readonly;
            if(this.html) {
                this.html.readOnly = this.readonly;
            }
        }
        return this;
    };
    /**
     * Returns true if the control is enabled for read only.
     * @return {Boolean}
     */
    TextControl.prototype.isReadOnly = function() {
        return this.readonly;
    };

    TextControl.prototype.getMaxLength = function () {
        return this.maxLength;
    }

    PMUI.extendNamespace('PMUI.control.TextControl', TextControl);

    // Declarations created to instantiate in NodeJS environment
    if (typeof exports !== "undefined") {
        module.exports = TextControl;
    }
}());
(function(){
    /**
     * @class PMUI.control.PasswordControl
     * Class for handle the HTML native password input control.
     * @extends PMUI.control.HTMLControl
     *
     * Quick usage example:
     *
     *      @example
     *      myTextbox = new PMUI.control.PasswordControl({
     *          name: "my_name",
     *          value: "John Doe",
     *          maxLength: 12,
     *          disabled: false,
     *          onChange: function(currentValue, previousValue) {
     *              if(previousValue !== "") {
     *                  alert("Your name is not \"" + previousValue + "\" anymore.\nNow it's \"" + currentValue + "\"");
     *              } else {
     *                  alert("Now your name is " + currentValue);
     *              }
     *          }
     *      });
     *
     *      document.body.appendChild(myTextbox.getHTML());
     *
     * @constructor
     * Creates a new instance of the PasswordControl object.
     * @param {Object} [settings=null] A JSON object with the config options.
     *
     * @cfg {Number} [maxLength=524288] A number which specifies the maximum character length the control can accept.
     * @cfg {Function} [onKeyUp=null] A callback function to be called every time a pressed key is released on the
     * control. For info about the callback parameters please read the
     * {@link PMUI.control.PasswordControl#event-onKeyUp onKeyUp event} section.
     */
    var PasswordControl = function(settings) {
        PasswordControl.superclass.call(this, settings);
//        /**
//         * @property {String} [placeholder=""]
//         * The control's placeholder (the text to be shown inside the control when there is not any text in it).
//         * @readonly
//         */
//        this.placeholder = null;
        /**
         * @property {Number} [maxLength=524288] The maximum character length the control accepts.
         * @readonly
         */
        this.maxLength = null;
        /**
         * @event onKeyUp
         * Fired when a pressed key is released.
         * @param {Object} event The event object.
         */
        this.onKeyUp = null;
        PasswordControl.prototype.init.call(this, settings);
    };

    PMUI.inheritFrom('PMUI.control.HTMLControl', PasswordControl);

    PasswordControl.prototype.type  = "PasswordControl";
    /**
     * Initializes the object.
     * @param  {Object} [settings=null] A JSON object with the config options.
     * @private
     */
    PasswordControl.prototype.init = function(settings) {
        var defaults = {
        //    placeholder: "",
            maxLength: 524288,
            onKeyUp: null,
            height : 30
        };

        $.extend(true, defaults, settings);

        this.setMaxLength(defaults.maxLength)
            //.setPlaceholder(defaults.placeholder)
            .setOnKeyUpHandler(defaults.onKeyUp)
            .setHeight(defaults.height);
    };
    /**
     * Sets the callback function to be called when the keyup event occurs.
     * @chainable
     */
    PasswordControl.prototype.setOnKeyUpHandler = function(handler) {
        if(typeof handler === 'function') {
            this.onKeyUp = handler;
        }

        return this;
    };
//    /**
//     * Sets the placeholder test to show in the control when there's not ant value in it.
//     * @param {String} placeholder
//     * @chainable
//     */
//    PasswordControl.prototype.setPlaceholder = function(placeholder) {
//        if(typeof placeholder === 'string') {
//            this.placeholder = placeholder;
//            if(this.html) {
//                this.html.placeholder = placeholder;
//            }
//        }
//
//        return this;
//    };
    /**
     * Sets the maximun character number to be accepted in the control.
     * @param {Number} maxLength The number must be an integer.
     * If the value is minor or equal to 0 then the maxLength property is set to the default (524288).
     * @chainable>
     */
    PasswordControl.prototype.setMaxLength = function(maxLength) {
        if(typeof maxLength === 'number' && maxLength % 1 === 0) {
            this.maxLength = maxLength;
            if(this.html) {
                this.html.maxLength = maxLength > 0 ? maxLength : 524288;
            }
        } else {
            throw new Error("method setMaxLength() only accepts integer values.");
        }

        return this;
    };
    /**
     * Set the events for the object.
     * @chainable
     */
    PasswordControl.prototype.defineEvents = function() {
        var that = this;

        PasswordControl.superclass.prototype.defineEvents.call(this);
        this.addEvent('keyup').listen(this.html, function(e) {
            if(typeof that.onKeyUp === 'function') {
                that.onKeyUp(e);
            }
        });
        return this;
    };
    /**
     * Creates the HTML element for the object
     * @return {HTMLElement}
     */
    PasswordControl.prototype.createHTML = function() {
        PasswordControl.superclass.prototype.createHTML.call(this);
        this.html.type = "password";
        this.setMaxLength(this.maxLength);
            //.setPlaceholder(this.placeholder);

        return this.html;
    };

    PMUI.extendNamespace('PMUI.control.PasswordControl', PasswordControl);

    // Declarations created to instantiate in NodeJS environment
    if (typeof exports !== "undefined") {
        module.exports = PasswordControl;
    }
}());
(function(){
    /**
     * @class PMUI.control.DropDownListControl
     * @extends PMUI.control.HTMLControl
     * Class to handle the Select HTML form element.
     * 
     * Example usage:
     *
     *      @example
     *      var control;
     *      $(function() {
     *          var settings = {
     *              name: "myList",
     *              options: [
     *                  {
     *                      label: "one",
     *                      value: 1
     *                  },
     *                  {
     *                      label: "two",
     *                      value: 2,
     *                      disabled: true
     *                  },
     *                  {
     *                      label: "three",
     *                      value: 3,
     *                      selected: true
     *                  }, 
     *                  {
     *                      label: "Letters",
     *                      options: [
     *                          {
     *                              value: "A"
     *                          },
     *                          {
     *                              value: "B"
     *                          }
     *                      ]
     *                  },
     *                  {
     *                      label: "months",
     *                      disabled: true,
     *                      options: [
     *                          {
     *                              value: "january"
     *                          },
     *                          {
     *                              value: "february"
     *                          }
     *                      ]
     *                  }
     *              ],
     *              value: 2
     *          };
     *          control = new PMUI.control.DropDownListControl(settings);
     *          document.body.appendChild(control.getHTML());
     *      });
     *
     * @constructor
     * Creates a new instance of the DropDownListControl.
     * @param {Object} [settings=null] An JSON object with the config options.
     */
    /**
     * @cfg {Array} [options=[]]
     * An array with all the options to be contained by the control.
     * 
     * Each element in the array is a JSON object, this JSON object can represent an option group 
     * or an option item. 
     * 
     * In case to represent an option item it can contain the next properties:
     *
     * - value {String} (required): the value for the option.
     * - label {String} (optional): the label for the option, if isn't specified the value is used instead.
     * - selected {Boolean} (optional): if the option is selected. #Note. If the configuration object has the
     * "value" propery set then this "selected" property will be
     * - disabled {Boolean} (optional): if the option is disabled or not.
     *
     * On the other hand, in case to represent an option group, it can contain the next properties:
     *
     * - label {String} (required): The name for the option group.
     * - disabled {Boolean} (optional): If the group is disabled or not.
     * - options {Array} (required): An array in which each element is a JSON object representing an option item,
     * so every item must have the structure explained above (for represent option items). #Note. This propery makes 
     * the difference between an option and a option group. If the "options" property is not specified or if it isn't 
     * an array then it will treated like a option item.
     * @cfg {String|Number} [value=null] The value of the option that is wanted to be selected. It must be one of the values
     * of the list options, otherwise it will be set to "".
     */
    var DropDownListControl = function(settings) {
        DropDownListControl.superclass.call(this, settings);
        /**
         * @property {Array} [options] An array with all the options/option groups form the control.
         * @private
         */
        this.options = [];
        /**
         * @property {String} [elementTag='input'] The tag for the HTML element to be created.
         * @private
         */
        this.elementTag = 'select';
        DropDownListControl.prototype.init.call(this, settings);
    };

    PMUI.inheritFrom('PMUI.control.HTMLControl', DropDownListControl);

    DropDownListControl.prototype.type = "DropDownListControl";    
    /**
     * Initializes the object.
     * @param  {Oject} [settings=null] A JSON opbject with the config options.
     * @private
     */
    DropDownListControl.prototype.init = function(settings) {
        var defaults = {
            options: [],
            value: null,
            height : 30
        };

        jQuery.extend(true, defaults, settings);

        this.setOptions(defaults.options);
        this.setHeight(defaults.height);

        if(defaults.value !== null) {
            this.setValue(defaults.value);
        }
    };
    /**
     * Clear all the options from the control.
     * @chainable
     */
    DropDownListControl.prototype.clearOptions = function() {
        this.options = [];
        if(this.html) {
            jQuery(this.html).empty();
        }
        this.value = "";
        return this;
    };
    /**
     * Enables or disables one or more options/option groups.
     * @param  {Boolean} disabled If the function will disable (use true) or enable (use false)
     * the options/option groups.
     * @param  {String|Number|Object} option It can be a string, a number or a JSON object.
     * 
     * - In case to be a String, it will be enabled/disabled the options that match the string in its value and the 
     * option groups that match the string in its label. In this case more than one single item can be 
     * enabled/disabled.
     * - In case to be a Number, it will be enabled/disabled the option/option group which index position matches the 
     * number. Obviously, in this case only one item will be enabled/disabled.
     * - In case to be an object you can specify if the change will be applied only to options or option groups, 
     * it should have two properties:
     *     - criteria {String}: The value (in case of options) or the label (in case of option groups) the items needs 
     *     to match for apply the changes.
     *     - applyTo {String} (optional), it can take the following values:
     *         - "groups", the change will be applied only to the option groups.
     *         - "options", the change will be applied only to the options (direct child of the object).
     *         - [any other string value], the default value, it indicates that the change will be applied to both 
     *         options/option groups that matches the criteria in its value/label respectly.
     *
     * @param  {String} [group] It it is specified must be an String making reference to an existing option group label.
     * Using this parameter, the elements to be match by the second parameter will be search only in the option groups 
     * that match this parameter in its label.
     * @chainable
     * @private
     */
    DropDownListControl.prototype.enableDisableOption = function(disabled, option, group) {
        var dataGroupTarget = this, htmlGroupTarget = jQuery(this.html), i, j, suboptions, 
            objectDefaults = {
                applyTo: 'all'
            };
        disabled = !!disabled;

        if(group) {
            for(i = 0; i < this.options.length; i += 1) {
                if(this.options[i].isGroup && this.options[i].label === group) {
                    dataGroupTarget = this.options[i];
                    break;
                }
            }
            if(dataGroupTarget === this.options) {
                throw new Error('disableOption(): the group "' + group + '" wasn\'t found.');
            }
            htmlGroupTarget = jQuery(this.html).find('>optgroup[label="' + group + '"]');
        }

        if(typeof option === 'number') {
            dataGroupTarget.options[option].disabled = disabled;
            htmlGroupTarget.find(">*").eq(option).attr("disabled", disabled);
        } else if(typeof option === 'string') {
            for(i = 0; i < dataGroupTarget.options.length; i += 1) {
                if(!dataGroupTarget.options[i].isGroup && dataGroupTarget.options[i].value === option) {
                    dataGroupTarget.options[i].disabled = disabled;
                } else if(dataGroupTarget.options[i].isGroup && dataGroupTarget.options[i].label === option) {
                    dataGroupTarget.options[i].disabled = disabled;
                    suboptions = dataGroupTarget.options[i].options;
                    for(j = 0; j < suboptions.length; j += 1) {
                        if(suboptions[j].value === option) {
                            suboptions[j].disabled = true;
                        }
                    }
                }
            }
            jQuery(htmlGroupTarget).find('option[value="' + option + '"]').add('optgroup[label="' + option + '"]')
                .attr("disabled", disabled);
        } else if(typeof option === 'object') {
            jQuery.extend(true, objectDefaults, option);
            if(objectDefaults.applyTo === 'groups') {
                for(i = 0; i < dataGroupTarget.options.length; i += 1) {
                    if(dataGroupTarget.options[i].isGroup && dataGroupTarget.options[i].label === option.criteria) {
                        dataGroupTarget.options[i].disabled = disabled;
                    }
                }
                jQuery(htmlGroupTarget).find('optgroup[label="' + option.criteria + '"]').attr("disabled", disabled);
            } else if(objectDefaults.applyTo === 'options') {
                for(i = 0; i < dataGroupTarget.options.length; i += 1) {
                    if(!dataGroupTarget.options[i].isGroup && dataGroupTarget.options[i].value === option.criteria) {
                        dataGroupTarget.options[i].disabled = disabled;
                    }
                }
                jQuery(htmlGroupTarget).find('option[value="' + option.criteria + '"]')
                    .attr("disabled", disabled.criteria);
            } else {
                option = option.criteria;
                for(i = 0; i < dataGroupTarget.options.length; i += 1) {
                    if(!dataGroupTarget.options[i].isGroup && dataGroupTarget.options[i].value === option) {
                        dataGroupTarget.options[i].disabled = disabled;
                    } else if(dataGroupTarget.options[i].isGroup && dataGroupTarget.options[i].label === option) {
                        dataGroupTarget.options[i].disabled = disabled;
                        suboptions = dataGroupTarget.options[i].options;
                        for(j = 0; j < suboptions.length; j += 1) {
                            if(suboptions[j].value === option) {
                                suboptions[j].disabled = true;
                            }
                        }
                    }
                }
                jQuery(htmlGroupTarget).find('option[value="' + option + '"]').add('optgroup[label="' + option + '"]')
                    .attr("disabled", disabled);
            }
        } else {
            throw new Error('disableOption(): the first parameter must be a Number or a String.');
        }

        return this;
    };
    /**
     * Disables one or more options/option groups.
     * @param  {String|Number|Object} option It can be a string, a number or a JSON object.
     * 
     * - In case to be a String, it will be disabled the options that match the string in its value and the option 
     * groups which match the string in its label. In this case more than one single item can be disabled.
     * - In case to be a Number, it will be disabled the option/option group which index position matches the number.
     * Obviously, in this case only one item will be disabled.
     * - In case to be an object you can specify if the change will be applied only to options or option groups, 
     * it should have two properties:
     *     - criteria {String}: The value (in case of options) or the label (in case of option groups) the items needs 
     *     to match for apply the changes.
     *     - applyTo {String} (optional), it can take the following values:
     *         - "groups", the change will be applied only to the option groups.
     *         - "options", the change will be applied only to the options (direct child of the object).
     *         - [any other string value], the default value, it indicates that the change will be applied to both 
     *         options/option groups that matches the criteria in its value/label respectly.
     *
     * @param  {String} [group] It it is specified must be an String making reference to an existing option group label.
     * Using this parameter, the elements to be match by the first parameter will be search only in the option groups 
     * that match this parameter in its label.
     * @chainable
     */
    DropDownListControl.prototype.disableOption = function(option, group) {
        return this.enableDisableOption(true, option, group);
    };
    /**
     * Enables one or more options/option groups.
     * @param  {String|Number|Object} option It can be a string, a number or a JSON object.
     * 
     * - In case to be a String, it will be enabled the options that match the string in its value and the option 
     * groups which match the string in its label. In this case more than one single item can be enabled.
     * - In case to be a Number, it will be enabled the option/option group which index position matches the number.
     * Obviously, in this case only one item will be enabled.
     * - In case to be an object you can specify if the change will be applied only to options or option groups, 
     * it should have two properties:
     *     - criteria {String}: The value (in case of options) or the label (in case of option groups) the items needs 
     *     to match for apply the changes.
     *     - applyTo {String} (optional), it can take the following values:
     *         - "groups", the change will be applied only to the option groups.
     *         - "options", the change will be applied only to the options (direct child of the object).
     *         - [any other string value], the default value, it indicates that the change will be applied to both 
     *         options/option groups that matches the criteria in its value/label respectly.
     *
     * @param  {String} [group] It it is specified must be an String making reference to an existing option group label.
     * Using this parameter, the elements to be match by the first parameter will be search only in the option groups 
     * that match this parameter in its label.
     * @chainable
     */
    DropDownListControl.prototype.enableOption = function(option, group) {
        return this.enableDisableOption(false, option, group);
    };
    /**
     * Removes one or more option/option groups.
     * @param  {String|Number|Object} option It can be a string, a number or a JSON object.
     * 
     * - In case to be a String, it will be removed the options that match the string in its value and the option 
     * groups which match the string in its label. In this case more than one single item can be removed.
     * - In case to be a Number, it will be removed the option/option group which index position matches the number.
     * Obviously, in this case only one item will be removed.
     * - In case to be an object you can specify if the change will be applied only to options or option groups, 
     * it should have two properties:
     *     - criteria {String}: The value (in case of options) or the label (in case of option groups) the items needs 
     *     to match for apply the changes.
     *     - applyTo {String} (optional), it can take the following values:
     *         - "groups", the change will be applied only to the option groups.
     *         - "options", the change will be applied only to the options (direct child of the object).
     *         - [any other string value], the default value, it indicates that the change will be applied to both 
     *         options/option groups that matches the criteria in its value/label respectly.
     *
     * @param  {String} [group] It it is specified must be an String making reference to an existing option group label.
     * Using this parameter, the elements to be match by the first parameter will be search only in the option groups 
     * that match this parameter in its label.
     * 
     * ##Note. Removing an option group implies removing all its child options.
     * @chainable
     */
    DropDownListControl.prototype.removeOption = function(option, group) {
        var dataGroupTarget = this, htmlGroupTarget = jQuery(this.html), i, j, suboptions,
            objectDefaults = {
                applyTo: 'all'
            };

        if(group) {
            for(i = 0; i < this.options.length; i += 1) {
                if(this.options[i].isGroup && this.options[i].label === group) {
                    dataGroupTarget = this.options[i];
                    break;
                }
            }
            if(dataGroupTarget === this.options) {
                throw new Error('disableOption(): the group "' + group + '" wasn\'t found.');
            }
            htmlGroupTarget = jQuery(this.html).find('>optgroup[label="' + group + '"]');
        }

        if(typeof option === 'number') {
            dataGroupTarget.options.splice(option, 1);
            htmlGroupTarget.find(">*").eq(option).remove();
        } else if(typeof option === 'string') {
            for(i = 0; i < dataGroupTarget.options.length; i += 1) {
                if(!dataGroupTarget.options[i].isGroup && dataGroupTarget.options[i].value === option) {
                    dataGroupTarget.options.splice(i, 1);
                    i -= 1;
                } else if(dataGroupTarget.options[i].isGroup && dataGroupTarget.options[i].label === option) {
                    suboptions = dataGroupTarget.options[i].options;
                    for(j = 0; j < suboptions.length; j += 1) {
                        if(suboptions[j].value === option) {
                            suboptions.splice(j, 1);
                            j -= 1;
                        }
                    }
                    dataGroupTarget.options.splice(i, 1);
                    i -= 1;
                }
            }
            jQuery(htmlGroupTarget).find('option[value="' + option + '"]').add('optgroup[label="' + option + '"]')
                .remove();
        } else if(typeof option === 'object') {
            jQuery.extend(true, objectDefaults, option);
            if(objectDefaults.applyTo === 'groups') {
                for(i = 0; i < dataGroupTarget.options.length; i += 1) {
                    if(dataGroupTarget.options[i].isGroup && dataGroupTarget.options[i].label === option.criteria) {
                        dataGroupTarget.options.splice(i, 1);
                        i -= 1;
                    }
                }
                jQuery(htmlGroupTarget).find('optgroup[label="' + option.criteria + '"]').remove();
            } else if(objectDefaults.applyTo === 'options') {
                for(i = 0; i < dataGroupTarget.options.length; i += 1) {
                    if(!dataGroupTarget.options[i].isGroup && dataGroupTarget.options[i].value === option.criteria) {
                        dataGroupTarget.options.splice(i, 1);
                        i -= 1;
                    }
                }
                jQuery(htmlGroupTarget).find('option[value="' + option.criteria + '"]').remove();
            } else {
                option = option.criteria;
                for(i = 0; i < dataGroupTarget.options.length; i += 1) {
                    if((!dataGroupTarget.options[i].isGroup && dataGroupTarget.options[i].value === option) || 
                        (dataGroupTarget.options[i].isGroup && dataGroupTarget.options[i].label === option)) {
                        dataGroupTarget.options.splice(i, 1);
                        i -= 1;
                    }
                }
                jQuery(htmlGroupTarget).find('option[value="' + option + '"]').add('optgroup[label="' + option + '"]')
                    .remove();
            }
        } else {
            throw new Error('disableOption(): the first parameter must be a Number or a String.');
        }

        return this;
    };
    /**
     * Adds a new option group to the control
     * @param {Object} optionGroup A JSON object with the following properties:
     *
     * - label {String} (required): the label for the option group.
     * - disabled {Boolean}(optional): if the option group will be disabled or not.
     * it defaults to false.
     * - options {Array} (optional): An array of JSON object, each one represents an option and
     * should have the same structure than the "option" paremeter for the 
     * {@link PMUI.control.DropDownListControl#addOption addOption() method}.
     * @chainable
     */
    DropDownListControl.prototype.addOptionGroup = function(optionGroup) {
        var newOptionGroup = {}, optionGroupHTML, i;

        if(!optionGroup.label) {
            throw new Error ("addOptionGroup(): a label for the new option group is required!");
        }

        newOptionGroup.label = optionGroup.label;
        newOptionGroup.disabled = !! optionGroup.disabled;
        newOptionGroup.isGroup = true;
        newOptionGroup.options = [];

        this.options.push(newOptionGroup);

        if(this.html) {
            optionGroupHTML = PMUI.createHTMLElement('optgroup');
            optionGroupHTML.label = newOptionGroup.label;
            optionGroupHTML.disabled = newOptionGroup.disabled;
            this.html.appendChild(optionGroupHTML);
        }
        if(!jQuery.isArray(optionGroup.options)) {
            optionGroup.options = [];
        }
        for(i = 0; i < optionGroup.options.length; i += 1) {
            this.addOption(optionGroup.options[i], newOptionGroup.label);
        }

        return this;
    };
    /**
     * Adds a new option to the control or to an option group.
     * @param {Object} option An object with ther settings for the new option.
     * this object can have the following properties:
     * 
     * - value {String} (required): the value for the option.
     * - label {String} (optional): the label for the option, if isn't specified the value is used instead.
     * - selected {Boolean} (optional): if the option is selected. #Note. If the configuration object has the
     * "value" propery set then this "selected" property will be
     * - disabled {Boolean} (optional): if the option is disabled or not.
     * 
     * @param {String} group The name of the option group in which the new option will be added. If it doesn't exist
     * it will be created.
     *
     * @chainable
     */
    DropDownListControl.prototype.addOption = function(option, group) {
        var newOption = {}, optionHTML, i, groupHTML, flag = false;

        newOption.value = option.value !== null && option.value !== undefined && option.value.toString ? option.value : (option.label || "");
        newOption.value = newOption.value.toString();
        newOption.label = option.label || newOption.value;
        newOption.label = newOption.label.toString();
        newOption.disabled = !! option.disabled;
        newOption.isGroup = false;

        if(!group) {
            this.options.push(newOption);
        } else {
            for(i = 0; i < this.options.length; i += 1) {
                if(this.options[i].isGroup && this.options[i].label === group) {
                    this.options[i].options.push(newOption);
                    flag = true;
                    break;
                }
            }
            if(!flag) {
                this.addOptionGroup({
                    label: group
                });
                this.options[this.options.length -1].options.push(newOption);
            }
        }

        if(this.html) {
            optionHTML = PMUI.createHTMLElement('option');
            optionHTML.value = newOption.value;
            optionHTML.selected = !!option.selected;
            optionHTML.label = newOption.label;
            optionHTML.disabled = newOption.disabled;
            optionHTML.textContent = optionHTML.label;

            if(group) {
                groupHTML = jQuery(this.html).find('optgroup[label="' + group + '"]');
                if(groupHTML.length) {
                    groupHTML.get(0).appendChild(optionHTML);
                } else {
                    throw new Error("addOption(): the optiongroup \"" + group + "\" wasn't found");
                }
            } else {
                jQuery(this.html).append(optionHTML);
            }
        }

        if(option.selected) {
            this.value = newOption.value;
        }
        if(this.getOptions().length == 1){
            this.value=newOption.value;        
        }
        return this;
    };
    /**
     * Returns the label from the option currently selected.
     * @return {String}
     */
    DropDownListControl.prototype.getSelectedLabel = function() {
        var i;
        if(this.html) {
            return jQuery(this.html).find('option:selected').attr('label');
        }
        for(i = 0; i < this.options.length; i += 1) {
            if(this.options[i].value === this.value) {
                return this.options[i].label;
            }             
        }

        return "";
    };
    /**
     * Determines if a value exists in any of the list options.
     * @param  {String|Number} value The value to be searched
     * @return {Boolean} It returns true if the value was found in any of the list options, otherwise it returns false.
     */
    DropDownListControl.prototype.valueExistsInOptions = function(value) {
        var i, j, options, optionsLength, subOptions, subOptionsLength;
        optionsLength = (options = this.options|| []).length;
        for(i = 0; i < optionsLength; i++) {
            if(options[i].isGroup) {
                subOptionsLength = (subOptions = options[i].options || []).length;
                for(j = 0; j < subOptionsLength; j++) {
                    if(subOptions[j].value == value) {
                        return true;
                    }
                }
            } else if (options[i].value == value) {
                return true;   
            }
        }
        return false;
    };
    /**
     * Returns the first available option in the list.
     * @return {Object|null} It returns a object literal with the label and value properties from the found option.
     * @private
     */
    DropDownListControl.prototype.getFirstAvailableOption = function() {
        var i, j, options, optionsLength, subOptions, subOptionsLength;
        optionsLength = (options = this.options || []).length;
        for(i = 0; i < optionsLength; i++) {
            if(options[i].isGroup) {
                return (options[i].options.length && options[i].options[0]) || null;
            } else {
                return options[i];
            }
        }
        return null;  
    }
    /**
     * Sets the options/option groups for the control.
     * @param {Array} options An array with the same structure that the 
     * {@link PMUI.control.DropDownListControl#cfg-options "options"} property in the 
     * Config options section.
     * @chainable
     */
    DropDownListControl.prototype.setOptions = function(options) {
        var i, valueExists, firstOption;
        if(jQuery.isArray(options)) {
            this.clearOptions();
            for(i = 0; i < options.length; i += 1) {
                if(jQuery.isArray(options[i].options)) {
                    this.addOptionGroup(options[i]);
                } else {
                    this.addOption(options[i]);
                }
            }
            if(!this.valueExistsInOptions(this.value)) {
                firstOption = this.getFirstAvailableOption(); 
                this.value = (firstOption && firstOption.value) || "";
            }
        }

        return this;
    };
    /**
     * Returns the options/option groups from the field
     * @param  {Boolean} [includeGroups=false] If it's evaluated as true then it will include 
     * the option groups with its child elements, otherwise it will return only the option items.
     * @return {Array}
     * 
     * example 
     *    list.getOptions(false);
     *     [La Paz][Cochabamba][Santa Cruz][Buenos Aires][Santa Fe][Cordoba][Santiago][.][.][Mexico D.F]
     *
     *    list.getOptions(true)
     *      [BOLIVIA
     *          [La Paz][Cochabamba][SantaCruz]
     *      ]
     *      [ARGENTINA
     *          [Buenos Aires][Santa Fe][Cordoba]
     *       ]
     *      [CHILE
     *         [x][y][z]
     *      ]
     *      [New York]
     *      [Mexico D.F.]
     */
    DropDownListControl.prototype.getOptions = function(includeGroups) {
        var options = [], i, j;
        if(includeGroups) {
            return this.options.slice(0);
        }
        for(i = 0; i < this.options.length; i += 1) {
            if(!this.options[i].isGroup) {
                options.push(this.options[i]);
            } else {
                for(j = 0; j < this.options[i].options.length; j += 1) {
                    options.push(this.options[i].options[j]);
                }
            }
        }

        return options;
    };
    /**
     * Defines the events for the control
     * @chainable
     */
    DropDownListControl.prototype.defineEvents = function() {
        var that = this;
        DropDownListControl.superclass.superclass.prototype.defineEvents.call(this);
        if(this.html) {
            this.addEvent('change').listen(this.html, function() {
                that.onChangeHandler();
            });    
        }
        return this;
    };
    /**
     * Sets the selected option in the dropdown.
     * @param {String|Number} value The value of the option that is wanted to be selected. It must be one of the values
     * of the list options, otherwise it will be set to "".
     */
    DropDownListControl.prototype.setValue = function(value) {
        var firstOption;
        if(!this.valueExistsInOptions(value)) {
            firstOption = this.getFirstAvailableOption();
            value = (firstOption && firstOption.value) || "";
        }
        return DropDownListControl.superclass.prototype.setValue.call(this, value);
    };
    /**
     * Creates the HTML element for the control.
     * @return {HTMLElement}
     */
    DropDownListControl.prototype.createHTML = function() {
        var value;
        if(this.html) {
            return this.html;
        }
        value = this.value;
        DropDownListControl.superclass.prototype.createHTML.call(this);
        this.setOptions(this.options.slice(0))
            .setValue(value);

        return this.html;
    };

    PMUI.extendNamespace('PMUI.control.DropDownListControl', DropDownListControl);
}());
(function(){
    /**
     * @class PMUI.control.TextAreaControl
     * Class to handle a HTML TextArea control.
     * @extends PMUI.control.TextControl
     *
     * Usage example:
     *
     *      @example
     *          var myTextArea;
     *          $(function() {
     *              myTextArea = new PMUI.control.TextAreaControl({
     *                  name: "some_text",
     *                  value: "John Doe",
     *                  maxLength: 12,
     *                  readonly: false,
     *                  width: 300,
     *                  height: 200,
     *                  placeholder: "insert some text",
     *                  disabled: false,
     *                  onChange: function(currentValue, previousValue) {
     *                      if(previousValue !== "") {
     *                          alert("the text is not \"" + previousValue + "\" anymore.\nNow it's \"" 
     *                              + currentValue + "\"");
     *                      } else {
     *                          alert("Now your name is " + currentValue);
     *                      }
     *                  }
     *              });
     *      
     *              document.body.appendChild(myTextArea.getHTML());
     *              
     *              myTextArea.defineEvents();
     *          });
     *
     * @constructor
     * Creates a new instance
     * @param {Object} [settings] A JSON object that can contain the properties specified in the
     * config options section.
     *
     * @cfg {Boolean} [readonly=false] If the control will be readonly.
     */
    var TextAreaControl = function(settings) {
        TextAreaControl.superclass.call(this, settings);
        /**
         * @property {Boolean} readonly A Boolean that specifies if the control is enabled for read only.
         * @readonly
         */
        this.readonly = null;
        /**
         * @property {String} [elementTag='input'] The tag for the HTML element to be created.
         * @private
         */
        this.elementTag = 'textarea';
        TextAreaControl.prototype.init.call(this, settings);
    };

    PMUI.inheritFrom('PMUI.control.TextControl', TextAreaControl);
    
    TextAreaControl.prototype.type = "TextAreaControl";
    /**
     * Initializes the object.
     * @param  {Object} [settings=null] A JSON object with the config options.
     * @private
     */
    TextAreaControl.prototype.init = function(settings) {
        var defaults = {
            readonly: false
        };

        jQuery.extend(true, defaults, settings);

        this.setReadOnly(defaults.readonly);
    };
    /**
     * Creates the HTML element for the object
     * @return {HTMLElement}
     */
    TextAreaControl.prototype.createHTML = function() {
        TextAreaControl.superclass.prototype.createHTML.call(this);

        this.setReadOnly(this.readonly);

        return this.html;
    };
    /**
     * Sets the maximun character number to be accepted in the control.
     * @param {Number} maxLength The number must be an integer. 
     * If the value is minor or equal to 0 then the maxLength property is set to the default (524288).
     * @chainable>
     */
    TextAreaControl.prototype.setMaxLength = function(maxLength) {
        var that =this;       
              
        
                if(typeof maxLength === 'number' && maxLength % 1 === 0) {
                    this.maxLength = maxLength;
                    if(this.html) {
                            if (navigator.appName == "Microsoft Internet Explorer") {
                                this.html.onkeypress = function (e){
                                        //code
                                        if (that.html.value.length > that.maxLength-1) {
                                            //code
                                            e.preventDefault();
                                        }
                                    };       
                            }
                            else{
                                this.html.maxLength = maxLength > 0 ? maxLength : 524288;         
                            }                       
                    }
                } else {
                throw new Error("method setMaxLength() only accepts integer values.");
                }
        
        return this;
    };
    PMUI.extendNamespace('PMUI.control.TextAreaControl', TextAreaControl);

    if (typeof exports !== "undefined") {
        module.exports = TextAreaControl;
    }
}());
(function(){
    /**
     * @class PMUI.control.SelectableControl
     * A checkbox is a graphical component that can be in either an "on" (true) or "off" (false) state. 
     * Clicking on a check box changes its state from "on" to "off," or from "off" to "on."
     * @extends PMUI.control.HTMLControl
     *
     * Usage example:
     *
     *      @example
     *      var a;
     *      $(function() {
     *          a = new PMUI.control.SelectableControl({
     *              name: "music",
     *              label: "Do you like music?",
     *              value: true,
     *              mode: 'checkbox', //it also can be "radio"
     *              onSelect: function() {
     *                  console.log("checked");
     *              },
     *              onDeselect: function() {
     *                  console.log("unchecked");
     *              }
     *          });
     *          document.body.appendChild(a.getHTML());
     *          a.defineEvents();
     *      });
     *
     * @constructor
     * Creates a new instance of the SelectableControl class.
     * @param {Object} [settings=null] A JSON object with the config options.
     *
     * @cfg {Boolean} [selected=false] If the control will be selected initially.
     * @cfg {String} [label=""] The label for the control.
     * @cfg {String} [value=""] The value for the control.
     * @cfg {Function} [onSelect=null] The function to be call when the item will be selected. For info about the 
     * parameters please read the {@link PMUI.control.SelectableControl#event-onSelect onSelect event} documentation.
     * @cfg {Function} [onDeselect=null] The function to be call when the item will be deselected 
     * (only supported when the mode is set to "checkbox"). For info about the callback parameters please read the 
     * {@link PMUI.control.SelectableControl#event-onDeselect onDeselect event} documentation.
     * @cfg {String} [mode="checkbox"] The mode for the control, it can be: "checkbox" (default) 
     * for a checkbox control or "radio" for a readio button.
     */
    var SelectableControl = function(settings) {
        SelectableControl.superclass.call(this, settings);
        /**
         * If the control is selected or not.
         * @type {Boolean}
         * @readonly
         */
        this.selected = null;
        /**
         * The control's label.
         * @type {String}
         * @readonly
         */
        this.label = null;
        /**
         * @event onSelect
         * Fired when the control is selected.
         */
        this.onSelect = null;
        /**
         * @event onDeselect
         * Fired when the control is deselected.
         */
        this.onDeselect = null;
        /**
         * The control's selection mode.
         * @type {String}
         * @readonly
         */
        this.mode = null;
        /**
         * The interactive control's HTML element.
         * @type {HTMLElement}
         * @private
         */
        this.control = null;
        /**
         * The control's HTML element that will contain the label text.
         * @type {[type]}
         */
        this.textContainer = null;
        SelectableControl.prototype.init.call(this, settings);
    };

    PMUI.inheritFrom('PMUI.control.HTMLControl', SelectableControl);
    /**
     * The object's type.
     * @type {String}
     */
    SelectableControl.prototype.type = 'SelectableControl';
    /**
     * Initializes the object.
     * @param  {Object} [settings=null] A JSON object with the config options.
     * @private
     */
    SelectableControl.prototype.init = function(settings) {
        var defaults = {
            selected: false,
            mode: 'checkbox',
            label: '',
            value: "",
            onSelect: null,
            onDeselect: null
        };

        jQuery.extend(true, defaults, settings);

        if(defaults.mode === 'checkbox' || defaults.mode === 'radio') {
            this.mode = defaults.mode; 
        } else {
            throw new Error('SelectableControl: it only accepts "checkbox" and "radio" as value for the "mode"' 
                + ' property');
        }
        this.onSelect = defaults.onSelect;
        this.onDeselect = defaults.onDeselect;

        this.setLabel(defaults.label);

        if(defaults.selected) {
            this.select();
        } else {
            this.deselect();
        }
    };
    /**
     * Sets the control's name
     * @param {String} name
     * @chainable
     */
    SelectableControl.prototype.setName = function(name) {
        SelectableControl.superclass.prototype.setName.call(this, name);
        if(this.control) {
            this.control.name = name;
        }

        return this;
    };
    /**
     * Sets the value for the control
     * @param {String|Boolean|Number} setValue This param value is evaluated as boolean.
     * @chainable
     */
    SelectableControl.prototype.setValue = function(value) {
        SelectableControl.superclass.superclass.prototype.setValue.call(this, value);
        if(this.control) {
            this.control.value = value;
        }
        return this;
    };
    /**
     * Returns a boolean that specifies if the checkbox/radio button is selected.
     * @return {Boolean}
     */
    SelectableControl.prototype.isSelected = function() {
        if(this.html) {
            this.selected = jQuery(this.control).is(':checked');
        }
        return this.selected;
    };
    /**
     * Sets deselected the checkbox/radio button.
     * @chainable
     */
    SelectableControl.prototype.deselect = function() {
        this.selected = false;

        if(this.control) {
            this.control.checked = false;
        }

        return this;
    };
    /**
     * Sets selected the checkbox/radio button.
     * @chainable
     */
    SelectableControl.prototype.select = function() {
        this.selected = true;

        if(this.control) {
            this.control.checked = true;
        }

        return this;
    };
    /**
     * A method which is called everytime the checked state from the control changes.
     * 
     * This method is used internally by the object, so in most of the cases you won't need to invocated. 
     * To execute instructions when the control changes, please use the 
     {@link PMUI.control.Control#setOnChangeHandler setOnChangeHandler()} method.
     * 
     * @chainable
     * @private
     */
    SelectableControl.prototype.onChangeHandler = function() {
        this.selected = $(this.control).is(":checked");
        if(typeof this.onChange === 'function') {
            this.onChange(this.value, this.value);
        }
        if(this.selected && typeof this.onSelect === 'function') {
            this.onSelect();
        } else if(!this.selected && typeof this.onDeselect === 'function') {
            this.onDeselect();
        }

        return this;
    };
    /**
     * Sets the label for the control.
     * @param {String} label
     * @chainable
     */
    SelectableControl.prototype.setLabel = function(label) {
        if(typeof label !== 'string') {
            throw new Error("setLabel(): it only accepts string type values.");
        }
        this.label = label;
        if(this.html) {
            this.textContainer.textContent = label;
        }

        return this;
    };
    /**
     * Disables/enables the control
     * @param {Boolean} disable If the value is evaluated as true then the control 
     is disabled, otherwise the control is enabled.
     * @chainable
     */
    SelectableControl.prototype.disable = function(disable) {
        SelectableControl.superclass.superclass.prototype.disable.call(this, disable);
        if(this.html) {
            this.control.disabled = this.disabled;
        }

        return this;
    };
    /**
     * @inheritDoc
     */
    SelectableControl.prototype.defineEvents = function() {
        var that = this, stopPropagation = function(e) {
                e.stopPropagation();
            };
        this.removeEvents();
        this.eventsDefined = true;
        if(this.html) {
            this.addEvent('change').listen(this.html, function() {
                that.onChangeHandler();
            });
        }
        return this;
    };
    /**
     * Creates the HTML element for the control.
     * @return {HTMLElement}
     */
    SelectableControl.prototype.createHTML = function() {
        var label, textContainer, control;
        if(this.html) {
            return this.html;
        }
        label = PMUI.createHTMLElement('label');
        control = PMUI.createHTMLElement('input');
        control.type = this.mode;
        textContainer = PMUI.createHTMLElement('span');
        textContainer.contentText = this.label;

        label.appendChild(control);
        label.appendChild(textContainer);
        this.control = control;
        this.textContainer = textContainer;
        this.html = label;

        this.html.id = this.id;

        this.setName(this.name)
            .setValue(this.value)
            .disable(this.disabled)
            .setLabel(this.label);

        if(this.selected) {
           this.select();
        } else {
            this.deselect();
        }

        this.applyStyle();

        return this.html;
    };

    SelectableControl.prototype.setFocus = function () {
        if (this.html){
            this.control.focus();
        } 
    };

    PMUI.extendNamespace('PMUI.control.SelectableControl', SelectableControl);

    // Declarations created to instantiate in NodeJS environment
    if (typeof exports !== "undefined") {
        module.exports = SelectableControl;
    }
} ());
(function() {
    /**
     * @class PMUI.control.HiddenControl
     * The HTML input tag with the type attribute set to "hidden".
     * @extends PMUI.control.HTMLControl
     * 
     * Usage example:
     *      
     *     var hiddenControl = new PMUI.control.HiddenControl();
     *
     *     document.body.appendChild(hiddenControl.getHTML());
     *
     * @constructor
     * Creates a new instance of the HiddenControl class.
     * @param {Object} [settings=null] A JSON object with the config options.
     */
    var HiddenControl = function(settings) {
        HiddenControl.superclass.call(this, settings);
    };

    PMUI.inheritFrom('PMUI.control.HTMLControl', HiddenControl);

    HiddenControl.prototype.type = "HiddenControl";

    /**
     * Creates the HTML element for the control.
     * @return {HTMLElement}
     */
    HiddenControl.prototype.createHTML = function() {
        HiddenControl.superclass.prototype.createHTML.call(this);
        this.html.type = 'hidden';

        return this.html;
    };

    PMUI.extendNamespace('PMUI.control.HiddenControl', HiddenControl);

    if(typeof exports !== "undefined"){
        module.exports = HiddenControl;
    }
}());
(function() {
    /**
     * @class  PMUI.control.DateTimeControl
     * A control to handle dates and times.
     * @extends {PMUI.control.HTMLControl}
     *
     * 
     * Usage example:
     *
     *      @example
     *      var dateTimePicker;
     *      
     *      $(function() {
     *          dateTimePicker = new PMUI.control.DateTimeControl({
     *              dateFormat: 'M dd yy',
     *              minDate: -90,
     *              maxDate: "+1y -1m -4d",
     *              firstDay: 1,
     *              months: {
     *                  "january": {
     *                      name: "Enero",
     *                      shortname: "Ene"
     *                  },
     *                  "february": {
     *                      name: "Febrero",
     *                      shortname: "Feb"
     *                  },
     *                  "march": {
     *                      name: "Marzo",
     *                      shortname: "Mar"
     *                  },
     *                  "april": {
     *                      name: "Abril",
     *                      shortname: "Abr"
     *                  },
     *                  "may": "May",
     *                  "june": "Junio",
     *                  "july": "July",
     *                  "august": "Agosto",
     *                  "september": "Septiembre",
     *                  "october": "Octubre",
     *                  "november": "Noviembre",
     *                  "december": "Diciembre"
     *              },
     *              days: {
     *                  "sunday": {
     *                      name: "Domingo",
     *                      shortname: "Do"
     *                  },
     *                  "monday": {
     *                      name: "Lunes",
     *                      shortname: "Lu"
     *                  },
     *                  "tuesday": {
     *                      name: "Martes",
     *                      shortname: "Ma"
     *                  },
     *                  "wednesday": {
     *                      name: "Miércoles",
     *                      shortname: "Mi"
     *                  },
     *                  "thursday": {
     *                      name: "Jueves",
     *                      shortname: "Ju"
     *                  },
     *                  "friday": "Viernes",
     *                  "saturday": "Sábado"
     *              }
     *          });
     *          document.body.appendChild(dateTimePicker.getHTML());
     *          dateTimePicker.defineEvents();
     *      });
     *
     * @cfg {Boolean} [datetime=false] If the control will have time supporting.
     * @cfg {String} [dateFormat="yy-mm-dd HH:ii:ss"|"yy-mm-dd"] The format for the date to show in the text box. 
     * It defaults to:
     *
     * - "yy-mm-dd HH:ii:ss" if {@link PMUI.control.DateTimeControl#cfg-datetime datetime} is set to true.
     * - "yy-mm-dd" if {@link PMUI.control.DateTimeControl#cfg-datetime datetime} is set to false.
     *
     * You can set a customized date format using the following wildcards:
     *
     * - d, day of month (no leading zero).
     * - dd, day of month (two digit).
     * - o, day of the year (no leading zeros).
     * - oo, day of the year (three digit).
     * - D, day name short.
     * - DD, day name long.
     * - m, month of year (no leading zero).
     * - mm, month of year (two digit).
     * - M, month name short.
     * - MM, month name long.
     * - y, year (two digit).
     * - yy, year (four digit).
     * - P, period (AM or PM).
     * - H, hours (0-23).
     * - HH, hours (00-23).
     * - h, hours (1-12).
     * - hh, hours (01-12).
     * - i, minutes (0-59).
     * - ii, minutes (00-59).
     * - s, seconds (0-59).
     * - ss, seconds (00-59).
     * - @, Unix timestamp (ms since 01/01/1970).
     * - !, Windows ticks (100ns since 01/01/0001).
     * - '...', literal text.
     * - '', single quote.
     * - anything else, literal text.
     *
     * The wildcards can be used together in the same string.
     *
     * @cfg {Object} [months={"january": "January", "february": "February", "march": "March", "april": "April", 
     * "may": "May", "june": "June", "july": "July", "august": "August", "september": "September", 
     * "october": "October", "november": "November", "december": "December"}] 
     * A JSON object to set the names and shortnames for every month in year. Each property of this object can be:
     *
     * - A string, in this case the name for the month is set to this string, and the shortname is set using 
     * the first 3 characters of the string.
     * - A JSON object, in this case the JSON may have two properties: 
     *     - "name", will be used as the name for the month.
     *     - "shortname", will be used as the shortname for the month, if it is not specified the the shortname for 
     *     the month will be set using the first 3 characters of the object's "name" property.
     *
     * @cfg {Object} [days={"sunday": "Sunday","monday": "Monday","tuesday": "Tuesday","wednesday": "Wednesday",
     * "thursday": "Thursday","friday": "Friday","saturday": "Saturday"}]
     * A JSON object to set the name and shortname for every day of week. Each property of this object can be: 
     *
     * - A string, in this case the name for the day is set to this string, and the shortname is set using 
     * the first 3 characters of the string.
     * - A JSON object, in this case the JSON may have two properties: 
     *     - "name", will be used as the name for the day.
     *     - "shortname", will be used as the shortname for the day, if it is not specified the the shortname for 
     *     the day will be set using the first 3 characters of the object's "name" property.
     *
     * @cfg {String|Number} [minDate=-365] A value which sets the minimum selectable date for the calendar. It can be:
     *
     * - a Date object.
     * - a String with the following format: "([+-]\d+[dmyw]\s)?". It will represent an addition or substraction of 
     * time units to the current date, for example the expression "+1y -2m +3d" means 
     * "the current date plus 1 year, minus 2 months plus 3 days". The prefixes you can use are:
     *     - "d" for days.
     *     - "w" for weeks.
     *     - "m" for months.
     *     - "y" for years.
     * 
     *     You can also use one single unit, i.e. "+3y", "-1m", or a combination of two: "-2y +3d", etc.
     * - a Number, in this case the number is taken as the number of days that will be sum/substracted from the 
     * current date. This number can be positive (for add days) or negative (for substract days).
     * - an Object, in this case the object must have the following structure:
     *
     *      {
     *          year: 2013, 
     *          month: 5, 
     *          day: 4,
     *          hours: 20,
     *          minutes: 15,
     *          seconds: 3, 
     *          millisenconds
     *      }
     *
     * In this case only year and month are required.
     *
     * @cfg {String|Number} [maxDate=365] A value which sets the maximum selectable date for the calendar. It can take 
     * the same type of values that the {@link PMUI.contriol.DateTimeControl#cfg-minDate minDate config option}.
     *
     * - a Date object.
     * - a String with the following format: "([+-]\d+[dmyw]\s)?". It will represent an addition or substraction of 
     * time units to the current date, for example the expression "+1y -2m +3d" means 
     * "the current date plus 1 year, minus 2 months plus 3 days". The prefixes you can use are:
     *     - "d" for days.
     *     - "w" for weeks.
     *     - "m" for months.
     *     - "y" for years.
     * 
     *     You can also use one single unit, i.e. "+3y", "-1m", or a combination of two: "-2y +3d", etc.
     * - a Number, in this case the number is taken as the number of days that will be sum/substracted from the 
     * current date. This number can be positive (for add days) or negative (for substract days).
     * - an Object, in this case the object must have the following structure:
     *
     *      {
     *          year: 2013, 
     *          month: 5, 
     *          day: 4,
     *          hours: 20,
     *          minutes: 15,
     *          seconds: 3, 
     *          millisenconds
     *      }
     *
     * In this case only year and month are required.
     *
     * @cfg {Number} [firstDay=0] Sets the first day of week. You can use numbers from 0 to 6, 0 means Sunday, 
     * 1 means Monday and so on.
     */
    var DateTimeControl = function(settings) {
        DateTimeControl.superclass.call(this, settings);
        /**
         * @property {Object} dom A JSON object that contains the control's DOM elements.
         * @private
         */
        this.dom = {};
        /**
         * @property {Boolean} datetime If the calendar has time supporting.
         * @readonly
         */
        this.datetime = null;
        /**
         * @property {String} dateFormat The format for the date to be shown on the control's textbox.
         * @readonly
         */
        this.dateFormat = null;
        /**
         * @property {Date} dateObject The control's date object.
         * @private
         */
        this.dateObject = null;
        /**
         * @property {Date} minDate The minimum selectable date.
         * @private
         */
        this.minDate = null;
        /**
         * @property {Date} maxDate The maximum selectable date.
         * @private
         */
        this.maxDate = null;
        /**
         * @property {Number} firstDay The first day of the week beginning with 0: Sunday and ending with 6: Saturday.
         * @readonly
         */
        this.firstDay = null;
        DateTimeControl.prototype.init.call(this, settings);
    };

    PMUI.inheritFrom('PMUI.control.HTMLControl', DateTimeControl);
    
    DateTimeControl.prototype.type = "DateTimeControl"; 

    /**
     * An array with the name of each day of the week.
     * @type {Array}
     * @private
     */
    DateTimeControl.prototype.daysOrder = [
        "sunday", "monday", "tuesday", "wednesday", "thursday", "friday", "saturday"
    ];
    /**
     * An array with the name fo the months of year.
     * @type {Array}
     * @private
     */
    DateTimeControl.prototype.monthsOrder = [
        "january", "february", "march", "april", "may", "june", "july", "august", "september", "october", 
        "november", "december"
    ];
    /**
     * An Object that contains the name, shortname and code for the days of the week.
     * @type {Object}
     * @private
     */
    DateTimeControl.prototype.days = {
        "sunday": {
            value: 0
        },
        "monday": {
            value: 1
        },
        "tuesday": {
            value: 2
        },
        "wednesday": {
            value: 3
        },
        "thursday": {
            value: 4
        },
        "friday": {
            value: 5
        },
        "saturday": {
            value: 6
        }
    };
    /**
     * An Object that contains the name, shortname and code for the months of year.
     * @type {Object}
     * @private
     */
    DateTimeControl.prototype.months = {
        "january": {
            value: 0
        },
        "february": {
            value: 1
        },
        "march": {
            value: 2
        },
        "april": {
            value: 3
        },
        "may": {
            value: 4
        },
        "june": {
            value: 5
        },
        "july": {
            value: 6
        },
        "august": {
            value: 7
        },
        "september": {
            value: 8
        },
        "october": {
            value: 9
        },
        "november": {
            value: 10
        },
        "december": {
            value: 11
        }
    };
    /**
     * Initialize the object.
     * @param  {Object} settings An object with the config options.
     * @private
     */
    DateTimeControl.prototype.init = function(settings) {
        var defaults = {
            datetime: false,
            dateFormat: settings && settings.datetime ? 'yy-mm-dd HH:ii:ss' : 'yy-mm-dd',
            months: {
                "january": "January", 
                "february": "February", 
                "march": "March", 
                "april": "April", 
                "may": "May", 
                "june": "June", 
                "july": "July", 
                "august": "August", 
                "september": "September", 
                "october": "October", 
                "november": "November", 
                "december": "December"
            },
            days: {
                "sunday": "Sunday",
                "monday": "Monday",
                "tuesday": "Tuesday",
                "wednesday": "Wednesday",
                "thursday": "Thursday",
                "friday": "Friday",
                "saturday": "Saturday"
            },
            minDate: -365,
            maxDate: 365,
            firstDay: 0,
            height : 30

        };

        jQuery.extend(true, defaults, settings);

        this.setFirstDay(defaults.firstDay)
            .setDateFormat(defaults.dateFormat)
            .setMonths(defaults.months)
            .setDays(defaults.days)
            .setMinDate(defaults.minDate)
            .setMaxDate(defaults.maxDate)
            .visibleDateTime(defaults.datetime)
            .setHeight(defaults.height);
    };
    /**
     * @inheritdoc
     */
    DateTimeControl.prototype.setID = function(id) {
        DateTimeControl.superclass.prototype.setID.call(this, id);
        if(this.dom) {
            this.dom.calendar.id = 'pmui-datepicker-calendar-' + this.id;
        }
        return this;
    };
    /**
     * Returns the day index of the first day of week.
     * @return {Number} A number refering a day: 0 for Sunday, 1 for Monday and so on.
     */
    DateTimeControl.prototype.getFirstDay = function() {
        return this.firstDay;
    };
    /**
     * Returns the control's minimum selectable date.
     * @param {String} [format="UTC"] The format to applied to the returning date.
     * @return {String} The minimum selectable date in string format.
     */
    DateTimeControl.prototype.getMinDate = function(format) {
        return this.formatDate(this.minDate, format || "UTC");
    };
    /**
     * Returns the control's maximum selectable date.
     * @param {String} [format="UTC"] The format to applied to the returning date.
     * @return {String} The maximum selectable date in string format.
     */
    DateTimeControl.prototype.getMaxDate = function(format) {
        return this.formatDate(this.maxDate, format || "UTC");
    };
    /**
     * Enables/disabled the calendar's time supporting.
     * @param  {Boolean} visible If it's true, then the time supporting is enabled, otherwise it's disabled.
     * @chainable
     */
    DateTimeControl.prototype.visibleDateTime = function(visible) {
        visible = !!visible;
        this.datetime = visible;
        if(this.html) {
            this.dom.footer.style.display = this.datetime ? 'block' : 'none';
        }

        return this;
    };
    /**
     * @method  setValue
     * Sets the value for the Control.
     * @param {Date|Number|String} value The date to set to the control, it can be:
     * 
     *  - A Date object, is used the date set in this object.
     *  - A Number, the number is used as the timestamp for the date (set in UTC).
     *  - A String, in this case the string must have one of the following formats:
     *  
     *      - "" (empty string), the value is set to empty.
     *      - "\d", the value is parsed to a number to be used as the timestamp for the date.
     *      - "\d{4}\-\d{2}\-\d{2}", this date is taken as "yyyy-mm-dd" and it's parsed.
     *      - "\d{4}\-\d{2}\-\d{2}T\d{2}\:\d{2}\:\d{2}[\\+\-]\d{2}\:\d{2}", the date is taken as 
     *      "yyyy-mm-ddTHH:ii:ss+HH:ii" (the timezone offset is used).
     *
     * @param {Boolean} [utc=false] If the date will be set as a UTC date. 
     * (Only applicable if the first parameter is String and has the format "\d{4}\-\d{2}\-\d{2}").
     * @chainable
     */
    DateTimeControl.prototype.setValue = function(value, utc) {
        var newDate, aux, aux2, y, m, d, h, i, s, hd, id, datetime = false, error = false;
        if(value instanceof Date) {
            newDate = value;
        } else if(typeof value === 'number') {
            newDate = new Date(value);
        } else if(typeof value === 'string') {
            if(value === "") {
                this.dateObject = null;
            } else if(/^\d+$/.test(value)) {
                newDate = new Date(parseInt(value, 10));
            } else if(/^\d{4}\-\d{2}\-\d{2}$/.test(value) 
                || /^\d{4}\-\d{2}\-\d{2}T\d{2}\:\d{2}\:\d{2}[\\+\-]\d{2}\:\d{2}$/.test(value)) {
                newDate = new Date();
                aux = value.split(/T|\+|\-(?=\d{2}\:\d{2})/);
                aux2 = aux[0].split("-");
                y = parseInt(aux2[0], 10);
                m = parseInt(aux2[1], 10);
                d = parseInt(aux2[2], 10);
                if(aux[1]) {
                    datetime = true;
                    aux2 = aux[1].split(":");
                    h = parseInt(aux2[0], 10);
                    i = parseInt(aux2[1], 10);
                    s = parseInt(aux2[2], 10);
                    aux2 = aux[2].split(":");
                    hd = (value[19] === '+' ? -1 : 1) * parseInt(aux2[0], 10);
                    id = (hd < 0? -1 :  1) * parseInt(aux2[1], 10);    
                } else {
                    h = i = s = hd = id = 0;
                }

                if(!this.isValidDateTime(y, m, d, h, i, s)) {
                    throw new Error("setValue(): invalid date time.");
                }

                newDate.setFullYear(y);
                newDate.setMonth(m - 1);
                newDate.setDate(d);
                newDate.setHours(h);
                newDate.setMinutes(i);
                newDate.setSeconds(s);
                newDate.setMilliseconds(0);

                newDate.setHours(newDate.getHours() + hd);
                newDate.setMinutes(newDate.getMinutes() + id);

                if(!this.dateObject) {
                    this.dateObject = new Date();
                }

                if(datetime || utc) {
                    this.dateObject.setUTCFullYear(newDate.getFullYear());
                    this.dateObject.setUTCMonth(newDate.getMonth());
                    this.dateObject.setUTCDate(newDate.getDate());
                    this.dateObject.setUTCHours(newDate.getHours());
                    this.dateObject.setUTCMinutes(newDate.getMinutes());
                    this.dateObject.setUTCSeconds(newDate.getSeconds());
                } else {
                    this.dateObject.setFullYear(y);
                    this.dateObject.setMonth(m - 1);
                    this.dateObject.setDate(d);
                    this.dateObject.setHours(h);
                    this.dateObject.setMinutes(i);
                    this.dateObject.setSeconds(s);
                    this.dateObject.setMilliseconds(0);
                }
                newDate = null;
            } else {
                error = true;
            }
        } else {
            error = true;
        }

        if(error) {
            throw new Error("setValue(): Invalid parameter format/data type.");
        }
        if(value !== "") {
            if(!this.dateObject) {
                this.dateObject = new Date();
            }
            if(newDate) {
                this.dateObject.setFullYear(newDate.getFullYear());
                this.dateObject.setMonth(newDate.getMonth());
                this.dateObject.setDate(newDate.getDate());
                this.dateObject.setHours(newDate.getHours());
                this.dateObject.setMinutes(newDate.getMinutes());
                this.dateObject.setSeconds(newDate.getSeconds());
            }
        }

        if(this.html) {
            if(this.dateObject) {
                this.html.value = this.formatDate(this.dateObject, this.dateFormat);
            } else {
                this.html.value = "";
            }
        }
        this.value = this.getValue('UTC');

        return this;
    };
    /**
     * Sets the first day of week.
     * @param {Number} day Use 0 for Sunday, 1 for Monday, 2 for Tuesday and so on!.
     * @chainable
     */
    DateTimeControl.prototype.setFirstDay = function(day) {
        if(typeof day === 'number') {
            if(day >= 0 && day < 7) {
                this.firstDay = Math.floor(day);
                if(this.html) {
                    this.setDays(this.days);
                    this.buildDaysTable();
                }
                return this;
            }
        }

        throw new Error("setFirstDay(): The parameter must be a number between 0 and 6.");
    };
    /**
     * Sets the format for the date to be displayed in the control's textbox.
     * @param {String} dateFormat A string that contains wildcards that represent a date part (day, month, etc.). 
     * The valid wildcards are the same ones that are specified in the 
     * {@link PMUI.control.DateTimeControl#cfg-dateFormat dateFormat config option}.
     * @chainable
     */
    DateTimeControl.prototype.setDateFormat = function(dateFormat) {
        if(typeof dateFormat === 'string') {
            this.dateFormat = dateFormat;
            this.updateValue();
        } else {
            throw new Error("setDateFormat(): The parameter must be a string.");
        }
        return this;
    };
    /**
     * Returns the date format to be displayed in the control's textbox.
     * @return {String}
     */
    DateTimeControl.prototype.getDateFormat = function() {
        return this.dateFormat;
    };
    /**
     * Returns the date using the format set by {@link PMUI.control.DateTimeControl#setDateFormat setDateFormat()} 
     * method.
     * @return {String}
     */
    DateTimeControl.prototype.getFormatedDate = function() {
        return this.html.value;
    };
    /**
     * Returns the selected date.
     * @param  {String} format Specifies the date format that will be used for the returning date.
     * The valid values are:
     *
     * - "UTC", returns a date in the format: yyyy-mm-ddTHH:ii:ss-HH:mm (i.e. 2013-08-31T00:08:00+04:00).
     * - "@" or "timestamp", returns a date in timestamp format.
     * - [any other valid format string], will return the date using the string as the dateformat. This string can 
     * contain any of the wilcards specified in the 
     * {@link PMUI.control.DateTimeControl#cfg-dateFormat dateFormat config option}.
     * @return {String}
     */
    DateTimeControl.prototype.getValue = function(returningFormat) {
        var res;
        returningFormat = returningFormat || "UTC";
        if(this.dateObject) {
            switch(returningFormat) {
                case 'timestamp':
                case '@':
                    res = this.formatDate(this.dateObject, '@');
                    break;
                default:
                    res = this.formatDate(this.dateObject, returningFormat);
            }
        } else {
            res = "";
        }

        return res;
    };
    /**
     * Returns true if the parameter is a leap year, otherwise returns false.
     * @param  {Number}  year The year to be evaluated.
     * @return {Boolean}
     * @private
     */
    DateTimeControl.prototype.isLeapYear = function(year) {
        if(year % 400 === 0) {
            return true;
        } 
        if(year % 100 === 0) {
            return false;
        } 
        if(year % 4 === 0) {
            return true;
        } 
        return false;
    };
    /**
     * Returns true if the bunch of arguments represents a valid date otherwise it returns false.
     * @param  {Number}  year
     * @param  {Number}  month The month number (a number from 1 to 12)
     * @param  {Number}  day
     * @param  {Number}  [hours]
     * @param  {Number}  [minutes]
     * @param  {Number}  [seconds]
     * @param  {Number}  [milliseconds]
     * @return {Boolean}
     * @private    
     */
    DateTimeControl.prototype.isValidDateTime = function(year, month, day, hours, minutes, seconds, milliseconds) {
        if(!(typeof year === 'number' && typeof month === 'number' && typeof day === 'number')) {
            return false;
        }

        hours = hours || 0;
        minutes = minutes || 0;
        seconds = seconds || 0;
        milliseconds = milliseconds || 0;

        if(!(typeof hours === 'number' && typeof minutes === 'number' && typeof seconds === 'number' 
            && typeof milliseconds === 'number')) {
            return false;
        }

        if(!(hours >= 0 && hours <= 23)) {
            return false;
        }

        if(!(minutes >= 0 && minutes <= 59)) {
            return false;
        }

        if(!(seconds >= 0 && seconds <= 59)) {
            return false;
        }

        if(!(milliseconds >= 0 && milliseconds <= 999)) {
            return false;
        }

        if(day < 1 || day > 31 || month < 1 || month > 12) {
            return false;
        }

        switch(month) {
            case 4:
            case 6:
            case 9:
            case 11:
                if(day > 30) {
                    return false;
                }
                break;
            case 2:
                if(this.isLeapYear(year)) {
                    return day <= 29;
                }
                return day <= 28;
        }

        return true;
    };
    /**
     * It parses the argument into a date.
     * @param  {String|Date} date It can be a String or a Date:
     *
     * - Date, in this case the argument isn't parse and it is returned.
     * - String, in this case the string must have the following format: "[-+]\d+[dwmy]". For example "+1y -4m +3d" 
     * means that te returning date will be 1 year minus 4 months plus 3 days from today.
     * @return {Date}
     * @private
     */
    DateTimeControl.prototype.parseDate = function(date) {
        var startDate, aux, i, amount;

        if(typeof date === 'number') {
            startDate = new Date();
            startDate.setDate(startDate.getDate() + date);        
        } else if(typeof date === 'string') {
            startDate = new Date();
            date = " " + jQuery.trim(date);
            if(/^(\s[\\+|\-]\d+[y|m|d|w])+$/.test(date)) {
                date = date.match(/[\\+|\-]\d+[y|m|d]/g);
                for(i = 0; i < date.length; i += 1) {
                    aux = date[i].match(/[\-|\\+]|\d+|[m|d|y]/g);
                    amount = parseInt(aux[1], 10);
                    if(aux[0] === '-') {
                        amount *= -1;
                    }
                    switch(aux[2]) {
                        case 'd':
                            startDate.setDate(startDate.getDate() + amount);
                            break;
                        case 'm':
                            startDate.setMonth(startDate.getMonth() + amount);
                            break;
                        case 'y':
                            startDate.setFullYear(startDate.getFullYear() + amount);
                            break;
                        case 'w':
                            startDate.setDate(startDate.getDate() + (amount * 7));
                            break;
                    }
                }
            }
        } else if(date instanceof Date){
            startDate = date;
        } else if(typeof date === 'object') {
            startDate = new Date(date.year, date.month -1, date.day || 1, date.hours || 0, date.minutes || 0, date.seconds || 0, 
                date.milliseconds || 0);
        } else {
            throw new Error("parseDate(): invalid parameter.");
        }

        return startDate;
    };
    /**
     * Sets the minimum date the control can accept as a selectable one.
     * @param {String|Date|Number|Object} date It can be: 
     * 
     * - a Date object.
     * - a String with the following format: "([+-]\d+[dmyw]\s)?". It will represent an addition or substraction of 
     * time units to the current date, for example the expression "+1y -2m +3d" means 
     * "the current date plus 1 year, minus 2 months plus 3 days". The prefixes you can use are:
     *     - "d" for days.
     *     - "w" for weeks.
     *     - "m" for months.
     *     - "y" for years.
     * 
     *     You can also use one single unit, i.e. "+3y", "-1m", or a combination of two: "-2y +3d", etc.
     * - a Number, in this case the number is taken as the number of days that will be sum/substracted from the 
     * current date. This number can be positive (for add days) or negative (for substract days).
     * - an Object, in this case the object must have the following structure:
     *
     *      {
     *          year: 2013, 
     *          month: 5, 
     *          day: 4,
     *          hours: 20,
     *          minutes: 15,
     *          seconds: 3, 
     *          millisenconds
     *      }
     *
     * In this case only year and month are required.
     *      
     * @chainable
     */
    DateTimeControl.prototype.setMinDate = function(date) {
        var minDate = this.parseDate(date), d;

        if(this.maxDate) {
            if(minDate > this.maxDate) {
                throw new Error("setMinDate(): The min date can't be major than the max date");       
            }
        }
        this.minDate = minDate;
        this.fillYearSelector();
        this.fillMonthSelector();
        this.buildDaysTable();

        if(this.html) {
            d = new Date();
            this.dom.todayButton.disabled = !(d >= minDate && (this.maxDate && d <= this.maxDate));
        }

        return this;
    };
    /**
     * Sets the maximum date the control can accept as a selectable one.
     * @param {String|Date|Number|Object} date It can be: 
     * 
     * - a Date object.
     * - a String with the following format: "([+-]\d+[dmyw]\s)?". It will represent an addition or substraction of 
     * time units to the current date, for example the expression "+1y -2m +3d" means 
     * "the current date plus 1 year, minus 2 months plus 3 days". The prefixes you can use are:
     *     - "d" for days.
     *     - "w" for weeks.
     *     - "m" for months.
     *     - "y" for years.
     * 
     *     You can also use one single unit, i.e. "+3y", "-1m", or a combination of two: "-2y +3d", etc.
     * - a Number, in this case the number is taken as the number of days that will be sum/substracted from the 
     * current date. This number can be positive (for add days) or negative (for substract days).
     * - an Object, in this case the object must have the following structure:
     *
     *      {
     *          year: 2013, 
     *          month: 5, 
     *          day: 4,
     *          hours: 20,
     *          minutes: 15,
     *          seconds: 3, 
     *          millisenconds
     *      }
     *
     * In this case only year and month are required.
     * @chainable
     */
    DateTimeControl.prototype.setMaxDate = function(date) {
        var maxDate = this.parseDate(date), d;

        if(this.minDate) {
            if(maxDate < this.minDate) {
                throw new Error("setMaxDate(): The max date can't be minor than the min date");       
            }
        }
        this.maxDate = maxDate;
        this.fillYearSelector();
        this.fillMonthSelector();
        this.buildDaysTable();

        if(this.html) {
            d = new Date();
            this.dom.todayButton.disabled = !(d <= maxDate && (this.minDate && d >= this.minDate));
        }

        return this;
    };
    /**
     * Sets the months names/shortnames to be used by the calendar.
     * @param {Object} months An object with the same structure that the 
     * {@link PMUI.control.DateTimeControl#cfg-months months config option}.
     * @chainable
     */
    DateTimeControl.prototype.setMonths = function(months) {
        var key, changed;

        for(key in this.months) {
            if(this.months.hasOwnProperty(key)) {
                if(months.hasOwnProperty(key)) {
                    if(typeof months[key] === 'string') {
                        changed = true;
                        this.months[key].name = months[key];
                        this.months[key].shortname = months[key].charAt(0).toUpperCase() + months[key].slice(1, 3);
                    } else if(typeof months[key] === 'object') {
                        this.months[key].name = months[key].name;
                        this.months[key].shortname = months[key].shortname;
                        changed = true;
                    }  
                }
            }
        }

        if(changed) {
            this.fillMonthSelector();
        }

        return this;
    };
    /**
     * Fills the months selector.
     * @chainable
     * @private
     */
    DateTimeControl.prototype.fillMonthSelector = function() {
        var currentMonth, option, year, i, j;

        if(this.html) {
            year = this.dom.yearSelector.value;
            currentMonth = this.dom.monthSelector.value || (new Date()).getMonth().toString();
            jQuery(this.dom.monthSelector).empty();
            i = 0;
            j = this.monthsOrder.length - 1;
            if(year === this.minDate.getFullYear().toString()) {
                i = this.minDate.getMonth();
            }
            if(year === this.maxDate.getFullYear().toString()) {
                j = this.maxDate.getMonth();
            }
            for(i; i <= j; i += 1) {
                option = PMUI.createHTMLElement('option');
                option.label = this.months[this.monthsOrder[i]].name;
                option.textContent = option.label;
                option.value = this.months[this.monthsOrder[i]].value;
                option.selected = currentMonth === option.value.toString();
                this.dom.monthSelector.appendChild(option);
            }
        }

        return this;
    };
    /**
     * Sets the name/shortnames for the days to be used in the calendar.
     * @param {Object} days A JSON object with the same structure than the 
     * {@PMUI.control.DateTimeControl#cfg-days days config option}.
     * @chainable
     */
    DateTimeControl.prototype.setDays = function(days) {
        var key, changed, i, cell, daysLength = this.daysOrder.length, aux;

        for(key in this.days) {
            if(this.days.hasOwnProperty(key)) {
                if(days.hasOwnProperty(key)) {
                    if(typeof days[key] === 'string') {
                        this.days[key].name = days[key];
                        this.days[key].shortname = days[key].substr(0, 3);
                    } else if(typeof days[key] === 'object'){
                        this.days[key].name = days[key].name || this.days[key].name || "";
                        this.days[key].shortname = (days[key].shortname && days[key].shortname.substr(0, 3)) 
                            || (this.days[key].name && this.days[key].name.substr(0, 3)) || "";
                    } else {
                        throw new Error("setDays(): The argument supplied must be a string or an object.");
                    }
                    changed = true;
                }
            }
        }       

        if(changed && this.html) {
            jQuery(this.dom.daysHeader).empty();
            aux = 0;
            i = this.firstDay;
            while(aux < daysLength) {
                if(i === daysLength) {
                    i = 0;
                }
                cell = PMUI.createHTMLElement('th');
                cell.textContent = this.days[this.daysOrder[i]].shortname;
                cell.setAttribute("data-value", this.days[this.daysOrder[i]].value);
                this.dom.daysHeader.appendChild(cell);
                i += 1;
                aux += 1;
            }
        } 

        return this;
    };
    /**
     * Fill the year selector.
     * @chainable
     * @private
     */
    DateTimeControl.prototype.fillYearSelector = function() {
        var minYear, maxYear, option, selectedYear;
        if(this.html) {
            selectedYear = this.dom.yearSelector.value || (this.dateObject && this.dateObject.getFullYear()) 
                || (new Date()).getFullYear();
            selectedYear = selectedYear.toString();
            jQuery(this.dom.yearSelector).empty();
            minYear = this.minDate.getFullYear();
            maxYear = this.maxDate.getFullYear();
            for(minYear; minYear <= maxYear; minYear += 1) {
                option = PMUI.createHTMLElement('option');
                option.textContent = option.label = option.value = minYear;
                if(option.value.toString() === selectedYear) {
                    option.selected = true;
                } 
                this.dom.yearSelector.appendChild(option);
            }
        }

        return this;
    };
    /**
     * Returns the number of days in a month
     * @param  {Number} y The year (it's necessary to determine the days in February.)
     * @param  {Number} m The month
     * @return {Number}   The number of days.
     * @private
     */
    DateTimeControl.prototype.daysInMonth = function(y, m) {
        if(m < 1 || m > 12) {
            return 0;
        }
        switch (m) {
            case 4:
            case 6:
            case 9:
            case 11:
                return 30;
            case 2:
                if(this.isLeapYear(y)) {
                    return 29;
                }
                return 28;
            default:
                return 31;
        }
    };
    /**
     * Returns the day of week for a date.
     * @param  {Number} y The year.
     * @param  {Number} m The month.
     * @param  {Number} d The day of month.
     * @return {Number}   A number from 0 to 6. 0 means Sunday, 1, Monday and so on.
     * @private
     */
    DateTimeControl.prototype.whichDay = function(y, m, d) {
        var centuryCode, monthCode, lastTwoYearDigits = y % 100,
            gregorianCenturyCodes = [6, 4, 2, 0], i;

        if(!this.isValidDateTime(y, m, d)) {
            throw new Error("whichDay(): invalid date.");
        }

        switch(m) {
            case 3:
            case 11:
                monthCode = 3;
                break;
            case 4:
            case 7:
                monthCode = 6;
                break;
            case 5:
                monthCode = 1;
                break;
            case 6:
                monthCode = 4;
                break;
            case 8:
                monthCode = 2;
                break;
            case 9:
            case 12:
                monthCode = 5;
                break;
            case 10:
                monthCode = 0;
                break;
            case 1:
                if(this.isLeapYear(y)) {
                    monthCode = -1;
                } else {
                    monthCode = 0;
                }
                break;
            case 2:
                if(this.isLeapYear(y)) {
                    monthCode = 2;
                } else {
                    monthCode = 3;
                }
                break;
        }

        centuryCode = Math.floor(y / 100);

        i = 0;
        while(centuryCode % 4 !== 0) {
            centuryCode += 1;
            i += 1;
        }
        if(i === 0) {
            centuryCode = gregorianCenturyCodes[0];
        } else {
            centuryCode = gregorianCenturyCodes[gregorianCenturyCodes.length - i];    
        }
        
        return (d + monthCode +lastTwoYearDigits + Math.floor(lastTwoYearDigits / 4) + centuryCode    ) % 7;
    };
    /**
     * Build the days table for the calendar
     * @chainable
     * @private
     */
    DateTimeControl.prototype.buildDaysTable = function() {
        var dayOfTheWeek, daysInMonth, y, m, i, day = 1, row, cell, limit, link, selectableDate;

        if(this.html) {
            jQuery(this.dom.tableBody).empty();
            y = parseInt(this.dom.yearSelector.value, 10);
            m = parseInt(this.dom.monthSelector.value, 10) + 1;
            dayOfTheWeek = this.whichDay(y, m, 1) - this.firstDay;
            if(dayOfTheWeek < 0) {
                dayOfTheWeek = 7 + dayOfTheWeek;
            }
            daysInMonth = this.daysInMonth(y, m);
            limit = Math.ceil((dayOfTheWeek + daysInMonth) / 7) * 7;
            row = PMUI.createHTMLElement('tr');

            for(i = 0; i < limit; i += 1) {
                cell = PMUI.createHTMLElement("td");
                if(i - dayOfTheWeek >= 0 && day <= daysInMonth) {
                    selectableDate = true;
                    if((m - 1) === this.maxDate.getMonth() && y === this.maxDate.getFullYear()) {
                        if(day > this.maxDate.getDate()) {
                            selectableDate = false;
                        }
                    } else if ((m - 1) === this.minDate.getMonth() && y === this.minDate.getFullYear()) {
                        if(day < this.minDate.getDate()) {
                            selectableDate = false;
                        }
                    }
                    if(selectableDate){
                        link = PMUI.createHTMLElement("a");
                        link.href = "#";
                        link.setAttribute("data-date", day);
                        link.textContent = day;
                        cell.appendChild(link);
                    } else {
                        cell.className = 'pmui-datepicker-disabled-date';
                        cell.textContent = day;
                    }
                    day  += 1;
                }
                row.appendChild(cell);
                if((i + 1) % 7 === 0) {
                    this.dom.tableBody.appendChild(row);
                    row = PMUI.createHTMLElement('tr');
                }
            }
            this.dom.tableBody.appendChild(row);
        }

        return this;
    };
    /**
     * Shows the control's calendar.
     * @chainable
     * @private
     */
    DateTimeControl.prototype.showCalendar = function() {
        var position, inputHeight, maxZIndex;

        if(this.html) {
            position = jQuery(this.html).offset();
            inputHeight = jQuery(this.html).outerHeight();

            jQuery(document.body).find(">*").each(function() {
                var n;

                n = parseInt(jQuery(this).css("z-index"));
                if(!isNaN(n)) {
                    maxZIndex = n > maxZIndex ? n : maxZIndex;
                }
            });

            document.body.appendChild(this.dom.calendar);
            this.dom.calendar.style.zIndex = maxZIndex ? maxZIndex + 1 : 999; 
            this.dom.calendar.style.left = position.left + 'px';
            this.dom.calendar.style.top = (position.top + inputHeight) + 'px';
            this.dom.calendar.style.display = "";
        }

        return this;
    };
    /**
     * Hide's the control's calendar.
     * @return {HTMLElement} The object's html.
     * @private
     */
    DateTimeControl.prototype.hideCalendar = function() {
        if(this.html) {
            jQuery(this.dom.calendar).detach();
        }

        return this.html;
    };
    /**
     * Returns the day number of a date in a whole year.
     * @param  {Number} y The year.
     * @param  {Number} m The month.
     * @param  {Number} d The day of month.
     * @return {Number}   The day of year.
     * @private
     */
    DateTimeControl.prototype.getDayOfYear = function(y, m, d) {
        var day = 0, month = 1;
        if(!(typeof y === 'number' && typeof m === 'number' && typeof d === 'number')) {
            throw new Error("getDayOfYear(): invalid parameters.");
        }
        if(!this.isValidDateTime(y, m, d)) {
            throw new Error("getDayOfYear(): invalid date.");
        }

        while(month < m) {
            day += this.daysInMonth(y, m);
            month += 1;
        }

        day += d;

        return day;
    };
    /**
     * Formats a date using the specified format string.
     * @param  {Date} dateObject a Date object.
     * @param  {String} format     String that contains wildcards specified in the 
     * {@link PMUI.control.DateTimeControl#cfg-dateFormat dateFormat config option}.
     * @return {String}            The formatted date as a string.
     * @private
     */
    DateTimeControl.prototype.formatDate = function(dateObject, format) {
        var finalValue,
            d = dateObject.getDate(),
            o = this.getDayOfYear(dateObject.getFullYear(), dateObject.getMonth() +1 , dateObject.getDate()),
            m = dateObject.getMonth() + 1,
            h = dateObject.getHours(),
            h12 = h > 12 ? h - 12 : h,
            i = dateObject.getMinutes(),
            s = dateObject.getSeconds(),
            yy = dateObject.getFullYear(), formatArray, timeOffset, timeOffsetString, aux;

            timeOffset = dateObject.getTimezoneOffset();

            timeOffsetString = timeOffset < 0 ? '+' : '-';
            timeOffset = Math.sqrt(timeOffset * timeOffset);
            aux = Math.floor(timeOffset / 60);
            timeOffset = timeOffset - (aux * 60);
            timeOffsetString += (aux < 10 ? '0' : '') + aux + ':' + (timeOffset < 10 ? '0' : '') + timeOffset;

            formatArray = [
                {
                    regExp: /dd/g, 
                    value: d < 10 ? "0" + d : d
                },
                {
                    regExp: /d/g, 
                    value: d
                },
                {
                    regExp: /oo/g, 
                    value: o < 10 ? "00" + o: (o < 100 ? "0" + o: o)
                },
                {
                    regExp: /o/g, 
                    value: o
                },
                {
                    regExp: /mm/g, 
                    value: m < 10 ? "0" + m :  m
                },
                {
                    regExp: /m/g, 
                    value: m
                },
                {
                    regExp: /yy/g, 
                    value: yy
                },
                {
                    regExp: /y/g, 
                    value: yy % 100
                },
                {
                    regExp: /\@/g,
                    value: dateObject.getTime()
                },
                {
                    regExp: /\!/g,
                    value: (dateObject.getTime() * 10000) + 621355968000000000
                },
                {
                    regExp: /MM/g, 
                    value: this.months[this.monthsOrder[dateObject.getMonth()]].name
                },
                {
                    regExp: /M/g, 
                    value: this.months[this.monthsOrder[dateObject.getMonth()]].shortname
                },
                {
                    regExp: /HH/g,
                    value: (h < 10 ? '0' : '') + h
                },
                {
                    regExp: /H/g,
                    value: h
                },
                {
                    regExp: /hh/g,
                    value: (h12 < 10 ? '0' : '') + h12
                },
                {
                    regExp: /h/g,
                    value: h12
                },
                {
                    regExp: /ii/g,
                    value: (i < 10 ? '0' : '') + i
                },
                {
                    regExp: /i/g,
                    value: i
                },
                {
                    regExp: /ss/g,
                    value: (s < 10 ? '0' : '') + s
                },
                {
                    regExp: /s/g,
                    value: s
                },
                {
                    regExp: /UTC/g,
                    value: yy + '-' + (m < 10 ? '0' : '') + m + '-' + (d < 10 ? '0' : '') + d + "T"
                        + (h < 10 ? '0' : '') + h + ':' + (i < 10 ? '0' : '') + i + ':' + (s < 10 ? '0' : '') + s
                        + timeOffsetString
                },
                {
                    regExp: /P/g,
                    value: h > 12 ? 'PM' : 'AM'
                },
                {
                    regExp: /DD/g, 
                    value: this.days[this.daysOrder[dateObject.getDay()]].name
                },
                {
                    regExp: /D/g, 
                    value: this.days[this.daysOrder[dateObject.getDay()]].shortname
                }
            ];

        finalValue = format || this.dateFormat;

        for(i = 0; i < formatArray.length; i += 1) {
            finalValue = finalValue.replace(formatArray[i].regExp, formatArray[i].value);
        }

        return finalValue;
    };
    /**
     * Updates the control's date text.
     * @chainable
     * @private
     */
    DateTimeControl.prototype.updateValue = function() {
        var finalValue;

        if(this.dateObject) {
            finalValue = this.formatDate(this.dateObject, this.dateFormat);
        } else {
            finalValue = "";
        }

        this.value = finalValue;

        if(this.html) {
            this.html.value = finalValue;
        }

        return this;
    };
    /**
     * Gets the control's value directly from the element.
     * @return {String}
     */
    DateTimeControl.prototype.getValueFromRawElement = function() {
        return this.getValue('UTC');
    };
    /**
     * The handler to be executed everytime the control's value changes.
     * @chainable
     * @private
     */
    DateTimeControl.prototype.onChangeHandler = function(hide) {
        var prevValue = this.value, newValue, 
            resCallback, date, month, year, auxDate, valueChanged = false;

        auxDate = new Date();
        date = parseInt($(this.dom.tableBody).find("a.selected").text(), 10);
        if(isNaN(date)) {
            return this;
        }
        month = parseInt(this.dom.monthSelector.value, 10);
        year = parseInt(this.dom.yearSelector.value, 10);
        auxDate.setYear(year);
        auxDate.setMonth(month);
        auxDate.setDate(date);
        if(this.datetime) {
            auxDate.setMilliseconds(0);
            auxDate.setSeconds(parseInt(this.dom.secondsInput.value, 10));
            auxDate.setMinutes(parseInt(this.dom.minutesInput.value, 10));
            auxDate.setHours(parseInt(this.dom.hoursInput.value, 10));
            this.dom.doneButton.disabled = true;
            jQuery(this.dom.tableBody).find('.selected').removeClass('selected');
        } else {
            if(!this.dateObject) {
                this.dateObject = new Date();
            }
            auxDate.setMilliseconds(0);
            auxDate.setSeconds(0);
            auxDate.setMinutes(0);
            auxDate.setHours(0);
        }
        if(auxDate.getTime() !== this.dateObject.getTime()) {
            valueChanged = true;
        }
        if(typeof this.onBeforeChange === 'function' && valueChanged) {
            resCallback = this.onBeforeChange(this.formatDate(auxDate), prevValue);
        }
        if(resCallback !== false) {
            this.dateObject = auxDate;
        } else {
            jQuery(this.dom.tableBody).find('a.selected').removeClass('selected');
            if(this.dom.monthSelector.value == this.dateObject.getMonth() 
                && this.dom.yearSelector.value == this.dateObject.getFullYear()) {

                jQuery(this.dom.tableBody).find('a[data-date="' + this.dateObject.getDate() + '"]').addClass('selected');
            }
        }
        newValue = this.getValueFromRawElement();
        this.updateValue();
        if(hide) {
            this.hideCalendar();
        }
        if(typeof this.onChange === 'function' && this.value !== prevValue) {
            this.onChange(this.value, prevValue);
        }

        return this;
    };
    /**
     * Defines the events for the control
     * @chainable
     */
    DateTimeControl.prototype.defineEvents = function() {
        var that = this, preselectedDate, specialCloseHandler = function(e) {
            if((e.target !== that.html && !$(e.target).parents('#pmui-datepicker-calendar-' + that.id).length) 
                || (e.type === 'keyup' && !jQuery(document.activeElement).parents('#pmui-datepicker-calendar-' 
                    + that.id).length)) {
                that.hideCalendar();
            }
        };
        //DateTimeControl.superclass.prototype.defineEvents.call(this);
        if(this.dom.todayButton) {
            this.addEvent('change').listen(this.dom.yearSelector, function() {
                that.fillMonthSelector();
                that.buildDaysTable();
            });
            this.addEvent('change').listen(this.dom.monthSelector, function() {
                that.buildDaysTable();
            });
            this.addEvent('focusin').listen(this.html, function() {
                that.showCalendar();
            });
            this.addEvent('click').listen(this.dom.todayButton, function() {
                preselectedDate = new Date();
                that.dom.doneButton.disabled = false;
                that.dom.yearSelector.value = preselectedDate.getFullYear();
                that.dom.monthSelector.value = preselectedDate.getMonth();
                that.buildDaysTable();
                jQuery(that.dom.tableBody).find('a[data-date=' + preselectedDate.getDate() + ']').addClass("selected");
            });
            this.addEvent('click keyup').listen(document, specialCloseHandler)
                .listen('.pmui-window', specialCloseHandler);
            //In the future, if the new Event objects model is implemented the following line should be listening to an
            //event close or windowClose. Currently a dirty implementation will fix he problem.
            this.addEvent('click').listen('.pmui-button.pmui-window-close', function() {
                that.hideCalendar(); 
            });
            this.addEvent('click').listen(this.dom.doneButton, function() {
                that.onChangeHandler(true);
            });
            jQuery(this.dom.tableBody).on('click', 'a', function(e) {
                e.preventDefault();
                jQuery(that.dom.tableBody).find("a").filter(".selected").removeClass("selected");
                jQuery(this).addClass("selected");
                var date, month, year;

                date = parseInt(this.textContent, 10);
                month = parseInt(that.dom.monthSelector.value, 10);
                year = parseInt(that.dom.yearSelector.value, 10);
                if(that.datetime) {
                    that.dom.doneButton.disabled = false;
                } else {
                    that.onChangeHandler(true);
                }
            });
        }
        return this;
    };
    /**
     * Creates the HTML for the control.
     * @return {HTMLElement}
     */
    DateTimeControl.prototype.createHTML = function() {
        var header, table, tableHeader, monthSelector, yearSelector, row, cell, tableBody, calendar, footer, button, input;
        if(this.html) {
            return this.html;
        }

        DateTimeControl.superclass.prototype.createHTML.call(this);

        this.html.type = 'text';
        this.html.readOnly = true;

        calendar = PMUI.createHTMLElement('div');
        calendar.className = 'pmui-datepicker';
        calendar.id = 'pmui-datepicker-calendar-' + this.id;
        header = PMUI.createHTMLElement('div'); 
        header.className = 'pmui-datepicker-header';
        table = PMUI.createHTMLElement('table');
        table.className = 'pmui-datepicker-head-table';
        monthSelector = PMUI.createHTMLElement("select");
        yearSelector = PMUI.createHTMLElement("select");
        row = PMUI.createHTMLElement("tr");
        cell = PMUI.createHTMLElement("td");
        cell.className = 'pmui-datepicker-month-selector';
        cell.appendChild(monthSelector);
        row.appendChild(cell);
        cell = PMUI.createHTMLElement("td");
        cell.className = 'pmui-datepicker-year-selector';
        cell.appendChild(yearSelector);
        row.appendChild(cell);
        table.appendChild(row);
        header.appendChild(table);

        this.dom.yearSelector = yearSelector;
        this.dom.monthSelector = monthSelector;

        table = PMUI.createHTMLElement('table');
        table.className = 'pmui-datepicker-table';
        tableHeader = PMUI.createHTMLElement('thead');
        row = PMUI.createHTMLElement('tr');
        this.dom.daysHeader = row;
        tableHeader.appendChild(row);
        table.appendChild(tableHeader);

        tableBody = PMUI.createHTMLElement('tbody');
        this.dom.tableBody = tableBody;
        table.appendChild(tableBody);

        footer = PMUI.createHTMLElement('div');
        footer.className = 'pmui-datepicker-footer';
        cell = PMUI.createHTMLElement('div');
        button = PMUI.createHTMLElement('button');
        button.textContent = 'Today';
        button.className = 'pmui-datepicker-button';
        this.dom.todayButton = button;
        cell.appendChild(button);
        footer.appendChild(cell);
        cell = PMUI.createHTMLElement('div');
        input = PMUI.createHTMLElement('input');

        //input.type = 'number';
        input.setAttribute("type","number");
        input.size = 2;
        input.min = 0;
        input.max = 23;
        input.value = 0;
        input.placeholder = "hrs.";
        this.dom.hoursInput = input;
        cell.appendChild(input);
        input = input.cloneNode(false);
        input.max = 59;
        input.placeholder = "min.";
        this.dom.minutesInput = input;
        cell.appendChild(input);
        input = input.cloneNode(false);
        input.max = 59;
        input.placeholder = "sec.";
        this.dom.secondsInput = input;
        cell.appendChild(input);
        footer.appendChild(cell);
        this.dom.timeCell = cell;
        cell = cell.cloneNode(false);
        footer.appendChild(cell);
        button = button.cloneNode(true);
        button.textContent = "Done";
        button.disabled = true;
        this.dom.doneButton = button;
        cell.appendChild(button);
        footer.appendChild(cell);

        calendar.appendChild(header);
        calendar.appendChild(table);
        calendar.appendChild(footer);

        this.dom.calendar = calendar;
        this.dom.footer = footer;

        this.setMinDate(this.minDate)
            .setMaxDate(this.maxDate)
            .setDays(this.days)
            .visibleDateTime(this.datetime);

        if(this.eventsDefined) {
            this.defineEvents();
        }

        return this.html;
    };

    PMUI.extendNamespace('PMUI.control.DateTimeControl', DateTimeControl);

    if(typeof exports !== "undefined"){
        module.exports =  DateTimeControl;
    }
}());
(function() {
/**
     * @class PMUI.ui.Window
     * @extend PMUI.core.Panel
     *    Usage example: 
     *
     *      @example
     *      
     *        f = new PMUI.form.Form({
     *               items: [
     *                   {
     *                       pmType: "text",
     *                       label: "Name",
     *                       id: "123",
     *                       value: "",
     *                       placeholder: "insert your name",
     *                       name: "name",
     *                       helper: "Introduce your name",
     *                       required : true,
     *                       controlsWidth: 200,
     *                       valueType: 'string'
     *                   },
     *                   {   
     *                       pmType: "text",
     *                       label: "test Number",
     *                       id: "1234",
     *                       value: 10,
     *                       placeholder: "insert your number",
     *                       name: "numberTest",
     *                       helper: "Introduce your number",
     *                       required : true,
     *                       controlsWidth: 200,
     *                       valueType: 'number'
     *                   },
     *                   {
     *                       pmType: "datetime",
     *                       visible : true,
     *                       label: "birth date",
     *                       name: "birthdate",
     *                       valueType: 'date',
     *                       required: true
     *                   }
     *               ]
     *           });
     *
     *      w = new PMUI.ui.Window({
     *               title: "Window Example",
     *               width: 500,
     *               height: 230,
     *               modal: true,
     *               buttons: [
     *                   {
     *                       pmType: 'button',
     *                       text: 'Save',
     *                       handler: function() {
     *                           alert("Saved!");
     *                       }
     *                   },
     *                   {
     *                       pmType: 'label',
     *                       text: " or "
     *                   },
     *                   {
     *                       pmType: 'button',
     *                       text: "Close",
     *                       handler : function () {
     *                           w.close();
     *                       },
     *                       border: 'link'
     *                   }
     *               ],
     *               closable: true,
     *               footerItems: [
     *                   {
     *                       text: "Close"
     *                   }
     *               ],
     *               visibleFooter: true,
     *               buttonPanelPosition: 'top'
     *           });
     *           w.addItem(f);
     *           w.open();
     *           w.defineEvents();
     *
     * Create a new instace of the Window class
     * 
     * @cfg {String}  [title = [Untitled window]] Title The title for the Window
     * @cfg {Boolean} [modal = true] Defines the  window property  is modal or not 
     * @cfg {String|Number} [height = "auto"] window height can be a number or a string. height 
     * is the div 
     * Element which contains a header, body and footer. When using a string that you can only use 
     * 'auto' or 'inherit' or # # # # px or% or # # # # em when is a number
     * @cfg {String|Number} [width = 400] The width of the main window which contains a html element 
     * Element 'div' header, body and footer 
     * @cfg {String|Number} [footerHeight = "auto"] the height of the Element div footer (px, auto,
     * number)which default 'auto'
     * @cfg {PMUI.util.ArrayList} [buttons = []], defines an array of objects that are buttons 
     * settings may in the fotter Element div.
     * @deprecated This config optio will be removed soon, please use the {@link #cfg-footerItems footerItems} config 
     * option instead.
     * @cfg {boolean} [visibleCloseButton = true], the button close are can define or not in the window.
     * @cfg {Array} [footerItems=[]] Sets the elements in the window footer, this elements can be instances of Button 
     * and/or instances of Label. The value for this config option must be an array in which each element can be:
     *
     * - An object literal, in this case the object literal must have the property "pmType" with its value set to 
     * "button" (if you want the element be a {@link PMUI.ui.Button Button}) or "label" (if you want the element be a 
     * {@link PMUI.ui.TextLabel Label}). Optionally you can add the respective config options for each case.
     *
     * - A PMUI object, in this case it must be an instance of {@link PMUI.ui.Button Button} or an instance of
     * {@link PMUI.ui.TextLabel Label}.
     *
     * @cfg {String} [footerAlign="center"] Sets the horizontal alignment for the items in the wondow footer, the 
     * possible values are "center", "left", "right".
     * @cfg {Boolean} [visibleFooter="false"] indicating whether the footer of the Window is visible or was hidden
     * @cfg {String}  [buttonPanelPosition="bottom"] indicates that position does the panel of buttons, you can be 
     * above or below body.
     * @cfg {Function|null} [onOpen=null] The callback function to be executed everytime the 
     * {@link #event-onOpen onOpen} event.
     * @cfg {Function|null} [onClose=null] The callback function to be executed everytime the
     * {@link #event-onClose onClose} event.
     */ 
    var Window = function(settings) {
        Window.superclass.call(this, jQuery.extend(true, {
            positionMode: 'absolute'
        }, settings));
        /**
         * The window's header.
         * @type {HTMLElement}
         * @private
         */
        this.header = null;
        /**
         * The window's body.
         * @type {HTMLElement}
         * @private
         */
        this.body =  null;
        /**
         * The window's footer.
         * @type {PMUI.panel.ButtonPanel}
         * @private
         */
        this.footer = null;
        /**
         * @property {String} title
         * The window's title. Set by the {@link #cfg-title title} config option and the 
         * {@link #method-setTitle setTitle()} method.
         * @readonly
         */
        this.title = null;
        /**
         * The windo's close button.
         * @type {PMUI.ui.Button}
         * @private
         */
        this.closeButton = null;
        /**
         * If the window is opened or not.
         * @type {Boolean}
         * @readonly
         */
        this.isOpen = false;
        /**
         * is defined if button panes Object is visible
         * @type {[type]}
         */
        this.visibleFooter = null;
        /**
         * If the window's close button is visible or not. Set by the 
         * {@link #cfg-visibleCloseButton visibleCloseButton} config option and the 
         * {@link #method-showCloseButton showCloseButton()} and {@link #method-hideCloseButton hideCloseButton()}
         * methods.
         * @type {Boolean}
         * @readonly
         */
        this.visibleCloseButton = null;
        /**
         * If the window will be a modal window. Set by the {@link #cfg-modal modal} config option and the 
         * {@link #method-setModal setModal()} method.
         * @type {Boolean}
         * @readonly
         */
        this.modal = null;
        /**
         * The window's footer height.
         * @type {Number|String}
         */
        this.footerHeight = null;
        /**
         * The window's footer alignment.
         * @type {String}
         */
        this.footerAlign = null;
        /**
         * defines the position of the buttons panel on the window
         * @type {String}
         */
        this.buttonPanelPosition = null;
        /**
         * @event onOpen
         * Fired everytime the window opens.
         * @param {PMUI.ui.Window} window The window that was opened.
         */
        this.onOpen = null;
        /**
         * @event onClose
         * Fired everytime the window closes.
         * @param {PMUI.ui.Window} window The window that was closed.
         */
        this.onClose = null;
        this.dom = {};

        Window.prototype.init.call(this, settings);
    };

    PMUI.inheritFrom('PMUI.core.Panel', Window);
    /**
     * Defines the object's type
     * @type {String}
     */
    Window.prototype.type = 'Window';
    /**
     * Defines the object's family
     * @type {String}
     */
    Window.prototype.family = 'ui';

    Window.prototype.init = function(settings) {
        var defaults = {
            height: 'auto',
            width: 'auto', 
            title: '[Untitled window]',
            modal: true,
            footerHeight: 'auto',
            zOrder: 100,
            footerItems: [],
            footerAlign: 'center',
            visibleCloseButton : true,
            visibleFooter: false,
            buttons: [],
            buttonPanelPosition : 'bottom',
            onOpen: null,
            onClose: null
        };

        jQuery.extend(true, defaults, settings);

        this.footer = new PMUI.panel.ButtonPanel({
            style: {
                cssClasses: ['pmui-window-footer']
            }
        });

        this.setTitle(defaults.title)
            .setModal(defaults.modal)
            .setWidth(defaults.width)
            .setHeight(defaults.height)
            .setFooterHeight(defaults.footerHeight)
            .setFooterAlign(defaults.footerAlign)
            .setZOrder(defaults.zOrder)
            .setButtonPanelPosition(defaults.buttonPanelPosition)
            .setFooterItems(defaults.footerItems)
            .setOnOpenHandler(defaults.onOpen)
            .setOnCloseHandler(defaults.onClose);

        if(defaults.visibleCloseButton) {
            this.showCloseButton();
        } else {
            this.hideCloseButton();
        }
        if(defaults.visibleFooter) {
            this.showFooter();
        } else {
            this.hideFooter();
        }
        //TODO tell the guys to replace its 'buttons' config option by FooterItems
        if (defaults.footerItems && defaults.footerItems.length) {
            this.setFooterItems(defaults.footerItems);
        } else {
            this.setFooterItems(defaults.buttons);
        }
    };
    /**
     * Sets the callback function to be executed everytime the {@link #event-onClose onClose} event fires.
     * @param {Function|null} handler It can be a function or null.
     */
    Window.prototype.setOnCloseHandler = function(handler) {
        if(!(handler === null || typeof handler === 'function')) {
            throw new Error("setOnCloseHandler(): The parameter must be a function or null.");
        }
        this.onClose = handler;
        return this;
    };
    /**
     * Sets the callback function to be executed everytime the {@link #event-onOpen onOpen} event fires.
     * @param {Function|null} handler It can be a function or null.
     */
    Window.prototype.setOnOpenHandler = function(handler) {
        if(!(handler === null || typeof handler === 'function')) {
            throw new Error("setOnOpenHandler(): The parameter must be a function or null.");
        }
        this.onOpen = handler;
        return this;
    };
    /**
     * updates the dimensions of the height of the HTMLElement window, body and foot when 
     it is modified by the user
     * @chainable
     */
    Window.prototype.updateDimensionsAndPosition = function() {
        var bodyHeight, footerHeight, headerHeight, windowHeight = this.height, windowWidth;
        if(!this.footer || !this.html || !this.isOpen) {
            return this;
        }
        if(this.footerHeight === 'auto') {
            this.footer.setHeight('auto');
        } else {
            this.footer.setHeight(this.footerHeight);
        }
        
        if(windowHeight === 'auto') {
            this.body.style.height = 'auto';
            this.body.style.minHeight = '150px';
        } else {
            if(/^\d+(\.\d+)?em$/.test(windowHeight)) {
                windowHeight = PMUI.emToPx(parseInt(windowHeight, 10), this.modalObject);
            } else if(/^\d+(\.\d+)?%$/.test(windowHeight)) {
                windowHeight = jQuery(this.html).outerHeight();
            }
            footerHeight = this.visibleFooter ? jQuery(this.footer.getHTML()).outerHeight() : 0;
            headerHeight = jQuery(this.header).outerHeight();
            bodyHeight = windowHeight - footerHeight - headerHeight;
            if(bodyHeight <= 0) {
                bodyHeight = 0;
            }
            this.body.style.minHeight = '';
            this.body.style.height = bodyHeight + "px";
        }

        windowWidth = jQuery(this.header).width();
        windowWidth = windowWidth - (this.visibleCloseButton ? jQuery(this.closeButton.getHTML()).outerWidth() : 0);
        this.dom.titleContainer.style.width = windowWidth < 0 ? 0 : windowWidth + 'px';

        windowWidth = jQuery(this.html).outerWidth();
        windowHeight = jQuery(this.html).outerHeight();

        this.addCSSProperties({
            left: '50%',
            "margin-left": (windowWidth / -2) + "px",
            top: '50%',
            "margin-top": (windowHeight / -2) + "px"
        });

        return this;
    };
    /**
     * shows the footer where the buttons are located
     * @chainable
     */
    Window.prototype.showFooter = function () {
        this.footer.setVisible(this.visibleFooter = true);
        return this.updateDimensionsAndPosition();
    };
    /**
     * hides the footer where the buttons are located
     * @chainable
     */
    Window.prototype.hideFooter = function () {
        this.footer.setVisible(this.visibleFooter = false);
        return this.updateDimensionsAndPosition();
    };
    /**
     * @method setTitle
     * Sets the window title in the header part of the window.
     * @param {String} title
     * @chainable
     */
    Window.prototype.setTitle = function(title) {
        if(typeof title === 'string') {
            this.title = title;
            if(this.dom.titleContainer) {
                this.dom.titleContainer.textContent = title;
                this.dom.titleContainer.title = title;
            }
        } else {
            throw new Error("The setTitle() method accepts only string values.");
        }
        return this;
    };
    /**
    * @method getTitle
    * its title of teh window
    * @return {String}
    */
    Window.prototype.getTitle = function () {
        return this.title;
    };
    /**
     * @method setModal
     * Set if the window is modal or not.
     * @param {Boolean} modal
     * @chainable
     */
    Window.prototype.setModal = function(modal) {
        if(typeof modal !== 'undefined') {
            this.modal = !!modal;
        }
        return this;
    };
    /**
     * @inheritdoc
     */
    Window.prototype.setWidth = function(width) {
        Window.superclass.prototype.setWidth.call(this, width);
        return this.updateDimensionsAndPosition();
    };
    /**
     * @inheritdoc
     */
    Window.prototype.setHeight = function(height) {
        Window.superclass.prototype.setHeight.call(this, height);
        return this.updateDimensionsAndPosition();
    };
    /**
     * @method  setFooterHeight
     * high fixed size for the footer of the window
     * @param {Number|String} height it can be a number or a string.
      In case of using a String equals to 'auto','inherit','##px','##%','##em'
     * @chainable
     */
    Window.prototype.setFooterHeight = function(footerHeight) {
        if(typeof footerHeight === 'number') {
            this.footerHeight = footerHeight;
        } else if(/^\d+(\.\d+)?px$/.test(footerHeight)) {
            this.footerHeight = parseInt(footerHeight, 10);
        } else if(footerHeight === 'auto') {
            this.footerHeight = footerHeight;
        } else {
            throw new Error('setFooterHeight: footerHeight param is not valid.');
        }
        if(this.footer) {
            this.footer.style.height = this.footerHeight + 'px';
        }
        if(this.isOpen) {
            this.updateDimensionsAndPosition();   
        }

        return this;
    };
    /**
     * Sets the alignment for the items on the footer.
     * @param {String} align Possible values: "center", "left", "right".
     * @chainable
     */
    Window.prototype.setFooterAlign = function(align) {
        this.footer.setAlignment(align);
        this.footerAlign = align;
        return this;
    };
    /**
     * buttonPanelPosition parameter sets that have the panel of buttons, within the window
     * @param {String} buttonPanelPosition Accepted values ​​are 'top' or 'bottom'
     */
    Window.prototype.setButtonPanelPosition = function (buttonPanelPosition) {
        if(!(buttonPanelPosition === 'top' || buttonPanelPosition === 'bottom')){
            throw  new Error('setButtonPanelPosition(): the value is not valid, should be a "top" or "bottom"');
        }
        this.buttonPanelPosition = buttonPanelPosition;
        if(this.html){
            if(buttonPanelPosition === 'top') {
                this.html.insertBefore(this.footer.html, this.body);
            } else {
                this.html.appendChild(this.footer.html);
            }
        }
        return this;
    };
    /**
     * Clear the items in the window footer.
     * @chainable
     */
    Window.prototype.clearFooterItems = function() {
        this.footer.clearItems();
        return this;
    };
    /**
     * Adds an item to the window footer.
     * @param {Object|PMUI.ui.Button|PMUI.ui.TextLabel} item This parameter can be:
     *
     * - An object literal, in this case the object literal must have the property "pmType" with its value set to 
     * "button" (if you want the element be a {@link PMUI.ui.Button Button}) or "label" (if you want the element be a 
     * {@link PMUI.ui.TextLabel Label}). Optionally you can add the respective config options for each case.
     *
     * - A PMUI object, in this case it must be an instance of {@link PMUI.ui.Button Button} or an instance of
     * {@link PMUI.ui.TextLabel Label}.
     * @chainable
     */
    Window.prototype.addFooterItem = function(item) {
        this.footer.addItem(item);
        return this;
    };
    /**
     * Sets the items for the window's footer.
     * @param {Array} items This accepted value for this parameter is the same than the one for the 
     * {@link #cfg-footerItems footerItems} config option.
     * @chainable
     */
    Window.prototype.setFooterItems = function(items) {
        var i;
        if (!jQuery.isArray(items)) {
            throw new Error("setFooterItems(): The parameter must be an array.");
        }
        this.clearFooterItems();
        for (i = 0; i < items.length; i += 1) {
            this.addFooterItem(items[i]);
        }
        return this;
    };
    /**
     * @method close
     * if the window is open, sets the status to closed
     * @chainable
     */  
    Window.prototype.close = function ( ) {
        jQuery(this.modalObject).detach();
        jQuery(this.html).detach();
        jQuery(this.closeButton).detach();
        if(typeof this.onClose === 'function') {
            this.onClose(this);
        }
        this.isOpen = false;
        return this;
    };
    /**
     * Shows the window's close button.
     * @chainable
     */
    Window.prototype.showCloseButton = function() {
        this.visibleCloseButton = true;
        if(this.closeButton) {
            this.closeButton.setVisible(true);
        }
        return this;
    };
    /**
     * Adds an child item to the body HTMLElement
     * @param {PMUI.core.Window|Object} item It can be one of the following data types:
     * - {PMUI.core.Element} the object to add
     * - {Object} a JSON object with the settings for the Container to be added
     * @chainable
     */    
    Window.prototype.addItem = function(item) {
        var itemToBeAdded;
  
        if (this.factory) {
            itemToBeAdded = this.factory.make(item);
        }
        if (itemToBeAdded && !this.isDirectParentOf(itemToBeAdded)) {
            itemToBeAdded.parent = this;
            this.items.insert(itemToBeAdded);
            if (this.body) {
                this.body.appendChild(itemToBeAdded.getHTML());   
                if(this.eventsDefined) {
                    itemToBeAdded.defineEvents();
                }
            }
        }
        return this;
    };
    /**
     * @method open
     * If the window is closed, sets the status to open and displays the values ​​set in the properties window, header, body and footer
     * @chainable
     */
    Window.prototype.open = function() {
        var the_window;
        if(this.isOpen) {
            return this;
        }
        the_window = this.getHTML();
        if(this.modal) {
            this.modalObject.appendChild(the_window);
            document.body.appendChild(this.modalObject);
            jQuery(the_window).draggable({ containment: '#' + this.modalObject.id, scroll: false });
        } else {
            document.body.appendChild(the_window);
            jQuery(this.getHTML()).draggable();
        }
        if(typeof this.onOpen === 'function') {
            this.onOpen(this);
        }
        this.isOpen = true;
        this.updateDimensionsAndPosition();
        this.setVisible(true);
        this.defineEvents();


        return this;
    };
    /**
     * Hides the window's close button.
     * @chainable
     */
    Window.prototype.hideCloseButton = function() {
        this.visibleCloseButton = false;
        if(this.closeButton) {
            this.closeButton.setVisible(false);
        }
        return this;
    };
    /**
     * Shows the window's close button.
     * @chainable
     */
    Window.prototype.showCloseButton = function() {
        this.visibleCloseButton = true;
        if(this.closeButton) {
            this.closeButton.setVisible(true);
        }
        return this;
    };
    /**
     * @method updateModalDimensions
     * Updates the status of modal HTMLElement and 
     fixed all styles required to show
     */
    Window.prototype.updateModalDimensions = function() {
        if(document && this.modalObject) {
            this.modalObject.style.height = this.modalObject.style.width = "0px";
            this.modalObject.style.width = window.innerWidth + "px";
            this.modalObject.style.height = window.innerHeight + "px";
        }
        return this;
    };
    /**
     * Sets the padding for the windows body.
     * @param {Number|String} bodyPadding A number or a string with a pixel-unit formatted text.d
     */
    Window.prototype.setBodyPadding = function (bodyPadding) {
        if(typeof bodyPadding === 'number') {
            this.bodyPadding = bodyPadding
        } else if(/^\d+(\.\d+)?px$/.test(height)) {
            this.bodyPadding = parseInt(bodyPadding, 10);
        } else {
            throw new Error('setHeight: height param is not valid.');
        }

        this.bodyPadding = bodyPadding; 
        if(this.html){
            this.body.style.padding = bodyPadding +'px' ;
        }
        return this;
    };
    /**
     * @inheritdoc
     */
    Window.prototype.paintItems = function() {
        if(this.body) {
            Window.superclass.prototype.paintItems.call(this);
        }
        return this;
    };
    /**
     * @method defineEvents
     * Defines the events associated with the Window
     */
    Window.prototype.defineEvents = function() {
        var i,
            j,
            children,
            that = this, 
            html = this.html, 
            modal = this.modal, 
            stopPropagation,
            updateDimensions,
            cancelWheeling = function(e) {
                var height = that.body.clientHeight, 
                    scrollHeight = that.body.scrollHeight,
                    scrollTop = that.body.scrollTop;
                e.stopPropagation();
                if((height +  scrollTop >= scrollHeight && e.deltaY > 0) || (scrollTop === 0 && e.deltaY < 0)) {
                    e.preventDefault();
                }
            };

        Window.superclass.prototype.defineEvents.call(this);
        stopPropagation = new PMUI.event.Action({
            handler: function (e) {
                e.stopPropagation();
            }
        }),
        updateDimensions = new PMUI.event.Action({
            handler: function() {
                that.updateModalDimensions();
            }
        });
        this.addEvent('mousedown').listen(this.body, stopPropagation);
        //this.addEvent('mousedown').listen(this.footer, stopPropagation);
        that.addEvent('click').listen(this.html, stopPropagation);
        that.addEvent('mouseover').listen(this.html, stopPropagation);
        that.addEvent('mouseout').listen(this.html, stopPropagation);
        //that.addEvent('mouseup').listen(modal, stopPropagation);
        that.addEvent('mousedown').listen(this.html, stopPropagation);
        that.addEvent('resize').listen(window, updateDimensions);
        this.addEvent('keydown').listen(this.html, stopPropagation);
        this.addEvent('keypress').listen(this.html, stopPropagation);
        this.addEvent('keyup').listen(this.html, stopPropagation);
        
        this.modalObject.addEventListener('wheel', function(e){
            e.preventDefault();
            e.stopPropagation();
        }, false);

        this.body.addEventListener('wheel', cancelWheeling, false);
        this.footer.defineEvents();
        that.closeButton.defineEvents();  
        this.updateModalDimensions();         
        return this;
    };
    /**
     * @method createHTML
     * Creates natural elements to the window html: head, body and foot
     * @return {HTMLElement} returns a HTML element
     */
    Window.prototype.createHTML = function() {
        var header, 
            body, 
            titleContainer, 
            closeButton, 
            modal,
            that = this;

        if(this.html) {
            return this.html;
        }

        Window.superclass.prototype.createHTML.call(this);

        header = PMUI.createHTMLElement('div');
        header.className = 'pmui-window-header';
        body = PMUI.createHTMLElement('div');
        body.className = 'pmui-window-body';
        modal = PMUI.createHTMLElement('div');
        modal.className = 'pmui-window-modal';
        modal.id = this.id + "-modal";

        titleContainer = PMUI.createHTMLElement('span');
        titleContainer.className = 'pmui-window-title';
        titleContainer.style.display = "inline-block";

        closeButton = new PMUI.ui.Button({
            style: {
                cssClasses: ['pmui-window-close']
            },
            text: "",
            width: '16px',
            handler: function(event) {
                that.close();                
            }
        });

        this.modalObject = modal;
        header.appendChild(titleContainer);
        header.appendChild(closeButton.getHTML());

        this.html.appendChild(header);
        this.html.appendChild(body);
        this.html.appendChild(this.footer.getHTML());

        this.dom.titleContainer = titleContainer;
        this.header = header;
        this.body = this.containmentArea = body;
        this.closeButton = closeButton;

        this.setTitle(this.title)
            .setFooterHeight(this.footerHeight)
            .setItems(this.items.asArray().slice(0))
            .setButtonPanelPosition(this.buttonPanelPosition);

        if(this.visibleCloseButton) {
            this.showCloseButton();
        } else {
            this.hideCloseButton();
        }
        if(this.visibleFooter) {
            this.showFooter();
        } else {
            this.hideFooter();
        }

        if(this.eventsDefined) {
            this.defineEvents();
        }

        return this.html;
    };

    PMUI.extendNamespace('PMUI.ui.Window', Window);

    if (typeof exports !== "undefined") {
        module.exports = Window;
    }
}());
(function() {   
    var MessageWindow = function(settings) {
        MessageWindow.superclass.call(this, settings);
        /**
         * The message for the message window.
         * @type {String}
         * @readonly
         */
        this.message = null;
        /**
         * An object for store the necessary dom elements to build the message window HTML.
         * @type {Object}
         * @private
         */
        this.dom = {};
        this.iconClass = null; 
        this.windowMessageType = null;
        MessageWindow.prototype.init.call(this, settings);
    };

    PMUI.inheritFrom('PMUI.ui.Window', MessageWindow);


    MessageWindow.prototype.type = 'WindowMessage';

    MessageWindow.prototype.family = 'ui';

    MessageWindow.prototype.init = function(settings) {
        var padding;
        var defaults = {
            message: '[not-message]',
            iconClass : '',
            windowMessageType : 'default',
            title : ''
        };
        jQuery.extend(true, defaults, settings);

        this.setMessage(defaults.message);
        this.setWindowMessageType(defaults.windowMessageType);
        this.setTitle(defaults.title); 

    };

    MessageWindow.prototype.setMessage = function (message) {
        if(typeof message !== 'string') {
            throw new Error('setMessage(): the parameter must be a string.');
        }

        this.message = jQuery.trim(message);

        if(this.dom.messageContainer) {
            this.dom.messageContainer.textContent = message;
        }

        return this;
    };
    MessageWindow.prototype.getMessage = function(){
        return this.message;
    };

    MessageWindow.prototype.setWindowMessageType = function(type){
        var classMessage = "";
        if(typeof type !== "string"){
            throw  new Error ("settype(): the type value is not valid");
        }
        this.windowMessageType = type;
        this.style.removeClasses(["pmui-windowmessage-default","pmui-windowmessage-error","pmui-windowmessage-warning","pmui-windowmessage-success"]);
        classMessage = 'pmui-windowmessage-'+this.windowMessageType;
        switch  (this.windowMessageType) {
            case 'error':
                    this.style.addClasses([classMessage]);
                break;
            case 'warning':
                    this.style.addClasses([classMessage]);
                break;
            case 'success':
                    this.style.addClasses([classMessage]);
                break;
            case 'default':
                    this.style.addClasses([classMessage]);
                break;
        }
        return this;
    };

    MessageWindow.prototype.getWindowMessageType = function () {
        return this.windowMessageType;
    };

    MessageWindow.prototype.addItem = function() {
        return this;
    };

    MessageWindow.prototype.createHTML = function() {
        var table, row, cellIcon, cellLabel, icon, p, textContent;

        MessageWindow.superclass.prototype.createHTML.call(this);

        table = PMUI.createHTMLElement('table');
        table.className = 'pmui-messagewindow-container';
        row = PMUI.createHTMLElement('tr');
        cellIcon = PMUI.createHTMLElement('td');
        cellLabel = PMUI.createHTMLElement('td');
        cellLabel.style.textAlign = "center";
        cellLabel.className = "pmui-content-label"
        icon = PMUI.createHTMLElement('span');
        icon.className = 'pmui-messagewindow-icon';
        textContent = PMUI.createHTMLElement('span');
        textContent.className = 'pmui-messagewindow-message';

        cellIcon.appendChild(icon);
        row.appendChild(cellIcon);

        
        cellLabel.appendChild(textContent);
        row.appendChild(cellLabel);
        table.appendChild(row);

        this.dom.icon = icon;
        this.dom.messageContainer = textContent;

        this.body.appendChild(table);

        this.setMessage(this.message);
        this.body.style.padding = "15px";
        cellLabel.style.width =  (typeof this.width == 'number')?this.width*(0.84)+"px":"70%"; 
        this.body.style.minHeight = "inherit";
        return this.html;
    };

    MessageWindow.prototype.updateDimensionsAndPosition = function (){
                var bodyHeight, footerHeight, headerHeight, windowHeight = this.height, windowWidth;
        if(!this.footer || !this.html || !this.isOpen) {
            return this;
        }
        if(this.footerHeight === 'auto') {
            this.footer.setHeight('auto');
        } else {
            this.footer.setHeight(this.footerHeight);
        }
        
        if(windowHeight === 'auto') {
            this.body.style.height = 'auto';
            this.body.style.minHeight = 'inherit';
        } else {
            if(/^\d+(\.\d+)?em$/.test(windowHeight)) {
                windowHeight = PMUI.emToPx(parseInt(windowHeight, 10), this.modalObject);
            } else if(/^\d+(\.\d+)?%$/.test(windowHeight)) {
                windowHeight = jQuery(this.html).outerHeight();
            }
            footerHeight = this.visibleFooter ? jQuery(this.footer.getHTML()).outerHeight() : 0;
            headerHeight = jQuery(this.header).outerHeight();
            bodyHeight = windowHeight - footerHeight - headerHeight;
            if(bodyHeight <= 0) {
                bodyHeight = 0;
            }
            this.body.style.minHeight = '';
            this.body.style.height = bodyHeight + "px";
        }

        windowWidth = jQuery(this.header).width();
        windowWidth = windowWidth - (this.visibleCloseButton ? jQuery(this.closeButton.getHTML()).outerWidth() : 0);
        this.dom.titleContainer.style.width = windowWidth < 0 ? 0 : windowWidth + 'px';

        windowWidth = jQuery(this.html).outerWidth();
        windowHeight = jQuery(this.html).outerHeight();

        this.addCSSProperties({
            left: '50%',
            "margin-left": (windowWidth / -2) + "px",
            top: '50%',
            "margin-top": (windowHeight / -2) + "px"
        });
        return this;
    }

    PMUI.extendNamespace('PMUI.ui.MessageWindow', MessageWindow);

    if (typeof exports !== "undefined") {
        module.exports = MessageWindow;
    }
}());

(function() {
    /**
     * @class PMUI.ui.ErrorMessageWindow
     * A window to show a error message.
     * @extends {PMUI.ui.MessageWindow}
     *
     * Usage example:
     *
     *      @example
     *      var message_window = new PMUI.ui.ErrorMessageWindow({
     *          message: 'This is an error message.'
     *      });
     *      message_window.open();
     *
     * @constructor
     * Creates a new instance of the class ErrorMessageWindow
     * @param {Object} [settings={}] An object with the config options.
     *
     * @cfg {String} [title='Error'] The message to show in the message window.
     */
    var ErrorMessageWindow = function(settings) {
        ErrorMessageWindow.superclass.call(this, settings);
        ErrorMessageWindow.prototype.init.call(this, settings);
    };

    PMUI.inheritFrom('PMUI.ui.MessageWindow', ErrorMessageWindow);
    /**
     * Initializes the object.
     * @param  {Object} settings The object with the config options.
     * @private
     */
    ErrorMessageWindow.prototype.init = function(settings) {
        var defaults = {
            title: 'Error'
        };

        jQuery.extend(true, defaults, settings);

        this.setTitle(defaults.title);
    };
    /**
     * Creates the message window HTML.
     * @return {HTMLElement}
     */
    ErrorMessageWindow.prototype.createHTML = function() {
        ErrorMessageWindow.superclass.prototype.createHTML.call(this);

        jQuery(this.dom.icon).addClass('pmui-messagewindow-icon-error');

        return this.html;
    };
    
    PMUI.extendNamespace('PMUI.ui.ErrorMessageWindow', ErrorMessageWindow);

    if (typeof exports !== "undefined") {
        module.exports = ErrorMessageWindow;
    }
}());
(function() {
    /**
     * @class PMUI.ui.WarningMessageWindow
     * A window to show a warning message.
     * @extends {PMUI.ui.MessageWindow}
     *
     * Usage example:
     *
     *      @example
     *      var message_window = new PMUI.ui.WarningMessageWindow({
     *          message: 'This is a warning message.'
     *      });
     *      message_window.open();
     *
     * @constructor
     * Creates a new instance of the class WarningMessageWindow
     * @param {Object} [settings={}] An object with the config options.
     *
     * @cfg {String} [title='Warning'] The message to show in the message window.
     */
    var WarningMessageWindow = function(settings) {
        WarningMessageWindow.superclass.call(this, settings);
        WarningMessageWindow.prototype.init.call(this, settings);
    };

    PMUI.inheritFrom('PMUI.ui.MessageWindow', WarningMessageWindow);
    /**
     * Initializes the object.
     * @param  {Object} settings The object with the config options.
     * @private
     */
    WarningMessageWindow.prototype.init = function(settings) {
        var defaults = {
            title: 'Warning'
        };

        jQuery.extend(true, defaults, settings);

        this.setTitle(defaults.title);
    };
    /**
     * Creates the message window ĤTML.
     * @return {HTMLElement}
     */
    WarningMessageWindow.prototype.createHTML = function() {
        WarningMessageWindow.superclass.prototype.createHTML.call(this);

        jQuery(this.dom.icon).addClass('pmui-messagewindow-icon-warning');

        return this.html;
    };

    PMUI.extendNamespace('PMUI.ui.WarningMessageWindow', WarningMessageWindow);

    if (typeof exports !== "undefined") {
        module.exports = WarningMessageWindow;
    }
}());
(function() {
    /**
     * @class PMUI.ui.SuccessMessageWindow
     * A window to show a success message.
     * @extends {PMUI.ui.MessageWindow}
     *
     * Usage example:
     *
     *      @example
     *      var message_window = new PMUI.ui.SuccessMessageWindow({
     *          message: 'This is a success message.'
     *      });
     *      message_window.open();
     *
     * @constructor
     * Creates a new instance of the class SuccessMessageWindow
     * @param {Object} [settings={}] An object with the config options.
     *
     * @cfg {String} [title='Success'] The message to show in the message window.
     */
    var SuccessMessageWindow = function(settings) {
        SuccessMessageWindow.superclass.call(this, settings);
        SuccessMessageWindow.prototype.init.call(this, settings);
    };

    PMUI.inheritFrom('PMUI.ui.MessageWindow', SuccessMessageWindow);
    /**
     * Initializes the object.
     * @param  {Object} settings The object with the config options.
     * @private
     */
    SuccessMessageWindow.prototype.init = function(settings) {
        var defaults = {
            title: 'Success'
        };

        jQuery.extend(true, defaults, settings);

        this.setTitle(defaults.title);
    };
    /**
     * Creates the message window ĤTML.
     * @return {HTMLElement}
     */
    SuccessMessageWindow.prototype.createHTML = function() {
        SuccessMessageWindow.superclass.prototype.createHTML.call(this);

        jQuery(this.dom.icon).addClass('pmui-messagewindow-icon-success');

        return this.html;
    };

    PMUI.extendNamespace('PMUI.ui.SuccessMessageWindow', SuccessMessageWindow);

    if (typeof exports !== "undefined") {
        module.exports = SuccessMessageWindow;
    }
}());
(function() {
    /**
     * @class PMUI.ui.InfoMessageWindow
     * A window to show an info message.
     * @extends {PMUI.ui.MessageWindow}
     *
     * Usage example:
     *
     *      @example
     *      var message_window = new PMUI.ui.InfoMessageWindow({
     *          message: 'This is an info message.'
     *      });
     *      message_window.open();
     *
     * @constructor
     * Creates a new instance of the class InfoMessageWindow
     * @param {Object} [settings={}] An object with the config options.
     *
     * @cfg {String} [title='Information'] The message to show in the message window.
     */
    var InfoMessageWindow = function(settings) {
        InfoMessageWindow.superclass.call(this, settings);
        InfoMessageWindow.prototype.init.call(this, settings);
    };

    PMUI.inheritFrom('PMUI.ui.MessageWindow', InfoMessageWindow);
    /**
     * Initializes the object.
     * @param  {Object} settings The object with the config options.
     * @private
     */
    InfoMessageWindow.prototype.init = function(settings) {
        var defaults = {
            title: 'Info'
        };

        jQuery.extend(true, defaults, settings);

        this.setTitle(defaults.title);
    };
    /**
     * Creates the message window HTML.
     * @return {HTMLElement}
     */
    InfoMessageWindow.prototype.createHTML = function() {
        InfoMessageWindow.superclass.prototype.createHTML.call(this);

        jQuery(this.dom.icon).addClass('pmui-messagewindow-icon-info');

        return this.html;
    };

    PMUI.extendNamespace('PMUI.ui.InfoMessageWindow', InfoMessageWindow);

    if (typeof exports !== "undefined") {
        module.exports = InfoMessageWindow;
    }
}());
(function() {
    /**
     * @class PMUI.ui.ConfirmMessageWindow
     * A window to show a question message. It haas built-in Yes and No buttons, but they can be overrode by some 
     * custom ones.
     * @extends {PMUI.ui.MessageWindow}
     *
     * Usage example:
     *
     *      @example
     *      var message_window = new PMUI.ui.ConfirmMessageWindow({
     *          message: 'Do you want to display an alert?',
     *              onYes: function() {
     *                  alert("this is an alert");
     *                  message_window.close();
     *              }, 
     *              onNo: function() {
     *                  message_window.close();
     *              }
     *      });
     *      message_window.open();
     *
     * @constructor
     * Creates a new instance of the class ConfirmMessageWindow
     * @param {Object} [settings={}] An object with the config options.
     *
     * @cfg {String} [title='Error'] The message to show in the message window.
     * @cfg {String} [yesText='Yes'] The text for the window's built-in Yes button.
     * @cfg {String} [noText='No'] The text for the window's built-in No button.
     * @cfg {Function|null} [onYes=null] The handler to be executed when the built-in Yes button is clicked.
     * @cfg {Function|null} [onNo=null] The handler to be executed when the built-in No button is clicked.
     * @cfg footerItems
     * The items to be shown in the message window's footer, the accepted values for this are the same that the ones 
     * for the Window's footerItems config option.
     * 
     * By default this is an empty array, if you modify this then the built-in Yes and No buttons won't be used, 
     * therefore the yesText, noText config options and onYes, onNo handlers won't be used either.
     */
    var ConfirmMessageWindow = function(settings) {
        ConfirmMessageWindow.superclass.call(this, settings);
        ConfirmMessageWindow.prototype.init.call(this, settings);
    };

    PMUI.inheritFrom('PMUI.ui.MessageWindow', ConfirmMessageWindow);
    /**
     * Initializes the object.
     * @param  {Object} settings The object with the config options.
     * @private
     */
    ConfirmMessageWindow.prototype.init = function(settings) {
        var yesButton = new PMUI.ui.Button(),
            noButton = new PMUI.ui.Button(),
            defaults = {
                title: 'Confirm',
                yesText: 'Yes',
                noText: 'No',
                onYes: null,
                onNo: null,
                footerItems: []
            };

        jQuery.extend(true, defaults, settings);

        this.setTitle(defaults.title);

        if(!defaults.footerItems || !defaults.footerItems.length) {
            defaults.footerItems = [
                yesButton.setHandler(defaults.onYes).setText(defaults.yesText),
                noButton.setHandler(defaults.onNo).setText(defaults.noText)
            ];
        }
        this.setFooterItems(defaults.footerItems);
    };
    /**
     * @inheritdoc
     */
    ConfirmMessageWindow.prototype.createHTML = function() {
        ConfirmMessageWindow.superclass.prototype.createHTML.call(this);

        jQuery(this.dom.icon).addClass('pmui-messagewindow-icon-confirm');

        return this.html;
    };
    
    PMUI.extendNamespace('PMUI.ui.ConfirmMessageWindow', ConfirmMessageWindow);

    if (typeof exports !== "undefined") {
        module.exports = ConfirmMessageWindow;
    }
}());
(function () {
    /**
     * @class PMUI.ui.Button
     * @extend PMUI.core.Element
     *   * Usage example (only for subclasses since this is an abstract class):
     *
     *          @example
     *          //Remember, this is an abstract class so it shouldn't be instantiate,
     *          //anyway we are instantiating it just for this example
     *
     *          var a;
     *      
     *           a = new PMUI.ui.Button({
     *               handler : function () {
     *                   alert("hi!");     
     *               },
     *               text : 'closed',
     *               iconClass:"pmui-gridpanel-pager_previous",
     *               iconPosition : "left",
     *               messageTooltip : 'this button to close the window'
     *           });
     *
     *           document.body.appendChild(a.getHTML());
     *           a.defineEvents();
     *      
     * @constructor
     * Create a new instace of the Window class 
     * @param {Object} settigs Constructor object 
     */
    var Button = function (settings, parent) {
        Button.superclass.call(this, jQuery.extend(settings, {elementTag: "a"}));
        /**
         * @property icon for the Button, it can a string 
         * @type {String}
         */
        this.icon = null;
        /**
         * @property {PMUI.event.Action} Defines the Action Object to handle he action windows
         * @type {Action}
         */
        this.action = null;
        /**
         * @property parent 
         * @type {String}
         */
        this.parent = null;
        /**
         * @property text of the Button 
         * @type {String}
         */
        this.text = null;
        /**
         * @property alias Button for idetified to alias 
         * @type {String}
         */
        this.aliasButton = null;
        /**
         * [iconPosition description]
         * @type {[type]}
         */
        this.iconPosition = null; 
        /**
         * [messageTooltip description]
         * @type {[type]}
         */
        this.messageTooltip = null;

        //this.buttonClass = null;

        //this.foreignClassName = null;

        this.linkStyle = null;

        this.labelVisible = null;
        this.iconVisible = null;
        this.disabled = null;
        this.dom  = null ;
        this.buttonType = null;

        Button.prototype.init.call(this, settings, parent);
    };

    PMUI.inheritFrom('PMUI.core.Element', Button);

    Button.prototype.type = 'Button';
    Button.prototype.family = 'Button';

    Button.prototype.init = function (settings, parent) {
        var defaults;
        defaults = {
            iconClass : '',
            aliasButton : null,
            parent : parent || null,
            height: 28,
            width: "auto",
            handler : function (){},
            text : '[button]',
            iconPosition : "right",
            tooltip : false,
            messageTooltip : "",
            //buttonClass : '',
            iconVisible: true,
            labelVisible: true,
            disabled : false,
            //cls: 'pmui-button',
            //linkStyle: false,
            //foreignClassName : "",
            //useForeignClass : false
            buttonType : 'default'
        };
        this.dom = {};

        jQuery.extend(true, defaults, settings);

        this.setAliasButton(defaults.aliasButton)
            .setParent(defaults.parent)
            .setText(defaults.text)
            .setIcon(defaults.iconClass)
            .setWidth(defaults.width)
            .setHeight(defaults.height)
            .setIconPosition(defaults.iconPosition)
            .setHandler(defaults.handler)
            .setTooltipMessage(defaults.messageTooltip)
            //.setTooltip(defaults.tooltip)
            //.setforeignClassName(defaults.foreignClassName, defaults.useForeignClass)
            //.setButtonClass(defaults.buttonClass)
            //.setLinkStyle(defaults.linkStyle)
            .setDisabled(defaults.disabled)
            .setButtonType(defaults.buttonType);

    };

    Button.prototype.setButtonType = function (buttonType){
        var classButton = "";
        if(typeof buttonType !== "string"){
            throw  new Error ("setButtonType(): the type value is not valid");
        }
        this.buttonType = buttonType;
        this.style.removeClasses(["pmui-error","pmui-warning","pmui-success","pmui-info","pmui-link"]);
        classButton = 'pmui-'+this.buttonType;
        switch  (this.buttonType) {
            case 'error':
                    this.style.addClasses([classButton]);
                break;
            case 'warning':
                    this.style.addClasses([classButton]);
                break;
            case 'success':
                    this.style.addClasses([classButton]);
                break;
            case 'info':
                    this.style.addClasses([classButton]);
                break;
            case 'link':
                    this.style.addClasses([classButton]);
                break;
        }
        return this;
    };

    Button.prototype.showLabel = function() {
        this.labelVisible = true;
        if(this.html) {
            this.label.style.display = '';
        }
        return this;
    };

    Button.prototype.setDisabled = function (value) {
        if(typeof value !== 'boolean') {
            throw new Error ('setDisabled(): the parameter is not valid, should be type boolean');
        }
        this.disabled = value;
        if (this.html) {
            if (this.disabled) {
                this.disable();
            } else {
                this.enable();
            }
        }
        return this;
    }

    Button.prototype.disable = function () {
        this.disabled = true;
        this.style.addClasses(['pmui-disabled']);
        if(this.eventsDefined) {
            this.defineEvents();
        }
        return this;
    };
    /**
     * @method enable
     * button can be enabled if it was disabled before
     * @chainable
     */
    Button.prototype.enable = function () {
        this.disabled = false;
        this.style.removeClasses(['pmui-disabled']);
        if(this.eventsDefined) {
            this.defineEvents();
        }
        return this;
    };
/*
    Button.prototype.hideLabel = function() {
        this.labelVisible = false;
        if(this.html) {
            this.label.style.display = 'none';
        }
        return this;
    };

    Button.prototype.showIcon = function() {
        this.iconVisible = true;
        if(this.html) {
            this.icon.style.display = '';
        }
        return this;
    };

    Button.prototype.hideIcon = function() {
        this.iconVisible = false;
        if(this.html) {
            this.icon.style.display = 'none';
        }
        return this;
    };*/

    /**
     * Performs the click action on the button.
     * @chainable
     */
    Button.prototype.click = function() {
        jQuery(this.html).trigger('click');
        return this;
    };

    /*
    Button.prototype.setButtonClass = function (buttonClass) {
        var value;
        if(! (typeof buttonClass == "string")) {
            throw new Error ("buttonClass(): the property buttonClass should be a type string");
        }  
        this.buttonClass = buttonClass;
        value = !!buttonClass;
        if(this.html){
            if(value) {
                this.html.className = this.buttonClass;
            } else {
                this.applyStyle();
            }
        }
        value = !!buttonClass;

        return this;
    };*/
    
    /**
     * @method setHandler
     * Sets the handler for the button
     * @param {Function} handler is a Function. 
     * @chainable
     */
    
    Button.prototype.setHandler = function (handler) {
        if(typeof handler !== "function" && handler !== null) {
            throw new Error ("setHandler(): the parameter should be a function or null;");
        }
        this.handler = handler;
        this.action = new PMUI.event.Action({
            actionText : this.aliasButton,
            handler: this.handler
        });
        return this;
    };

    Button.prototype.removeEvent = function (type) {
        jQuery(this.html).unbind(type);
        return this;
    };



    Button.prototype.removeEvents = function () {
        var x;
        for ( x in  this.events ) {
            this.removeEvent(this.events[x].eventName);
        }

        /*var that = this;
        jQuery.each(this.events, function(event,object){
            that.removeEvent(object.eventName);
        });*/

        this.events = {};
        return this;
    };

    /**
     * Sets the text for the label Button.
     * @param {String} is a String. 
     * @chainable
     */
    Button.prototype.setText = function (text) {
        this.text = text;
        if (this.html && this.dom.spanText) {
            if(jQuery.trim(text) !== "") {
                this.dom.spanText.textContent = this.text;
            } else {
                this.dom.spanText.inerHTML = '&nbsp;';
            }
        }
        return this;
    };
    /**
     * Sets the icon for the Button.
     * @param {String} is a String. 
     * @chainable
     */
    Button.prototype.setIcon = function (iconClass) {
        var icon;
        if (!(typeof iconClass == "string")) {
            throw new Error ("setIcon(): the property should bo string");
        }
        this.icon = iconClass;
        if (this.html && this.dom.icon){
            this.dom.icon.className = this.dom.icon.className + " " + this.icon;  
        }
        return this;
    };
    /**
     * @method setAliasButton
     * Sets the alias for the label Button
     * @param {String} is a String. lue
     * @chainable
     */
    Button.prototype.setAliasButton = function (alias) {
        this.aliasButton = alias;
        return this;
    };
    /**
     * @method setParent
     * Sets the parent for the Button.
     * @param {Object} is a HTML Element. 
     * @chainable
     */
    Button.prototype.setParent = function (parent){
        this.parent = parent;
        return this;
    };
    /**
     * @method createHTML
     * Create HTML for the element
     * @return {HTMLElement} HTML Generated  
     */    

    Button.prototype.createHTML = function (){
        var spanText, icon;
        if (this.html) {
            return this.html;
        }
        Button.superclass.prototype.createHTML.call(this);
        this.html.href = '#';
        //***conatainer for text
        spanText = PMUI.createHTMLElement('span');
        spanText.className = "button-label";
        this.dom.spanText = spanText;
        this.html.appendChild(spanText); 

        // container for icon
        icon = PMUI.createHTMLElement('span');
        icon.className = 'button-icon';
        this.dom.icon = icon;
        this.html.appendChild(icon);

        this.setIconPosition(this.iconPosition);
        this.setTooltipMessage(this.messageTooltip);

        this.setDisabled(this.disabled);
        this.setText(this.text);
        this.setIcon(this.icon);
        this.applyStyle();

        return this.html;
    };
    /**
     * @method defineEvents
     * Define events click for the Button, listening the function action  
     */
    
    Button.prototype.defineEvents = function () {
        var that = this,
            fnClick;
        Button.superclass.prototype.defineEvents.call(this);
        if(this.html) {
            if(this.disabled) {
                this.addEvent('click').listen(this.html, function(e) {
                    e.preventDefault();
                    e.stopPropagation();
                });
            } else {
                this.addEvent('click').listen(this.html, function (e){
                    e.preventDefault();
                    e.stopPropagation();
                    this.focus();
                    if(typeof that.handler === 'function') {
                        that.handler(that);    
                    }
                }); 
            }
        }
        return this;
    };
    /**
     * @method setHeight
     * Sets the height for the button.
     * @param {String, number} height
     * This parameter can take the following types 
     * - number 
     * - string 
     */
    Button.prototype.setHeight = function(height) {
        if(typeof height === 'number') {
            this.height = height + "px";   
        } else if(/^\d+(\.\d+)?px$/.test(height)) {
            this.height = height;
        } else if(/^\d+(\.\d+)?%$/.test(height)) {
            this.height = height;
        } else if(/^\d+(\.\d+)?em$/.test(height)) {
            this.height = height;
        } else if(height === 'auto' || height === 'inherit') {
            this.height = height;
        } else {
            throw new Error('setHeight: height param is not a number');
        }
        if (this.height !== 'auto') {
            this.style.addProperties({"line-height": parseInt(this.height, 10) + "px"});
        } else {
            this.style.addProperties({"line-height": 'normal'});
        }
        this.applyStyle();
       // this.centerLabel();
        return this;
    };
    /**
     * @method setIconPosition
     * Creates the HTML element for the control.
     * @param {String} position
     * This parameter can take the following values: left, right
     */
    Button.prototype.setIconPosition = function (position) {
        if (position == 'right' || position == "left"  || position == "top"  || position == "bottom" ) {
            this.iconPosition = position;
            if(this.html){
                if (this.iconPosition == 'left' || this.iconPosition=='top'){
                    $(this.html).prepend(this.dom.icon);
                    if (this.iconPosition == 'left'){
                        this.dom.icon.style.display = 'inline-block';
                        this.dom.icon.style.marginLeft = '';

                    }else{
                        this.dom.icon.style.display = 'block';
                        this.dom.icon.style.marginLeft = '50%';
                    }
                }else{
                    $(this.html).append(this.dom.icon);
                    if (this.iconPosition == 'right'){
                        this.dom.icon.style.display = 'inline-block';
                        this.dom.icon.style.marginLeft = '';
                    }else{
                        this.dom.icon.style.display = 'block';
                        this.dom.icon.style.marginLeft = '50%';
                    }
                }
            }

        } else {
            throw new Error ("setIconPosition(): the parameter is not valid, should be 'right','left', 'top' or 'bottom'");
        }
        return this;
    };
    /**
     * @method setTooltipMessage
     * Sets the tooltip message
     * @param {String} messageTooltip     
     */
    Button.prototype.setTooltipMessage = function (messageTooltip) {
        if ( typeof messageTooltip == "string") {
            this.messageTooltip = messageTooltip;
            if (this.html) {
                this.html.setAttribute('title',messageTooltip);
            }
        }
        return this;
    };
    /**
     * @method removeMessageTooltip
     * remove the tooltip message of the button
     * @param {String} messageTooltip     
     */
   /*Button.prototype.removeMessageTooltip = function () {
        if(this.html){
            this.html.removeAttribute('title');
        }   
        return this;
    };*/
    
    PMUI.extendNamespace("PMUI.ui.Button", Button);
        
        if (typeof exports !== "undefined") {
            module.exports = Button;
        }
}());


(function() {
    /**
     * @class PMUI.ui.TooltipMessage
     * @extends PMUI.core.Element
     * Class to handle tooltip elements and alert messages
     *
     * Usage example (creates 4 TooltipMessage instances):
     *
     *      @example
     *      //usage example
     *      //creates 4 TooltipMessage instances:
     *      var a, b, c, d;
     *      $(function() {
     *          // Instantiates a simple info message
     *          a = new PMUI.ui.TooltipMessage({
     *              message: "Hi folks! this is a info",
     *              category: "info",
     *              displayMode: "block",
     *              mode: "normal"
     *          });
     *          //Instantiates an error tooltip with mouse tracking option enabled
     *          b = new PMUI.ui.TooltipMessage({
     *              message: "Hi folks! this is an error",
     *              category: "error",
     *              track: true,
     *              displayMode: "block"
     *          });
     *          //Instantiates a warning tooltip with a slide down effect when the tooltip is opened
     *          c = new PMUI.ui.TooltipMessage({
     *              message: "Hi folks! this is a warning",
     *              category: "warning",
     *              displayMode: "block",
     *              showEffect: {
     *                  effect: "slideDown",
     *                  delay: 250
     *              }
     *          });
     *          //Instantiates a help tooltip with a explode effect when the tooltip is hidden
     *          d = new PMUI.ui.TooltipMessage({
     *              message: "Hi folks! this is a help",
     *              category: "help",
     *              displayMode: "block",
     *              hideEffect: {
     *                  effect: "explode",
     *                  delay: 250
     *              }
     *          });
     *          document.body.appendChild(a.getHTML());
     *          document.body.appendChild(b.getHTML());
     *          document.body.appendChild(c.getHTML());
     *          document.body.appendChild(d.getHTML());
     *      });
     * @constructor
     * Create a new instace of the class 'TooltipMessage'
     * @param {Object} settings 
     * A JSON object which can contain the following fields: 
     * 
     * @cfg {String} [message = ""] The message to show
     * @cfg {String} [category = 'help'] The object's category, it can be "info", "help", "warning" or "error"
     * @cfg {String} [displayMode = 'inline'] Determines if the object will have an "inline" or "block" displaying
     * @cfg {String} [mode="tooltip"] When is set to "tooltip" the object has a tooltip behavior, but when is set to "normal" it 
     * has a regular message behavior
     * @cfg {String} [tooltipClass='pmui-tooltip-message'] the css class for the HTML element which will contain the message
     * @cfg {Object} [tooltipPosition = { 
                my: "left top+15", 
                at: "left bottom", 
                collision: "flipfit" 
            }] specifies the position for the tooltip, read the 
      {@link PMUI.ui.TooltipMessage#setTooltipPosition .setTooltipPosition()} for more info
     * @cfg {Object} [showEffect=null] specifies the animation to apply when the tooltip is shown, read the 
      {@link PMUI.ui.TooltipMessage#setShowEffect .setShowEffect()} for more info
     *  @cfg {Object} [hideEffect=null] specifies the animation to apply when the tooltip is hidden, read the 
      {@link PMUI.ui.TooltipMessage#setHideEffect .setHideEffect()} for more info
     * @cfg {Function} [onOpen] a callback function to be invoked when the tooltip is shown
     * @cfg {Function} [onClose] a callback functio to be invoked when the tooltip is hidden
     * @cfg {Boolean} [track = false] a turns on/off the tooltip to follow the mouse movement
     * Note: tooltipClass, tooltipPosition, showEffect, hideEffect, onOpen, onClose and track options 
     only are applied when the mode is set to "tooltip"  
     */
    var TooltipMessage = function(settings) {
        TooltipMessage.superclass.call(this, settings);
        /**
         * @property {PMUI.core.Element} icon
         * An {@link PMUI.core.Element} which is used to put the icon in
         * @private
         */
        this.icon = null;
        /**
         * An Element
         * @type {PMUI.core.Element} which is used to display the message when the mode is set to "normal"
         * @private
         */
        this.messageArea = null;
        /**
         * @property {String} [message=""] 
         * The message to be displayed
         * @readonly
         */
        this.message = null;
        /**
         * @property {String} [category="help"] 
         * The category for the message
         * @readonly
         */
        this.category = null;
        /**
         * @property {String} [displayMode="inline"]
         * The displaying mode for the object's HTML element
         * @readonly
         */
        this.displayMode = null;
        /**
         * @property {String} [mode="tooltip"]
         * The object's behavior mode
         * @readonly
         */
        this.mode = null;
        /**
         * @property {Object} [tooltipPosition={ 
                my: "left top+15", 
                at: "left bottom", 
                collision: "flipfit" 
            }]
         * The tooltip position
         * @readonly
         */
        this.tooltipPosition = null;
        /**
         * @property {String} [tooltipClass="pmui-tooltip-message"]
         * The CSS class name for the tooltip HTML element
         * @readonly
         */
        this.tooltipClass = null;
        /**
         * @property {Object|Boolean|String|Number} [showEffect=null]
         * The effect to apply when the tooltip is shown
         */
        this.showEffect = null;
        /**
         * @property {Object|Boolean|String|Number} [hideEffect=null]
         * The effect to apply when the tooltip is hidden
         */
        this.hideEffect = null;
        /**
         * @property {Function} [onOpen=null]
         * The function callback to be invoked when the tooltip is shown
         */
        this.onOpen = null;
        /**
         * @property {Function} [onClose=null]
         * The function callback to be invoked when the tooltip is hidden
         * @type {Function}
         */
        this.onClose = null;
        /**
         * @property {Boolean} [track=false]
         * A boolean that specifies if the the tooltip element must follow the mouse position
         */
        this.track = null;
        TooltipMessage.prototype.init.call(this, settings);
    };

    PMUI.inheritFrom('PMUI.core.Element', TooltipMessage);

    TooltipMessage.prototype.type = 'PMUITooltipMessage';

    TooltipMessage.prototype.family = 'PMUITooltipMessage';

    TooltipMessage.prototype.init = function(settings) {
        var defaults = {
            message: "",
            category: "help",
            displayMode: "inline",
            mode: "tooltip",
            tooltipClass: 'pmui-tooltip-message',
            tooltipPosition: { 
                my: "left top+15", 
                at: "left bottom", 
                collision: "flipfit" 
            },
            showEffect: null,
            hideEffect: null, 
            onOpen: null, 
            onClose: null,
            track: false
        };

        $.extend(true, defaults, settings);

        this.onClose = defaults.onClose;
        this.onOpen = defaults.onOpen;

        this.setTooltipClass(defaults.tooltipClass)
            .setTooltipPosition(defaults.tooltipPosition)
            .setShowEffect(defaults.showEffect)
            .setHideEffect(defaults.hideEffect)
            .setTrack(defaults.track)
            .setMessage(defaults.message)
            .setCategory(defaults.category)
            .setDisplayMode(defaults.displayMode)
            .setMode(defaults.mode);
    };
    /**
     * Set the CSS class for the HTML element which will contain the tooltip message.
     * This only takes effect when the mode property is set to "tooltip"
     * @param {String} tooltipClass
     */
    TooltipMessage.prototype.setTooltipClass = function(tooltipClass) {
        this.tooltipClass =  tooltipClass;
        return this;
    };
    /**
     * Establish if the HTML element which will be contain the tooltip message should track 
     (follow) the mouse position when it appears
     * * This only takes effect when the mode property is set to "tooltip"
     * @param {Boolean} [track=true]
     */
    TooltipMessage.prototype.setTrack = function(track) {
        this.track = !! track;
        if(this.html) {
            this.setMode(this.mode);
        }
        return this;
    };
    /**
     * [setHideEffect description]
     * If and how to animate the hiding of the tooltip.
     * This only takes effect when the mode property is set to "tooltip"
     * @param {Object|Boolean|String|Number} effect
     * - Boolean: When set to false, no animation will be used and the tooltip will be hidden 
     immediately. When set to true, the tooltip will fade out with the default duration and the 
     default easing.
     * - Number: The tooltip will fade out with the specified duration and the default easing.
     * - String: The tooltip will be hidden using the specified effect. The value can either be the 
     name of a built-in jQuery animation method, such as "slideUp", or the name of a jQuery UI effect, 
     such as "fold". In either case the effect will be used with the default duration and the default easing.
     * - Object: If the value is an object, then effect, delay, duration, and easing properties may be 
     provided. If the effect property contains the name of a jQuery method, then that method will be used; 
     otherwise it is assumed to be the name of a jQuery UI effect. When using a jQuery UI effect that supports 
     additional settings, you may include those settings in the object and they will be passed to the effect. 
     If duration or easing is omitted, then the default values will be used. If effect is omitted, 
     then "fadeOut" will be used. If delay is omitted, then no delay is used.
     */
    TooltipMessage.prototype.setHideEffect = function(effect) {
        this.hideEffect = effect;
        if(this.html) {
            this.setMode(this.mode);
        }
        return this;
    };
    /**
     * If and how to animate the showing of the tooltip.
     * This only takes effect when the mode property is set to "tooltip"
     * @param {Object|Boolean|String|Number} effect
     * - Boolean: When set to false, no animation will be used and the tooltip will be shown immediately. 
     When set to true, the tooltip will fade in with the default duration and the default easing.
     * - Number: The tooltip will fade in with the specified duration and the default easing.
     * - String: The tooltip will be shown using the specified effect. The value can either be the name of 
     a built-in jQuery animation method, such as "slideDown", or the name of a jQuery UI effect, such as 
     "fold". In either case the effect will be used with the default duration and the default easing.
     * - Object: If the value is an object, then effect, delay, duration, and easing properties may be 
     provided. If the effect property contains the name of a jQuery method, then that method will be used; 
     otherwise it is assumed to be the name of a jQuery UI effect. When using a jQuery UI effect that 
     supports additional settings, you may include those settings in the object and they will be passed 
     to the effect. If duration or easing is omitted, then the default values will be used. If effect is 
     omitted, then "fadeIn" will be used. If delay is omitted, then no delay is used.
     */
    TooltipMessage.prototype.setShowEffect = function(effect) {
        this.showEffect = effect;
        if(this.html) {
            this.setMode(this.mode);
        }
        return this;
    };
    /**
     * Identifies the position of the tooltip in relation to the associated target element. 
     * The of option defaults to the target element, but you can specify another element to position 
     against. You can refer to the [jQuery UI Position][1] utility for more details about the various options.
     * [1]: http://api.jqueryui.com/position
     * @param {Object} position
     */
    TooltipMessage.prototype.setTooltipPosition = function(position) {
        this.tooltipPosition = position;
        if(this.html) {
            this.setMode(this.mode);
        }
        return this;
    };
    /**
     * Sets the message to be shown in both normal and tooltip modes
     * @param {String} message
     */
    TooltipMessage.prototype.setMessage = function(message) {
        if(typeof message === 'string') {
            this.message = message;
            if(this.html) {
                if(this.messageArea) {
                    this.messageArea.getHTML().textContent = message;
                }

                if(this.mode === 'tooltip') {
                    this.icon.html.title = message;
                } else {
                    this.icon.html.title = "";
                }
            }
        } else {
            throw new Error("setMessage() method only accepts string values.");
        }
        return this;
    };
    /**
     * Set the category for the tooltip/message object
     * @param {String} category the value can be one of the following:
     * 
     * - help
     * - info
     * - error
     * - warning
     */
    TooltipMessage.prototype.setCategory = function(category) {
        var validCategories = [
                "help", "info", "error", "warning"
            ];
        if(typeof category === 'string' && validCategories.indexOf(category) > -1) {
            this.category = category;
            if(this.icon && this.messageArea) {
                this.icon.style.removeAllClasses();
                this.icon.style.addClasses(['pmui-icon', 'pmui-icon-' + category]);
                this.messageArea.className = 'pmui-tooltip-message pmui-tooltip-' + category + '-message';
            }
            if(this.html) {
                if(this.category === "error") {
                    this.style.addClasses(["pmui-tooltip-category-error"]);
                } else {
                    this.style.removeClasses(["pmui-tooltip-category-error"]);
                }
            }
        } else {
            throw new Error('setCategory() method only accepts' + 
                ' one of the following values: "help", "info", "warning", "info".');
        }

        return this;
    };
    /**
     * Sets the CSS displaying mode
     * @param {String} displayMode It can take the "block" or "inline" values
     */
    TooltipMessage.prototype.setDisplayMode = function(displayMode) {
        if(displayMode === 'block' || displayMode === 'inline') {
            this.displayMode = displayMode;
            if(this.html) {
                this.style.addProperties({"display": displayMode});
            }
        } else {
            throw new Error('setDisplayMode() method only accepts "inline" or "block" values');
        }

        return this;
    };
    /**
     * Sets the mode for the object, it determines if the current object has a tooltip or notification message behavior
     * @param {String} mode It only can take one of the following values:
     * - "tooltip"
     * - "normal"
     */
    TooltipMessage.prototype.setMode = function(mode) {
        if(mode === 'tooltip' || mode === 'normal') {
            this.mode = mode;
            if(this.html) {
                $(this.html).addClass('pmui-tooltip-mode-' + mode);
                if(mode === 'tooltip') {
                    this.messageArea.setVisible(false);
                    this.icon.html.title = this.message;
                    $(this.icon.html).tooltip({
                        tooltipClass: this.tooltipClass,
                        position: this.tooltipPosition,
                        show: this.showEffect,
                        hide: this.hideEffect,
                        open: this.onOpen,
                        track: this.track,
                        close: this.onClose
                    });
                } else {
                    try {
                        $(this.icon.html).tooltip('destroy');
                    } catch(ex) {}
                    this.icon.html.title = "";
                    this.messageArea.setVisible(true);
                }
            }
        } else {
            throw new Error('setMode() method only accepts "tooltip" or "normal" values');
        }

        return this;
    };
    /**
     * It creates the HTML Element for the current object
     * @return {HTMLElement}
     */
    TooltipMessage.prototype.createHTML = function() {
        var html, messageArea, icon;
        if(this.html) {
            return this.html;
        }

        html = PMUI.createHTMLElement("span");
        html.className = 'pmui-tooltip';
        icon = new PMUI.core.Element({
            elementTag: 'span',
            width: 18,
            height: 18,
            style: {
                cssClasses: ['pmui-icon'],
                cssProperties: {
                    display: "inline-block",
                    "vertical-align": "middle"
                }
            }
        });
        messageArea = new PMUI.core.Element({
            elementTag: 'span',
            style: {
                cssClasses: ['pmui-tooltip-message']
            }       
        });

        html.appendChild(icon.getHTML());
        html.appendChild(messageArea.getHTML());
        this.icon = icon;
        this.messageArea = messageArea;
        this.html = html;

        this.applyStyle();
        
        this.setCategory(this.category);
        this.setMessage(this.message);
        this.setMode(this.mode);

        return this.html;
    };

    PMUI.extendNamespace('PMUI.ui.TooltipMessage', TooltipMessage);

    if (typeof exports !== "undefined") {
        module.exports = TooltipMessage;
    }
}());

(function() {
    /**
     * @class PMUI.ui.TextLabel
     * A text label that can contain a plain text or an HTML formatted text.
     * @extends {PMUI.core.Element}
     *
     * Usage example:
     *
     *      @example
     *      var plainText, htmlText;
     *      plainText = new PMUI.ui.TextLabel({
     *          textMode: 'plain',
     *          text: 'Hello! this is a label with a plain text in it',
     *          style: {
     *              cssProperties: {
     *                  background: "red"
     *              }
     *          }
     *      });
     *      htmlText = new PMUI.ui.TextLabel({
     *          textMode: 'html',
     *          text: '<h1>Hello!</h1> this ia a label with a <b> HTML</b> formatted text'
     *      });
     *      document.body.appendChild(plainText.getHTML());
     *      document.body.appendChild(htmlText.getHTML());
     *
     * @constructor
     * Creates a new instance of the class.
     * @param {Object} [settings={}] An object literal with the config options to config the object.
     *
     * @cfg {String} [textMode='plain'] The rendering mode for the text in the label. Possible values: 
     * 
     * - "plain": for plain text rendering.
     * - "html": for html text rendering. Only the <b>, <i>, <h1> to <h6> tags are allowed, any other tags will be 
     * removed. 
     * @cfg {String} [elementTag='span'] The HTML tag to be used as the text label.
     * @cfg {String} [text=""] The label's text.
     */
    var TextLabel = function(settings) {
        TextLabel.superclass.call(this, settings);
        /**
         * The text rendering mode. Set by the {@link #cfg-textMode textMode config option} and the 
         * {@link #method-setTextMode setTextMode() method}.
         * @type {String}
         * @readonly
         */
        this.textMode = null;
        /**
         * The current displayed label's text. Set by the {@link #cfg-text text config option} and the 
         * {@link #method-setText setText() method}.
         * @type {String}
         * @readonly
         */
        this.text = null;
        /**
         * The original set text.
         * @type {String}
         * @private
         */
        this.rawText = null;
        TextLabel.prototype.init.call(this, settings);
    };

    PMUI.inheritFrom('PMUI.core.Element', TextLabel);
    /**
     * The object's type.
     * @type {String}
     */
    TextLabel.prototype.type = 'TextLabel';
    /**
     * Initializes the object.
     * @param  {Object} [settings={}] An object literal with the config options the new object will be initialized 
     * with.
     * @chainable
     * @private
     */
    TextLabel.prototype.init = function(settings) {
        var defaults = {
            textMode: 'plain',
            elementTag: 'span',
            text: ''
        };

        jQuery.extend(true, defaults, settings);

        this.setElementTag(defaults.elementTag)
            .setTextMode(defaults.textMode)
            .setText(defaults.text);

        return this;
    };
    /**
     * Sets the label's text.
     * @param {String} text
     */
    TextLabel.prototype.setText = function(text) {
        var tags;
        if(typeof text !== 'string') {
            throw new Error('setText(): the parameter must be a string.');
        }
        this.text = this.rawText = text;
        if(this.textMode === 'html') {
            allowed = ("<b><i><h1><h2><h3><h4><h5><h6>".match(/<[a-z][a-z0-9]*>/g) || []).join('');
            tags = /<\/?([a-z][a-z0-9]*)\b[^>]*>/gi,
            commentsAndPhpTags = /<!--[\s\S]*?-->|<\?(?:php)?[\s\S]*?\?>/gi;
            this.text = text.replace(commentsAndPhpTags, '').replace(tags, function ($0, $1) {
                return allowed.indexOf('<' + $1.toLowerCase() + '>') > -1 ? $0 : '';
            });
        }

        if(this.html) {
            if(this.textMode === 'html') {
                this.html.innerHTML = this.text;
            } else {
                this.html.textContent = this.text;
            }
        }
        return this;
    };
    /**
     * Sets the text rendering mode.
     * @param {String} textMode Possible values: 
     * 
     * - "plain": for plain text rendering.
     * - "html": for html text rendering. Only the <b>, <i>, <h1> to <h6> tags are allowed, any other tags will be 
     * removed. 
     */
    TextLabel.prototype.setTextMode = function(textMode) {
        if(textMode !== 'plain' && textMode !== 'html') {
            throw Error("setTextMode(): The parameter must be \"plain\" or \"html\".");
        }
        this.textMode = textMode;
        if(typeof this.rawText === 'string') {
            this.setText(this.rawText);
        }
        return this;
    };
    /**
     * Creates the label's HTML.
     * @return {HTMLElement}
     */
    TextLabel.prototype.createHTML = function() {
        if(this.html) {
            return this.html;
        }
        TextLabel.superclass.prototype.createHTML.call(this);
        this.setText(this.rawText);
        return this.html;
    };

    PMUI.extendNamespace('PMUI.ui.TextLabel', TextLabel);

    // Declarations created to instantiate in NodeJS environment
    if (typeof exports !== "undefined") {
        module.exports = TextLabel;
    }
}());
    (function() {
    /**
     * @class PMUI.ui.FlashMessage
     * A message to display for a while.
     * @extends {PMUI.core.Element}
     *
     * Usage example:
     *
     *      @example
     *      var message = new PMUI.ui.FlashMessage({
     *          message: "Hi!, this is a flash message!",
     *          duration: 5000,
     *          severity: 'info'
     *      });
     *
     *      message.show();
     *
     * @constructor
     * Creates a new instance of the class.
     * @param {Object} [settings] The config options.
     *
     * @cfg {String|Array} [message=""] The object's message. It can be a single string or an array of strings. In the 
     * llatter case the mesage will be showed in a list format.
     * @cfg {Number} [duration=3000] The time in milliseconds the message will be displayed.
     * @cfg {HTMLElement|PMUI.ui.Element} [appendTo=document.body] The html element/PMUI's object the flash message
     * will be displayed.
     * @cfg {String} [display="absolute"] The display mode for the object's html.
     * @cfg {String} [position="absolute"] The position mode for the object's html.
     * @cfg {String} [severity="info"] The severity for the message. Valid values: 'info', 'success', 'error'.
     */
    var FlashMessage = function(settings) {
        FlashMessage.superclass.call(this, settings);
        /**
         * The object's message. Set by the {@link #cfg-message message} config option and the 
         * {@link #method-setMessage setMessage()} method.
         * @type {String|Array}
         * @readonly
         */
        this.message = null;
        /**
         * The duration in milliseconds to show the message. Set by the {@link #cfg-duration duration} config option 
         * and the {@link #method-setDuration setDuration()} method.
         * @type {Number}
         * @readonly
         */
        this.duration = null;
        /**
         * The html element/PMUI's object the message will be displayed in. Set by the {@link #cfg-appendTo appendTo} 
         * config option and the {@link #method-setAppendTo setAppendTo()} method.
         * @type {HTMLElement|PMUI.core.Element}
         */
        this.appendTo = null;
        /**
         * The message's severity. Set by the {@link #cfg-severity severity} config option and the 
         * {@link #method-setSeverity setSeverity()} method.
         * @type {String}
         * @readonly
         */
        this.severity = null;
        FlashMessage.prototype.init.call(this, settings);
    };

    PMUI.inheritFrom('PMUI.core.Element', FlashMessage);
    /**
     * The object's type.
     * @type {String}
     */
    FlashMessage.prototype.type = 'FlashMessage';
    /**
     * The initializer method.
     * @param  {Object} [settings] The config options
     * @private
     */
    FlashMessage.prototype.init = function(settings) {
        var defaults = {
            message: "",
            duration: 3000,
            appendTo: document.body,
            display: 'inline-block',
            positionMode: 'absolute',
            severity: 'info'
        };

        jQuery.extend(true, defaults, settings);
        this.setMessage(defaults.message)
            .setDuration(defaults.duration)
            .setAppendTo(defaults.appendTo)
            .setPositionMode(defaults.positionMode)
            .setDisplay(defaults.display)
            .setSeverity(defaults.severity);
    };
    /**
     * Sets the severity for the message.
     * @param {String} severity Valid values: "info", "success" or "error".
     * @chainable
     */
    FlashMessage.prototype.setSeverity = function(severity) {
        if(!(severity === 'success' || severity === 'error' || severity === 'info')) {
            throw new Error('setSeverity(): the parameter must be "success" or "error" or "info"');
        }
        this.severity = severity;
        this.style.removeClasses(['pmui-info', 'pmui-error', 'pmui-success']).addClasses(['pmui-' + severity]);

        return this;
    };
    /**
     * Sets the html element/PMUI's object in which the message will be displayed.
     * @param {HTMLElement|PMUI.core.Element} appendTo
     * @chainable
     */
    FlashMessage.prototype.setAppendTo = function(appendTo) {
        if(!(PMUI.isHTMLElement(appendTo) || appendTo instanceof PMUI.core.Element)) {
            throw new Error("setAppendTo(): The parameter must be a HTML element or an instance of PMUI.ui.Element."); 
        }
        this.appendTo = appendTo;
        return this;
    };
    /**
     * Sets the duration for display the message.
     * @param {Number} duration The duration in milliseconds.
     * @chainable
     */
    FlashMessage.prototype.setDuration = function(duration) {
        if(typeof duration !== 'number') {
            throw new Error('setDuration(): The parameter must be a number.');
        }
        this.duration = duration;
        return this;
    };
    /**
     * Sets the message for the object.
     * @param {String|Array} message It can be a string or an array of strings, in the latter case the message will be 
     * showed in a list format.
     * @chainable
     */
    FlashMessage.prototype.setMessage = function(message) {
        var ul, li, i;
        if(typeof message !== 'string' && !jQuery.isArray(message)) {
            throw new Error('setMessage(): The parameter must be a message.');
        }
        this.message = (typeof message === 'string') ? jQuery.trim(message) : message;
        if(this.html) {
            jQuery(this.html).empty();
            if(typeof message === 'string') {
                this.html.textContent = message;                
            } else {
                ul = PMUI.createHTMLElement('ul');
                ul.className = 'pmui-flashmessage-list';
                for(i = 0; i < message.length; i++) {
                    li = PMUI.createHTMLElement('li');
                    li.textContent =  message[i];
                    ul.appendChild(li); 
                }
                this.html.appendChild(ul);
            }

        }
        return this;
    };
    /**
     * Shows the message in the html element/PMUI's object specified by the {@link #cfg-appendTo appendTo} config 
     * option or the {@link #method-setAppendTo setAppendTo()} method.
     * @chainable
     */
    FlashMessage.prototype.show = function() {
        var targetHTML = this.appendTo, html = this.html, top = 50, w, pw;
        if(!PMUI.isHTMLElement(targetHTML)) {
            targetHTML = targetHTML.html;
        }
        if(targetHTML) {
            if(!html) {
                html = this.getHTML();
            }
            jQuery(html).fadeTo(1, 0).get(0).style.top = top + "px";
            document.body.appendChild(html);
            w = jQuery(html).outerWidth();
            targetHTML.appendChild(html);
            this.style.addProperties({
                left: '50%',
                'margin-left': w/-2
            });
            jQuery(html).finish().css({
                'top': '50px'
            }).fadeTo(1, 0).animate({
                top: "-=" + top,
                opacity: 1
            }, 400, 'swing').delay(this.duration).animate({
                top: "+=" + top,
                opacity: 0
            });
        }
        return this;
    };
    /**
     * @inheritDoc
     */
    FlashMessage.prototype.createHTML = function() {
        FlashMessage.superclass.prototype.createHTML.call(this);
        this.setMessage(this.message);
        return this.html;
    };

    PMUI.extendNamespace('PMUI.ui.FlashMessage', FlashMessage);

    if (typeof exports!== 'undefined'){
        module.exports = FlashMessage;
    }
}());
(function () {
    var Menu = function(settings) {
        settings = jQuery.extend(true, settings, {
            elementTag: 'ul',
            positionMode: 'absolute'
        });
        Menu.superclass.call(this, settings);
        this.targetElement = null;
        this.parent = null;
        this.onShow = null;
        this.onHide = null;
        this.displayed = null;
        this.factory = new PMUI.menu.MenuItemFactory();
        this.items = new PMUI.util.ArrayList();
        this.onOutsideClickHandler = null;
        Menu.prototype.init.call(this, settings);
    };

    PMUI.inheritFrom('PMUI.core.Element', Menu);

    Menu.prototype.type = 'Menu';

    Menu.prototype.init = function(settings) {
        var defaults = {
            targetElement: null,
            parent: null,
            onShow: null,
            onHide: null,
            items: []
        };

        jQuery.extend(true, defaults, settings);

        this.onShow = defaults.onShow;
        this.onHide = defaults.onHide;
        this.setParent(defaults.parent)
            .setItems(defaults.items)
            .setTargetElement(defaults.targetElement)
            .setOutsideClickHandler();
    };

    Menu.prototype.setOutsideClickHandler = function() {
        var that = this;
        this.onOutsideClickHandler = function(e) {
            if(!jQuery(e.currentTarget).parents('#' + that.id).length) {
                that.hide();
            }
        };
        return this;
    };

    Menu.prototype.setTargetElement = function(target) {
        var rootMenu,
            targetElement = (this.targetElement && this.targetElement.html) || document.body;
        $(targetElement).removeClass('pmui-contextmenu-target');
        if(target instanceof PMUI.core.Element || null) {
            rootMenu = this.getRootMenu();
            if(rootMenu.targetElement) {
                    if(rootMenu.targetElement.menu === rootMenu) {
                    rootMenu.targetElement.menu = null;
                }    
            }
            rootMenu.targetElement = target;
        }
        return this;
    };

    Menu.prototype.getTargetElement = function() {
        var rootMenu = this.getRootMenu();
        return rootMenu.targetElement;
    };

    Menu.prototype.addItem = function(item) {
        var newItem = this.factory.make(item);
        if(newItem) {
            newItem.setParent(this);
            this.items.insert(newItem);
            if(this.html) {
                this.html.appendChild(newItem.getHTML());
            }
        }
        return this;
    };

    Menu.prototype.removeItem = function(item) {
        var itemToRemove;
        if(typeof item === 'string') {
            itemToRemove = this.items.find('id', item);
        } else if(typeof item === 'number') {
            itemToRemove = this.items.get(item);
        } else if (item instanceof PMUI.item.MenuItem && this.items.contains(item)) {
            itemToRemove = item;
        }
        if(itemToRemove) {
            this.items.remove(itemToRemove);
            jQuery(itemToRemove.html).detach();
        }
        return this;
    };

    Menu.prototype.clearItems = function() {
        var i = 0;
        while(this.items.getSize()) {
            this.removeItem(0);
        }
        return this;
    };

    Menu.prototype.setItems = function(items) {
        var i;
        if(!jQuery.isArray(items)) {
            throw new Error('setItems(): The parameter must be an array.');
        }
        this.clearItems();
        for(i = 0; i < items.length; i++) {
            this.addItem(items[i]);
        }
        return this;
    };

    Menu.prototype.getItems = function() {
        return this.items.asArray().slice(0);
    };

    Menu.prototype.setParent = function(parent) {
        if(!(parent === null || parent instanceof PMUI.menu.MenuItem)) {
            throw new Error('setParent(): The parameter must be an instance of PMUI.item.MenuItem or null.');
        }
        this.parent = parent;
        return this;
    };

    Menu.prototype.defineEventListeners = function() {
        var that = this;
        this.removeEvents();
        this.addEvent('contextmenu').listen(this.html, function(e) {
            e.stopPropagation();
            e.preventDefault();
        });
        this.addEvent('click').listen(this.html, function(e) {
            e.stopPropagation();
        });
        this.addEvent('mousedown').listen(this.html, function(e) {
            e.stopPropagation();
        });

        return this;
    };
 
    Menu.prototype.createHTML = function() {
        if(this.html) {
            return this.html;
        }

        Menu.superclass.prototype.createHTML.call(this); 

        this.setItems(this.items.asArray().slice(0));

        this.defineEventListeners();

        return this.html;
    };

    Menu.prototype.isRoot = function() {
        return !this.parent;
    };

    Menu.prototype.getRootMenu = function() {
        if(this.isRoot()) {
            return this;
        } else {
            return this.parent.parent.getRootMenu();
        }
    };

    Menu.prototype.show = function(x, y) {
        var rootMenu = this.getRootMenu(), 
            targetElement = (this.targetElement && this.targetElement.html) || document.body,
            zIndex = $(targetElement).zIndex();

        x = x || 0;
        y = y || 0;
        rootMenu.setPosition({
            x: x,
            y: y
        });
        PMUI.removeCurrentMenu();
        rootMenu.setZOrder(zIndex + 1);
        $(targetElement).addClass('pmui-contextmenu-target');
        document.body.appendChild(rootMenu.getHTML());
        this.addEvent('mousedown', 'clickOutside').listen(document, this.onOutsideClickHandler);
        this.displayed = true;
        PMUI.currentContextMenu = this;
        if(typeof this.onShow === 'function') {
            this.onShow(this);
        }
        return this;
    };

    Menu.prototype.hide = function() {
        var rootMenu = this.getRootMenu(),
            targetElement = (this.targetElement && this.targetElement.html) || document.body;
        this.removeEvent('clickOutside');
        $(targetElement).removeClass('pmui-contextmenu-target');
        jQuery(rootMenu.html).detach();
        rootMenu.displayed = false;
        PMUI.currentContextMenu = null;
        if(typeof rootMenu.onHide === 'function') {
            rootMenu.onHide(rootMenu);
        }
        return this;
    };

    Menu.prototype.setContextMenu = function() {
        return this;
    };

    PMUI.extendNamespace('PMUI.menu.Menu', Menu);

    // Declarations created to instantiate in NodeJS environment
    if (typeof exports !== "undefined") {
        module.exports = Menu;
    }
}());
(function() {
    var MenuItem = function(settings) {
        MenuItem.superclass.call(this, settings);
        this.dom = {};
        this.parent = null;
        MenuItem.prototype.init.call(this, settings);
    };

    PMUI.inheritFrom('PMUI.core.Element', MenuItem);

    MenuItem.prototype.type = 'MenuItem';

    MenuItem.prototype.init = function(settings) {
        var defaults = {
            parent: null
        }

        jQuery.extend(true, defaults, settings);

        this.setParent(defaults.parent);
    };

    MenuItem.prototype.setParent = function(parent) {
        if(!(parent === null || parent instanceof PMUI.menu.Menu)) {
            throw new Error('setParent(): The parameter must be an instance of PMUI.ui.Menu or null.');
        }
        this.parent = parent;
        return this;
    };

    MenuItem.prototype.getParent = function() {
        return this.parent;
    };

    MenuItem.prototype.getRootMenu = function() {
        var parent = this.parent;
        if(this.parent) {
            return this.parent.getRootMenu();
        }
        return parent;
    };

    MenuItem.prototype.isLeaf = function() {
        throw new Error("isLeaf() is being called from an abstract class.");
    };

    MenuItem.prototype.getMenu = function() {
        return this.parent;
    };

    MenuItem.prototype.setContextMenu = function() {
        return this;
    };

    MenuItem.prototype.getMenuTargetElement = function() {
        var rootMenu = this.getRootMenu();
        if(rootMenu) {
            return rootMenu.getTargetElement();
        }
        return rootMenu
    };

    MenuItem.prototype.createHTML = function() {
        if(this.html) {
            return this.html;
        }
        MenuItem.superclass.prototype.createHTML.call(this);
        return this.html;
    };

    PMUI.extendNamespace('PMUI.menu.MenuItem', MenuItem);

}());
(function() {
    var MenuOption = function(settings) {
        MenuOption.superclass.call(this, settings);
        this.onClick = null;
        this.text = null;
        this.hideOnClick = null;
        this.disabled = null;
        MenuOption.prototype.init.call(this, settings);
    };

    PMUI.inheritFrom('PMUI.menu.MenuItem', MenuOption);

    MenuOption.prototype.type = 'MenuOption';

    MenuOption.prototype.init = function(settings) {
        var defaults = {
            onClick: null,
            elementTag: 'li',
            text: '[option]',
            hideOnClick: true,
            disabled: false
        };
        jQuery.extend(true, defaults, settings);
        this.setElementTag(defaults.elementTag)
            .setText(defaults.text)
            .setOnClickHandler(defaults.onClick)
            .hideOnClick = !!defaults.hideOnClick;  

        if(defaults.disabled) {
            this.disable();
        } else{
            this.enable();
        }
    };

    MenuOption.prototype.enable = function() {
        this.disabled = false;
        this.style.removeClasses(['pmui-disabled']);
        return this;
    };

    MenuOption.prototype.disable = function() {
        this.disabled = true;
        this.style.addClasses(['pmui-disabled']);
        return this;
    };

    MenuOption.prototype.setText = function(text) {
        if(typeof text !== 'string') {
            throw new Error('setText(): the parameter must be a srting.');
        }
        this.text = text;
        if(this.dom.textContainer) {
            this.dom.textContainer.textContent = text;
        }
        return this;
    };

    MenuOption.prototype.setOnClickHandler = function(onClick) {
        if(!(onClick === null || typeof onClick === 'function')) {
            throw new Error('setOnClickHandler(): The parameter must be a function or null.');
        }
        this.onClick = onClick;
        return this;
    };

    MenuOption.prototype.onClickHandler = function() {
        var that = this;
        return function(e) {
            e.preventDefault();
            e.stopPropagation();
            if(!that.disabled) {
                if(typeof that.onClick === 'function') {
                    that.onClick(that);
                }
                if(that.hideOnClick) {
                    that.parent.hide();
                }
            }
        };
    };

    MenuOption.prototype.remove = function() {
        this.parent.removeItem(this);
        return this;
    };

    MenuOption.prototype.defineEventListeners = function() {
        this.removeEvents();
        this.addEvent('click').listen(this.dom.title, this.onClickHandler());
        return this;
    };

    MenuOption.prototype.createHTML = function() {
        var link, textContainer, iconContainer;
        if(this.html) {
            return this.html;
        }
        MenuOption.superclass.prototype.createHTML.call(this);

        link = PMUI.createHTMLElement('a');
        link.href = '#';
        link.className = 'pmui-menuoption-title';

        textContainer = PMUI.createHTMLElement('span');
        textContainer.className = 'pmui-menuoption-text';

        iconContainer = PMUI.createHTMLElement('i');
        iconContainer.className = 'pmui-menuoption-text-icon';

        this.dom.title = link;
        this.dom.textContainer = textContainer;
        this.dom.iconContainer = iconContainer;

        link.appendChild(iconContainer);
        link.appendChild(textContainer);
        this.html.appendChild(link);

        this.setText(this.text);

        this.defineEventListeners();

        return this.html;
    };

    PMUI.extendNamespace('PMUI.menu.MenuOption', MenuOption);
}());
(function() {
    var MenuRegularOption = function(settings) {
        MenuRegularOption.superclass.call(this, settings);
        this.childMenu = null;
        MenuRegularOption.prototype.init.call(this, settings);
    };

    PMUI.inheritFrom('PMUI.menu.MenuOption', MenuRegularOption);

    MenuRegularOption.prototype.init = function(settings) {
        var defaults = {
            items: []
        };

        jQuery.extend(true, defaults, settings);

        this.childMenu = new PMUI.menu.Menu({
            positionMode: 'absolute',
            parent: this
        });

        this.setItems(defaults.items);
    };

    MenuRegularOption.prototype.setItems = function(items) {
        this.childMenu.setItems(items);
        if(this.childMenu.getItems().length) {
            this.style.addClasses(['pmui-father']);
        }
        return this;
    };

    MenuRegularOption.prototype.getItems = function() {
        return this.childMenu.getItems();
    };

    MenuRegularOption.prototype.onClickHandler = function() {
        var that = this;
        return function(e) {
            e.preventDefault();
            e.stopPropagation();
            if(!that.disabled) {
                if(typeof that.onClick === 'function') {
                    that.onClick(that);
                }
                if(that.hideOnClick && that.childMenu.getItems().length === 0) {
                    that.parent.hide();
                }   
            }
        };
    };

    MenuRegularOption.prototype.createHTML = function() {
        if(this.html) {
            return this.html;
        }
        MenuRegularOption.superclass.prototype.createHTML.call(this);
        this.html.appendChild(this.childMenu.getHTML());
        this.setItems(this.getItems());
        return this.html;
    };

    PMUI.extendNamespace('PMUI.menu.MenuRegularOption', MenuRegularOption);
}());
(function (){
    var MenuSeparatorItem = function(settings) {
        MenuSeparatorItem.superclass.call(this, jQuery.extend(true, settings, {
            elementTag: 'div'
        }));
        MenuSeparatorItem.prototype.init.call(this, settings);
    };

    PMUI.inheritFrom('PMUI.menu.MenuItem', MenuSeparatorItem);

    MenuSeparatorItem.prototype.type = 'MenuSeparatorItem';

    MenuSeparatorItem.prototype.isLeaf = function() {
        return true;
    };

    PMUI.extendNamespace('PMUI.menu.MenuSeparatorItem', MenuSeparatorItem);
}());
(function() {
    var MenuItemFactory = function(settings) {
        MenuItemFactory.superclass.call(this, settings);
        MenuItemFactory.prototype.init.call(this, settings);
    };

    PMUI.inheritFrom('PMUI.util.Factory', MenuItemFactory);

    MenuItemFactory.prototype.type = "MenuItemFactory";

    MenuItemFactory.prototype.init = function(settings) {
        var defaults = {
            products: {
                "menuRegularOption": PMUI.menu.MenuRegularOption,
                "menuSeparatorItem": PMUI.menu.MenuSeparatorItem
            },
            defaultProduct: "menuRegularOption"
        };

        this.setDefaultProduct(defaults.defaultProduct)
            .setProducts(defaults.products);
    };

    PMUI.extendNamespace('PMUI.menu.MenuItemFactory', MenuItemFactory);
}());
(function(){
    /**
     * @class PMUI.form.Field
     * Abstract class that encapsulates the field behavior
     * @extends PMUI.core.Element
     * @abstract
     *
     * @constructor
     * While it is true that this class must not be instantiated, 
     * it is useful to mention the settings parameter for the constructor function 
     * (which will be used for the non abstract subclasses).
     * @param {Object} [settings] A JSON object, it can be contain the following fields:
     * 
     * - {@link PMUI.form.Field#cfg-name name}.
     * - {@link PMUI.form.Field#cfg-label label}.
     * - {@link PMUI.form.Field#cfg-value value}.
     * - {@link PMUI.form.Field#cfg-helper helper}.
     * - {@link PMUI.form.Field#cfg-showHelper showHelper}.
     * - {@link PMUI.form.Field#cfg-validators validators}.
     * - {@link PMUI.form.Field#cfg-valueType valueType}.
     * - {@link PMUI.form.Field#cfg-controlPositioning controlPositioning}.
     * - {@link PMUI.form.Field#cfg-labelWidth labelWidth}.
     * - {@link PMUI.form.Field#cfg-showColon showColon}
     *
     * @cfg {String} name The name for the field.
     * @cfg {String} label The text to be shown as the field's label
     * @cfg {String} value The initial value to be set to the control.
     * @cfg {String} helper The helper text to be shown in the helper tooltip
     * @cfg {Boolean} showHelper A boolean value which determines if the helper tootltip will be shown or not.
     * @cfg {Object} validators An array where each array's item is a JSON object (with the validator setting data)
     *  or a {@link PMUI.form.Validator Validator} object. 
     *      {
     *          validators: [
     *              {
     *                  type: "maxlength",
     *                  criteria: 5
     *              },
     *              new LengthValidator({
     *                  min: 0,
     *                  max: 5
     *              })
     *          ]
     *      }
     * In example above, "validators" is an array in which their first element is an JSON object 
     * and the second one is a {@link PMUI.form.Validator Validator} object.
     * @cfg {String} valueType A string which specifies the data type for the Field value.
     * @cfg {String} controlPositioning A formatted string that specifies the order output for the control(s). 
     * A string  which specifies the output order for the field's controls. 
     * Basically this string uses a wildcard with the format "[cx]", each one is replaced by the control:
     *      "[c0]-[c1]-[c2]"
     * If you apply the string above to the controlPositioning property, it will render
     * the first three field's controls, each one separated from the other by a "-".
     *
     * Another wildcard is [c*], this represents all the controls or the ones that haven't been included yet:
     *      "[c*]-"
     * The example above will render all the controls and at the end it will add a "-".
     *
     *      "[c2]-[c*]"
     * The example above will render first the second control and then the other ones (starting from the first one).
     * @cfg {String} labelWidth The width for the label.
     * The width label should be a String with the following format "X%"
     * 
     * Note that this% should not exceed 80%, because 20% is the tooltip
     * @cfg {Boolean} showColon If a colon is shown after the label text.
     * @cfg {Boolean} [labelVisible=true] If the label is visible or not. 
     * @cfg {String} [labelPosition="left"] Sets the position for the label, the value can take one of the following
     * options:
     *
     * - "left" (default).
     * - "right".
     * - "top".
     * - "bottom"
     *
     * @cfg {String|Number|Array} [controlsWidth] Determines the width for the field's controls, but the width of the 
     * field must be a number or a string with the format "#px", otherwise the width for the controls will be set to 
     * "auto".It can be:
     *
     * - A String, in this case the string must have the following format "##px", otherwise must be "auto".
     * - A Number, in this case the value received is parsed into a px units value.
     * - An Array in which each element can be a String or a Number (following the rules above). 
     * 
     * This will be applied respectively to each control in the field.
     * @cfg {Function} [onClick=null] Description here.
     * @cfg {Function|null} [onBeforeChange] The callback function to be executed before the field's value changes. To
     * avoid the change the callback function must return false. For info about the parameters sent to the calback
     * function please read the {@link #event-onBeforeChange onBeforeChange event} documentation.
     */
    var Field = function(settings) {
        Field.superclass.call(this, settings);
        /**
         * @property {String} [name=[The object's id]] The field's name.
         * @readonly
         */
        this.name = null;
        /**
         * @property {String} [label = "[field]"] The field's label text.
         * @readonly
         */
        this.label = null;
        /**
         * @property {String} [value=""] The field's value.
         * @readonly
         */
        this.value = null;
        /**
         * @property {PMUI.ui.TooltipMessage} [helper] The field's help tooltip
         * @readonly
         */
        this.helper = null;
        /**
         * @property {PMUI.ui.TooltipMessage} [message] A {@link PMUI.ui.TooltipMessage TooltipMessage} 
         * object to show a message related to the field (i.e. validation error messages)
         * @private
         */
        this.message = null;
        /**
         * @property {Array} controls An array, it will contain all the necessary Control objects
         * @private
         */
        this.controls = [];
        /**
         * @property {Object} validators An JSON object which will contain all the 
         * {@link PMUI.form.Validator Validators} object.
         * @private
         */
        this.validators = {};
        /**
         * @property {String} [controlPositioning="[c*]"] A formatted string that specifies 
         * the order output for the field's controls.
         * @readonly
         */
        this.controlPositioning = null;
        /**
         * @property {Object} [dom] A JSON object which will contain important DOM object 
         * for the Field object.
         * @private
         */
        this.dom = null;
        /**
         * @property {Boolean} [helperIsVisible=false] A Boolean that let us 
         * know if the help tooltip will be shown or not.
         * @readonly
         */
        this.helperIsVisible = null;
        /**
         * @property {String} [labelWidth="30%"]
         * The width for the label. This property should be a String with the following format "X%"
         * @readonly
         */
        this.labelWidth = null;
        /**
         * @property {Boolean} [visibleColon=true]
         * If a colon is shown after the label text.
         * @readonly
         */
        this.visibleColon = null;
        /**
         * @property {Boolean} labelVisible
         * If the label is visible or not.
         * @readonly
         */
        this.labelVisible = null;
        /**
         * @property {String} [labelPosition="String"] The position for the field's label.
         * @readonly
         */
        this.labelPosition = null;
        /**
         * @event onBeforeChange
         * Fired before the field's value changes.
         * @param {String} newValue The new value to be set.
         * @param {String} oldValue The old field's value.
         */
        this.onBeforeChange = null;
        /**
         * @event onChange
         * Fired when the field's value changes.
         * @param {String} newValue The field's new value.
         * @param {String} oldValue The previous field's value.
         */
        this.onChange = null;
        /**
         * @property {String} valueType The value data type for the field.
         */
        this.valueType = null;
        /**
         * @property {PMUI.form.ValidatorFactory} validatorFactory The factory object for validator production.
         */
        this.validatorFactory = null;
        /**
         * @property {Boolean} [required] If the field is required or not
         */
        this.required = null;
        /**
         * @property {String} [requiredMessage="This field is required."]
         * The message to display when the validation for required property fails.
         */
        this.requiredMessage = null;
        /**
         * @property {Boolean} [validAtChange=true] 
         * If the validation must be executed every time the field's value changes.
         */
        this.validAtChange = null;
        /**
         * The width for the control(s)
         * @type {Number|String|Array}
         * @readonly
         */
        this.controlsWidth = [];
        /**
         * @property {PMUI.form.Form} form
         * The form the field belongs to.
         */
        this.form = null;
        /**
         * @property {Boolean} eventsDefined
         * If the events for the object have been defined.
         */
        this.eventsDefined = null;
        /**
         * @property {PMUI.data.DataSet} data
         * The field's data object.
         * @private
         */
        this.data = null;
        /**
         * The initial value the field will be set when its reset() method is called.
         * @type {String}
         * @private
         */
        this.initialValue = null;
        /**
         * [onClick description]
         * @type {[type]}
         */
        this.onClick = null;
        /**
         * If the field is disabled or not. Notice that if it is disabled the validation won't be executed.
         * @type {Boolean}
         */
        this.disabled = null;

        /**
         * @protected
         * Valid Types
         * @type {Array}
         */
        this.dependentFields = [];   
        this.dependencyHandler = null;
        
        this.validTypes = {'string':true,'number':true,'boolean':true,'date':true,'object':true};
        Field.prototype.init.call(this, settings);
    };

    PMUI.inheritFrom('PMUI.core.Element', Field);

    Field.prototype.type = "Field";

    Field.prototype.init = function(settings) {
        var defaults = {
            name: this.id,
            label: '[field]',
            value: '',
            helper: '',
            showHelper: !!(settings && settings.helper),
            validators: [],
            valueType: 'string',
            controlPositioning: '[c*]',
            labelWidth: '23.5%',
            width: '100%',
            showColon: true,
            validatorFactory: new PMUI.form.ValidatorFactory(),
            onBeforeChange: null,
            onChange: null,
            required: false,
            validAtChange: true,
            requiredMessage: 'This field is required.',
            labelVisible: true,
            labelPosition: 'left',
            form: null,
            controlsWidth: "auto",
            disabled: false,
            dependentFields : [],
            dependencyHandler : function(){},
            onClick: null
        };

        //this.dependentFields = new PMUI.util.ArrayList();
        this.dependentFields = [];
        this.dependencyFields = new PMUI.util.ArrayList();
        $.extend(true, defaults, settings);

        this.initialValue = defaults.value;

        this.helper = new PMUI.ui.TooltipMessage({
            category: 'help'
        });

        this.dom = {};
        this.data = new PMUI.data.DataField();

        this.message = new PMUI.ui.TextLabel({
            text:'This field is required.',
            displayMode: 'block',
            mode: 'normal',
            visible: false
        });

        this.setForm(defaults.form)
            .setValidAtChange(defaults.validAtChange)
            .setRequired(defaults.required)
            .setRequiredMessage(defaults.requiredMessage)
            .setValidatorFactory(defaults.validatorFactory)
            .setValidators(defaults.validators)
            .setName(defaults.name)
            .setLabel(defaults.label)
            .setValue(defaults.value)
            .setHelper(defaults.helper)
            .setValueType(defaults.valueType)
            .setControlPositioning(defaults.controlPositioning)
            .setOnBeforeChangeHandler(defaults.onBeforeChange)
            .setOnChangeHandler(defaults.onChange)
            .setWidth(defaults.width)
            .setLabelWidth(defaults.labelWidth)
            .setLabelPosition(defaults.labelPosition)
            .setControlsWidth(defaults.controlsWidth)
            .setOnClickHandler(defaults.onClick)
            .setControls();
        this.setDependencyHandler(defaults.dependencyHandler);
        this.setDependentFields(defaults.dependentFields);
            //
        if(defaults.showHelper) {
            this.showHelper();  
        } else {
            this.hideHelper();
        }
        if(defaults.showColon) {
            this.showColon();
        } else {
            this.hideColon();
        }
        if(defaults.labelVisible) {
            this.showLabel();
        } else {
            this.hideLabel();
        }
        if(defaults.disabled) {
            this.disable();
        } else {
            this.enable();
        }
    };
    Field.prototype.onBeforeChangeHandler = function(handler) {
        var that = this;
        return function(newValue, oldValue) {
            var callbackResponse, futureValue = "", i;
            if(typeof that.onBeforeChange === 'function') {
                for(i = 0; i < that.controls.length; i += 1) {
                    if(that.controls[i] === this) {
                        futureValue += ' ' + newValue;
                    } else {
                        futureValue += ' ' + that.controls[i].getValue();
                    }
                }
                futureValue = futureValue.substr(1);
                callbackResponse = that.onBeforeChange(futureValue, that.getValue());
            }
            return callbackResponse;
        };
    };
    Field.prototype.setOnBeforeChangeHandler = function(handler) {
        if(!(handler === null || typeof handler === 'function')) {
            throw new Error("setOnBeforeChangeHandler(): The parameter must be a function or null.");
        }
        this.onBeforeChange = handler;
        return this;
    };
    Field.prototype.setOnClickHandler = function(handler) {
        if(!(handler === null || typeof handler === 'function')) {
            throw new Error("setOnClickHandler(): The parameter must be a function or null.");
        }
        this.onClick = handler;
        return this;
    };
    Field.prototype.setDependentFields = function(dependentFields) {
        if(!jQuery.isArray(dependentFields)) {
            throw new Error("setDependentFields(): The parameter must be an array.");
        }
        this.dependentFields = dependentFields;
        if(this.form) {
            this.form.updateDependencies();
        }
        return this;
    };

    Field.prototype.removeDependentField = function(FieldName) {
        var i;
        for(i = 0; this.dependentFields.length; i++) {
            if(this.dependentFields[i] === FieldName) {
                this.dependentFields.splice(i, 1);
                i--;
            }
        }
        return this;
    };

    Field.prototype.setDependencyHandler = function(handler){
        if (typeof handler == 'function' || handler == null) {
            this.dependencyHandler = handler;
        }
        return this;
    };

    /**
     * Disables the field. Notice that when a field is disabled it is not validated and it is not returned when its 
     * form's getData() method is invoked.
     * @chainable
     */
    Field.prototype.disable = function() {
        var i;
        this.disabled = true;
        this.style.addClasses(["pmui-disabled"]);
        for(i = 0; i < this.controls.length; i++) {
            this.controls[i].disable(true);
        }

        return this;
    };
    /**
     * Enables the field.
     * @chainable
     */
    Field.prototype.enable = function() {
        var i;
        this.disabled = false;
        this.style.removeClasses(["pmui-disabled"]);
        for(i = 0; i < this.controls.length; i++) {
            this.controls[i].disable(false);
        }

        return this;
    };
    /**
     * Resets the field to its original value.
     * @chainable
     */
    Field.prototype.reset = function() {
        this.setValue(this.initialValue);
        this.hideMessage();
        if(this.eventsDefined) {
            this.onChangeHandler();
        }
        return this;
    };
    /**
     * Determines the width for the field's controls, but the width of the field must be a number or a string with 
     * the format "#px", otherwise the width for the controls will be set to "auto".
     * @param {Number|String|Array} width  It can be:
     *
     * - A String, in this case the string must have the following format "##px", otherwise must be "auto".
     * - A Number, in this case the value received is parsed into a px units value.
     * - An Array in which each element can be a String or a Number (following the rules above). 
     * 
     * This will be applied respectively to each control in the field.
     * @chainable
     */
    Field.prototype.setControlsWidth = function(width) {
        var i, labelWidthPx, auxWidth, autoGerateWidthForControls = false, fieldWidthIsAbsolute = false;

        if(!(width === 'auto' || typeof width === 'number' || jQuery.isArray(width) || /^\d+(\.\d+)?px/.test(width))) {
            throw new Error("setControlsWidth(): The parameter must be a string \"auto\", or a number or an array.");
        }

        this.controlsWidth = width;

        if(width === 'auto') {
            autoGerateWidthForControls = true;
        }

        if(typeof this.width === 'number' || /^\d+(\.\d+)?px/.test(this.width)) {
            fieldWidthIsAbsolute = true;
        }

        if(!autoGerateWidthForControls) {
            for(i = 0; i < this.controls.length; i++) {
                if(jQuery.isArray(width)) {
                    this.controls[i].setWidth(width[i] || "auto");
                } else {
                    this.controls[i].setWidth(width);
                }
            }            
        } else if(fieldWidthIsAbsolute) {
            auxWidth = parseInt(this.width, 10);
            labelWidthPx = auxWidth * (parseInt(this.labelWidth,10)/100) + 4;
            labelWidthPx = auxWidth - 83 - labelWidthPx;
            for(i = 0; i < this.controls.length; i++) {
                this.controls[i].setWidth(labelWidthPx);
            }
        } else {
            for(i = 0; i < this.controls.length; i++) {
                this.controls[i].setWidth("auto");
            }
        }     
        
        return this;
    };
    /**
     * Sets the form the field belongs to.
     * @param {PMUI.form.Form} form
     * @chainable
     */
    Field.prototype.setForm = function(form) {
        if(form instanceof PMUI.form.Form) {
            this.form = form;
        }

        return this;
    };
    /**
     * Turns on/off the validation when the field's value changes.
     * @param {Boolean} validAtChange
     * @chainable
     */
    Field.prototype.setValidAtChange = function(validAtChange) {
        this.validAtChange = !!validAtChange;
        return this;
    };
    /**
     * Sets the position for the label.
     * @param {String} position It can take one of the following values: "top", "right", "bottom", "left".
     * @chainable
     */
    Field.prototype.setLabelPosition = function(position) {
        if(position === 'top' || position === 'right' || position === 'bottom' || position === 'left' ) {
            this.labelPosition = position;
            if(this.html) {
                if(position === 'top' || position === 'left') {
                    this.html.insertBefore(this.dom.labelTextContainer ,this.dom.controlContainer);
                    this.dom.labelTextContainer.style.display = position === 'top' ? 'block' : 'inline-block';
                    $(this.dom.controlContainer).css("float",'');  
                    $(this.dom.labelTextContainer).css("float", position == "left" ? 'left' : '');
    
                    
                } else {
                    this.html.insertBefore(this.dom.controlContainer, this.dom.labelTextContainer);
                    this.dom.labelTextContainer.style.display = position === 'bottom' ? 'block' : 'inline-block';
                    $(this.dom.labelTextContainer).css("float",'');   
                    $(this.dom.controlContainer).css("float", position == "rigth" ? 'rigth' : '');

                }
            }
        } else {
            throw new Error('setLabelPosition(): it only accepts "top", "rigth", "left" or "bottom" as value for ' +
                'the parameter');
        }

        return this;
    };
    /**
     * Sets the factory which must be produce {@link PMUI.form.Validator Validator} objects.
     * @param {PMUI.util.Factory|Object} factory
     * It can be a:
     *
     * - a {@link PMUI.util.Factory} object
     * - a JSON object: in this case a new {@link PMUI.form.ValidatorFactory ValidatorFactory} will be created
     * using the JSON object as the constructor parameter.
     * @private
     * @chainable
     */
    Field.prototype.setValidatorFactory = function(factory) {
        if(factory instanceof PMUI.util.Factory) {
            this.validatorFactory = factory;
        } else {
            this.validatorFactory  = new PMUI.form.ValidatorFactory(factory);
        }

        return this;
    };
    /**
     * Sets the message to show when the required field validation fails.
     * @param {String} requiredMessage
     * @chainable
     */
    Field.prototype.setRequiredMessage = function(requiredMessage) {
        if(typeof requiredMessage === 'string') {
            this.requiredMessage = requiredMessage;
        } else {
            throw new Error("the setRequiredMessage() method only accepts string values.");
        }

        return this;
    };
    /**
     * Sets if the fields is required or not.
     * @param {Boolean} required
     * @chainable
     */
    Field.prototype.setRequired = function(required) {
        this.required = !!required;
        if(this.dom.fieldRequired) {
            if(this.required){
                this.showRequired();
            }else{
                this.hideRequired();
            }            
        }
        return this;
    };

    Field.prototype.hideRequired = function (){
        this.dom.fieldRequired.style.display = 'none';
        return this;
    };

    Field.prototype.showRequired = function (){
        this.dom.fieldRequired.style.display = 'inline-block';
        return this;
    };
    /**
     * Removes all the validators from the field.
     * @chainable
     */
    Field.prototype.clearValidators = function() {
        var key;
        for(key in this.validators) {
            if(this.validators.hasOwnProperty(key)) {
                this.validators[key] = null;
                delete this.validators[key];   
            }
        }

        return this;
    };
    /**
     * Adds a validator to the field.
     * @param {String|PMUI.form.Validator|Object} validator
     * It can be:
     *
     * - a String: it must be on of the supported pmTypes by the 
     * {@link PMUI.form.ValidatorFactory ValidatorFactory} class.
     * - a {@link PMUI.form.Validator Validator} object.
     * - a JSON object: in this case the JSON structure must be the same as the needed one to create 
     * the desired validator, additionally must have the respective pmType.
     *
     * Note. All the supported pmTypes are specified in the 
     * {@link PMUI.form.ValidatorFactory ValidatorFactory documentation}.
     * @chainable
     */
    Field.prototype.addValidator = function(validator) {
        var newValidator;
        if(this.validatorFactory) {
            if(this.validatorFactory.isValidClass(validator) || this.validatorFactory.isValidName(validator.pmType)) {
                newValidator = this.validatorFactory.make(validator);
            } else {
                throw new Error('Invalid validator to add.');
            }
        }

        if(newValidator && newValidator instanceof PMUI.form.Validator) {
            newValidator.setParent(this);
            this.validators[newValidator.type] = newValidator;
        }

        return this;
    };
    /**
     * Sets the validators for the field.
     * @param {Array} validators An array in which each element can be one of the accepted types in the 
     * {@link PMUI.form.Field#addValidator addValidator() method}.
     * @chainable
     */
    Field.prototype.setValidators = function(validators) {
        var i = 0;
        if(jQuery.isArray(validators)) {
            this.clearValidators();
            for(i = 0; i < validators.length; i++) {
                this.addValidator(validators[i]);
            }
        }

        return this;
    };
    /**
     * Shows a colon after the label text.
     * @chainable
     */
    Field.prototype.showColon = function() {
        this.visibleColon = true;
        //return this.setLabel(this.label);
        if(this.dom.fieldColon) {
            this.dom.fieldColon.style.display = ''; 
        }
    };
    /**
     * Hides the colon after the label text.
     * @chainable
     */
    Field.prototype.hideColon = function() {
        this.visibleColon = false;
        if(this.dom.fieldColon){
            this.dom.fieldColon.style.display = "none";
        }
        return this;
    } 
    /**
     * Sets the width for the label.
     * @param {String} width It can be a Number or a String. 
     * This parameter should be a String with the following format "X%"; 
     * @chainable
     */
    Field.prototype.setLabelWidth = function(width) {
        if (/^\d+(\.\d+)?(px|em|%)$/.test(width)) {
            this.labelWidth = width;
            this.setControlsWidth(this.controlsWidth);
        } else {
            throw new Error('setLabelWidth(): invalid "width" parameter');
        }
        if(this.dom.labelTextContainer) {
            this.dom.labelTextContainer.style.width = this.labelWidth;
            if(this.labelVisible && this.labelPosition != "top")
                $(this.dom.messageContainer).css({"margin-left": parseInt(this.labelWidth)+ 1 + "%" });
        }
        return this;
    };
    /**
     * Sets the callback function to be called when the field's value changes.
     * @param {Function} handler
     * @chainable
     */
    Field.prototype.setOnChangeHandler = function(handler) {
        if(typeof handler === 'function') {
            this.onChange = handler;
        }

        return this;
    };
    /**
     * Sets the helper tooltip visible.
     * @chainable
     */
    Field.prototype.showHelper = function() {
        this.helperIsVisible = true;
        this.helper.setVisible(true);
        return this;
    };
    /**
     * Sets the helper tooltip non visible.
     * @chainable
     */
    Field.prototype.hideHelper = function() {
        this.helperIsVisible = false;
        this.helper.setVisible(false);
        return this;
    };
    /**
     * Returns an array with all the field's controls.
     * @return {Array}
     */
    Field.prototype.getControls = function() {
        return this.controls;
    };  
    /**
     * Returns an index based field's control.
     * @param  {Number} index An integer value.
     * @return {PMUI.control.Control}
     */
    Field.prototype.getControl = function(index) {
        index = index || 0;
        return this.controls[index];
    };
    /**
     * Sets the controlPositioning property.
     * @param {String} positioning The string must have the same format that the 
     * {@link PMUI.form.Field#cfg-controlPositioning controlPositioning} config option.
     */
    Field.prototype.setControlPositioning = function(positioning) {
        var pos, controlPos, i, j, k, controls, span, addControl, that = this;
        if(typeof positioning === 'string') {
            this.controlPositioning = positioning;
            if(this.html && this.controls.length) {
                for(i = 0; i < this.controls.length; i++) {
                    jQuery(this.controls[i].getHTML()).detach();
                }
                $(this.dom.controlContainer).empty();
                if(positioning !== "") {
                    controls = this.controls.slice();
                    pos = positioning.split(/\[c[\d|\*]\]/);
                    controlPos = positioning.match(/\[c[\d|\*]\]/g);
                    addControl = function(c) {
                        var k;
                        if(c === '[c*]') {
                            for(k = 0; k < controls.length; k++) {
                                if(controls[k] !== null) {
                                    that.dom.controlContainer.appendChild(controls[k].getHTML());
                                    controls[k] = null; 
                                }
                            }
                        } else {
                            k = c.match(/\d+/);
                            k = parseInt(k[0], 10);
                            if(controls[k] !== null) {
                                that.dom.controlContainer.appendChild(controls[k].getHTML());
                                controls[k] = null; 
                            }
                        }   
                    };

                    j = 0;
                    for(i = 0; i < pos.length; i++) {
                        if(pos[i] === "" && j < controlPos.length) {
                            addControl(controlPos[j]);
                            j++;
                        } else {
                            span = PMUI.createHTMLElement('span');
                            span.textContent = pos[i];
                            this.dom.controlContainer.appendChild(span);
                        }
                        if(j < controlPos.length) {
                            addControl(controlPos[j]);
                                j++;
                        }
                    }
                    this.dom.labelTextContainer.setAttribute('for',this.controls[0].id);
                    this.dom.controlContainer.appendChild(this.helper.getHTML());
                }
            }
        } else {
            throw new Error("The setControlPositioning() method only accepts string values.");
        }

        return this;
    };
    /**
     * Sets the controls for the field.
     *
     * Since this is an abstract method, it must be implemented in its non-abstract subclasses
     * @abstract
     * @private
     */
    Field.prototype.setControls = function() {
    };
    /**
     * Sets the name for the Field
     * @param {String} name
     */
    Field.prototype.setName = function(name) {
        if(typeof name === 'string') {
            this.name = name;
            this.data.setKey(name);
        } else {
            throw new Error('The setName() method only accepts string values!');
        }
        return this;
    };
    /**
     * Returns the field's name
     * @return {String}
     */
    Field.prototype.getName = function() {
        return this.name;
    };
    /**
     * Sets the text for the field's label.
     * @param {String} label
     */
    Field.prototype.setLabel = function(label) {
        if(typeof label === 'string') {
            this.label = label;
        } else {
            throw new Error("The setLabel() method only accepts string values!");
        }
        if(this.dom.fieldTextLabel) {
            this.dom.fieldTextLabel.textContent = this.label;
        }
        if(this.dom.fieldRequired){
            this.dom.fieldRequired.textContent = '*';
            this.setRequired(this.required); 
        }   
        if(this.dom.fieldColon){
            this.dom.fieldColon.textContent = ":";
        }
        return  this;
    };
    /**
     * Returns the text from the field's label.
     * @return {String}
     */
    Field.prototype.getLabel = function() {
        return this.label;
    };
    /**
     * Sets the value to the field's controls.
     * @protected
     * @param {String} value
     * @chainable
     */
    Field.prototype.setValueToControls = function(value) {
        var i;
        for(i = 0; i < this.controls.length; i += 1) {
            this.controls[i].setValue(value);
        }
        return this;
    };
    /**
     * Returns the current value from the field's controls without affect the field's value.
     * @return {String}
     * @private
     */
    Field.prototype.getValueFromControls = function() {
        var value = '', i;

        for(i = 0; i < this.controls.length; i++) {
            value += ' ' + this.controls[i].getValue();
        }

        return value.substr(1);
    };
    /**
     * Update the field's value property from the controls
     * @protected
     * @chainable
     */
    Field.prototype.updateValueFromControls = function() {
        var value = this.getValueFromControls();

        this.value = value;
        this.data.setValue(value);

        return this;
    };
    /**
     * Sets the field's value.
     * @param {String} value
     */
    Field.prototype.setValue = function(value) {
        if(typeof value === 'number') {
            value = value.toString();
        }
        if(typeof value === 'string') {
            this.value = value;
        } else {
            throw new Error("The setValue() method only accepts string values!");
        }
        this.data.setValue(this.value);
        this.setValueToControls(this.value);
        return this;
    };
    /**
     * Returns the field's value.
     * @param [format] Defines the return format
     * @returns {*}
     */
    Field.prototype.getValue = function(format) {
        var castFormat = format || this.valueType;
        return PMUI.castValue(this.value, castFormat);
    };
    /**
     * Sets the helper text.
     * @param {String} helper
     */
    Field.prototype.setHelper = function(helper) {
        this.helper.setMessage(helper);

        return this;
    };
    /**
     * Shows a message below the control.
     * @param  {String} message  The message
     * @param  {String} [category="info"] The message category,
     * It can be one of the accepted values for the 
     * {@link PMUI.ui.TooltipMessage#setCategory TooltipMessage's setCategory() method}, it defaults to "info".
     * @chainable
     */
    Field.prototype.showMessage = function() {
        this.message.setVisible(true)
        return this;
    };
    /**
     * Hides the message below the control.
     * @chainable
     */
    Field.prototype.hideMessage = function() {
        this.message.setVisible(false);

        return this;
    };
    /**
     * Sets the value type for the field.
     * @param {String} type
     */
    Field.prototype.setValueType = function(type) {
        var newType;
        if(typeof type === 'string') {
            newType = this.validTypes[type] ? type : 'object';
            this.valueType = newType;
            this.data.setType(newType);
        } else {
            throw new Error("The setValueType() method only accepts string values!");
        }

        return this;
    };
    /**
     * Returns the field's value type.
     * @return {String}
     */
    Field.prototype.getValueType = function() {
        return this.valueType;
    };
    /**
     * The onChange event handler constructor.
     * @private
     * @return {Function} The handler.
     */
    Field.prototype.onChangeHandler = function() {
        var that = this, i, dependentFields = this.dependentFields, form = this.form;

        return function() {
            var previousValue = that.value, newValue = that.getValueFromControls();

            if(newValue === previousValue) {
                return this;
            }

            that.value = newValue;
            that.data.setValue(newValue);

            if(that.validAtChange) {
                that.isValid();
            }
            //PMUI.triggerEvent(this, 'change', this);
            if(typeof that.onChange === 'function') {
                that.onChange(that.getValue(), previousValue);
            }
            if(that.form) {
                (that.form.onChangeHandler())(that, that.value, previousValue);
            //PMUI call a event for dependency [dependencyHandler]   
                for(i = 0; i < dependentFields.length; i += 1) {
                    dependentField = form.getField(dependentFields[i]);
                    if(!dependentField) {
                        that.removeDependentField(dependentFields[i]);
                        continue;
                    }
                    dependentField.fireDependencyHandler();
                }
            }
        };
    };

    Field.prototype.fireDependencyHandler = function() {
        var form = this.form, dependsOf, obj = {}, i, that = this;
        if(form) {
            dependsOf = form.dependencies[that.name];
            for(i = 0; i < dependsOf.length; i++) {
                obj[dependsOf[i].name] = dependsOf[i];
            }
            if(typeof this.dependencyHandler === 'function') {
                this.dependencyHandler(that,obj);
            }
        }
        return this;
    };
    /**
     * Attach the event listeners to the HTML element
     * @private
     * @chainable
     */
    Field.prototype.defineEvents = function() {
        var i, that = this;
        Field.superclass.prototype.defineEvents.call(this);
        for(i = 0; i < this.controls.length; i++) {
            this.controls[i].setOnChangeHandler(this.onChangeHandler())
                .defineEvents();
            this.controls[i].setOnBeforeChangeHandler(this.onBeforeChangeHandler()).defineEvents();
        }
        this.addEvent('click').listen(this.html, function(e) {
            if(typeof that.onClick === 'function') {
                that.onClick(that);
            }
        });
        this.addEvent('click').listen(this.dom.labelTextContainer, function(e) {
            e.stopPropagation();    
        });
        this.eventsDefined = true;

        return this;
    };
    /**
     * Returns true if the label is visible, otherwise returns false.
     * @return {Boolean}
     */
    Field.prototype.isLabelVisible = function() {
        return this.labelVisible;
    };
    /**
     * Hides the field's label.
     * @chainable
     */
    Field.prototype.hideLabel = function() {
        jQuery(this.dom.labelTextContainer).css('visibility','hidden');
        this.labelVisible = false;
        return this;
    };
    /**
     * Shows the field's label.
     * @chainable
     */
    Field.prototype.showLabel = function() {
        jQuery(this.dom.labelTextContainer).css('visibility','');
        this.labelVisible = true;
        return this;
    };
    /**
     * Evaluates the required validation.
     * @return {Boolean} Returns true if the validation passes otherwise returns false.
     */
    Field.prototype.evalRequired = function() {
        var valid = true;
        if(this.required){
            if(this.valueType !== 'number'){
                if(this.getValue() === "" || this.getValue() ==="[]") {
                    valid = false;
                }   
            } else {
                if(this.getValue().toString() === 'NaN' ) {
                    valid = false; 
                }
            }
            if(valid){
                this.hideMessage();
            } else {
                this.showMessage();
            }
        }
        return valid;
    };
    /**
     * Executes the validations and returns true if all of them passes, otherwise returns false.
     * @return {Boolean}
     */
    Field.prototype.isValid = function() {
        var valid = true, validator;
        valid = valid && this.evalRequired();
        if(!valid) {
            this.controls[0].style.addClasses(['error']);
            return valid;
        }
        this.controls[0].style.removeClasses(['error']);
        for(validator in this.validators) {
            if(this.validators.hasOwnProperty(validator)) {
                valid = valid && this.validators[validator].isValid();
                if(!valid) {
                    this.message.setText(this.validators[validator].errorMessage);
                    return valid;
                }
            }
        }
        return valid;
    };
    /**
     * Create the HTML Element for the Field.
     * @protected
     * @return {HTMLElement}
     */
    Field.prototype.createHTML = function() {
        var html, label, 
            labelTextContainer, 
            controlContainer, 
            messageContainer,
            fieldColon,
            fieldRequired,
            fieldTextLabel;

        if(!this.html) {
            //html = PMUI.createHTMLElement("div");
            this.style.addClasses(['pmui-field']);
            Field.superclass.prototype.createHTML.call(this);
  

            labelTextContainer = PMUI.createHTMLElement("label");
            labelTextContainer.className = 'pmui-field-label'; // contenedor de todos los labels

            //labelText
            fieldTextLabel = PMUI.createHTMLElement('span');
            fieldTextLabel.className = "pmui-field-textLabel";
            //asterisk for required
            fieldRequired = PMUI.createHTMLElement ('span');
            fieldRequired.className = "pmui-field-required";
            //colon
            fieldColon = PMUI.createHTMLElement('span');
            fieldColon.className = "pmui-field-colon";


            controlContainer = PMUI.createHTMLElement("span");
            controlContainer.className = 'pmui-field-control';
            messageContainer = PMUI.createHTMLElement("span");
            messageContainer.className = 'pmui-field-message';
            messageContainer.style.display = "block"



            //labelTextContainer append tree childs label, * and :
            labelTextContainer.appendChild(fieldTextLabel);
            labelTextContainer.appendChild(fieldRequired);
            labelTextContainer.appendChild(fieldColon);

            //messageContainer append the helper
            messageContainer.appendChild(this.message.getHTML());

            //this html append label, control and helper tag
            this.html.appendChild(labelTextContainer);
            this.html.appendChild(controlContainer);
            this.html.appendChild(messageContainer);

            this.dom.labelTextContainer = labelTextContainer;
            this.dom.fieldTextLabel = fieldTextLabel;
            this.dom.fieldColon = fieldColon;
            this.dom.fieldRequired = fieldRequired;
            this.dom.controlContainer = controlContainer;
            this.dom.messageContainer = messageContainer;
            //this.html = html;

            this.setControlPositioning(this.controlPositioning);
            this.setLabelWidth(this.labelWidth);
            this.setLabel(this.label);
            this.setValue(this.value);
            this.setLabelPosition(this.labelPosition);
            if(this.labelVisible) {
                this.showLabel();
            } else {
                this.hideLabel();
            }
            if(this.visibleColon) {
                this.showColon();
            } else {
                this.hideColon();
            }
        }

        return this.html;
    };
    /**
     * Set the width for the field.
     * @param {Number|String} width height it can be a number or a string.
     * In case of using a String you only can use 'auto' or ##px or ##% or ##em when ## is a number.
     * @chainable
     */
    Field.prototype.setWidth = function (width) {
        Field.superclass.prototype.setWidth.call(this, width);
        if(this.labelWidth){
            this.setLabelWidth(this.labelWidth);    
        }    
        return this;
    };
    /**
     * @method setFocus
     * set the focus on field for to control
     * @param {Number} [index] if the field has more than one control and needs to focus on a specific Control
     * @chainable
     */
    Field.prototype.setFocus = function (index){
        var i = index || 0, j, controls = this.getControls();

        if(this.controls[i]) {
            this.controls[i].setFocus();
        }
        return this;                
    };   

    /*Field.prototype.addDependentField = function (field) {
        if (field instanceof PMUI.form.Field){
            this.dependentFields.insert(field);
            field.dependencyFields.insert(this);
        } else {
            throw new Error ("addDependentField(): the parameter is not an instance of field");
        }
        return this;
    };*/
    /*
    Field.prototype.removeDependentField = function (fieldDependent) {

        if (fieldDependent instanceof PMUI.form.Field){
            this.dependentFields.remove(fieldDependent);
        } else {
            throw new Error ("addDependentField(): the parameter is not an instance of field");
        }
        return this;
    };*/

    PMUI.extendNamespace('PMUI.form.Field', Field);

    if (typeof exports !== "undefined") {
        module.exports = Field;
    }
}());
(function(){
    /**
     * @class PMUI.field.TextField
     * @extends PMUI.form.Field
     * Class to handle a {@link PMUI.control.TextControl} field.
     *
     * Usage example:
     *
     *      @example
     *          var a;
     *          $(function() {
     *              a = new PMUI.field.TextField({
     *                  label: "Some Text",
     *                  placeholder: 'a text here',
     *                  required: true,
     *                  helper: "Introduce a text (6 chars. max.)",
     *                  validators: [
     *                      {
     *                          pmType: 'textLength',
     *                          criteria: {
     *                              maxLength: 6
     *                          }
     *                      }
     *                  ]
     *              });
     *              document.body.appendChild(a.getHTML());
     *              a.defineEvents();
     *          });
     *          
     *
     * The code above will generate a required TextField with "Some Text" as label, 
     * the control will have a placeholder with the text "a text here". 
     * Also the field will contain a helper with the text "Introduce a text (6 chars. max.)" 
     * and a validator that will control the maximum length for the text to be entered.
     *
     * @cfg {String} [placeholder=""] The text to show as placeholder for the field's control
     * @cfg {Number} [maxLength=0] An integer that specifies the maximum character length for the text to be entered. 
     * 0 means no max length. 
     * 
     * #Note: This setting doesn't act as a validator, it simply set the max character length 
     * for the field's control.
     * @cfg {Boolean} [trimOnBlur=true] A boolean that specifies if the value entered will be trimmed 
     * when the field loses focus.
     */
    var TextField = function(settings) {
        TextField.superclass.call(this, settings);
        /**
         * @property {Boolean} [trimOnBlur=true] If the field's value must be trimmed every time it loses focus.
         */
        this.trimOnBlur = null;
        TextField.prototype.init.call(this, settings);
    };

    PMUI.inheritFrom('PMUI.form.Field', TextField);

    TextField.prototype.type = "TextField";

    TextField.prototype.init = function(settings) {
        var defaults = {
            placeholder: "",
            maxLength: 0,
            trimOnBlur: true,
            readOnly : false
        };

        $.extend(true, defaults, settings);

        this.setPlaceholder(defaults.placeholder)
            .setMaxLength(defaults.maxLength)
            .setTrimOnBlur(defaults.trimOnBlur)
            .setReadOnly(defaults.readOnly)
            .setDependentFields(this.dependentFields);
    };
    /**
     * Sets the value for the field.
     * @param {String} value
     * @chainable
     */
    TextField.prototype.setValue = function(value) {
        if(this.trimOnBlur) {
            value = jQuery.trim(value);
        }
        TextField.superclass.prototype.setValue.call(this, value);

        return this;
    };
    /**
     * Sets the placeholder for the control. Note that this feature is only supported 
     * by browsers which support the "placeholder" for input elements.
     * @param {String} placeholder
     * @chainable
     */
    TextField.prototype.setReadOnly = function(readonly) {
        this.controls[0].setReadOnly(readonly);
        return this;
    };
    /**
     * [setPlaceholder description]
     * @param {[type]} placeholder [description]
     */
    TextField.prototype.setPlaceholder = function(placeholder) {
        this.controls[0].setPlaceholder(placeholder);
        return this;
    };
    /**
     * [getPlaceholder description]
     * @return {[type]} [description]
     */
    TextField.prototype.getPlaceholder = function() {
        return this.controls[0].getPlaceholder();
    };
    /**
     * [setMaxLength description]
     * @param {[type]} maxLength [description]
     */
    TextField.prototype.setMaxLength = function(maxLength) {
        this.controls[0].setMaxLength(maxLength);
        return this;
    };
    /**
     * [getMaxLength description]
     * @return {[type]} [description]
     */
    TextField.prototype.getMaxLength = function() {
        return this.controls[0].getMaxLength();
    };
    /**
     * [isReadOnly description]
     * @return {Boolean} [description]
     */
    TextField.prototype.isReadOnly = function() {
        return this.controls[0].isReadOnly();
    };
    /**
     * Switches on/off the value trimming for the field's value when it loses focus.
     * @param {Boolean} trimOnBlur Send true for switch on or false to switch off.
     * @chainable
     */
    TextField.prototype.setTrimOnBlur = function(trimOnBlur) {
        this.trimOnBlur = !! trimOnBlur;
        return this;
    };
    /**
     * Returns a boolean value that indicates if the trimming function is enabled/disabled
     * @return {Boolean}
     */
    TextField.prototype.getTrimOnBlur = function() {
        return this.trimOnBlur;
    };
    /**
     * Sets the control for the TextField
     * @chainable
     * @private
     */
    TextField.prototype.setControls = function() {
        if(this.controls.length) {
            return this;
        }

        this.controls.push(new PMUI.control.TextControl());

        return this;
    };
    /**
     * Establish the handler function for the internal onChange event
     * @return {Function}
     * @private
     */
    TextField.prototype.onChangeHandler = function() {
        var that = this, i, dependentFields = this.dependentFields, form = this.form;
        return function() {
            var previousValue = that.value, value;
            if(that.trimOnBlur) {
                value = that.controls[0].getValue();
                value = jQuery.trim(value);
                that.controls[0].setValue(value);
            }
            that.updateValueFromControls();
            if(that.validAtChange) {
                that.isValid();
            }
            if(typeof that.onChange === 'function') {
                that.onChange(that.getValue(), previousValue);
            }
            if(that.form) {
                (that.form.onChangeHandler())(that, that.getValue(), previousValue);
            }
            if(that.form) {
                for(i = 0; i < dependentFields.length; i += 1) {
                    dependentField = form.getField(dependentFields[i]);
                    if(!dependentField) {
                        that.removeDependentField(dependentFields[i]);
                        continue;
                    }
                    dependentField.fireDependencyHandler();
                }
            }
        };
    };

    PMUI.extendNamespace('PMUI.field.TextField', TextField);

    if (typeof exports !== "undefined") {
        module.exports = TextField;
    }
}());
(function(){
    /**
     * @class PMUI.field.PasswordField
     * @extends PMUI.form.Field
     * Class to handle a {@link PMUI.control.PasswordControl} field.
     *
     * Usage example:
     *
     *      @example
     *          var a;
     *          $(function() {
     *              a = new PMUI.field.PasswordField({
     *                  label: "Some Text",
     *                  required: true,
     *                  helper: "Introduce a text (6 chars. max.)",
     *                  validators: [
     *                      {
     *                          pmType: 'textLength',
     *                          criteria: {
     *                              maxLength: 6
     *                          }
     *                      }
     *                  ]
     *              });
     *              document.body.appendChild(a.getHTML());
     *              a.defineEvents();
     *          });
     *
     *
     * The code above will generate a required PasswordField with "Some Text" as label,
     * the control will have a placeholder with the text "a text here".
     * Also the field will contain a helper with the text "Introduce a text (6 chars. max.)"
     * and a validator that will control the maximum length for the text to be entered.
     *
     * @cfg {Number} [maxLength=0] An integer that specifies the maximum character length for the text to be entered.
     * 0 means no max length.
     *
     * #Note: This setting doesn't act as a validator, it simply set the max character length
     * for the field's control.
     * @cfg {Boolean} [trimOnBlur=true] A boolean that specifies if the value entered will be trimmed
     * when the field loses focus.
     */
    var PasswordField = function(settings) {
        PasswordField.superclass.call(this, settings);
        /**
         * @property {Boolean} [trimOnBlur=true] If the field's value must be trimmed every time it loses focus.
         */
        this.trimOnBlur = null;
        PasswordField.prototype.init.call(this, settings);
    };

    PMUI.inheritFrom('PMUI.form.Field', PasswordField);

    PasswordField.prototype.type = "PasswordField";

    PasswordField.prototype.init = function(settings) {
        var defaults = {
            //placeholder: "",
            maxLength: 0,
            trimOnBlur: true
        };

        $.extend(true, defaults, settings);

        this.setMaxLength(defaults.maxLength)
            //.setPlaceholder(defaults.placeholder)
            .setTrimOnBlur(defaults.trimOnBlur);
    };
    /**
     * Sets the value for the field.
     * @param {String} value
     * @chainable
     */
    PasswordField.prototype.setValue = function(value) {
        if(this.trimOnBlur) {
            value = jQuery.trim(value);
        }
        PasswordField.superclass.prototype.setValue.call(this, value);

        return this;
    };
//    /**
//     * Sets the placeholder for the control. Note that this feature is only supported
//     * by browsers which support the "placeholder" for input elements.
//     * @param {String} placeholder
//     * @chainable
//     */
//    PasswordField.prototype.setPlaceholder = function(placeholder) {
//        this.controls[0].setPlaceholder(placeholder);
//        return this;
//    };

//    PasswordField.prototype.getPlaceholder = function() {
//        return this.controls[0].getPlaceholder();
//    };

    PasswordField.prototype.setMaxLength = function(maxLength) {
        this.controls[0].setMaxLength(maxLength);
        return this;
    };

    PasswordField.prototype.getMaxLength = function() {
        return this.controls[0].getMaxLength();
    };
    /**
     * Switches on/off the value trimming for the field's value when it loses focus.
     * @param {Boolean} trimOnBlur Send true for switch on or false to switch off.
     * @chainable
     */
    PasswordField.prototype.setTrimOnBlur = function(trimOnBlur) {
        this.trimOnBlur = !! trimOnBlur;
        return this;
    };
    /**
     * Returns a boolean value that indicates if the trimming function is enabled/disabled
     * @return {Boolean}
     */
    PasswordField.prototype.getTrimOnBlur = function() {
        return this.trimOnBlur;
    };
    /**
     * Sets the control for the PasswordField
     * @chainable
     * @private
     */
    PasswordField.prototype.setControls = function() {
        if(this.controls.length) {
            return this;
        }

        this.controls.push(new PMUI.control.PasswordControl());

        return this;
    };
    /**
     * Establish the handler function for the internal onChange event
     * @return {Function}
     * @private
     */
    PasswordField.prototype.onChangeHandler = function() {
        var that = this;

        return function() {
            var previousValue = that.value, value;
            if(that.trimOnBlur) {
                value = that.controls[0].getValue();
                value = jQuery.trim(value);
                that.controls[0].setValue(value);
            }
            that.updateValueFromControls();
            if(that.validAtChange) {
                that.isValid();
            }
            if(typeof that.onChange === 'function') {
                that.onChange(that.getValue(), previousValue);
            }
            if(that.form) {
                (that.form.onChangeHandler())(that, that.getValue(), previousValue);
            }
        };
    };

    PMUI.extendNamespace('PMUI.field.PasswordField', PasswordField);

    if (typeof exports !== "undefined") {
        module.exports = PasswordField;
    }
}());
(function () {
    /**
     * @class PMUI.field.TextAreaField
     * @extends PMUI.field.TextField 
     * Class to handle a {@link PMUI.control.TextAreaControl}.
     *
     * Usage example:
     *
     *      @example 
     *           var a;
     *           $(function() {
     *                a = new PMUI.field.TextAreaField({
     *                     id:'12345',
     *                     label: "coment",
     *                     cols: 300,   
     *                     rows: 200,
     *                     name: 'coments',
     *                     placeholder: 'make your comment here...........',
     *                     required: true,
     *                     labelposition:'left',
     *                     labelVerticalPosition:'top',
     *                     tooltipPosition:'bottom',
     *                     helper: "Introduce a text (200 chars. max.)",
     *                     validators: [
     *                         {
     *                             pmType: 'textLength',
     *                             criteria: {
     *                                 maxLength: 200
     *                             }
     *                         }
     *                     ]
     *                 });
     *                 document.body.appendChild(a.getHTML());
     *             });
     *             
     *  
     * TextAreaField, is a field for very long text strings
     * to generate the above code on TextAreaField requires, one id, label, name, placeholder, required, 
     labelposition, 
     * labelVerticalPosition, tooltipPosition, helper.
     *
     * In case to modified label position or tooltip position it can contain the next properties:
     * @cfg {'string'} [labelVerticalPosition='top'] in case that the position of the label left or right we 
     fixed this 
     * property in three cases ('top', 'center', 'bottom')
     * @cfg {'string'} [tooltipPosition='top'] this property can set the tooltip in two cases ('top' or 'bottom') 
     for TextAreaField
     */

    var TextAreaField = function (settings){
        
        TextAreaField.superclass.call(this, settings);
        /**
         * @property {string} [labelVerticalPosition='top'], in case that the position of the label left 
         or right we fixed this 
         * property in three cases ('top', 'center', 'bottom')
         * @readonly
         */
        this.labelVerticalPosition = null;
        /**
         * @property {string} [tooltipPosition='top'] this property can set the tooltip in two cases ('top' or 
         'bottom').
         * @readonly
         */
        this.tooltipPosition = null;
        this.cols = null;
        this.rows = null;
        
        TextAreaField.prototype.init.call(this, settings);
    };

    PMUI.inheritFrom('PMUI.field.TextField', TextAreaField);

    TextAreaField.prototype.type = "TextAreaField";

    TextAreaField.prototype.init = function (settings){
        var defaults = {
            labelVerticalPosition : 'top',
            tooltipPosition : 'top',
            cols : 'auto',
            rows: 'auto'
        };

        jQuery.extend(true, defaults, settings);

        this.setLabelVerticalPosition(defaults.labelVerticalPosition);
        this.setTooltipPosition(defaults.tooltipPosition);
        this.setRows(defaults.rows);
        this.setCols(defaults.cols);

    };  

    /**
     * Sets the labelVerticalPosition for the Label of the TextAreaField. 
     * input values ​​(top, center and bottom) only affect the field if the label of TextAreaField is in 
     * left or right position.
     * @param {String} labelVerticalPosition
     * @chainable
     */

    TextAreaField.prototype.setLabelVerticalPosition = function (value) {
        var x, result;
        if (value === 'top' || value ==='bottom' || value === 'center'){
            this.labelVerticalPosition = value;

                if(this.controls[0] && this.html){
                    this.dom.labelTextContainer.style.position = 'relative';
                    if(this.labelPosition === 'left' || this.labelPosition === 'right'){
                        if(this.labelVerticalPosition ===  'top'){
                            this.dom.labelTextContainer.style.top = '0px';
                            this.dom.labelTextContainer.style.verticalAlign = 'top';
                        }
                        if(this.labelVerticalPosition ===  'bottom'){
                            this.dom.labelTextContainer.style.top = '0px';

                            this.dom.labelTextContainer.style.verticalAlign = 'bottom';
                        }               
                        if(this.labelVerticalPosition === 'center'){

                            this.dom.labelTextContainer.style.verticalAlign = 'top';
                            result = ($(this.controls[0].getHTML()).outerHeight()/2) - 
                                ($(this.dom.labelTextContainer).outerHeight()/2);
        
                            this.dom.labelTextContainer.style.top = (result) +'px';
                
                        }
                    }
                }
        }
        else
        {
            throw new Error ("The value is not 'top' or 'bottom' please enter these values");
        }

        return this;
    };
    /**
     * Sets the label position
     * input values(top, bottom, left, right) 
     * @param {String} labelPosition
     * @chainable
     */
    TextAreaField.prototype.setLabelPosition = function(position) {
        if(this.html && (position === 'top' || position === 'bottom')) {
            this.dom.labelTextContainer.style.top = '0px';
        }
        TextAreaField.superclass.prototype.setLabelPosition.call(this, position);

        return this;
    };
    /**
     * Sets the tooltipPosition for the Label of the TextAreaField. 
     * input values ​​(top and bottom) only affect the field if the label of TextAreaField
     * @param {String} tooltipPosition
     * @chainable
     */
    TextAreaField.prototype.setTooltipPosition = function (value) {

        if (value === 'top' || value ==='bottom'){
            this.tooltipPosition = value;

                if(this.controls[0] && this.html){
        
                    
                        if(this.tooltipPosition ===  'top'){
                            this.helper.icon.style.addProperties({'vertical-align':'top'});
                        }
                        if(this.tooltipPosition ===  'bottom'){
            
                            this.helper.icon.style.addProperties({'vertical-align':'bottom'});
                        }                                   
                }
        }
        else
        {
            throw new Error ("The value is not 'top' or 'bottom' please enter these values");
        }

        return this;
    };
    /**
     * Sets the  setHeight for the {PMUI.controls.TextAreaControl}. 
     * @param {String|number} [PMUI.controls.TextAreaControl]
     * @chainable
     */
    TextAreaField.prototype.setControlHeight = function (height){
            this.controls[0].setHeight(height);
            return this;
    };
    /**
     * Sets the  setWidth for the {PMUI.controls.TextAreaControl}. 
     * @param {String|number} [PMUI.controls.TextAreaControl]
     * @chainable
     */
    TextAreaField.prototype.setControlWidth = function (width){
            this.controls[0].setWidth(width);
            return this;
    };

    /**
     * Sets the text for the TextAreaField label.
     * @param {String} label
     */

    TextAreaField.prototype.setLabel = function (label) {

        if(typeof label === 'string' ) {
            this.label = label;
        } else {
            throw new Error("The setLabel() method only accepts string values!");
        }
     
        if(this.dom.labelTextContainer) {
            this.dom.labelTextContainer.textContent = this.label + (this.colonVisible ? ':' : '');
        }
        return  this;
    };
    /**
     * Sets the control for the TextField
     * @chainable
     * @private
     */
    TextAreaField.prototype.setControls = function() {
        if(this.controls.length) {
            return this;
        }

        this.controls.push(new PMUI.control.TextAreaControl());

        return this;
    };
        
    /**
     * Disables the field
     * @param {Boolean} Disable, if the field, then control is activated, if the control is enabled
     * @chainable
     */
    TextAreaField.prototype.disableField = function (){
        this.controls[0].disable(true);
        return this;
    };
    /**
     * Disables/enables the control
     * @param {Boolean} Enable, if the field, then control is activated, if the control is diabled
     * @chainable
     */
    TextAreaField.prototype.enableField = function (){
        this.controls[0].disable(false);
        return this;
    };
    /**
     * Creates the HTML element for the textAreaField
     * @return {HTMLElement} 
     */
    TextAreaField.prototype.createHTML = function() {
        TextAreaField.superclass.prototype.createHTML.call(this);       
        this.setLabelVerticalPosition(this.labelVerticalPosition);
        this.setTooltipPosition(this.tooltipPosition);
        return this.html;
    };
    /**
     * assign hight as rows to textArea,
     * @param {number|String} rows will be assigned as high in PX, the entered values ​​can 
     * be numbers, 'number + px', 'auto' or percentage
     * @chainable
     */
    TextAreaField.prototype.setRows = function(rows){
        this.controls[0].setHeight(rows);
        return this;
    };
    /**
     * assign width as Cols to textArea,
     * @param {number|String} Cols will be assigned as high in PX, the entered values ​​can 
     * be numbers, 'number + px', 'auto' or percentage
     * @chainable
     */
    TextAreaField.prototype.setCols = function(cols){
        this.controls[0].setWidth(cols);
        return this;
    };

    PMUI.extendNamespace('PMUI.field.TextAreaField', TextAreaField);

    if (typeof exports !== "undefined") {
        module.exports = TextAreaField;
    }
}());
(function (){
    /**
     * @class PMUI.field.DropDownListField
     * @extends PMUI.form.Field 
     * Class to handle a {@link PMUI.control.DropDownListControl} field.
     *
     * Usage example:
     *
     *      @example
     *              var settings, list;
     *              $(function(){
     *                  settings = {
     *                   id : '123',
     *                  label : "Pais",
     *                  name : 'List Select',
     *                  required: true,
     *                  helper: "selected one opcion of the list",
     *                  options: [                                
     *                  {
     *                   label: "BOLIVIA",
     *                        options: [
     *                        {
     *                              label: "La Paz",
     *                              value: 1,
     *                              disabled : true
     *                          },
     *                          {
     *                              label: "Cochabamba",
     *                              value: 2
     *                          },
     *                          {
     *                              label: "SantaCruz",
     *                              value: 3
     *                          },
     *                      ]
     *                   },
     *                   {
     *                        label: "ARGENTINA",
     *                      options: [
     *                          {
     *                              label: "Buenos Aires",
     *                              value: 4
     *                          },
     *                          {
     *                              label: "Santa Fe",
     *                              value: 5
     *                          },
     *                          {
     *                                  value: "Cordoba"
     *                          }
     *                      ]
     *                   },
     *                   {
     *                        label: "CHILE",
     *                      options: [
     *                          {
     *                              label: "Arica",
     *                              value: 6
     *                          },
     *                          {
     *                              label: "Santiago",
     *                              value: 7
     *                          },
     *                          {
     *                                  value: "Concepcion"
     *                          }
     *                      ]
     *                   },
     *                   {
     *                        label: "New York",
     *                        value: 8
     *                   },
     *                   {
     *                        label: "Mexico D.F.",
     *                        value: 9
     *                   }
     *               ]
     *           }
     *           list = new PMUI.field.DropDownListField(settings);
     *           document.body.appendChild(list.getHTML());
     *         });
     *         
     *
      * DropDownListField typo is a selection field, enclose the values â€‹â€‹that we can choose from a list 
      * of options. The attributes that accompany the opening tag are:
      * The code above will generate a DropDownListField, required one id, name, requiere, 
      * helper, text  for list the options, one opcion array  with the list options
      * 
      * @constructor Creates an instance of the class DropDownListField.
      * @param {Object} options Initialization options.
      * @cfg {Array} [options=[]] An array with all the options to be contained by the 
      {PMUI.control.DropDownListControl}.
     * 
     * Each element in the array is a JSON object, this JSON object can represent an option group 
     * or an option item. 
     * 
     * In case to represent an option item it can contain the next properties:
     *
     * - value {String} (required): the value for the option.
     * - label {String} (optional): the label for the option, if isn't specified the value is used instead.
     * - selected {Boolean} (optional): if the option is selected. #Note. If the configuration object has the
     * "value" propery set then this "selected" property will be
     * - disabled {Boolean} (optional): if the option is disabled or not.
     *
     * On the other hand, in case to represent an option group, it can contain the next properties:
     *
     * - label {String} (required): The name for the option group.
     * - disabled {Boolean} (optional): If the group is disabled or not.
     * - options {Array} (required): An array in which each element is a JSON object representing an option item,
     * so every item must have the structure explained above (for represent option items). #Note. This propery makes 
     * the difference between an option and a option group. If the "options" property is not specified or if it isn't
     * an array then it will treated like a option item.
     * @cfg {String|Number} [listWidth='auto'] This value sets the width of the list
     */


     var DropDownListField = function (settings){

          DropDownListField.superclass.call(this, settings);

          /**
         * @property {Boolean} [disabled=false] If the field is disabled or not.
         * @readonly
         */
          
          this.listWidth = null;
          //this.defaultValue = null;

          DropDownListField.prototype.init.call(this, settings);

     }; 

     PMUI.inheritFrom("PMUI.form.Field", DropDownListField);

     DropDownListField.prototype.type = "DropDownListField";

     DropDownListField.prototype.init = function (settings) {
        var defaults = {
            options : [],
            listWidth : 'auto',
            value: null
        };

        jQuery.extend(true, defaults, settings);

        this.setOptions(defaults.options);
        this.setListWidth(defaults.listWidth);
        this.updateValueFromControls();

        if(defaults.value !== null) {
            this.setValue(defaults.value);
        }
     };
     /**
     * Sets the options/option groups for the control.
     * @param {Array} options An array with the same structure that the 
     * {@link PMUI.field.DropDownListField#cfg-options "options"} property in the 
     * Config settings section.
     */
     DropDownListField.prototype.setOptions = function (options) {
        var i;
        if(jQuery.isArray(options)){
           this.controls[0].setOptions(options);
        }
        if(!this.value && options) {
            for(i = 0; i < options.length; i++) {
                if(options[i].selected) {
                    this.initialValue = options[i].value || options[i].label || "";
                    break;
                }
            }
        }
        this.value = this.controls[0].getValue();
        return this;
     };
    /**
     * Returns the options/option groups from the field
     * @param  {Boolean} [includeGroups=false] If it's evaluated as true then it will include 
     * the option groups with its child elements, otherwise it will return only the option items.
     * @return {Array}
     * 
     *         example 
     *
     *          list.getOptions(false)
     *         [La Paz][Cochabamba][Santa Cruz][Buenos Aires][Santa Fe][Cordoba][Santiago][.][.][Mexico D.F]
     *    
     *        list.getOptions(true)
     *          [BOLIVIA
     *              [La Paz][Cochabamba][SantaCruz]
     *          ]
     *          [ARGENTINA
     *              [Buenos Aires][Santa Fe][Cordoba]
     *           ]
     *          [CHILE
     *             [x][y][z]
     *          ]
     *          [New York]
     *          [Mexico D.F.]
     *
     * @return {Array}
     */
     DropDownListField.prototype.getOptions = function (includeGroups) {
          return this.controls[0].getOptions(includeGroups);
     };
      /**
     * Clear all the options from the control.
     * @chainable
     */
     DropDownListField.prototype.clearOptions = function() {
        this.controls[0].clearOptions();
        this.value = this.controls[0].value;
        return this;
    };

    /**
     * Disables one or more options/option groups.
     * @param  {String|Number|Object} option It can be a string, a number or a JSON object.
     * 
     * - In case to be a String, it will be disabled the options that match the string in its value and the option 
     * groups which match the string in its label. In this case more than one single item can be disabled.
     * - In case to be a Number, it will be disabled the option/option group which index position matches the number.
     * Obviously, in this case only one item will be disabled.
     * - In case to be an object you can specify if the change will be applied only to options or option groups,
     * it should have two properties:
     *     - criteria {String}: The value (in case of options) or the label (in case of option groups) the items 
     needs 
     *     to match for apply the changes.
     *     - applyTo {String} (optional), it can take the following values:
     *         - "groups", the change will be applied only to the option groups.
     *         - "options", the change will be applied only to the options (direct child of the object).
     *         - [any other string value], the default value, it indicates that the change will be applied to both 
     *         options/option groups that matches the criteria in its value/label respectly.
     *
     * @param  {String} [group] It it is specified must be an String making reference to an existing option group 
     label.
     * Using this parameter, the elements to be match by the first parameter will be search only in the option groups 
     * that match this parameter in its label.
     * 
     *         example
     *              list =    [a
     *                             [a0]
     *                             [a1]
     *                        ]
     *                        [b
     *                             [b0]
     *                             [b1]
     *                        ]
     *                        [c]
      *   //disabling sending a number
     *              list.disableOption(0); -->disabled group a
     *              list.disableOption(4); -->disabled option b[b0]
     *    
     *    //disabling sending a string
      *             list.disableOption('a') --> disabled group a
      *             list.disableOption('b0') --> disabled option b[b0]

      *   //disabling sending a objects
                    list.disableOption({criteria:'a', applyTo:'groups'}) --> disabled group a       
      *  
     * @chainable
     */

     DropDownListField.prototype.disableOption = function (option, group) {
          this.controls[0].disableOption(option, group);
          return this;
     };

     /**
     * Enables one or more options/option groups.
     * @param  {String|Number|Object} option It can be a string, a number or a JSON object.
     * 
     * - In case to be a String, it will be enabled the options that match the string in its value and the option 
     * groups which match the string in its label. In this case more than one single item can be enabled.
     * - In case to be a Number, it will be enabled the option/option group which index position matches the number.
     * Obviously, in this case only one item will be enabled.
     * - In case to be an object you can specify if the change will be applied only to options or option groups,
     * it should have two properties:
     *     - criteria {String}: The value (in case of options) or the label (in case of option groups) the items needs
     *     to match for apply the changes.
     *     - applyTo {String} (optional), it can take the following values:
     *         - "groups", the change will be applied only to the option groups.
     *         - "options", the change will be applied only to the options (direct child of the object).
     *         - [any other string value], the default value, it indicates that the change will be applied to both
     *         options/option groups that matches the criteria in its value/label respectly.
     *
     * @param  {String} [group] It it is specified must be an String making reference to an existing option group 
     label.
     * Using this parameter, the elements to be match by the first parameter will be search only in the option groups
     * that match this parameter in its label.
     * @chainable
     */

     DropDownListField.prototype.enableOption = function (option, group) {
          this.controls[0].enableOption(option, group);
          return this;
     };

      /**
     * Removes one or more option/option groups.
     * @param  {String|Number|Object} option It can be a string, a number or a JSON object.
     * 
     * - In case to be a String, it will be removed the options that match the string in its value and the option
     * groups which match the string in its label. In this case more than one single item can be removed.
     * - In case to be a Number, it will be removed the option/option group which index position matches the number.
     * Obviously, in this case only one item will be removed.
     * - In case to be an object you can specify if the change will be applied only to options or option groups, 
     * it should have two properties:
     *     - criteria {String}: The value (in case of options) or the label (in case of option groups) the items 
     needs 
     *     to match for apply the changes.
     *     - applyTo {String} (optional), it can take the following values:
     *         - "groups", the change will be applied only to the option groups.
     *         - "options", the change will be applied only to the options (direct child of the object).
     *         - [any other string value], the default value, it indicates that the change will be applied to both 
     *         options/option groups that matches the criteria in its value/label respectly.
     *
     * @param  {String} [group] It it is specified must be an String making reference to an existing option group 
     label.
     * Using this parameter, the elements to be match by the first parameter will be search only in the option groups 
     * that match this parameter in its label.
     * 
     * ##Note. Removing an option group implies removing all its child options.
     * @chainable
     */
     DropDownListField.prototype.removeOption = function (option, group) {
          this.controls[0].removeOption(option, group);
          return this;
     };
    /**
     * Adds a new option group to the Field
     * @param {Object} optionGroup A JSON object with the following properties:
     *
     * - label {String} (required): the label for the option group.
     * - disabled {Boolean}(optional): if the option group will be disabled or not.
     * it defaults to false.
     * - options {Array} (optional): An array of JSON object, each one represents an option and
     * should have the same structure than the "option" paremeter for the 
     * {@link PMUI.field.DropDownListField#addOption addOption() method}.
     */
     DropDownListField.prototype.addOptionGroup = function (optionGroup) {
          this.controls[0].addOptionGroup(optionGroup);
          return this;
     };

     /**
     * Adds a new option to the list of the DropDownListField or to an option group.
     * @param {Object} option An object with ther settings for the new option.
     * this object can have the following properties:
     * 
     * - value {String} (required): the value for the option.
     * - label {String} (optional): the label for the option, if isn't specified the value is used instead.
     * - selected {Boolean} (optional): if the option is selected. #Note. If the configuration object has the
     * "value" propery set then this "selected" property will be
     * - disabled {Boolean} (optional): if the option is disabled or not.
     * 
     * @param {String} group The name of the option group in which the new option will be added. If it doesn't exist
     * it will be created.
     *    
     *         usage
     *         // add option a groupÃ§
      *
     *         list.addOption({value:'5', label:'Pando', selected: true, disabled:true },'BOLIVIA')
     *         
     *         //or add option
     *
     *         list.addOption({value:'paris', label:'Paris'})
     *
     * @chainable
     */
     DropDownListField.prototype.addOption = function (option, group) {
          this.controls[0].addOption(option, group);
          if(this.getOptions().length == 1){
            this.value = this.controls[0].value;
          }
          return this;
     };

     /**
     * Returns the label from the option currently selected.
     * @return {String}
     */
     DropDownListField.prototype.getSelectedLabel = function () {
          return this.controls[0].getSelectedLabel();
     };
     /**
      * Sets the control for the DropDownListField
      * @chainable
      * @private
      */
     DropDownListField.prototype.setControls = function () {
          if(this.controls.length) {
               return this;
          }
          this.controls.push(new PMUI.control.DropDownListControl());

          return this;
     };
      /**
     * Set the width for the HTML DropDownListControl(select)
     * @param {Number|String} width height it can be a number or a string.
      In case of using a String you only can use 'auto' or 'inherit' or ##px or ##% or ##em when ## is a number
     * @chainable
     */   

     DropDownListField.prototype.setListWidth = function (width) {
          this.listWidth = width;
          this.controls[0].setWidth(this.listWidth);
          return this;
     };
     

     DropDownListField.prototype.getListWidth = function (width) {
        return this.listWidth;
     };

     DropDownListField.prototype.setValue = function(value) {
        var val;
        if(this.controls[0]) {
            val = this.controls[0].setValue(value).getValue();
        }
        DropDownListField.superclass.prototype.setValue.call(this, val || value);
        return this;
     };

     PMUI.extendNamespace('PMUI.field.DropDownListField', DropDownListField);

     if (typeof exports !== "undefined"){
          module.exports = DropDownListField;
     }
}());
(function(){
        /**
     * @class  PMUI.field.CheckBoxGroupField
     * Field where the values ​​can be selected from a group of options.
     * @extends PMUI.form.Field
     *
     *  Usage example:
     *  
     *      @example
     *             var a;
     *             $(function() {
     *                 a = new PMUI.field.CheckBoxGroupField({
     *                     label: "development tools",
     *                     controlPositioning: 'vertical',
     *                     maxDirectionOptions: 2,
     *                     required: true,
     *                     value: '1',
     *                     helper: "what lanaguages you know?. Select others besides javascript",
     *                     options : [
     *                         {  label: 'Java',
     *                            disabled: false,
     *                            value: 'java',
     *                            selected :true
     *                         },
     *                         { label: 'JavaScript',
     *                            disabled: true,
     *                            value: 'javaScript',
     *                            selected :true
     *                         },
     *                         { label: 'Python',
     *                            disabled: false,
     *                            value: 'python'
     *                         },
     *                          {  label: 'C#',
     *                            disabled: true,
     *                            value: 'visualbasic'
     *                         }
     *                     ],
     *                      onChange: function(newVal, oldVal) {
     *                         console.log("The value for the field  \"" + this.getLabel() + 
     *                         "\": has change from \"" + oldVal + "\" to \"" + newVal + "\"");
     *                     },
     *                 });
     *                 document.body.appendChild(a.getHTML());
     *             });
     *             
     *      
     * The example above will generate a field with 4 CheckBox options.
     *
     * @constructor Creates an instance of the class CheckBoxGroupField.
     * @param {Object} options Initialization options.
     * @cfg {Array} [options=[]] An array in which every element is a JSON object with the same 
     structure required in the 
     * {@link PMUI.field.CheckBoxGroupField#addOption addOption() method}.
     * @cfg {String} [controlPositioning="vertical"] A string that determines the direction for the options to be added
     * in the field. This string can have one of the following values:
     * 
     * - "horizontal" (default), in this case all the options will be included  in horizontal order.
     * - "vertical", in this case all the options will be included in vertical order.
     * @cfg {Number} [maxDirectionOptions=1] The maximum number of options to be added in the current direction 
     * (set by the {@link PMUI.control.CheckBoxGroupField#cfg-controlPositioning controlPositioning config option}).
     *
     * @cfg {String} [value='[]'] The value that determines which options will be checked. the String must have an 
     * array format in which each element is the value of the options that are wanted to be checked.
     */
    var CheckBoxGroupField = function (settings){
        CheckBoxGroupField.superclass.call(this, settings);
        /**
         * @property {String} controlPositioning The direction for the options to be added in the field.
         * @readonly
         */
        this.controlPositioning = null;
        /**
         * @property {Number} maxDirectionOptions The max number of options that can be in the current options 
         * direction.
         * @readonly
         */
        this.maxDirectionOptions = null;
        /**
         * The status if the controls to be saved when the field is switch between enabled/disabled.
         * @type {Object}
         * @private
         */
        this.auxControlStates = {};

        this.controlTable = {};
        CheckBoxGroupField.prototype.init.call(this, settings);
    };

    PMUI.inheritFrom('PMUI.form.Field', CheckBoxGroupField);

    CheckBoxGroupField.prototype.type = "CheckBoxGroupField";

    CheckBoxGroupField.prototype.init = function(settings){
        var defaults = {
            options:[], 
            controlPositioning: "vertical",
            maxDirectionOptions: 1,
            value: '[]'
        };

        jQuery.extend(true, defaults, settings);
        this.setOptions(defaults.options)
            .setMaxDirectionOptions(defaults.maxDirectionOptions)
            .setControlPositioning(defaults.controlPositioning);
    };
    /**
     * Enables the field. Notice that the controls that initially were disabled will continue being disabled.
     * @chainable
     */
    CheckBoxGroupField.prototype.enable = function() {
        var key, i, controlsLength = this.controls.length, controls = this.controls;
        CheckBoxGroupField.superclass.prototype.enable.call(this);
        this.disabled = false;
        for(i = 0; i < controlsLength; i++) {
                controls[i].disable(this.auxControlStates[controls[i].id]);
        }
        return this;
    };
    /**
     * Sets the max number of controls that can be in the current direction (the direction is set by the 
     * CheckBoxGroupField's 
     * {@link PMUI.field.CheckBoxGroupField#setControlPositioning setControlPositioning() method}).
     * @param {Number} max It should be an integer, otherwise it will be floor rounded. If the value is equal or minor 
     * than 0 it means there's no limit.
     */
    CheckBoxGroupField.prototype.setMaxDirectionOptions = function(max) {
        if(typeof max === 'number') {
            this.maxDirectionOptions = Math.floor(max);
            if(this.html) {
                this.setControlPositioning(this.controlPositioning);
            }
        } else {
            throw new Error("setMaxDirectionOptions(): it only accepts number values.");
        }

        return this;
    };
    /**
     * It overrides the Field's {@link PMUI.form.Field#setControlPositioning setControlPositioning() method}.
     * This new implementation sets the direction for the control addition: horizontal or vertical.
     * @param {String} positioning It can be "horizontal" or "vertical".
     */
    CheckBoxGroupField.prototype.setControlPositioning = function(positioning) {
        var errorMessage = "The setControlPositioning() method only accepts \"horizontal\" or \"vertical\" as value.",
            table, tbody, cell, row, i, column, rowIndex;
        if(typeof positioning === 'string') {
            if(!(positioning === 'horizontal' || positioning === 'vertical')) {
                return this;
            }
            this.controlPositioning = positioning;
            if(this.html && this.controls) {
                for(i = 0; i < this.controls.length; i++) {
                    jQuery(this.controls[i].getHTML()).detach();
                }
                $(this.dom.controlContainer).empty();
                table = PMUI.createHTMLElement("table");
                table.className = 'pmui-field-control-table';
                tbody = PMUI.createHTMLElement("tbody");
                if(positioning === 'horizontal') {
                    row = PMUI.createHTMLElement("tr");
                    for(i = 0; i < this.controls.length; i++) {
                        cell = PMUI.createHTMLElement('td');
                        this.controls[i].getHTML();
                        this.controls[i].control.tabIndex = i;
                        cell.appendChild(this.controls[i].getHTML());
                        row.appendChild(cell);
                        if(this.maxDirectionOptions > 0 && (i + 1) % this.maxDirectionOptions === 0) {
                            tbody.appendChild(row);
                            row = PMUI.createHTMLElement("tr");
                        }
                    }
                    tbody.appendChild(row);
                } else {
                    column = 0;
                    for(i = 0; i < this.controls.length; i++) {
                        cell = PMUI.createHTMLElement('td');
                        this.controls[i].getHTML();
                        this.controls[i].control.tabIndex = i;
                        cell.appendChild(this.controls[i].getHTML());
                        rowIndex = this.maxDirectionOptions === 0 ? i : i % this.maxDirectionOptions;

                        row = jQuery(tbody).find('tr').eq(rowIndex).get(0);
                        if(!row) {
                            row = PMUI.createHTMLElement('tr');
                            tbody.appendChild(row);
                        }
                        row.appendChild(cell);
                    }
                }   
                this.controlTable = table;
                table.appendChild(tbody);
                this.dom.controlContainer.appendChild(table);
                this.dom.controlContainer.appendChild(this.helper.getHTML());
            }
        }

        return this;
    };
    /**
     * Sets all the options for the control for checkbox button.
     * @param {Array} options An array in which every element is a JSON object with the 
     same structure required in the 
     * {@link PMUI.field.CheckBoxGroupField#addOption addOption() method}.
     * @chainable
     */
    CheckBoxGroupField.prototype.setOptions = function (options){
        var i = 0;

        this.value = [];
        this.value = JSON.stringify(this.value);

        if(jQuery.isArray(options)){
            for(i = 0; i < options.length; i++) {
                this.addOption(options[i]);          
            }
        } else {
            throw new Error("setOptions(): the supplied argument must be an array.");
        } 
        
        return this;
    };
    /**
     * Adds a new option to the CheckBoxGroupField.
     * @param {Object} option A JSON object, which can have the same properties that the config options 
     * for the {@link PMUI.control.SelectableControl SelectableControl class} except by "mode", 
     it will always be override by the value "checkBox".
     * @chainable
     */
    CheckBoxGroupField.prototype.addOption = function (option) {
       var newOption;

        newOption = new PMUI.control.SelectableControl(jQuery.extend(true, option, {
            mode: 'checkbox', 
            name: this.controls.length + 1, 
            selected: option.selected
        }));

        if(newOption.isSelected()) {
            this.value = JSON.parse(this.value);
            this.value.push(newOption.getValue());
            this.value = JSON.stringify(this.value);
        }
        if(this.eventsDefined) {
            newOption.setOnChangeHandler(this.onChangeHandler()).getHTML();
            newOption.setOnBeforeChange(this.onBeforeChangeHandler());
            newOption.defineEvents();
        }
        this.auxControlStates[newOption.id] = newOption.disabled;
        this.controls.push(newOption);
        this.setControlPositioning(this.controlPositioning);
 
        return this;
    };
    /**
     * Removes one or more options (checkBox Button). 
     * @param  {String|Number|PMUI.control.SelectableControl} option It can be:
     *
     *  - A number, in that case the parameter is used as the index of the item to be removed.
     *  - A String, in that case the parameter is used as the item's value, 
     *  that means that everyone item that has that value will be removed.
     *  - An instance of {@link PMUI.control.SelectableControl SelectableControl} which must be a control of the field.
     * @chainable
     */
    CheckBoxGroupField.prototype.removeOption = function(item) {
        var itemToRemove, i;
        if (item instanceof PMUI.control.SelectableControl) {
            for(i = 0; i < this.controls.length; i++){
                if(this.controls[i] === item){
                    itemToRemove = i;
                    break;
                }
            }
        } else if (typeof item === 'string') {
            for(i = 0; i < this.controls.length; i++){
                if(this.controls[i].id === item){
                    itemToRemove = this.controls[i];
                    break;
                }
            }
        } else {
            itemToRemove = item;
        }
        if (typeof itemToRemove === 'number') {
            itemToRemove = this.controls[itemToRemove];
            delete this.auxControlStates[itemToRemove.id];
            jQuery(itemToRemove.html).detach();
            this.controls.splice(itemToRemove, 1);
            this.setControlPositioning(this.controlPositioning);
        }
        return this;
    };             
    /**
     * It clears all the options from the field.
     * @chainable
     */
    CheckBoxGroupField.prototype.clearOptions = function() {
        while(this.controls.length) {
            this.removeOption(0);
        }

        return this;
    };
    /**
     * Update the field's value property from the controls
     * the parameter value is a clean array;
     * @protected
     * @chainable
     */
    CheckBoxGroupField.prototype.updateValueFromControls = function() {
        var value = [], i;

        for(i = 0; i < this.controls.length; i++) {
            if(this.controls[i].isSelected()) {
                value.push(this.controls[i].getValue());
            }
        }

        this.value = JSON.stringify(value);
        this.data.setValue(this.value);

        return this;
    };
    /**
     * Sets the value to the field's controls.
     * @protected
     * @param {String} value
     * @chainable
     */
    CheckBoxGroupField.prototype.setValueToControls = function(value) {
        var i, j, values, controls, controlsLength;

        try {
            value = value.replace(/'/g,"\"");
            values = JSON.parse(value);
        } catch(e) {
            values = '[]';
        }

        controlsLength = (controls = this.controls.slice(0)).length;
        for(i = 0; i < controlsLength; i += 1){
            controls[i].deselect();
        }
        for(i = 0; i < values.length; i++) {
            for(j = 0; j < controlsLength; j++) {
                if(controls[j].getValue() === values[i]) {
                    controls[j].select();
                    controls.splice(j, 1);
                    controlsLength -= 1;
                    j -= 1;
                }   
            }
        }
        return this;
    };
    /**
     * Evaluates the required validation.
     * @return {Boolean} Returns true if the validation passes otherwise returns false.
     */
    CheckBoxGroupField.prototype.evalRequired = function() {
        var valid = true, value;
        value = JSON.parse(this.getValue());
        if(this.required && value.length === 0) {
            this.showMessage(this.requiredMessage, "error");
            valid = false;
        } else {
            this.hideMessage();
        }
        return valid;
    };  

    CheckBoxGroupField.prototype.isValid = function() {
        var valid = true, validator;
        valid = valid && this.evalRequired();
        if(!valid) {
            $(this.controlTable).addClass('error');
            return valid;
        }
            $(this.controlTable).removeClass('error');
        for(validator in this.validators) {
            if(this.validators.hasOwnProperty(validator)) {
                valid = valid && this.validators[validator].isValid();
                if(!valid) {
                    this.message.setText(this.validators[validator].errorMessage);
                    return valid;
                }
            }
        }
        return valid;
    };  
    /**
     * @inheritdoc
     */
    CheckBoxGroupField.prototype.getValue = function(format) {
        var castFormat = format || this.valueType,
            controls = this.controls, i, res = [];
        for(i = 0; i < controls.length; i += 1) {
            if(controls[i].isSelected()) {
                res.push(PMUI.castValue(controls[i].getValue(), castFormat));
            }
        }
        return JSON.stringify(res);
    };
    /**
     * Disable the option 
     * @param {number|string} disable If the value is part of group CheckBox
     * @chainable
     */
    CheckBoxGroupField.prototype.disableOption = function (value){
        var i;
        for (i=0 ; i<this.controls.length ; i+=1) {
            if (this.controls[i].value == value || this.controls[i].label == value || value == i){
                this.controls[i].disable(true);
                return this;
            }
        }
        throw new Error ('the value send is not part of group CheckBox');
    };
    /**
     * Enable the option 
     * @param {number|string} disable If the value is part of group CheckBox
     * Enables a field's option.
     */    
    CheckBoxGroupField.prototype.enableOption = function (value){
        var i;
        for (i=0 ; i<this.controls.length ; i+=1) {
            if (this.controls[i].value == value || this.controls[i].label == value || value == i){
                this.controls[i].disable(false);
                return this;
            }
        }
        throw new Error ('the value send is not part of group CheckBox');
    };

    CheckBoxGroupField.prototype.createHTML = function() {
        if (this.html){
            return this.html;
        }
        CheckBoxGroupField.superclass.prototype.createHTML.call(this)
        if(this.disabled){
            this.disable();
        }
        return this.html;
    };

    CheckBoxGroupField.prototype.checkedOption = function(option){
        var i;
        for (i = 0 ; i < this.controls.length ; i+=1) {
            if ( option === i || option === this.controls[i].label || option === this.controls[i].value) {
                if (!this.controls[i].selected){
                    this.controls[i].select();
                }else{
                    this.controls[i].deselect();
                }
                return this;
            }
        }
        return this;
    };

    CheckBoxGroupField.prototype.defineEvents = function () {
        var that = this;
        CheckBoxGroupField.superclass.prototype.defineEvents.call(this);
        this.addEvent('click').listen(this.dom.labelTextContainer, function (e){
            $(that.controls[0].html).find('input')[0].focus();
        });
        return this;
    };

    PMUI.extendNamespace('PMUI.field.CheckBoxGroupField',CheckBoxGroupField);

    if(typeof exports !== "undefined"){
        module.exports = CheckBoxGroupField;
    }
}());
(function() {
    /**
     * @class  PMUI.field.RadioButtonGroupField
     * Field whose value can be selected from a group of options.
     * @extends PMUI.form.Field
     * 
     *      Usage example:
     *
     *      @example
     *      var a;
     *      $(function() {
     *          a = new PMUI.field.RadioButtonGroupField({
     *              label: "Some Text",
     *              controlPositioning: 'vertical',
     *              maxDirectionOptions: 2,
     *              options: [
     *                  {
     *                      label: "opt1",
     *                      value: "1"
     *                  },
     *                  {
     *                      label: "opt2",
     *                      value: "2"
     *                  },
     *                  {
     *                      label: "opt3",
     *                      value: "3"
     *                  }
     *              ],
     *              onChange: function(newVal, oldVal) {
     *                  console.log("The value for the field  \"" + this.getLabel() 
     *                      + "\": has change from \"" + oldVal + "\" to \"" + newVal + "\"");
     *              },
     *              required: true,
     *              value: "2"
     *          });
     *          document.body.appendChild(a.getHTML());
     *      });
     *
     *
     * The example above will generate a field with 3 radio buttons.
     *
     * @cfg {Array} [options=[]] An array in which every element is a JSON object with the same structure required in 
     * the {@link PMUI.field.RadioButtonGroupField#addOption addOption() method}.
     * @cfg {String} [controlPositioning="vertical"] A string that determines the direction for the options to be added
     * in the field. This string can have one of the following values:
     * 
     * - "horizontal" (default), in this case all the options will be included  in horizontal order.
     * - "vertical", in this case all the options will be included in vertical order.
     * @cfg {Number} [maxDirectionOptions=1] The maximum number of options to be added in the current direction 
     * (set by the {@link PMUI.control.RadioButtonGroupField#cfg-controlPositioning controlPositioning config option}).
     */
    var RadioButtonGroupField = function(settings) {
        RadioButtonGroupField.superclass.call(this, settings);
        /**
         * @property {String} controlPositioning The direction for the options to be added in the field.
         * @readonly
         */
        this.controlPositioning = null;
        /**
         * @property {Number} maxDirectionOptions The max number of options that can be in the current options 
         * direction.
         * @readonly
         */
        this.maxDirectionOptions = null;
        /**
         * The status if the controls to be saved when the field is switch between enabled/disabled.
         * @type {Object}
         * @private
         */
        this.auxControlStates = {};
        this.controlTable = null;
        RadioButtonGroupField.prototype.init.call(this, settings);
    };

    PMUI.inheritFrom('PMUI.form.Field', RadioButtonGroupField);

    RadioButtonGroupField.prototype.type = "RadioButtonGroupField";

    RadioButtonGroupField.prototype.init = function(settings) {
        var defaults = {
            options: [],
            controlPositioning: "vertical",
            maxDirectionOptions: 1
        };

        jQuery.extend(true, defaults, settings);

        this.setOptions(defaults.options)
            .setMaxDirectionOptions(defaults.maxDirectionOptions)
            .setControlPositioning(defaults.controlPositioning);
    };
    /**
     * Enables the field. Notice that the controls that initially were disabled will continue being disabled.
     * @chainable
     */
    RadioButtonGroupField.prototype.enable = function() {
        var key, i, controlsLength = this.controls.length, controls = this.controls;
        RadioButtonGroupField.superclass.prototype.enable.call(this);
        this.disabled = false;
        for(i = 0; i < controlsLength; i++) {
                controls[i].disable(this.auxControlStates[controls[i].id]);
        }
        return this;
    };
    /**
     * Sets the max number of controls that can be in the current direction (the direction is set by the 
     * RadioButtonGroupField's 
     * {@link PMUI.field.RadioButtonGroupField#setControlPositioning setControlPositioning() method}).
     * @param {Number} max It should be an integer, otherwise it will be floor rounded. If the value is equal or minor 
     * than 0 it means there's no limit.
     */
    RadioButtonGroupField.prototype.setMaxDirectionOptions = function(max) {
        if(typeof max === 'number') {
            this.maxDirectionOptions = Math.floor(max);
            if(this.html) {
                this.setControlPositioning(this.controlPositioning);
            }
        } else {
            throw new Error("setMaxDirectionOptions(): it only accepts number values.");
        }

        return this;
    };
    /**
     * Removes one or more options (radio button) 
     * @param  {String|Number|PMUI.control.SelectableControl} option It can be:
     *
     *  - A number, in that case the parameter is used as the index of the item to be removed.
     *  - A String, in that case the parameter is used as the item's value, 
     *  that means that everyone item that has that value will be removed.
     *  - An instance of {@link PMUI.control.SelectableControl SelectableControl} which must be a control of the field.
     * @chainable
     */
    RadioButtonGroupField.prototype.removeOption = function(item) {
        var itemToRemove, i;
        if (item instanceof PMUI.control.SelectableControl) {
            for(i = 0; i < this.controls.length; i++){
                if(this.controls[i] === item){
                    itemToRemove = i;
                    break;
                }
            }
        } else if (typeof item === 'string') {
            for(i = 0; i < this.controls.length; i++){
                if(this.controls[i].id === item){
                    itemToRemove = this.controls[i];
                    break;
                }
            }
        } else {
            itemToRemove = item;
        }
        if (typeof itemToRemove === 'number') {
            itemToRemove = this.controls[itemToRemove];
            delete this.auxControlStates[itemToRemove.id];
            jQuery(itemToRemove.html).detach();
            this.controls.splice(itemToRemove, 1);
            this.setControlPositioning(this.controlPositioning);
        }
        return this;
    };
    /**
     * It clears all the options from the field.
     * @chainable
     */
    RadioButtonGroupField.prototype.clearOptions = function() {
        while(this.controls.length) {
            this.removeOption(0);
        }

        return this;
    };
    /**
     * It overrides the Field's {@link PMUI.form.Field#setControlPositioning setControlPositioning() method}.
     * This new implementation sets the direction for the control addition: horizontal or vertical.
     * @param {String} positioning It can be "horizontal" or "vertical".
     */
    RadioButtonGroupField.prototype.setControlPositioning = function(positioning) {
        var errorMessage = "The setControlPositioning() method only accepts \"horizontal\" or \"vertical\" as value.",
            table, tbody, cell, row, i, column, rowIndex;
        if(typeof positioning === 'string') {
            if(!(positioning === 'horizontal' || positioning === 'vertical')) {
                return this;
            }
            this.controlPositioning = positioning;
            if(this.html && this.controls) {
                for(i = 0; i < this.controls.length; i++) {
                    jQuery(this.controls[i].getHTML()).detach();
                }
                $(this.dom.controlContainer).empty();
                table = PMUI.createHTMLElement("table");
                table.className = 'pmui-field-control-table';
                tbody = PMUI.createHTMLElement("tbody");
                if(positioning === 'horizontal') {
                    row = PMUI.createHTMLElement("tr");
                    for(i = 0; i < this.controls.length; i++) {
                        cell = PMUI.createHTMLElement('td');
                        this.controls[i].getHTML();
                        this.controls[i].control.tabIndex = i;
                        cell.appendChild(this.controls[i].getHTML());
                        row.appendChild(cell);
                        if(this.maxDirectionOptions > 0 && (i + 1) % this.maxDirectionOptions === 0) {
                            tbody.appendChild(row);
                            row = PMUI.createHTMLElement("tr");
                        }
                    }
                    tbody.appendChild(row);
                } else {
                    column = 0;
                    for(i = 0; i < this.controls.length; i++) {
                        cell = PMUI.createHTMLElement('td');
                        this.controls[i].getHTML();
                        this.controls[i].control.tabIndex = i;
                        cell.appendChild(this.controls[i].getHTML());
                        rowIndex = this.maxDirectionOptions === 0 ? i : i % this.maxDirectionOptions;

                        row = jQuery(tbody).find('tr').eq(rowIndex).get(0);
                        if(!row) {
                            row = PMUI.createHTMLElement('tr');
                            tbody.appendChild(row);
                        }
                        row.appendChild(cell);
                    }
                }
                this.controlTable = table;
                table.appendChild(tbody);
                this.dom.controlContainer.appendChild(table);
                this.dom.controlContainer.appendChild(this.helper.getHTML());
            }
        }

        return this;
    };
    /**
     * Adds a new option to the field.
     * @param {Object} option A JSON object, which can have the same properties that the config options 
     * for the {@link PMUI.control.SelectableControl SelectablecControl class} except by "mode", 
     * it will always be override by the value "radio".
     * @chainable
     */
    RadioButtonGroupField.prototype.addOption = function(option) {
        var newOption,
            settings = {
                mode: 'radio', 
                name: this.name, 
                selected: option.selected
            };

        newOption = new PMUI.control.SelectableControl(jQuery.extend(true, option, settings));
        if(this.eventsDefined) {
            newOption.setOnChangeHandler(this.onChangeHandler()).getHTML();
            newOption.defineEvents();
        }
        this.auxControlStates[newOption.id] = newOption.disabled;
        this.controls.push(newOption);
        this.setControlPositioning(this.controlPositioning);

        return this;
    };
    /**
     * Sets all the options for the control.
     * @param {Array} options An array in which every element is a JSON object with the same structure required in the 
     * {@link PMUI.field.RadioButtonGroupField#addOption addOption() method}.
     * @chainable
     */
    RadioButtonGroupField.prototype.setOptions = function(options) {
        var i = 0;
        if(jQuery.isArray(options)) {
            for(i = 0; i < options.length; i += 1) {
                this.addOption(options[i]);
            }
        } else {
            throw new Error("setOptions(): the supplied argument must be an array.");
        }

        return this;
    };
    /**
     * Update the field's value property from the controls
     * @protected
     * @chainable
     */
    RadioButtonGroupField.prototype.updateValueFromControls = function() {
        var value = '', i;

        for(i = 0; i < this.controls.length; i += 1) {
            if(this.controls[i].isSelected()) {
                value = this.controls[i].getValue();
                break;
            }
        }
        
        this.value = value;
        this.data.setValue(this.value);

        return this;
    };
    /**
     * Sets the value to the field's controls.
     * @protected
     * @param {String} value
     * @chainable
     */
    RadioButtonGroupField.prototype.setValueToControls = function(value) {
        var i;
        for(i = 0; i < this.controls.length; i += 1) {
            if(this.controls[i].getValue() === value) {
                this.controls[i].select();   
            }
        }
        return this;
    };
    /**
     * @inheritdoc
     */
    RadioButtonGroupField.prototype.getValueFromControls = function() {
        var value = '', i;
        for(i = 0; i < this.controls.length; i++) {
            if(this.controls[i].isSelected()) {
                value += ' ' + this.controls[i].getValue();
            }
        }
        return value.substr(1);
    };
    RadioButtonGroupField.prototype.createHTML = function() {
        if (this.html){
            return this.html;
        }
        RadioButtonGroupField.superclass.prototype.createHTML.call(this)
        if(this.disabled){
            this.disable();
        }
        return this.html;
    };

    /**
     * @method selection Option
     * @param  {Number|String} value This is the index or the name of the option that will be selected
     * @chainable
     */
    RadioButtonGroupField.prototype.selectOption = function (value) {
        var i;
        for (i = 0; i<this.controls.length ; i+=1) {
            if (value == this.controls[i].label || value == i){
                this.setValue(this.controls[i].value);
                return this;
            }
        }
        return this;
    };

    RadioButtonGroupField.prototype.disableOption = function (value) {
        var i;
        for (i=0 ; i<this.controls.length ; i+=1) {
            if (this.controls[i].value === value || this.controls[i].label === value || value === i){
                this.controls[i].disable(true);
                return this;
            }
        }
        throw new Error ('the value send is not part of group Radio');
    };

    RadioButtonGroupField.prototype.enableOption = function (value) {
        var i;
        for (i=0 ; i<this.controls.length ; i+=1) {
            if (this.controls[i].value === value || this.controls[i].label === value || value === i){
                this.controls[i].disable(false);
                return this;
            }
        }
        throw new Error ('the value send is not part of group Radio');
    };

    RadioButtonGroupField.prototype.defineEvents = function () {
        var that = this;
        if(this.html && !this.eventsDefined){
            RadioButtonGroupField.superclass.prototype.defineEvents.call(this);
            this.addEvent('click').listen(this.dom.labelTextContainer, function (e){
                $(that.controls[0].html).find('input')[0].focus();
            });
            this.eventsDefined = true;
        }
        return this;
    };
    RadioButtonGroupField.prototype.isValid = function() {
        var valid = true, validator;
        valid = valid && this.evalRequired();
        if(!valid) {
            $(this.controlTable).addClass('error');
            return valid;
        }
            $(this.controlTable).removeClass('error');
        for(validator in this.validators) {
            if(this.validators.hasOwnProperty(validator)) {
                valid = valid && this.validators[validator].isValid();
                if(!valid) {
                    this.message.setText(this.validators[validator].errorMessage);
                    return valid;
                }
            }
        }
        return valid;
    }; 

    PMUI.extendNamespace('PMUI.field.RadioButtonGroupField', RadioButtonGroupField);

    if(typeof exports !== "undefined"){
        module.exports = RadioButtonGroupField;
    }
}());
(function(){
   /**
     * @class  PMUI.field.DateTimeField
     * @extends {PMUI.form.Field}
     * This field is to manipulate the [PMI.control.DateTimeControl]
     * 
     * example:
     *
     *      @example
     *      var dateTimePicker;
     *          $(function() {
     *              dateTimePicker = new PMUI.field.DateTimeField(                   
     *                  {   
     *                      label:'Calendar',
     *                      helper: 'This is calendar Gregorian',
     *                      value: new Date(), 
     *                      datetime : true,
     *                      dateFormat: 'M dd yy',
     *                      minDate: -90,
     *                      maxDate: "+1y -1m -4d",
     *                      firstDay: 1,
     *                      months: {
     *                          "january": {
     *                              name: "Enero",
     *                              shortname: "Ene"
     *                          },
     *                          "february": {
     *                              name: "Febrero",
     *                              shortname: "Feb"
     *                          },
     *                          "march": {
     *                              name: "Marzo",
     *                              shortname: "Mar"
     *                          },
     *                          "april": {
     *                              name: "Abril",
     *                              shortname: "Abr"
     *                          },
     *                          "may": "May",
     *                          "june": "Junio",
     *                          "july": "July",
     *                          "august": "Agosto",
     *                          "september": "Septiembre",
     *                          "october": "Octubre",
     *                          "november": "Noviembre",
     *                          "december": "Diciembre"
     *                      },
     *                      days: {
     *                          "sunday": {
     *                              name: "Domingo",
     *                              shortname: "Do"
     *                          },
     *                          "monday": {
     *                              name: "Lunes",
     *                              shortname: "Lu"
     *                          },
     *                          "tuesday": {
     *                              name: "Martes",
     *                              shortname: "Ma"
     *                          },
     *                          "wednesday": {
     *                              name: "Miércoles",
     *                              shortname: "Mi"
     *                          },
     *                          "thursday": {
     *                              name: "Jueves",
     *                              shortname: "Ju"
     *                          },
     *                          "friday": "Viernes",
     *                          "saturday": "Sábado"
     *                      }
     *                  }
     *              );
     *              document.body.appendChild(dateTimePicker.getHTML());
     *              dateTimePicker.defineEvents();
     *          });
     *          
     * @constructor Creates an instance of the class DateTimeField.
     * @param {Object} options Initialization options.
     * @cfg {Boolean} [datetime=false] If the [PMUI.control.DateTimeControl] You can see more about the configuration
     * in {@link PMUI.control.DateTimeControl#cfg-datetime datetime}
     * @cfg {String} [dateFormat="yy-mm-dd H:i:s"|"yy-mm-dd"],necessary to set the date format property of the control
     * [PMUI.control.DateTimeControl] 
     * You can see more about the configuration in  {@link PMUI.control.DateTimeControl#cfg-dateFormat dateFormat}
     * @cfg {Object} [months={"january": "January", "february": "February", "march": "March", "april": "April", 
     * "may": "May", "june": "June", "july": "July", "august": "August", "september": "September", 
     * "october": "October", "november": "November", "december": "December"}], A JSON object to set the names and
     * shortnames for every month in year.
     * You can see more about the configuration in {@link PMUI.control.DateTimeControl#cfg-months months}
     * @cfg {Object} [days={"sunday": "Sunday","monday": "Monday","tuesday": "Tuesday","wednesday": "Wednesday",
     * "thursday": "Thursday","friday": "Friday","saturday": "Saturday"}], A JSON object to set the name and shortname
     * for every day of week.
     * You can see more about the configuration in {@link PMUI.control.DateTimeControl#cfg-days days}
     * @cfg {String|Number} [minDate=-365] A value which sets the min selectable date for the calendar. You can see
     * more about the configuration in
     * {@link PMUI.control.DateTimeControl#cfg-minDate minDate}
     * @cfg {String|Number} [maxDate=365] A value which sets the max selectable date for the calendar.
     * @cfg {Number} [firstDay=0] Sets the first day of week. You can use numbers from 0 to 6, 0 means Sunday, 1 means
     * Monday and so on.
     * @cfg {String} [returnFormat="UTC"] Specifies the date format that will be used for the returning date.
     * The valid values are:
     *
     * - "UTC", returns a date in the format: yyyy-mm-ddTHH:ii:ss-HH:mm (i.e. 2013-08-31T00:08:00+04:00).
     * - "@" or "timestamp", returns a date in timestamp format.
     * - [any other valid format string], will return the date using the string as the dateformat. This string can 
     * contain any of the wilcards specified in the 
     * {@link PMUI.control.DateTimeControl#cfg-dateFormat dateFormat config option}.
     */
    var DateTimeField = function(settings) {
        DateTimeField.superclass.call(this, settings);
        /**
         * The default date format to be used in the returning date. Set by the {@link #cfg-returnFormat} config option
         * and the {@link #method-setReturnFormat setReturnFormat()}  method.
         * @type {String}
         */
        this.returnFormat = null;
        DateTimeField.prototype.init.call(this, settings);
    };
    
    PMUI.inheritFrom('PMUI.form.Field', DateTimeField);

    DateTimeField.prototype.type = "DateTimeField";
    
    DateTimeField.prototype.init = function (settings) {
        var defaults = {
            datetime : false,
            dateFormat: settings && settings.datetime ? 'yy-mm-dd H:i:s' : 'yy-mm-dd',
            months: {
                "january": "January", 
                "february": "February", 
                "march": "March", 
                "april": "April", 
                "may": "May", 
                "june": "June", 
                "july": "July", 
                "august": "August", 
                "september": "September", 
                "october": "October", 
                "november": "November", 
                "december": "December"
            },
            days: {
                "sunday": "Sunday",
                "monday": "Monday",
                "tuesday": "Tuesday",
                "wednesday": "Wednesday",
                "thursday": "Thursday",
                "friday": "Friday",
                "saturday": "Saturday"
            },
            minDate: -365,
            maxDate: 365,
            firstDay: 0,
            returnFormat: 'UTC'
        }; 

        jQuery.extend(true, defaults, settings);

        this.setDateFormat(defaults.dateFormat)
            .setMonths(defaults.months)
            .setDays(defaults.days)
            .setMinDate(defaults.minDate)
            .setMaxDate(defaults.maxDate)
            .setFirstDay(defaults.firstDay)
            .setReturnFormat(defaults.returnFormat)
            .setValue(defaults.value)
            .visibleDateTime(defaults.datetime);
    };  
    /**
     * Enables/disabled the calendar's [PMUI.control.DateTimeControl] time supporting.
     * @param  {Boolean} visible If it's true, then the time supporting is enabled, otherwise it's disabled.
     * @chainable
     */
    DateTimeField.prototype.visibleDateTime = function(visible) {
        this.controls[0].visibleDateTime(visible);
        return this;
    };
    /**
     * Returns the [PMUI.control.DateTimeControl] maximum selectable date.
     * @param {String} [format="UTC"] The format to applied to the returning date.
     * @return {String} The maximum selectable date in string format.
     */
    DateTimeField.prototype.getMaxDate = function (format) {
        return this.controls[0].getMaxDate(format);
    };
    /**
     * Returns the [PMUI.control.DateTimeControl] minimun selectable date.
     * @param {String} [format="UTC"] The format to applied to the returning date.
     * @return {String} The maximum selectable date in string format.
     */
    DateTimeField.prototype.getMinDate = function (format) {
        return this.controls[0].getMinDate(format);
    };
    /**
     * Sets the date formatr to be used when you get the value from the object.
     * @param {String} format The valid values are:
     *
     * - "UTC", returns a date in the format: yyyy-mm-ddTHH:ii:ss-HH:mm (i.e. 2013-08-31T00:08:00+04:00).
     * - "@" or "timestamp", returns a date in timestamp format.
     * - [any other valid format string], will return the date using the string as the dateformat. This string can 
     * contain any of the wilcards specified in the 
     * {@link PMUI.control.DateTimeControl#cfg-dateFormat dateFormat config option}.
     * @chainable
     */
    DateTimeField.prototype.setReturnFormat = function(format) {
        if(typeof format === 'string') {
            this.returnFormat = format;
        } else {
            throw new Error("setReturnFormat(): the parameter must be a string.");
        }

        return this;
    };
    /**
     * Returns the day index of the first day of week.
     * @return {Number} A number refering a day: 0 for Sunday, 1 for Monday and so on.
     */
    DateTimeField.prototype.getFirstDay = function() {
        return this.controls[0].getFirstDay();
    };
    /**
     * Sets the value to the field's controls.
     * @protected
     * @param {String} value
     * @param {String} utc
     * @chainable
     */
    DateTimeField.prototype.setValueToControls = function(value, utc) {
        this.controls[0].setValue(value, utc);
        return this;
    };
    /**
     * Sets the field's value.
     * @param {String} value
     * @param {String} utc
     */
    DateTimeField.prototype.setValue = function (value, utc) {
        value = value || "";
        if(value instanceof Date || typeof value === 'string' || typeof value === 'number') {
            if(this.controls[0]) {
                this.setValueToControls(value, utc);
                this.value = this.controls[0].getValue(this.returnFormat);    
                this.data.setValue(this.value);    
            }
        } else {
            throw new Error("The setValue() method only accepts string values!");
        }
        return this;
    };
    /**
     * Sets the first day of week.
     * @param {Number} day Use 0 for Sunday, 1 for Monday, 2 for Tuesday and so on!.
     */
    DateTimeField.prototype.setFirstDay = function (day) {
        this.controls[0].setFirstDay(day);
        return this;
    };
    /**
    * Sets the format for the date to be displayed in the control's textbox control. 
    * You can see more about here [PMUI.control.DateTimeControl] in method 
    * {@link PMUI.control.DateTimeControl#setDateFormat setDateFormat}
    **/
    DateTimeField.prototype.setDateFormat = function (dateFormat) {
         this.controls[0].setDateFormat(dateFormat);
        return this;
    };
     /**
     * Sets the minimum date the [PMUI.control.DateTimeControl] can accept as a selectable one.
     * You can see more about here [PMUI.control.DateTimeControl] in method 
     * {@link PMUI.control.DateTimeControl#setMinDate setMinDate}
     */
    DateTimeField.prototype.setMinDate = function (date) {
        this.controls[0].setMinDate(date);
        return this;
    };
     /**
     * Sets the max date the [PMUI.control.DateTimeControl] can accept as a selectable one.
     * You can see more about here in the class [PMUI.control.DateTimeControl] in method
       {@link PMUI.control.DateTimeControl#setMaxDate setMaxDate}
     */
    DateTimeField.prototype.setMaxDate = function (date) {
        this.controls[0].setMaxDate(date);
        return this;
    };
    /**
     * Sets the months the [PMUI.control.DateTimeControl] names/shortnames to be used by the calendar.
     * @param {Object} months An object with the same structure that the [PMUI.control.DateTimeControl]
     * method {@link PMUI.control.DateTimeControl#setMonths setMonths}.
     * @chainable
     */
    DateTimeField.prototype.setMonths = function (months) {
        this.controls[0].setMonths(months);
        return this;
    };
    /**
     * Sets the [PMUI.control.DateTimeControl] name/shortnames for the days to be used in the calendar.
     * @param {Object} days A JSON object with the same structure than the [PMUI.control.DateTimeControl]
     * method {@link PMUI.control.DateTimeControl#setDays setDays}
     */
    DateTimeField.prototype.setDays = function (days) {
        this.controls[0].setDays(days);
        return this;
    };
    /**
     * Returns the formated date.
     * @return {String}
     */
    DateTimeField.prototype.getFormatedDate = function () {
        return this.controls[0].getFormatedDate();
    };
    /**
     * Returns the field's value.
     * @param  {String} [format] Specifies the date format that will be used for the returning date. If it isn't 
     * specified, the {@link #property-returnFormat returnFormat} will be used instead.
     * 
     * The valid values are:
     *
     * - "UTC", returns a date in the format: yyyy-mm-ddTHH:ii:ss-HH:mm (i.e. 2013-08-31T00:08:00+04:00).
     * - "@" or "timestamp", returns a date in timestamp format.
     * - [any other valid format string], will return the date using the string as the dateformat. This string can 
     * contain any of the wilcards specified in the 
     * {@link PMUI.control.DateTimeControl#cfg-dateFormat dateFormat config option}.
     * @return {String}
     */
    DateTimeField.prototype.getValue = function (format) {
        return this.controls[0].getValue(format || this.returnFormat);
    };
    /**
     * Sets the controls for the field.
     * Since this is an abstract method, it must be implemented in its non-abstract subclasses
     * @abstract
     * @private
     */
    DateTimeField.prototype.setControls = function() {
        if(this.controls.length) {
            return this;
        }
        this.controls.push(new PMUI.control.DateTimeControl());
        return this;
    };
    /**
     * Updates the current value from the controls which are part of the field
     * @chainable
     */
    DateTimeField.prototype.updateValueFromControls = function() {
        this.value = this.controls[0].getValue(this.returnFormat);
        this.data.setValue(this.value);
        return this;
    };

    PMUI.extendNamespace('PMUI.field.DateTimeField', DateTimeField);

    if (typeof exports !== "undefined") {
        module.exports = DateTimeField;
    }
}());
(function() {
    /**
     * @class  PMUI.form.FormItemFactory
     * @extend PMUI.util.Factory
     * Extends the factory class to produce objects to be included in containers for forms. 
     * 
     * Its default products are:
     * 
     * - {@link PMUI.form.FormPanel FormPanel} objects: using "panel".
     * - {@link PMUI.field.TextField TextField} objects: using "text".
     * - {@link PMUI.field.DropDownListField DropDownListField} objects: using "dropdown".
     * - {@link PMUI.field.RadioButtonGroupField RadioButtonGroupField} objects: using "radio".
     * - {@link PMUI.field.CheckBoxGroupField CheckBoxGroupField} objects: using "checkbox".
     * - {@link PMUI.field.TextAreaField TextAreaField} objects: using "textarea".
     * - {@link PMUI.field.DateTimeField DateTimeField} objects: using "datetime".
     *
     * @constructor
     * Creates a new instance od the class
     */
    var FormItemFactory = function() {
        FormItemFactory.superclass.call(this);
        FormItemFactory.prototype.init.call(this);
    };

    PMUI.inheritFrom('PMUI.util.Factory', FormItemFactory);

    FormItemFactory.prototype.init = function() {
        var defaults = {
            products: {
                "field" : PMUI.form.Field,
                "panel": PMUI.form.FormPanel,
                "text": PMUI.field.TextField,
                "password": PMUI.field.PasswordField,
                "dropdown": PMUI.field.DropDownListField,
                "radio": PMUI.field.RadioButtonGroupField,
                "checkbox": PMUI.field.CheckBoxGroupField,
                "textarea": PMUI.field.TextAreaField,
                "datetime": PMUI.field.DateTimeField,
                "optionsSelector" : PMUI.field.OptionsSelectorField,
                "upload" : PMUI.field.UploadField,
                "buttonField" : PMUI.field.ButtonField
            }, 
            defaultProduct: "panel"
        };
        this.setProducts(defaults.products)
            .setDefaultProduct(defaults.defaultProduct);
    };

    PMUI.extendNamespace('PMUI.form.FormItemFactory', FormItemFactory);

}());
(function(){
    /**
     * @class PMUI.form.FormPanel
     * @extends PMUI.core.Panel
     * Class to handle form containers for {@link PMUI.form.Field Field objects}, 
     * {@link PMUI.form.FormPanel FormPanel objects}, and {@link PMUI.form.TabPanel TabPanel objects}. 
     *
     *      @example
     *      var p;
     *      $(function() {
     *          p = new PMUI.form.FormPanel({
     *              width: 600, 
     *              height: 130,
     *              fieldset: true,
     *              legend: "my fieldset panel",
     *              items: [
     *                  {
     *                      pmType: "text",
     *                      label: "Name",
     *                      id: "123",
     *                      value: "",
     *                      placeholder: "insert your name",
     *                      name: "name"
     *                  },{
     *                      pmType: "text",
     *                      label: "Last name",
     *                      value: "",
     *                      placeholder: "your lastname here asshole!",
     *                      name: "lastname"
     *                  }, {
     *                      pmType: "panel",
     *                      layout: 'hbox',
     *                      items: [
     *                          {
     *                              pmType: "text",
     *                              label: "E-mail",
     *                              value: "",
     *                              name: "email"
     *                          },{
     *                              pmType: "text",
     *                              label: "Phone",
     *                              value: "555",
     *                              name: "phone"
     *                          }
     *                      ]
     *                  }
     *              ],
     *              layout: "vbox"
     *          });
     *          document.body.appendChild(p.getHTML());
     *          console.log("haha");
     *      });
     *  
     * @cfg {PMUI.form.Form} [form=null] The {@link PMUI.form.Form Form} the object belongs to.
     * @cfg {Array} [items=[]] The array with the items to be contained by the object.
     *      //example
     *      {
     *         ......
     *         items: [
     *             {
     *                 pmType: "text",
     *                 label: "Name",
     *                 id: "123",
     *                 value: "",
     *                 placeholder: "insert your name",
     *                 name: "name"
     *             }, {
     *                 pmType: "text",
     *                 label: "Last name",
     *                 value: "",
     *                 placeholder: "your lastname here asshole!",
     *                 name: "lastname"
     *             }, {
     *                 pmType: "panel",
     *                 layout: 'hbox',
     *                 items: [
     *                     {
     *                         pmType: "text",
     *                         label: "E-mail",
     *                         value: "",
     *                         name: "email"
     *                     },{
     *                         pmType: "text",
     *                         label: "Phone",
     *                         value: "555",
     *                         name: "phone"
     *                     }
     *                 ]
     *             }
     *         ],
     *         ......
     *     });
     * @cfg {Boolean} [fieldset=false] If the panel will apply the fieldset behavior.
     * @cfg {String} [legend=""] The text for the legend to show in case the object applies the fieldset behavior.
     * (only applicable if the the object will apply the fieldset behavior).
     * @cfg {Number} [legendSize=12] The size for the text in the fieldset's legend.
     */
    var FormPanel = function(settings) {
        /**
         * @property {PMUI.util.ArrayList} fields
         * Object that contains all the object's direct children that are {@link PMUI.form.Field Field} objects.
         * @private
         */
        this.fields = new PMUI.util.ArrayList();
        /**
         * @property {PMUI.util.ArrayList} formPanels 
         * Object that contains all the object's direct children that are {@link PMUI.form.FormPanel FormPanel} 
         objects.
         * @private
         */
        this.formPanels = new PMUI.util.ArrayList();
        /**
         * @property {PMUI.util.ArrayList} tabPanels 
         * Object that contains all the object's direct children that are {@link PMUI.form.TabPanel TabPanel} objects.
         * @private
         */
        this.tabPanels = new PMUI.util.ArrayList();

        FormPanel.superclass.call(this, settings);
        /**
         * @property {Boolean} [fieldset] 
         * If the panel has the fieldset behavior.
         * @readonly
         */
        this.fieldset = null;
        /**
         * @property {PMUI.form.Form} [form=null] The {@link PMUI.form.Form Form} the object belongs to.
         */
        this.form = null;
        /**
         * @property {String} [legend=""] The text for the legend to show in case of the fieldset behavior 
         will be applied.
         */
        this.legend = null;
        /**
         * @property {String} [legendElement=""] The HTML element for containt the legend to show in case of the 
         fieldset behavior will be applied.
         */
        this.legendElement = null;
        /**
         * @property {Number} [fieldsetLegendSize] The size for the text showed in the fieldset legend.
         * @readonly
         */
        this.legendSize = null;
        this.onChange = null;

        FormPanel.prototype.init.call(this, settings);
    };

    PMUI.inheritFrom('PMUI.core.Panel', FormPanel);
    
    FormPanel.prototype.type = 'FormPanel';

    FormPanel.prototype.init = function(settings) {
        var defaults = {
            form: null,
            fieldset: false,
            legend: "",
            padding: (settings && settings.fieldset) ? "5px 10px" : "10px",
            borderWidth: (settings && settings.fieldset) ? 1 : "", 
            legendSize: 12,
            onChange: null
        };

        jQuery.extend(true, defaults, settings);

        this.setPadding(defaults.padding)
            .setBorderWidth(defaults.borderWidth)
            .setForm(defaults.form);

        this.form = defaults.form;
        this.fieldset = defaults.fieldset;
        this.legend = defaults.legend;
        this.legendSize = defaults.legendSize;
        this.onChange = defaults.onChange;

        if(this.fieldset) {
            this.setElementTag("fieldset");
        }
    };
    /**
     * Returns a field from the form panel.
     * @param  {String} name The name of the field to find.
     * @return {PMUI.form.Field|null}
     */
    FormPanel.prototype.getField = function(name) {
        var i, items = this.getItems(), field = null;
        for(i = 0; i < items.length; i++) {
            if(items[i] instanceof PMUI.form.Field) {
                if(items[i].getName() === name) {
                    return items[i];
                }
            } else {
                field = items[i].getField(name);
                if(field) {
                    return field;
                }
            }
        }
        return field;
    };
    /**
     * Returns an array of items depending on the parameters the method receives.
     * @param  {String} [filter=undefined]
     * A string which specifiest the bunch of items that will be returned.
     * It defaults to {undefined}, that means that only the direct child items will be in the returned array.
     *
     * Alternatively this param can take one of the following values:
     *
     * - 'fields': it will return only the child {@link PMUI.form.Field Field} items.
     * - 'formPanels': it will return only the child {@link PMUI.form.FormPanel FormPanel} items.
     * - 'tabPanels': it will return only the child {@link PMUI.form.TabPanel TabPanel} items.
     * 
     * @param  {Boolean} [includeChildren=false] 
     *
     * If the value is evaluated as false only the direct child items will be returned, 
     * otherwise additionaly will be added the items for all child items.
     *
     * Note: This parameter only has effect when the [filter] parameter is provided.
     * @return {Array}
     */
    FormPanel.prototype.getItems = function(filter, includeChildren) {
        var res = [], items, size, i, targetArray, the_class;

        switch(filter) {
            case 'fields':
                the_class = PMUI.form.Field;
                targetArray = this.fields;
                break;
            case 'formPanels':
                the_class = PMUI.form.FormPanel;
                targetArray = this.formPanels;
                break;
            case 'tabPanels':
                the_class = PMUI.form.TabPanel;
                targetArray = this.tabPanels;
                break;
            default:
                return FormPanel.superclass.prototype.getItems.call(this);
        }

        if(includeChildren) {
            if(the_class) {
                items = this.items.asArray();
                size = items.length;
                for(i = 0; i < size; i += 1) {
                    if(items[i] instanceof the_class) {
                        res.push(items[i]);
                    } else {
                        res = res.concat(items[i].getItems(filter, true));
                    }
                }
            } else {
                throw new Error('getItems(): The valid values for the "filter" parameter are: "fields",' +
                    '"formPanels" or "tabPanels", received: ' + filter);
            }
        } else {
            res = targetArray.asArray().slice(0);
        }
        return res;
    };
    /**
     * Sets the form the form panel belongs to.
     * @param {PMUI.form.Form} form
     * @chainable
     */
    FormPanel.prototype.setForm = function(form) {
        if(form instanceof PMUI.form.Form) {
            this.form = form;
        }

        return this;
    };
    /**
     * Sets the {@link PMUI.util.Factory Factory} object for the FormPanel
     * @param {PMUI.util.Factory|Object} factory It can be a {@link PMUI.util.Factory Factory} object, 
     or a JSON object.
     * If it is a JSON object then a {@link PMUI.form.FormItemFactory FormItemFactory} 
     * is created using the JSON object as the config options.
     * @chainable
     * @private
     */
    FormPanel.prototype.setFactory = function (factory) {
        if (factory instanceof PMUI.util.Factory){
            this.factory = factory;
        } else {
            this.factory = new PMUI.form.FormItemFactory(factory);
        }
        return this;
    };
    /**
     * Clear all the items.
     * @chainable
     */
    FormPanel.prototype.clearItems = function() {
        FormPanel.superclass.prototype.clearItems.call(this);
        if(this.fields) {
            this.fields.clear();
            this.formPanels.clear();
            this.tabPanels.clear();
        }
        return this;
    };
    /**
     * Sets the text for the legend to be displayed just in case of the fieldset behavior is beign applied.
     * @param {String} legend
     * @chainable
     */
    FormPanel.prototype.setLegend = function(legend) {
        if(typeof legend === 'string') {
            this.legend = legend;
            if(this.legendElement) {
                this.legendElement.textContent = legend;
            }
        } else {
            throw new Error("setLegend(): this method accepts string values as only parameter.");
        }

        return this;
    };
    /**
     * Creates the HTML element for the FormPanel.
     * @chainable
     */
    FormPanel.prototype.createHTML = function() {
        var legendElement, html;
        if(this.html) {
            return this.html;
        }

        FormPanel.superclass.prototype.createHTML.call(this);

        if(this.fieldset) {
            legendElement = PMUI.createHTMLElement('legend');
            legendElement.textContent = this.legend;
            legendElement.className = 'pmui-formpanel-legend';
            legendElement.style.fontSize = this.legendSize + "px";
            this.legendElement = legendElement;
            jQuery(this.html).prepend(legendElement);
        }
        return this.html;
    };
    /**
     * Returns the object's usable height.
     * @return {Number|String}
     */
    FormPanel.prototype.getUsableHeight = function() {
        var height = FormPanel.superclass.prototype.getUsableHeight.call(this);
        if(isNaN(height)) {
            return height;
        }
        if(this.fieldset) {
            height -= this.legendSize;
        }

        return height;
    };
    /**
     * Sets the internal handler function for the onChange Event
     * @return {Function} The handler function.
     * @private
     */
    FormPanel.prototype.onChangeHandler = function() {
        var that = this;
        return function(a, b, c) {
            var target, newValue, previousValue;
            if(this instanceof PMUI.form.Field) {
                target = this;
                newValue = a;
                previousValue = b;
            } else {
                target = a;
                newValue = b;
                previousValue = c;
            }
            if(typeof that.onChange === 'function') {
                that.onChange(target, newValue, previousValue);
            }
            if(this instanceof PMUI.form.Field && this.form) {
                (this.form.onChangeHandler())(target, newValue, previousValue);
            }
        };
    };
    /**
     * Add an item to the object.
     * @param {Object|PMUI.form.Field|PMUI.form.FormPanel|PMUI.form.TabPanel} item 
     * The item parameter can be:
     *
     * - A JSON object, in that case, it must have at least a "pmType" property 
     * which can have any of the values specifed in the {@link PMUI.form.FormItemFactory FormItemFactory 
     documentation}.
     * - A {@link PMUI.form.Field Field} object.
     * - A {@link PMUI.form.FormPanel FormPanel} object.
     * - A {@link PMUI.form.TabPanel TabPanel} object.
     */
    FormPanel.prototype.addItem = function(item, index) {
        var itemToBeAdded;
        if(this.factory) {
            if(typeof item === 'object' && !item.layout) {
                item.layout = (this.layout && this.layout.type.toLowerCase()) || "vbox";
            }
            if(this.factory.isValidClass(item) || this.factory.isValidName(item.pmType)) {
                itemToBeAdded = this.factory.make(item);
            } else {
                throw new Error('Invalid item to add.');
            }
        }
        if(itemToBeAdded) {
            //itemToBeAdded.onChange = this.onChangeHandler();
            itemToBeAdded.setForm(this.form);
            itemToBeAdded.setDisplay("inline-block");
            if(itemToBeAdded instanceof PMUI.form.FormPanel) {
                this.formPanels.insert(itemToBeAdded);
            } else if(itemToBeAdded instanceof PMUI.form.Field) {
                this.fields.insert(itemToBeAdded);
                if(this.form) {
                    this.form.data.addItem(itemToBeAdded.data);
                }
            }
            FormPanel.superclass.prototype.addItem.call(this, itemToBeAdded, index);
        }

        return this;
    };
    /**
     * Removes an item from the form panel.
     * @param  {String|Number|PMUI.core.Element} item 
     * It can be:
     * - a String: in this case this string must be the id of the element to be removed.
     * - a Number: in this case this number must be the index of the element to be removed.
     * - a {@link PMUI.core.Element Element} object: in this case it will be removed 
     * (if it is contained by the current form panel).
     * @chainable
     */
    FormPanel.prototype.removeItem = function(item) {
        var itemToRemove;
        if (item instanceof PMUI.core.Element) {
            itemToRemove = item;
        } else {
            if (typeof item === 'string') {
                itemToRemove = this.items.find("id", item.id);
            } else if (typeof item === 'number') {
                itemToRemove = this.items.get(item);
            }
        }
        if(itemToRemove) {
            if(itemToRemove instanceof PMUI.form.Field) {
                if(this.form) {
                    this.form.data.removeItem(itemToRemove.data);
                }
            }
            FormPanel.superclass.prototype.removeItem.call(this, itemToRemove);
        }

        return this;
    };
    /**
     * Returns true if the validation passes otherwise it returns false.
     * @return {Boolean}
     */
    FormPanel.prototype.isValid = function() {
        var items = this.items.asArray(), i, valid = true;

        for(i = 0; i < items.length; i++) {
            if(!(items[i] instanceof PMUI.form.Field) || !items[i].disabled) {
                valid = valid && items[i].isValid();
            } else {
                continue;
            }
            if(!valid && items[i] instanceof PMUI.form.Field) {
                items[i].setFocus();
            }
            if(!valid) {
                return valid;
            } 
        }

        return valid;
    };

    PMUI.extendNamespace('PMUI.form.FormPanel', FormPanel);

    if (typeof exports !== "undefined") {
        module.exports = FormPanel;
    }
}());
(function() {
    /**
     * @class PMUI.form.Form
     * @extends PMUI.core.Panel
     *
     *  Usage example:
     *
     *      @example
     *      var f;
     *      $(function() {
     *          f = new PMUI.form.Form({
     *              onSubmit: function() {
     *                  console.log("submitting...");
     *              },
     *              onChange: function(field, newValue, previousValue) {
     *                  console.log("The field " + field.getName() + " has changed from \"" + previousValue + "\" to \"" + newValue + "\"");
     *              },
     *              items: [
     *                  {
     *                      pmType: "text",
     *                      label: "Name",
     *                      id: "123",
     *                      value: "",
     *                      placeholder: "insert your name",
     *                      name: "name",
     *                      helper: "Introduce your name",
     *                      required : true,
     *                      controlsWidth: 200
     *                  }, 
     *                  {
     *                      pmType: "datetime",
     *                      visible : false,
     *                      label: "birth date",
     *                      name: "birthdate"
     *                  },
     *                  {
     *                      pmType: "dropdown",
     *                      label: "age",
     *                      options: [
     *                          {
     *                              label: "",
     *                              value: ""
     *                          },
     *                          {
     *                              label: "from 0 to 7 years old",
     *                              value: "0-7"
     *                          },
     *                          {
     *                              label: "from 8 to 13 years old",
     *                              value: "8-13"
     *                          },
     *                          {
     *                              label: "from 14 to 19 years old",
     *                              value: "14-19"
     *                          },
     *                          {
     *                              label: "from 20 to 30 years old",
     *                              value: "20-30"
     *                          },
     *                          {
     *                              label: "from 31 to 45 years old",
     *                              value: "31-45"
     *                          },
     *                          {
     *                              label: "older than 45",
     *                              value: "46+"
     *                          }
     *                      ],
     *                      name: "age",
     *                      helper: "Select one of the options",
     *                      required: true,
     *                      onChange: function() {
     *                          console.log("The value has been changed");
     *                      },
     *                      controlsWidth: 100
     *                  },
     *                  {
     *                      pmType: "radio",
     *                      label: "Gender",
     *                      value: "m",
     *                      name: "gender",
     *                      required: true,
     *                      options: [
     *                          {
     *                              label: "Male",
     *                              value: "m"
     *                          },
     *                          {
     *                              label: "Female",
     *                              value: "f"
     *                          }
     *                      ]
     *                  }, 
     *                  {
     *                      pmType: "text",
     *                      label: "E-mail",
     *                      value: "",
     *                      name: "email",
     *                      helper: "you email here",
     *                      required: true,
     *                      validators: [
     *                          {
     *                              pmType: "regexp",
     *                              criteria: /^(([^<>()[\]\\.,;:\s@\"]+(\.[^<>()[\]\\.,;:\s@\"]+)*)|(\".+\"))@((\[[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\])|(([a-zA-Z\-0-9]+\.)+[a-zA-Z]{2,}))$/,
     *                              errorMessage: "Please enter a valid email address"
     *                          }
     *                      ]
     *                  }, 
     *                  {
     *                      pmType: "checkbox",
     *                      label: "Things you like to do",
     *                      value: "[\"music\"]",
     *                      name: "hobbies",
     *                      helper: "Check up to 3 options",
     *                      options: [
     *                          {
     *                              label: "Music",
     *                              value: "music"
     *                          },
     *                          {
     *                              label: "Programming",
     *                              value: "programming"
     *                          },
     *                          {
     *                              label: "Bike ridding",
     *                              value: "bike"
     *                          },
     *                          {
     *                              label: "Gastronomy",
     *                              value: "gastronomy"
     *                          },
     *                          {
     *                              label: "Movies",
     *                              value: "movies"
     *                          }
     *                      ],
     *                      validators: [],
     *                      required: true
     *                  },
     *                  {
     *                      pmType: "panel",
     *                      fieldset: true,
     *                      legend: "Another Info",
     *                      layout: "vbox",
     *                      items: [
     *                          {
     *                              pmType: "panel",
     *                              layout: 'hbox',
     *                              items: [
     *                                  {
     *                                      pmType: "text",
     *                                      label: "Country",
     *                                      value: "Bolivia",
     *                                      name: "country"
     *                                  },
     *                                  {
     *                                      pmType: "text",
     *                                      label: "City",
     *                                      name: "city"
     *                                  }
     *                              ]
     *                          },
     *                          {
     *                              pmType: "panel",
     *                              layout: 'hbox',
     *                              items: [
     *                                  {
     *                                      pmType: "text",
     *                                      label: "Address",
     *                                      name: "address",
     *                                      placeholder: "Your address here"
     *                                  },
     *                                  {
     *                                      pmType: "text",
     *                                      label: "Zip Code",
     *                                      name: "zip"
     *                                  }
     *                              ]
     *                          }, 
     *                          {
     *                              pmType: "panel",
     *                              layout: 'vbox',
     *                              items: [
     *                                  {
     *                                      pmType: "textarea",
     *                                      label: "About you",
     *                                      name: "about"
     *                                  }
     *                              ]
     *                          }
     *                      ]
     *                  }
     *              ],
     *              footerItems: [
     *                  {
     *                      pmType: "button",
     *                      text: "Submit",
     *                      handler: function() {
     *                          f.submit();
     *                          console.log("submitting form...");
     *                      }
     *                  }, {
     *                      pmType: "button",
     *                      text: "Reset",
     *                      handler: function() {
     *                          f.reset();
     *                      }
     *                  }
     *              ]
     *          });
     *          document.body.appendChild(f.getHTML());
     *      });
     *
     * The example above will generate a form with 8 fields: 
     * Name, Last Name, E-mail, Phone, Country, City, Address, Zip Code. 
     * The first four ones are required and each one has a helper.
     * The email and Phone fields also have a validator for control the input.
     *
     * The latest four ones are contained by a fieldset.
     *
     * @cfg {String} title The title for the form.
     * @cfg {name} [name=<the object's id>] The name for the form.
     * @cfg {String} [encType="application/x-www-form-urlencoded"] The value for the form's encType attribute .
     * @cfg {Array} [items=[]] An array in which every element can be 
     * a JSON Object or a supported object by Form:
     *
     * - In case to be a JSON object, it should includes a "pmType" property with a valid value. 
     * The valid values for the pmType property are specified in the 
     * {@link PMUI.form.FormItemFactory FormItemFactory documentation}.
     *
     * - In case of objects, use an object that the {@link PMUI.form.FormItemFactory FormItemFactory} can produce.
     *
     * Example:
     *     var form, items;
     *     
     *     items = [
     *          {
     *              pmType: "text",
     *              label: "Name",
     *              id: "123",
     *              value: "",
     *              placeholder: "insert your name",
     *              name: "name",
     *              helper: "Introduce your name",
     *              required: true
     *          }, 
     *          {
     *              pmType: "text",
     *              label: "Last name",
     *              value: "",
     *              placeholder: "your lastname here asshole!",
     *              name: "lastname",
     *              helper: "Introduce your lastname",
     *              required: true
     *          }
     *      ];
     *
     *     form = new PMUI.form.Form({
     *         name: "My form",
     *         items: items
     *     });
     *     
     * Depending on the type of element you add, it can contain a nested items property to define its child items:
     *
     *      f = new PMUI.form.Form({
     *          items: [
     *              .......
     *              {
     *                  pmType: "panel",
     *                  fieldset: true,
     *                  legend: "Another Info",
     *                  layout: "vbox",
     *                  items: [
     *                      {
     *                          pmType: "panel",
     *                          layout: 'hbox',
     *                          items: [
     *                              {
     *                                  pmType: "text",
     *                                  label: "Country",
     *                                  value: "Bolivia",
     *                                  name: "country"
     *                              },
     *                              {
     *                                  pmType: "text",
     *                                  label: "City",
     *                                  name: "city"
     *                              }
     *                          ]
     *                      },
     *                      {
     *                          pmType: "panel",
     *                          layout: 'hbox',
     *                          items: [
     *                              {
     *                                  pmType: "text",
     *                                  label: "Address",
     *                                  name: "address",
     *                                  placeholder: "Your address here"
     *                              },
     *                              {
     *                                  pmType: "text",
     *                                  label: "Zip Code",
     *                                  name: "zip"
     *                              }
     *                          ]
     *                      }
     *                  ]
     *              }
     *          ]
     *      }); 
     *
     * As you can see in the example above, only the container type objects can have an "item" property.
     * 
     * @cfg {Number} [fontSize=12] The size to be use for the font in the form.
     * @cfg {Number} [width=600] The width for the form.
     * In case of using a String you only can use 'auto' or 'inherit' or ##px or ##% or ##em when ## is a number.
     * @cfg {Number} [height=400] The height for the form.
     * In case of using a String you only can use 'auto' or 'inherit' or ##px or ##% or ##em when ## is a number.
     * @cfg {Boolean} [visibleHeader=true] If the header is visible or not.
     * @cfg {String} [layout='vbox'] The layout type to apply to the form.
     * @cfg {Array} [buttons=[]] An array in which each element is JSON Object or a 
     * {@link Form.ui.Button Button object}
     * @deprecated This config option will be removed soon, please use the {@link #cfg-footerItems footerItems} config 
     * option instead.
     * @cfg {Function} [onChange=null] A callback function to be called everytime a form's field changes.
     * This callback function will be called in the current object context and will receive three parameters:
     * 
     * - The field that changed.
     * - The new field's value.
     * - The previous field's value.
     *
     *      var f = new PMUI.form.Form({
     *          onChange: function(field, newValue, previousValue) {
     *              console.log("Action in the form named \"" + this.getName() + "\":");
     *              console.log("The field " + field.getName() + " has changed from \"" + previousValue + 
     "\" to \"" + newValue + "\"");
     *          },
     *          ......
     *      };
     * 
     * @cfg {Function} [onSubmit=null] A callback function to be called when the form is submitted.
     * This callback function will be called in the form object context.
     *
     *      var f = new PMUI.form.Form({
     *          onSubmit: function() {
     *              console.log("Submitting the form named \"" + this.getName() + "\"...");
     *          },
     *          ......
     *      };
     * @cfg {Array} [footerItems=[]] Sets the elements in the window footer, this elements can be instances of Button 
     * and/or instances of Label. The value for this config option must be an array in which each element can be:
     *
     * - An object literal, in this case the object literal must have the property "pmType" with its value set to 
     * "button" (if you want the element be a {@link PMUI.ui.Button Button}) or "label" (if you want the element be a 
     * {@link PMUI.ui.TextLabel Label}). Optionally you can add the respective config options for each case.
     *
     * - A PMUI object, in this case it must be an instance of {@link PMUI.ui.Button Button} or an instance of
     * {@link PMUI.ui.TextLabel Label}.
     * @cfg {Function} [onLoad=null] The callback function to be executed when the form is ready to use. for info about
     * the parameters received by the callback please read the {@link #event-onLoad onLoad event} documentation.
     * @cfg {Srting} [buttonPanelPosition='bottom'] The position for the form's footer. It can be 'bottom' or 'top'.
     */
    var Form = function(settings) {
        Form.superclass.call(this, settings);
        /**
         * The name for the form
         * @type {String}
         * @readonly
         */
        this.name = null;
        /**
         * The form's encType property
         * @type {String}
         * @readonly
         */
        this.encType = null;
        /**
         * The text for the form's title.
         * @type {String}
         * @readonly
         */
        this.title = null;
        /**
         * The size for the font in the form.
         * @type {Number}
         * @readonly
         */
        this.fontSize = null;
        /**
         * If the header is visible or not.
         * @type {Boolean}
         * @readonly
         */
        this.visibleHeader = null;
        /**
         * The callback function to be called when a form's field changes.
         * @type {Function}
         */
        this.onChange = null;
        /**
         * The callback functionto be called when the form is submitted.
         * @type {Function}
         */
        this.onSubmit = null;
        /**
         * The form's data object.
         * @type {PMUI.data.DataSet}
         * @private
         */
        this.data = null;
        /**
         * The DOM object that plays the role of title container in the form.
         * @type {HTMLElement}
         * @private
         */
        this.header = null;
        /**
         * The form footer.
         * @type {PMUI.panel.ButtonPanel}
         * @private
         */
        this.footer = null;
        /**
         * The height for the footer area in the Form
         * @type {Number}
         * @readonly
         */
        this.footerHeight = null;
        /**
         * @event onLoad
         * Fired when the form is ready to use.
         * @param {PMUI.form.Form} form The form.
         */
        this.onLoad = null;
        /**
         * If the form is dirty.
         * @type {Boolean}
         */
        this.dirty = null;
        /**
         * The position for the form's footer.
         * @type {String}
         */
        this.buttonPanelPosition = null;
        this.dependencies = {};
        this.alignmentButtons = null;
        Form.prototype.init.call(this, settings);
    };

    PMUI.inheritFrom('PMUI.core.Panel', Form);

    Form.prototype.type = 'PMForm';

    Form.prototype.family = 'PMForm';

    Form.prototype.init = function(settings) {
        var defaults = {
            title: "Untitled form",
            name: this.id,
            encType: "application/x-www-form-urlencoded",
            items: [],
            fontSize: 12,
            width: 600,
            height: 400,
            visibleHeader: true,
            layout: 'vbox',
            buttons: [],
            footerItems: [],
            buttonPanelPosition: 'bottom',
            onChange: null,
            onSubmit: null,
            onLoad: null,
            alignmentButtons:'right'
        };

        $.extend(true, defaults, settings);

        this.panel = new PMUI.form.FormPanel({
           form: this,
           width: defaults.width,
           layout: defaults.layout,
           style: {
                cssProperties: {
                     "overflow": "auto"
                }
           },
           onChange: this.onChangeHandler()
        });

        this.footer = new PMUI.panel.ButtonPanel({
            style: {
                cssClasses: ['pmui-form-footer']
            }
        });

        this.data = new PMUI.data.DataSet();

        this.buttons = new PMUI.util.ArrayList();

        this.visibleHeader = defaults.visibleHeader;
        this.onChange = defaults.onChange;
        this.onSubmit = defaults.onSubmit;

        this.setLayout(defaults.layout)
           .setFontSize(defaults.fontSize)
           .setWidth(defaults.width)
           .setTitle(defaults.title)
           .setName(defaults.name)
           .setEncType(defaults.encType)
           .setItems(defaults.items)
           .setOnLoadHandler(defaults.onLoad)
           .setButtonPanelPosition(defaults.buttonPanelPosition)
           .setAlignmentButtons(defaults.alignmentButtons);

        //TODO tell the guys about the deprecated methods
        if(defaults.footerItems && defaults.footerItems.length) {
           this.setFooterItems(defaults.footerItems);
        } else {
           this.setButtons(defaults.buttons);
        }
        if(typeof this.onLoad === 'function') {
           this.onLoad(this);
        }
    };

    Form.prototype.updateDependencies = function() {
        var dependencies = {}, i, j, fields = this.getFields(), dependent, dependents;

        for(i = 0; i < fields.length; i += 1) {
            dependents = fields[i].dependentFields;
            for(j = 0; j < dependents.length; j += 1) {
                if(this.getField(dependents[j])) {
                    if(!dependencies[dependents[j]]) {
                        dependencies[dependents[j]] = [];
                    }
                    dependencies[dependents[j]].push(fields[i]);
                }
            }
        }
        this.dependencies = dependencies;
        return this;
    };


    /**
     * Sets the position for the form's footer.
     * @param {String} position A string with the text 'bottom' or 'top', this sets the position for the form's footer.
     */
    Form.prototype.setButtonPanelPosition = function (position) {
        if(position === 'top' || position === 'bottom') {
            this.buttonPanelPosition = position;
            if(this.html) {
                if(position === 'top') {
                    this.html.insertBefore(this.footer.html, this.panel.html);       
                } else {
                    this.html.appendChild(this.footer.html);
                }
                
            }
        }
        return this;
    };
    /**
     * Sets the callback to be executed when the {@link #event-onLoad onLoad event} is fired.
     * @param {Function|null} handler The callback function or the null constant, in the latter case no callback 
     * function will be executed.
     * @chainable
     */
    Form.prototype.setOnLoadHandler = function (handler) {
         if(!(typeof handler === 'function' || handler === null)) {
               throw new Error('setOnLoadHandler(): the parameter must be a function or null.');
         }
         this.onLoad = handler;
         return this;
    }
    /**
     * Removes an iten from the form's footer.
     * @param  {PMUI.core.Element|String|Number} item It can be a string (id of the child to remove), 
     a number (index of the child to remove) or a {Element} object.
     * @chainable
     */
    Form.prototype.removeFooterItem = function(item) {
        this.footer.removeItem(item);
        return this;
    };
    /**
     * Removes all the items in the form's footer.
     * @chainable
     */
    Form.prototype.clearFooterItems = function() {
        this.footer.clearItems();
        return this;
    };
    /**
     * Adds an item into the forms footer.
     * @param {Object|PMUI.ui.Button|PMUI.ui.Button} item It can be:
     * 
     * - An object literal, in this case it can have the config options for create a {@link PMUI.ui.Button Button} or a 
     * {@link PMUI.ui.TextLabel Label}, additionally it must include the respective pmType ('button' for Button and 'label'
     * for Label).
     *
     * -A PMUI object, in this case it must be an instance of {@link PMUI.ui.Button Button} or an instance of {@link 
     * PMUI.ui.TextLabel Label}.
     */
    Form.prototype.addFooterItem = function(item) {
        this.footer.addItem(item);
        this.showFooter();
        return this;
    };
    /**
     * Sets the items to be shown in the form's footer.
     * @param {Array} items An array in which each element can be:
     *
     * - An object literal, in this case it can have the config options for create a {@link PMUI.ui.Button Button} or a 
     * {@link PMUI.ui.TextLabel Label}, additionally it must include the respective pmType ('button' for Button and 'label'
     * for Label).
     *
     * -A PMUI object, in this case it must be an instance of {@link PMUI.ui.Button Button} or an instance of {@link 
     * PMUI.ui.TextLabel Label}.
     */
    Form.prototype.setFooterItems = function(items) {
        var i;
        if(!jQuery.isArray(items)) {
            throw new Error('setFooterItems(): The parameter must be an array.');
        }
        this.clearFooterItems();
        this.hideFooter();
        for(i = 0; i < items.length; i++) {
            this.addFooterItem(items[i]);
            this.footer.getItem(i).setHeight(31);
        }

        return this;
    };
    /**
     * The method which sets the internal callback function for the onChange event.
     * @return {Function}
     * @private
     */
    Form.prototype.onChangeHandler = function() {
        var that = this;
        return function(field, newValue, previousValue) {
            if(field.initialValue === newValue){
                that.dirty = false;
            }else{
                that.dirty = true; 
            }
            if(typeof that.onChange === 'function') {
                that.onChange(field, newValue, previousValue);
            }
        };
    };
    /**
     * Sets the form to dirty/clean.
     * @param {Boolean} dirty True to set the form to dirty, False to set it to clean.
     * @chainable
     */
    Form.prototype.setDirty = function(dirty) {
        this.dirty = !!dirty;
        return this;
    };
    /**
     * Check if the form is dirty
     * @return {Boolean} Returns true if the for is dirty, otherwise it returns false.
     */
    Form.prototype.isDirty = function () {
        return this.dirty;
    };
    /**
     * Removes any button previously added to the form.
     * @param  {String|Number|PMUI.ui.Button} button
     * This param can be:
     *  - String: in that case the string must be the id of the button to remove.
     *  - Number: in that case the number is evaluated as the index of the button to remove.
     *  - Button: the button to remove.
     * @chainable
     * @deprecated This method will be removed soon, please use the {@link #method-removeFooterItem removeFooterItem()}
     * method instead.
     */
    Form.prototype.removeButton = function(button) {
        return this.removeFooterItem(button);
    };
    /**
     * Remove all the buttons from the form.
     * @chainable
     * @deprecated This method will be removed soon, please use the {@link #method-clearFooterItems clearFooterItems()}
     * method instead.
     */
    Form.prototype.clearButtons = function() {
        return this.clearFooterItems();
    };
    /**
     * Adds a button into the form. The button will be added to the form's footer.
     * @param {Object|PMUI.ui.Button} button
     * It can be a:
     *
     * - {@link PMUI.ui.Button Button} object.
     * - JSON Object: use the same JSON structure for create {@link PMUI.ui.Button Button} objects.
     *
     * @chainable
     * @deprecated This method will be removed soon, please use the {@link #method-addFooterItem addFooterItem()}
     * method instead.
     */
    Form.prototype.addButton = function(button) {
        return this.addFooterItem(button);
    };
    /**
     * Sets the buttons for the form. They will be added to the footer.
     * @param {Array} buttons An array in which each element is a JSON object or a {@link PMUI.ui.Button Button}.
     * @chainable
     * @deprecated This method will be removed soon, please use the {@link #method-setFooterItems setFooterItems()}
     * method instead.
     */
    Form.prototype.setButtons = function(buttons) {
        return this.setFooterItems(buttons);
    };
    /**
     * Updates the dimensions for the form's areas.
     * @chainable
     */
    Form.prototype.updateDimensions = function() {
        var headerHeight = 0, footerHeight = 0, bodyHeight;
        if(!this.panel) {
            return this;
        }
        if(this.getHeight() !== 'auto') {
            if(this.visibleHeader && this.title) {
                headerHeight = 2 * this.fontSize;
            }
            if(this.footer.getItems().length){
                footerHeight = (1.7 * this.fontSize) + 12;
            }
            bodyHeight =  this.getHeight() - headerHeight - footerHeight;
        }
        return this;
    };
    /**
     * Sets the form's height.
     * @param {Number|String} height it can be a number or a string.
     * In case of using a String you only can use 'auto' or 'inherit' or ##px or ##% or ##em when ## is a number
     * @chainable
     */
    Form.prototype.setHeight = function(height) {
        Form.superclass.prototype.setHeight.call(this, height);
        this.updateDimensions();
        return this;
    };
    /**
     * Sets the form's width.
     * @param {Number|String} width it can be a number or a string.
     * In case of using a String you only can use 'auto' or 'inherit' or ##px or ##% or ##em when ## is a number
     * @chainable
     */
    Form.prototype.setWidth = function (width) {
        Form.superclass.prototype.setWidth.call(this, width);
        this.updateDimensions();
        return this;
    };
    /**
     * Sets the size for the font toi be used in the form.
     * @param {Number} fontSize
     */
    Form.prototype.setFontSize = function(fontSize) {
        if(typeof fontSize === 'number') {
            this.fontSize = fontSize;
            this.style.addProperties({
                "font-size": fontSize + "px"
            });
            this.updateDimensions();
        } else {
            throw new Error("setFontSize(): this method only accepts a number as parameter.");
        }

        return this;
    };
    /**
     * Returns a field from the form.
     * @param  {String} name The name of the field to find.
     * @return {PMUI.form.Field|null}
     */
    Form.prototype.getField = function(name) {
          return this.panel.getField(name);
    };
    /**
     * Returns all the fields contained by the form.
     * @return {Array} An array in which each element is a {PMUI.form.Field}.
     */
    Form.prototype.getFields = function() {
        return this.panel.getItems("fields", true);
    };
    /**
     * Returns the items from the form's body.
     * @return {Array} An array with all the form's items.
     */
    Form.prototype.getItems = function() {
        return this.panel.getItems();
    };
    /**
     * Sets the layout mode for the form's root panel.
     * @param {String} layout it can take one of the following values:
     *
     * - "vbox" for vertical positioning.
     * - "hbox" for horizontal positioning.
     * - "box" without layout.
     */
    Form.prototype.setLayout = function(layout) {
        var factory;
        if(this.panel) {
            factory = new PMUI.layout.LayoutFactory();
            this.panel.layout = factory.make(layout);
            this.panel.layout.setContainer(this.panel);
            if (this.html) {
                this.panel.layout.applyLayout();
            }
        }
        return this;
    };
    /**
     * Adds an item to the form's root panel. It only creates one of 
     * the {@link PMUI.form.FormItemFactory FormItemFactory} supported products.
     * @param {String|Object|PMUI.form.FormPanel|PMUI.form.Field} item 
     * It can accept one of the following data types:
     *
     * - a String: one of the supported pmTypes by the {@link PMUI.form.FormItemFactory FormItemFactory} object.
     * - a JSON Object: use the JSON structure the object's constructor needs, additionally add the pmType property,
     * you can use any of the pmTypes supported by the {@link PMUI.form.FormItemFactory FormItemFactory}.
     * - a {@link PMUI.form.FormPanel FormPanel} object.
     * - a{@link PMUI.form.Field Field} object
     */
    Form.prototype.addItem = function(item) {
        if(this.panel) {
            this.panel.addItem(item);
            this.updateDependencies();
        }
        return this;
    };
    /**
     * Set items for the form's root panel.
     * @param {Array} items An array in which each element can be one of the accepted parameters 
     * in the {@link PMUI.form.Form#addItem addItem() method}.
     * @chainable
     */
    Form.prototype.setItems = function(items) {
        if(this.panel) {
            this.panel.setItems(items);
            this.updateDependencies();
        }
        return this;
    };
    /**
     * Sets the title text for the form.
     * @param {String} title
     * @chainable
     */
    Form.prototype.setTitle = function(title) {
        if(typeof title === 'string') {
            this.title =  title;
            if(this.header) {
                $(this.header).empty();
                if(title) {
                    $(this.header).append('<h2 class="pmui-form-title"></h2>').find('h2').text(title);
                } else {
                    $(this.header).append('<h2 class="pmui-form-title"></h2>').find('h2').html("&nbsp;");
                }
                this.updateDimensions();
            }
        }

        return this;
    };
    /**
     * Sets the name for the form
     * @param {String} name 
     * @chainable
     */
    Form.prototype.setName = function(name){
        this.name = name;
        if(this.html) {
            this.html.name = name;
        }

        return this;
    };
    /**
     * Sets the enctype property for the form
     * @param {String} encType
     * @chainable
     */
    Form.prototype.setEncType = function(encType) {
        this.encType = encType;
        if(this.html) {
            this.html.setAttribute("encType", encType);
        }

        return this;
    };
    /**
     * Shows the form header
     * @chainable
     */
    Form.prototype.showHeader = function() {
        this.visibleHeader = true;
        this.header.style.display = "";
        this.updateDimensions();

        return this;
    };
    /**
     * Hides the form's header
     * @chainable
     */
    Form.prototype.hideHeader = function() {
        this.visibleHeader = false;
        this.header.style.display = "none";
        this.updateDimensions();

        return this;
    };
    /**
     * Returns true if all the form's fields assert the their validations otherwise returns false.
     * @return {Boolean}
     */
    Form.prototype.isValid = function() {
        return this.panel.isValid();
    };
    /**
     * Submits the form.
     * @chainable
     */
    Form.prototype.submit = function() {
        if(this.isValid()) {
            if(typeof this.onSubmit === 'function') {
                this.onSubmit(this.getData());
            }
            //TODO: call send() method from proxy
        }

        return this;
    };
    /**
     * Resets the form to their initial values.
     * @chainable
     */
    Form.prototype.reset = function() {
        var fields, i;
        fields = this.getFields();
        for(i = 0; i < fields.length; i++) {
            fields[i].reset();
        }
        if(this.dirty){
          this.dirty = false;
        }
        return this;
    };
    /**
     * Returns the form's data.
     * @param {String} [format='object'] The format in which the data will be returned, it can take three values:
     * 
     * - 'xml', the data will be returned in xml format.
     * - 'json', the data will be returned in a json-formated string.
     * - 'object', the data will be returned in a object literal format.
     * - otherwise or nothing, the data will be returned in a object literal format but only containing the name and values from 
     * fields.
     * @return {Object|String} The data type of the returned item will depend on the input parameter, please read the 
     * method description for more info.
     */
    Form.prototype.getData = function(format) {
        var fields, i, res, data = new PMUI.data.DataSet();
        fields = this.getFields();
        if(format !== 'xml' && format !== 'json' && format !== 'object') {
            res = {};
            for(i = 0; i < fields.length; i++) {
                if(!fields[i].disabled) {
                    res[fields[i].getName()] = fields[i].getValue();
                }
            }
        } else {
            for(i = 0; i < fields.length; i++) {
                if(!fields[i].disabled) {
                    data.addItem(fields[i].data);
                }
            }
            if(format === 'xml') {
                return data.getXML();
            } else if(format === 'json') {
                return data.getJSON();
            } else if(format === 'object'){
                return data.getData();
            }
        }
        return res;
    };
    /**
     * Define the event listeners for the form.
     * @chainable
     */
    Form.prototype.defineEvents = function() {
          this.panel.defineEvents();
          this.footer.defineEvents();
          this.eventsDefined = true;

          return this;
    };
    /**
     * Creates the HTML for the form.
     * @return {HTMLElement}
     */
    Form.prototype.createHTML = function() {
        var html, header, footer;

        if(this.html) {
            return this.html;
        }
        html = PMUI.createHTMLElement('form');
        html.id = this.id;
        html.className = 'pmui-form';

        header = PMUI.createHTMLElement('div');
        header.className = 'pmui-form-header';

        this.header = header;

        html.appendChild(header);
        html.appendChild(this.panel.getHTML());
        html.appendChild(this.footer.getHTML());
        this.html = html;
        
        this.setTitle(this.title)
            .setName(this.name)
            .setEncType(this.encType);

        if(this.visibleHeader) {
            this.showHeader();
        } else {
            this.hideHeader();
        }
        //Padding default for FormPanel
        this.panel.setPadding('20px 10px 20px 10px');
        
        this.style.applyStyle();    
        if(this.layout) {
            this.layout.applyLayout();
        }
        if(this.getFields().length){
          this.setFocus(0);
        }
        this.setButtonPanelPosition(this.buttonPanelPosition);
        return this.html;
    };
    /**
     * clean all form fields
     * @chainable
     */
    Form.prototype.clearItems = function (){
        this.panel.clearItems(); 
        return this;
    };
    /**
     * clear all form fields
     * @chainable
     */
    Form.prototype.removeItem = function (item){
        var i = 0;
        Form.superclass.prototype.removeItem.call(this, item);
        this.panel.removeItem(item);        
        return this;
    };
    /**
     * hides the footer where the buttons are located
     * @chainable
     */
    Form.prototype.hideFooter = function () {
        this.footer.setVisible(false);
        return this; 
    };
    /**
     * shows the footer where the buttons are located
     * @chainable
     */
    Form.prototype.showFooter = function () {
        this.footer.setVisible(true);
        return this;
    };
    /**
     * @method setFocus
     * set the focus on a form field
     * @chainable
     */
    Form.prototype.setFocus = function () {
          var j, fields = this.getFields();          
          for (j = 0 ; j < fields.length; j += 1){
               if ((!fields[j].isReadOnly || !fields[j].isReadOnly()) && !fields[j].disabled && fields[j].isVisible()) {
                   fields[j].setFocus();
                   break;
               }               
          }
          return this;
     }          
    // Declarations created to instantiate in NodeJS environment
    if (typeof exports !== "undefined") {
        module.exports = Form;
    }
    Form.prototype.setAlignmentButtons = function (value) {
          if (typeof value == "string") {
               this.alignmentButtons = value;
               this.footer.setAlignment(value);
               return this;
          }
          throw new Error('setAlignmentButtons(), the value is no type valid')
    };



    PMUI.extendNamespace('PMUI.form.Form', Form);
}());
(function () {
    /**
     * @class PMUI.form.Validator
     * Handles the validations of the fields
     * @extend PMUI.core.Base
     *
     * @constructor
     * Create a new instance of the class
     * @param {Object} settings
     * @param {PMUI.form.Field} parent
     *
     * @cfg {*} [criteria=null] This is the criteria info to execute the validation,
     * each validator needs a different data type criteria, 
     * so please read about the criteria config option for every validator object.
     * @cfg {String} [errorMessage='the validation has failed'] This property defines 
     * the error message to display if validation fails when filled fields
     * @cfg {PMUI.form.Field} [parent=null] Defines the field the validator belongs to.
     */       
    var Validator = function (settings) {

       Validator.superclass.call(this, settings);
        /**
         * Defines the Field parent
         * @type {PMUI.form.Field}
         */
        this.parent = null;
        /**
         * Defines the criteria object
         * @type {Object}
         */
        this.criteria = null;
        /**
         * Defines if the object is validated
         * @type {Boolean}
         */
        this.validated = false;
        /**
         * Defines the validation state
         * @type {null/Boolean}
         */
        this.valid = null;
        /**
         * Defines the error message to show in case of the validation fails
         * @type {null/Boolean}
         */
        this.errorMessage = null;

        Validator.prototype.init.call(this, settings);
    };

    PMUI.inheritFrom('PMUI.core.Base', Validator);
    /**
     * Defines the object's type
     * @type {String}
     */
    Validator.prototype.type = 'Validator';
    /**
     * Defines the object's family
     * @type {String}
     */
    Validator.prototype.family = 'Validator';
    /**
     * Initializes the object with default values
     * @param {Object} settings
     * @param {PMUI.form.Field} parent
     */
    Validator.prototype.init = function (settings) {
        var defaults = {
            criteria: null,
            errorMessage: 'the validation has failed',
            parent: null
        };
        jQuery.extend(true, defaults, settings);
        this.setCriteria(defaults.criteria)
            .setParent(defaults.parent)
            .setErrorMessage(defaults.errorMessage);
    };
    /**
     * Sets the validation error message to show in case of the validation fails
     * @param {String} errorMessage
     * @chainable
     */
    Validator.prototype.setErrorMessage = function (errorMessage) {
        this.errorMessage = errorMessage;
        return this;
    };
    /**
     * Sets the validation error message to show in case of the validation fails
     * @param {String} errorMessage
     * @chainable
     */
    Validator.prototype.getErrorMessage = function () {
        return this.errorMessage;
    };
    /**
     * Sets the validation criteria
     * @param {Object} criteria
     * @chainable
     */
    Validator.prototype.setCriteria = function (criteria) {
        this.criteria = criteria;
        return this;
    };
    /**
     * Sets the parent field
     * @param {PMUI.form.Field|GridControlCell} parent
     * @chainable
     */
    Validator.prototype.setParent = function (parent) {
        if(parent) {
            if(parent instanceof PMUI.form.Field || parent instanceof PMUI.grid.GridControlCell) {
                this.parent = parent;
            } else {
                throw new Error("setParent() method only accepts a Field object as parameter.");
            }
        }
        return this;
    };

    /**
     * Returns the parent field
     * @return {PMUI.form.Field}
     */
    Validator.prototype.getParent = function () {
        return this.parent;
    };
    /**
     * Evaluates the validator
     */
    Validator.prototype.validate = function () {
        this.valid = true;
    };
    /**
     * Returns the validation response
     * @chainable
     */
    Validator.prototype.isValid = function () {
        this.validate();
        this.updateTooltip();
        return this.valid;
    };
    /**
     * Updates de the tooltip message
     */
    Validator.prototype.updateTooltip = function () {
        if (this.parent && this.parent.message) {
            if (this.valid) {
                this.parent.hideMessage();
            } else {
                this.parent.showMessage(this.errorMessage, "error");
            }
        }
    };
    // Declarations created to instantiate in NodeJS environment
    if (typeof exports !== "undefined") {
        module.exports = Validator;
    }

    PMUI.extendNamespace('PMUI.form.Validator', Validator);

}());
(function () {
    /**
     * @class  PMUI.form.TextLengthValidator
     * @extend PMUI.form.Validator
     *
     * Create a new instace abstrac of the TextLengthValidator class 
     * @param {Object} settings
     *
     * @cfg {String} [errorMessage='the validation has failed'] This property defines the error 
     * message to display if the validation text size entered is not valid
     */
    var TextLengthValidator = function (settings) {

        TextLengthValidator.superclass.call(this, settings);
        TextLengthValidator.prototype.init.call(this, settings);

    };

    PMUI.inheritFrom('PMUI.form.Validator', TextLengthValidator);
    /**
     * Defines the object's type
     * @type {String}
     */
    TextLengthValidator.prototype.type = 'TextLengthValidator';

    TextLengthValidator.prototype.init = function (settings) {
        var maxMessage = settings.criteria.maxLength ? " at most " + settings.criteria.maxLength + " characters" : "",
            minMessage = settings.criteria.minLength ? "at least " + settings.criteria.minLength + " characters" : "",
        defaults = {
            errorMessage: "The text length must have " + minMessage + (maxMessage ? " and " : "") + maxMessage 
        };

        jQuery.extend(true, defaults, settings);

        this.setErrorMessage(defaults.errorMessage);
    };
    /**
     * Evaluates the validator
     * this method is for validate the length of the string on control the value 
     * input in text field, cutting spaces white in use trim of the jquery function
     * @chainable
     */ 
    TextLengthValidator.prototype.validate = function () {
        var res = false,
            value = this.criteria.trim ? jQuery.trim(this.parent.value) : this.parent.value;

        this.valid = true;

        if (this.criteria.maxLength) {
            this.valid = value.length <= parseInt(this.criteria.maxLength, 10);
        }
        if (this.criteria.minLength) {
            this.valid = (this.valid !== null ? this.valid : true) &&
                value.length >= parseInt(this.criteria.minLength, 10);
        }

        return this;
    };

    PMUI.extendNamespace('PMUI.form.TextLengthValidator', TextLengthValidator);

    if (typeof exports !== "undefined") {
        module.exports = TextLengthValidator;
    }

}());
(function(){
    /**
     * @class PMUI.form.RegExpValidator
     * @extends PMUI.form.Validator
     * A text validator based in regular expressions.
     *
     * Usually this object won't be instatiate directly, it will be instantiate through a 
     * {@link PMUI.form.Field Field} object just like is shown in the 
     * {@link PMUI.field.TextField TextField documentation}. We recommend to use it this way.
     *
     * The other way to instantiate an object from this class:
     *
     *      var myValidator,
     *          field,
     *          callback = function() {
     *              this.isValid();
     *          };
     *
     *      field = new PMUI.field.TextField({
     *          onChange: callback
     *      });
     *
     *      myValidator = new RegExpValidator({
     *          criteria: /^\d+$/,
     *          errorMessage: "You must introduce only number digits."
     *      });
     *
     *      field.addValidator(myValidator);
     *
     *      document.body.appendChild(field.getHTML());
     *
     * @cfg {Object|RegExp} criteria The criteria config option can be:
     * 
     * - a JSON object: in this case it must have two porperties:
     *     - pattern (String): the pattern string to be use as the regular expression 
     (don't forget to escape special characters)
     *     - modifiers (String): this is optional, this is the modifier for the regular expression.
     * - a RegExp object 
     */
    var RegExpValidator = function (options) {
        RegExpValidator.superclass.call(this, options);
        RegExpValidator.prototype.init.call(this, options);
    };
    
    PMUI.inheritFrom('PMUI.form.Validator', RegExpValidator);

    RegExpValidator.prototype.type = "RegExpValidator";

    RegExpValidator.prototype.init = function (options) {
        var defaults = {
            errorMessage: "The text pattern doesn't match"
        };

        $.extend(true, defaults, options);

        this.setErrorMessage(defaults.errorMessage);
    };
    /**
     * Execute the validation.
     * @chainable
     * @private
     */
    RegExpValidator.prototype.validate = function () {
        var res = false, regExp;
        if (this.parent && this.parent.value) {
            if(this.criteria instanceof RegExp) {
                this.valid = this.criteria.test(this.parent.value);
            } else if(typeof this.criteria === 'string') {
                regExp = new RegExp(this.criteria.pattern, this.criteria.modifiers);
                this.valid = regExp.text(this.parent.value);
            }
        } else {
            this.valid = false;
        }

        return this;
    };

    PMUI.extendNamespace('PMUI.form.RegExpValidator', RegExpValidator);

    if (typeof exports !== "undefined") {
        module.exports = RegExpValidator;
    }
}());
(function() {
    /**
     * @class  PMUI.form.ValidatorFactory
     * @extend PMUI.util.Factory
     * Extends the factory class to produce field validators. 
     * 
     * Its default products are:
     * 
     * - {@link PMUI.form.TextLengthValidator TextLengthValidator} objects: using "textLength".
     * - {@link PMUI.form.RegExpValidator RegExpValidator} objects: using "regexp".
     *
     * @constructor
     * Creates a new instance od the class
     */
    var ValidatorFactory = function() {
        ValidatorFactory.superclass.call(this);
        ValidatorFactory.prototype.init.call(this);
    };

    PMUI.inheritFrom('PMUI.util.Factory', ValidatorFactory);

    ValidatorFactory.prototype.init = function() {
        var defaults = {
            products: {
                "textLength": PMUI.form.TextLengthValidator,
                "regexp": PMUI.form.RegExpValidator
            }, 
            defaultProduct: "textLength"
        };
        this.setProducts(defaults.products)
            .setDefaultProduct(defaults.defaultProduct);
    };

    PMUI.extendNamespace('PMUI.form.ValidatorFactory', ValidatorFactory);

    // Declarations created to instantiate in NodeJS environment
    if (typeof exports !== "undefined") {
        module.exports = ValidatorFactory;
    }

}());
(function (){
    /**
     * @class  PMUI.panel.LayoutPanel
     * @extends PMUI.core.Panel
     * Handles layouts to be inserted into panels. 
     * It is composed by five main elements: north, south, west, east and center
     *
     * Example:
     * 
     *           @example
     *          
     *           layout = new PMUI.panel.LayoutPanel({
     *               id: "myLayout",
     *               center: {
     *                   cssProperties: {
     *                       "background-color": "#91b3e0"    
     *                   },
     *                   cssClasses: ['my-class']
     *               },
     *               west: {
     *                   resizable: false,
     *                   closed: true
     *               },
     *               south: new PMUI.panel.LayoutPanel({
     *                   north:{
     *                       size: 50
     *                   }
     *               }),
     *               southConfig:{
     *                   size: 200
     *               },
     *               north: {
     *                   size: 50,
     *                   togglerOpen: 80,
     *                   overflow: false
     *               },
     *               east: {
     *                   closable: false
     *               }
     *           });
     *           document.body.appendChild(layout.getHTML());
     *           layout.render(true);
     *
     * @constructor
     * Create a new instance of the LayoutPanel class
     * @param {Object} options Contructor object
     */
    var LayoutPanel = function (options) {
        LayoutPanel.superclass.call(this, options);

        /**
         * @property {String} prefixPlugin
         * The prefix for the classes in jquery.layout plugin
         * @private
         */
        var prefixPlugin = 'ui-layout-',
            /**
             * @property {Array} positions
             * The array with the position accepted in the layout
             * @private
             */
            positions = ['north','center','south','east','west'],
            /**
             * @property {Object} settings
             * Defines the settings acceptted by the layout panel
             * @private
             */
            settings = {
                size: "size",
                closed: "initClosed",
                resizable: "resizable",
                closable: "closable",
                togglerOpen: "togglerLength_open",
                togglerClosed: "togglerLength_closed",
                overflow: "allowOverflow",
                hidden: "initHidden",
                cssProperties: "cssProperties",
                cssClasses: "cssClasses"
            };

        /**
         * @property {Object} panels
         * The settings for the plugin organized by section
         * @private
         */
        this.panels = {};

        /**
         * North panel pointer
         * @type {HTMLElement}
         */
        this.north = null;

        /**
         * Center panel pointer
         * @type {HTMLElement}
         */
        this.center = null;

        /**
         * South panel pointer
         * @type {HTMLElement}
         */
        this.south = null;

        /**
         * East panel pointer
         * @type {HTMLElement}
         */
        this.east = null;

        /**
         * West panel pointer
         * @type {HTMLElement}
         */
        this.west = null;

        /**
         * Javascript pointer to the jquery layout plugin instance
         * @type {Object}
         */
        this.instance = null;

        /**
         * Returns the config object
         * @return {Object} 
         */
        LayoutPanel.prototype.getConfig = function () {
            return {
                panels: this.panels,
                prefix: prefixPlugin,
                positions: positions,
                isSetting: function (value) {
                    var valid = settings[value];
                    return (typeof valid !== 'undefined');
                },
                transformSetting: function (value) {
                    return settings[value];
                }
            };
        };

        /**
         * Sets the panel settings
         * @param {String} pos      Position (north, center, south, east or west)
         * @param {Object} settings Object settings
         */
        LayoutPanel.prototype.setConfig = function (pos, settings) {
            this.panels[pos] = settings;
            return this;
        };

        LayoutPanel.prototype.init.call(this, options);
    };

    PMUI.inheritFrom('PMUI.core.Panel', LayoutPanel);

    /**
     * Defines the object's type
     * @type {String}
     */
    LayoutPanel.prototype.type = 'LayoutPanel';

    /**
     * @private
     * Initializes the object with default values
     * @param  {Object} options 
     */
    LayoutPanel.prototype.init = function (options) {
        var defaults = {
            center: {
                cssProperties: {},
                cssClasses: []
            },
            factory: {
                products: {
                    "layout": PMUI.panel.LayoutPanel,
                    "treePanel": PMUI.panel.TreePanel,
                    "panel": PMUI.core.Panel
                },
                defaultProduct: "panel"
            }
        };
        jQuery.extend(true, defaults, options);
        this.setFactory(defaults.factory)
            .setPluginSettings(defaults)
            .setPanels(defaults);
    };

    /**
     * Calculate and sets the settings for each panel
     * @param {Object} obj Object constructor
     */
    LayoutPanel.prototype.setPluginSettings = function (obj) {
        var config = this.getConfig(),
            that = this;
        jQuery.each(config.positions, function(index, pos){
            var panel = obj[pos],
                panelSettings = {};
            if (panel) {
                if (that.factory && that.factory.isValidClass(panel)) {
                    panelSettings = obj[pos + "Config"] || {};
                } else {
                    jQuery.each(panel, function(key, value){
                        if (config.isSetting(key)){
                            panelSettings[config.transformSetting(key)] = value;
                        }
                    }); 
                }
                //if (panelSettings !== {}) {
                if (!(panelSettings.cssClasses && jQuery.isArray(panelSettings.cssClasses))) {
                    panelSettings.cssClasses = [];
                }
                panelSettings.cssClasses.push(config.prefix + pos);
                that.setConfig(pos, panelSettings);
                //}
            } 
        });
        return this;
    };

    /**
     * Set panels into north, south, center, west and east pointers
     * @param {Object} obj Constructor object
     */
    LayoutPanel.prototype.setPanels = function (obj) {
        var config = this.getConfig(),
            that = this,
            treeOptions;
        jQuery.each(config.positions, function(index, pos){
            if (obj[pos]){
                
                that[pos] = that.factory.make(obj[pos]);

                if (that[pos] instanceof PMUI.panel.LayoutPanel) {
                    that.setConfig(pos, jQuery.extend(true, config.panels[pos], {instance: that[pos]}));
                }
                if (that[pos] instanceof PMUI.panel.TreePanel) {
                    treeOptions = {
                        cssProperties: that[pos].style.cssProperties,
                        cssClasses: that[pos].style.cssClasses
                    };
                    that.setConfig(pos, jQuery.extend(true, config.panels[pos], treeOptions));
                }
                
                that[pos].style.cssProperties = config.panels[pos].cssProperties || {};
                that[pos].style.cssClasses = config.panels[pos].cssClasses || [config.prefix + pos];    
                
                that.addItem(that[pos]);
            }
        });
        return this;
    };


    /**
     * Create html for each item to container 
     * @return {Object} the current object html
     */
    LayoutPanel.prototype.createHTML = function () {
        var i,
            sizeArray,
            items;

        items = this.items.asArray();
        LayoutPanel.superclass.prototype.createHTML.call(this);
        for (i = 0; i < items.length; i+=1) {
            $(this.html).append(items[i].getHTML());
        }
        return this.html;
    };
    /**
     * Render all the html object
     * @param  {Boolean} [recursive] Defines if the render must be recursive
     */
    LayoutPanel.prototype.render = function (recursive) {
        var that = this,
            options,
            renderRecursive = recursive || false;
        options = that.getConfig().panels;
        if (jQuery(that.html).height() === 0){
            jQuery(that.html).height(jQuery(document).height() - 20);
        }
        this.instance = jQuery(that.html).layout(options);

        jQuery.each(options, function(position, val) {
            if (options[position].instance && renderRecursive){
                options[position].instance.render(renderRecursive);
            }
            if (options[position].allowOverflow === true) {
                that.instance.allowOverflow(position);
            }
        });
    };

    /**
     * Remove each html related to the layout from the DOM and destroy the layout instance
     */
    LayoutPanel.prototype.destroy = function () {
        if (this.instance.destroy !== undefined) {
            this.instance.destroy(); 
        }
        if (this.html !== undefined) {
            jQuery(this.html).remove();
            this.html = null;
        }
        this.instance = null;
    };

    PMUI.extendNamespace('PMUI.panel.LayoutPanel', LayoutPanel);

    if (typeof exports !== 'undefined') {
        module.exports = LayoutPanel;
    }

}());
(function () {
    /**
     * @class PMUI.layout.Layout
     * Defines the properties for the different layouts in the panels
     * @abstract
     **/
    var Layout = function (options) {
        /**
         * Defines the container to apply the layout
         * @type {Object}
         */
        this.belongsTo = null;
        Layout.prototype.init.call(this, options);
    };

    /**
     * Defines the object's type
     * @type {String}
     */
    Layout.prototype.type = "Layout";

    /**
     * Defines the object's family
     * @type {String}
     */
    Layout.prototype.family = "Layout";

    Layout.prototype.init = function (options) {
        var defaults = {
            belongsTo :  null
        };

        jQuery.extend(true, defaults, options);
        this.setContainer(defaults.belongsTo);
    };

    /**
     * Applies the layout to the container
     * @abstract
     */
    Layout.prototype.applyLayout = function () {
        return this;
    };

    /**
     * Sets the parent container
     * @param {Object} parent Container Object
     */
    Layout.prototype.setContainer = function (parent) {
        this.belongsTo = parent;
        return this;
    };

    PMUI.extendNamespace('PMUI.layout.Layout', Layout);

    if (typeof exports !== 'undefined') {
        module.exports = Layout;
    }

}());
(function () {
    /**
     * @class  PMUI.layout.HBox
     * @extends PMUI.layout.Layout
     * Class created to handle Horizontal Box layout changes
     *
     * @constructor
     * Creates a new isntance of the object
     * @param {Object} options Constructor object
     */
    var HBox = function (options) {
        HBox.superclass.call(this, options);
        HBox.prototype.init.call(this, options);
    };

    PMUI.inheritFrom('PMUI.layout.Layout', HBox);
    
    /**
     * Defines the object's type
     * @type {String}
     */
    HBox.prototype.type = "HBox";

    /**
     * @private
     * Initializes the object with default values
     * @param  {Object} options 
     */
    HBox.prototype.init = function (options) {
        var defaults = {

        };
        jQuery.extend(true, defaults, options);
    };

    /**
     * Applies the layout to the current element
     */
    HBox.prototype.applyLayout = function () {
        // get the object of this layout
        var owner = this.belongsTo,
            items = owner.items,
            totalProportion = 0,
            usableWidth = owner.getUsableWidth(),
            usableHeight = owner.getUsableHeight(),
            i, width, item;
        var ownerParent = owner;
        // search the width in anysome parent
        if(owner != null)
        {
            if(owner.getWidth() != "auto" && owner.parent != null)
                 ownerParent=owner.parent;
             
                if(owner.html){
                    //var border =ownerParent.getBorderWidth()? parseFloat(ownerParent.getBorderWidth()):0;
                    //var paddingLeft = ownerParent.getPadding()[3]?ownerParent.getPadding()[3]:0;
                    //owner.html.style.width=ownerParent.getUsableWidth()+border+paddingLeft; 
                    owner.html.style.width=ownerParent.getUsableWidth(); 
                }

        }
        // compute totalProportion
        for (i = 0; i < items.getSize(); i += 1) {
            item = items.get(i);
            totalProportion += item.proportion;
        }

        // set the width of each object based on the width of its parent
        /*if(isNaN(usableWidth)) {
            for (i = 0; i < items.getSize(); i += 1) {
                items.get(i).setHeight(0); 
            }
            return this;
        }*/
        xPosition = 0;
        for (i = 0; i < items.getSize(); i += 1) {
            item = items.get(i);
            if(item.html) {
                item.setDisplay("inline-block");
            }
            width = usableWidth * (item.proportion / totalProportion);
            item.setWidth(width);
            item.setHeight(usableHeight);
        }
        return this;
    };

    PMUI.extendNamespace('PMUI.layout.HBox', HBox);

    if (typeof exports !== 'undefined') {
        module.exports = HBox;  
    }

}());
(function () {
    /**
     * @class  PMUI.layout.VBox
     * @extends PMUI.layout.Layout
     * Class created to handle Vertical Box layout changes
     *
     * @constructor
     * Creates a new isntance of the object
     * @param {Object} options Constructor object
     */
    var VBox = function (options) {
        VBox.superclass.call(this, options);
        VBox.prototype.init.call(this, options);
    };

    PMUI.inheritFrom('PMUI.layout.Layout', VBox);
    
    /**
     * Defines the object's type
     * @type {String}
     */
    VBox.prototype.type = "VBox";

    /**
     * @private
     * Initializes the object with default values
     * @param  {Object} options 
     */
    VBox.prototype.init = function (options) {
        var defaults = {

        };
        jQuery.extend(true, defaults, options);
    };

    /**
     * Applies the layout to the current element
     */
    VBox.prototype.applyLayout = function () {
        // get the object of this layout
        var owner = this.belongsTo,
            items = owner.items,
            totalProportion = 0,
            usableWidth = owner.getUsableWidth(),
            usableHeight = owner.getUsableHeight(),
            i, height,
            item;

        // compute totalProportion
        for (i = 0; i < items.getSize(); i += 1) {
            item = items.get(i);
            totalProportion += item.proportion;
        }

        // set the width of each object based on the width of its parent
        /*if(isNaN(usableHeight)) {
            return this;
        }*/
        for (i = 0; i < items.getSize(); i += 1) {
            item = items.get(i);
            if(item.html) {
                item.setDisplay("block");
            }
            height = usableHeight * (item.proportion / totalProportion);
            item.setHeight(height);
            item.setWidth(usableWidth);
        }
        return this;
    };

    PMUI.extendNamespace('PMUI.layout.VBox', VBox);

    if (typeof exports !== 'undefined') {
        module.exports = VBox;  
    }

}());
(function () {
    /**
     * @class  PMUI.layout.Box
     * @extends PMUI.layout.Layout
     * Class created to no layout changes
     *
     * @constructor
     * Creates a new isntance of the object
     * @param {Object} options Constructor object
     */
    var Box = function (options) {
        Box.superclass.call(this, options);
        Box.prototype.init.call(this, options);
    };

    PMUI.inheritFrom('PMUI.layout.Layout', Box);
    
    /**
     * Defines the object's type
     * @type {String}
     */
    Box.prototype.type = "Box";

    /**
     * @private
     * Initializes the object with default values
     * @param  {Object} options 
     */
    Box.prototype.init = function (options) {
        var defaults = {

        };
        jQuery.extend(true, defaults, options);
    };

    /**
     * Applies the layout to the current element
     */
    Box.prototype.applyLayout = function () {};

    PMUI.extendNamespace('PMUI.layout.Box', Box);

    if (typeof exports !== 'undefined') {
        module.exports = Box;   
    }

}());
(function (){
    /**
     * @class PMUI.layout.LayoutFactory
     * @extends {PMUI.util.Factory}
     * Extends the functionality of Factory to set the constructor for layouts
     * 
     * @constructor
     * Makes a new instance of the class
     * @param {Object} options 
     */
    var LayoutFactory = function (options) {
        LayoutFactory.superclass.call(this, options);
        LayoutFactory.prototype.init.call(this, options);
    };

    PMUI.inheritFrom('PMUI.util.Factory', LayoutFactory);

    /**
     * Defines the object's type
     * @type {String}
     */
    LayoutFactory.prototype.type = 'LayoutFactory';

    /**
     * @private
     * Initializes the object with default values
     * @param  {Object} options 
     */
    LayoutFactory.prototype.init = function (options) {
        var defaults = {
            products: {
                "hbox": PMUI.layout.HBox,
                "vbox": PMUI.layout.VBox,
                "box": PMUI.layout.Box
            },
            defaultProduct: "box"
        };
        jQuery.extend(true, defaults, options);
        this.setProducts(defaults.products)
            .setDefaultProduct(defaults.defaultProduct);
    };

    LayoutFactory.prototype.make = function(obj) {
        var product,
            productType = obj.pmType || '';
        if (this.isValidClass(obj)){
            product = obj;
        } else if (this.isValidName(productType)) {
            product = this.build.call(this, productType, obj);
        } else if(this.isValidName(obj)) {
            product = this.build.call(this, obj, {});
        } else {
            product = this.build.call(this, this.defaultProduct, obj);
        }
        return product;
    };

    PMUI.extendNamespace('PMUI.layout.LayoutFactory', LayoutFactory);

    if (typeof exports !== 'undefined') {
        module.exports = LayoutFactory;
    }

}());
(function () {
    /**
     * @class PMUI.panel.TreePanel
     * A panel that contains {@link PMUI.item.TreeNode TreeNode items}.
     * @extends {PMUI.core.Panel}
     *
     * An example of a TreePanel with descendant nodes (using the items {@link #cfg-items config option}):
     *
     *      @example
     *      var t = new PMUI.panel.TreePanel({
     *          style: {
     *              cssClasses: ['xxxxxxxx']
     *          },
     *          items: [
     *              {
     *                  label: "America",
     *                  items: [
     *                      {
     *                          label: "North America",
     *                          items: [
     *                              {
     *                                  label: "Canada"
     *                              },
     *                              {
     *                                  label: "USA"
     *                              },
     *                              {
     *                                  label: "Mexico"
     *                              }
     *                          ]
     *                      },
     *                      {
     *                          label: "Central America and Caribbean",
     *                          items: [
     *                              {
     *                                  label: "Guatemala"
     *                              },
     *                              {
     *                                  label: "Cuba"
     *                              },
     *                              {
     *                                  label: "Costa Rica"
     *                              }
     *                          ]
     *                      },
     *                      {
     *                          label: "South America",
     *                          items: [
     *                              {
     *                                  label: "Argentina"
     *                              },
     *                              {
     *                                  label: "Bolivia"
     *                              },
     *                              {
     *                                  label: "Brasil"
     *                              }
     *                          ]
     *                      }
     *                  ]
     *              }
     *          ]
     *      });
     *      document.body.appendChild(t.getHTML());
     *      t.defineEvents();
     *
     * An example of a TreePanel with descendant nodes (using the {@link #cfg-dataItems dataItems} config option):
     *
     *      @example
     *      t2 = new PMUI.panel.TreePanel({
     *          nodeDefaultSettings: {
     *              labelDataBind: 'name',
     *              itemsDataBind: 'regions',
     *              recursiveChildrenDefaultSettings: true,
     *              childrenDefaultSettings: {
     *                  labelDataBind: 'name',
     *                  itemsDataBind: 'countries',
     *                  autoBind: true
     *              },
     *              autoBind: true
     *          },
     *          dataItems: [
     *              {
     *                  name: 'America',
     *                  regions: [
     *                      {
     *                          name: 'North America',
     *                          countries: [
     *                              {
     *                                  name: 'Canada'
     *                              },
     *                              {
     *                                  name: 'USA'
     *                              },
     *                              {
     *                                  name: 'Mexico'
     *                              }
     *                          ]
     *                      },
     *                      {
     *                          name: 'Central America and Caribbean',
     *                          countries: [
     *                              {
     *                                  name: 'Guatemala'
     *                              },
     *                              {
     *                                  name: 'Cuba'
     *                              }, 
     *                              {
     *                                  name: 'Costa Rica'
     *                              }
     *                          ]
     *                      },
     *                      {
     *                          name: 'South America',
     *                          countries: [
     *                              {
     *                                  name: 'Argentina'
     *                              },
     *                              {
     *                                  name: 'Bolivia'
     *                              }, 
     *                              {
     *                                  name: 'Brasil'
     *                              }
     *                          ]
     *                      }
     *                  ]
     *              }
     *          ]
     *      });
     *      document.body.appendChild(t2.getHTML());
     *      t2.defineEvents();
     *
     * @constructor
     * Creates a new instance of the class.
     * @param {Object} settings The config options.
     *
     * @cfg {Object|null} [nodeDefaultSettings=null] The default settings for all the direct child nodes of the tree panel.
     * It is an object with the config options for {@link PMUI.item.TreeNode TreeNode}.
     * @cfg {Object|null} [dataItems=null] An object with the data for the items in the tree panel.
     * @cfg {Array} [data=[]] An array in which each element is an object with the config options for 
     * {@link PMUI.item.TreeNode TreeNode} an instance of it. If you use both dataItems and this config option, this 
     * config option will be ignored and the dataItems will be used instead.
     * @cfg {String} [filterPlaceholder="search"] The text to show as placeholder for the text control for filtering.
     * @cfg {Function|String|null} [emptyMessage=null] The message to show when the grid is showing no items. It 
     * can be:
     *
     * - A String, in this case the string will be displayed in both cases, when the treepanel has no items and when no 
     * treepanel's item meet the filter.
     * - A Function, in this case the function must return a string or an HTML Element to be displayed. The function 
     * will receive two parameters: the {@link PMUI.panel.TreePanel TreePanel}, a boolean If it is true it means the returned 
     * value will be used when a filter is applied, otherwise it means that the returned value will be used when 
     * there are no items in the treepanel.
     * - null, in this case a default message will be used for each situation.
     * @cfg {Boolean} [visibleTitle=false] If the title will be displayed.
     * @cfg {String} [title=""] The text for the title.
     * @cfg {Function} [onNodeClick=null] The callback function to be executed everytime the 
     * {@link #event-onNodeClick onNodeClick} event fires. For info about the parameters sent to the callback please 
     * read the {@link #event-onNodeClick onNodeClick} event documentation.
     * @cfg {Function|null} [onBeforeAppend=null] A callback function to be called everytime the 
     * {@link #event-onBeforeAppend onBeforeAppend} event is fired. For info about the parameters used for this 
     * callback please read the documentation for this event.
     * @cfg {Function|null} [onAppend=null] A callback function to be called everytime the 
     * {@link #event-onAppend onAppend} event is fired. For info about the parameters used for this callback please 
     * read the documentation for this event.
     */
    var TreePanel = function (settings) {
        TreePanel.superclass.call(this, jQuery.extend(settings, {
            factory: {
                products: {
                    'node': PMUI.item.TreeNode
                },
                defaultProduct: 'node'
            }
        }));
        /**
         * The default settings for all the direct child nodes of the tree panel.
         * @type {Object}
         */
        this.nodeDefaultSettings = null;
        /**
         * The filter criteria being applied currently.
         * @type {String}
         * @readonly
         */
        this.filterCriteria = "";
        /**
         * If the tree panel is filterable or not.
         * @type {Boolean}
         * @readonly
         */
        this.filterable = null;
        /**
         * The text control in which the user can introduce a text to filter.
         * @type {PMUI.control.TextControl}
         * @private
         */
        this.filterControl = null;
        /**
         * A string or a function or null. Determines the text/HTML Element to show when there are no items to show.
         * @type {Function|String|null}
         * @readonly
         */
        this.emptyMessage = null;
        /**
         * The title for the TreePanel
         * @type {String}
         */
        this.title = null;
        /**
         * If the title is visible or not.
         * @type {Boolean}
         * @readonly
         */
        this.visibleTitle = null;
        /**
         * @event onItemClick
         * Fired everytime an tree item is clicked.
         * @param {PMUI.panel.TreePanel} treePanel The TreePanel in which the event was fired.
         * @param {PMUI.item.TreeNode} node The node that was clicked.
         */
        this.onNodeClick = null;
        /**
         * An private object to store the DOM elements that compose the object's HTML.
         * @type {Object}
         * @private
         */
        this.dom = {};
        /**
         * @event onBeforeAppend
         * Fired before a child node is appended to the current node or to any of its children nodes.
         * @param {PMUI.item.TreeNode} node The node in which the event is being fired.
         * @param {PMUI.item.TreeNode} newNode The new node to be appended.
         */
        this.onBeforeAppend = null;
        /**
         * @event onAppend
         * Fired everytime a child node is appended to the current node or to any of its children nodes.
         * @param {PMUI.item.TreeNode} node The node in which the event is being fired.
         * @param {PMUI.item.TreeNode} newNode The appended node.
         */
        this.onAppend = null;
        TreePanel.prototype.init.call(this, settings);
    };

    PMUI.inheritFrom('PMUI.core.Panel', TreePanel);
    /**
     * The object's type.
     * @type {String}
     */
    TreePanel.prototype.type = 'TreePanel';
    /**
     * The class family.
     * @type {String}
     */
    TreePanel.prototype.family = 'Panel';
    /**
     * Initializes the object.
     * @param  {Object} settings The config options for the object.
     * @private
     */
    TreePanel.prototype.init = function(settings) {
        var defaults = {
            nodeDefaultSettings: null,
            dataItems: null,
            data: [],
            filterPlaceholder: 'search',
            filterable: false,
            emptyMessage: null,
            visibleTitle: false,
            title: "",
            onNodeClick: null,
            onAppend: null,
            onBeforeAppend: null,
            collapse: false
        };

        jQuery.extend(true, defaults, settings);

        this.setNodeDefaultSettings(defaults.nodeDefaultSettings)
            .setEmptyMessage(defaults.emptyMessage)
            .setTitle(defaults.title)
            .setOnNodeClickHandler(defaults.onNodeClick)
            .setOnAppendHandler(defaults.onAppend)
            .setOnBeforeAppendHandler(defaults.onBeforeAppend);
        this.filterControl = new PMUI.control.TextControl({
            onKeyUp: this.onFilterControlChangeHandler(),
            placeholder: defaults.filterPlaceholder
        });

        if(jQuery.isArray(defaults.dataItems)) {
            this.setDataItems(defaults.dataItems);
        } else {
            this.setItems(defaults.items);
        }
        if(defaults.filterable) {
            this.enableFiltering();
        } else {
            this.disableFiltering();
        }
        if(defaults.visibleTitle) {
            this.showTitle();
        } else {
            this.hideTitle();
        }
        if(defaults.collapse) {
            this.collapse(true);
        } else {
            this.expand(true);
        }
    };
    /**
     * Sets the callback function to be called everytime the {@link #event-onBeforeAppend onBeforeAppend} event is 
     * fired. For info about the parameters used for this callback please read the documentation for this event.
     * @param {Function|null} handler
     */
    TreePanel.prototype.setOnBeforeAppendHandler = function(handler) {
        if(!(handler === null || typeof handler === 'function')) {
            throw new Error("setOnBeforeAppendHandler(): The parameter must be a function or null.");
        }
        this.onBeforeAppend = handler;
        return this;
    };
    /**
     * Sets the callback function to be called everytime the 
     * {@link #event-onAppend onAppend} event is fired. For info about the parameters used for this callback please 
     * read the documentation for this event.
     * @param {Function|null} handler
     */
    TreePanel.prototype.setOnAppendHandler = function(handler) {
        if(!(handler === null || typeof handler === 'function')) {
            throw new Error("setOnAppendHandler(): The parameter must be a function or null.");
        }
        this.onAppend = handler;
        return this;
    };
    /**
     * Sets the callback function to be executed everytime the {@link #event-onNodeclick onNodeClick} is fired. For
     * info about the parameters sent to the callback read the document for that event.
     * @param {Function} handler It can be a function or null.
     */
    TreePanel.prototype.setOnNodeClickHandler = function(handler) {
        if(!(handler === null || typeof handler === 'function')) {
            throw new Error("setOnNodeClickHandler(): The parameter must be a function.");
        }   
        this.onNodeClick = handler;
        return this;
    };
    /**
     * @inheritdoc
     */
    TreePanel.prototype.clearItems = function() {
        TreePanel.superclass.prototype.clearItems.call(this);
        return this.showEmptyMessage();
    };
    /**
     * Hides the title bar.
     * @chainable
     */
    TreePanel.prototype.hideTitle = function() {
        this.visibleTitle = false;
        if(this.dom.title) {
            this.dom.title.style.display = 'none';
        }
        return this;
    };
    /**
     * Shows the title.
     * @chainable
     */
    TreePanel.prototype.showTitle = function() {
        this.visibleTitle = true;
        if(this.dom.title) {
            this.dom.title.style.display = '';
        }
        return this;
    };
    /**
     * Sets the title for the tree panel.
     * @param {String} title
     * @chainable
     */
    TreePanel.prototype.setTitle = function(title) {
        if(typeof title !== 'string') {
            throw new Error('showTitle(): The parameter must be a string.');
        }
        this.title = jQuery.trim(title);
        if(this.dom.title) {
            this.dom.title.textContent = this.title;
        }
        return this;
    };
    /**
     * Sets the message to display when there are no items to display in the tree panel.
     * @param {String|Function|null} emptyMessage It can be:
     *
     * - A String, in this case the string will be displayed in both cases, when the tree has no items and when no 
     * tree's items meet the filter.
     * - A Function, in this case the function must return a string or an HTML Element to be displayed. The function 
     * will receive two parameters: the {@link PMUI.panel.TreePanel tree panel}, a boolean If it is true it means the returned 
     * value will be used when a filter is applied, otherwise it means that the returned value will be used when 
     * there are no items in the tree panel.
     * - null, in this case a default message will be used for each situation.
     * @chainable
     */
    TreePanel.prototype.setEmptyMessage = function(emptyMessage) {
        if(!(emptyMessage === null || typeof emptyMessage === 'string' || typeof emptyMessage === 'function')) {
            throw new Error("setEmptyMessage(): the parameter must be a string, a function or null.");
        }
        this.emptyMessage = emptyMessage;
        return this;
    };
    /**
     * Sets the filter for the text control's placeholder.
     * @param {String} placeholder The text for the placeholder.
     * @chainable
     */
    TreePanel.prototype.setFilterPlaceholder = function(placeholder) {
        this.filterControl.setPlaceholder(placeholder);
        return this;
    };
    /**
     * Returns the placeholder for the text control for filtering.
     * @return {String}
     */
    TreePanel.prototype.getFilterPlaceholder = function() {
        return this.filterControl.placeholder;
    };
    /**
     * Returns the handler to be executed when the filter changes.
     * @return {Function}
     * @private
     */
    TreePanel.prototype.onFilterControlChangeHandler = function() {
        var that = this;
        return function() {
            var nextFilter = this.getHTML().value;
            if (that.filterCriteria !== nextFilter) {
                that.filter(nextFilter);
            }
        };
    };
    /**
     * Enables the filtering functionality.
     * @chainable
     */
    TreePanel.prototype.enableFiltering = function() {
        this.filterable = true;
        this.clearFilter().filterControl.setValue("");
        if(this.dom.toolbar) {
            this.dom.toolbar.style.display = '';
        }
        return this;
    };
    /**
     * Disables the filtering functionality.
     * @chainable
     */
    TreePanel.prototype.disableFiltering = function() {
        this.filterable = false;
        this.clearFilter().filterControl.setValue("");
        if(this.dom.toolbar) {
            this.dom.toolbar.style.display = 'none';
        }
        return this;
    };
    /**
     * Sets the default settings for the direct child nodes of the tree panel.
     * @param {Object} settings The config options to be applied to new child nodes to be added in case they are 
     * missing any config option.
     * @chainable
     */
    TreePanel.prototype.setNodeDefaultSettings = function(settings) {
        if(typeof settings !== 'object') {
            throw new Error('setNodeDefaultSettings(): The parameter must be an object.');
        }
        this.nodeDefaultSettings = settings;

        return this;
    };
    /**
     * @inheritdoc
     */
    TreePanel.prototype.setFactory = function() {
        this.factory = new PMUI.util.Factory({
            products: {
                'node': PMUI.item.TreeNode
            },
            defaultProduct: 'node'
        });

        return this;
    };
    /**
     * The private handler fot the {@link #event-onBeforeAppend onBeforeAppend} event.
     * @private
     * @chainable
     */
    TreePanel.prototype.onBeforeAppendHandler = function(treeNode) {
        if(typeof this.onBeforeAppend === 'function') {
            this.onBeforeAppend(this, treeNode);
        }
        return this;
    };
    /**
     * The private handler for the {@link #event-onAppend} event.
     * @private
     * @chainable
     */
    TreePanel.prototype.onAppendHandler = function(treeNode) {
        if(typeof this.onAppend === 'function') {
            this.onAppend(this, treeNode);
        }
        return this;
    };
    /**
     * Adds a new node to the tree panel.
     * @param {Object|PMUI.item.TreeNode} item An instance of {@link PMUI.item.TreeNode TreeNode}, or an object with 
     * the config options to create a new one. In the latter case, if any config option is missing then will be used 
     * the respective one in the set by the {@link #method-setNodeDefaultSettings setNodeDefaultSettings()} method.
     * @param {Number} [index] The position index in which will be added the new item. If not specified the will be 
     * added to the end.
     * @chainable
     */
    TreePanel.prototype.addItem = function(item, index) {
        var settings = {}, itemToAdd;
        if(item instanceof PMUI.item.TreeNode) {
            item.setParent(this);
        } else if(typeof item === 'object') {
            jQuery.extend(true, settings, this.nodeDefaultSettings || {}, item);
            settings.parent = this;
            item = settings;
        } else {
            throw new Error('addItem(): The parameter must be an instance of PMUI.item.TreeNode or an object.');
        }
        itemToAdd =  this.factory.make(item);
        if(itemToAdd) {
            this.hideEmptyMessage();
            this.onBeforeAppendHandler(itemToAdd);
            TreePanel.superclass.prototype.addItem.call(this, itemToAdd, index);
            this.onAppendHandler(itemToAdd);
        }
        return this;
    };
    /**
     * @inheritdoc
     */
    TreePanel.prototype.removeItem = function(item) {
        TreePanel.superclass.prototype.removeItem.call(this, item);
        if(!this.items.getSize()) {
            this.showEmptyMessage();
        }
        return this;
    };
    /**
     * @inheritdoc
     */
    TreePanel.prototype.setItems = function(items) {
        if(this.onAppend !== undefined) {
            TreePanel.superclass.prototype.setItems.call(this, items);
        }
        return this;
    };
    /**
     * Expands the child nodes.
     * @param  {Boolean} [expandAll=false] If the expand function will be applied to all the sub child items or not. 
     * @chainable
     */
    TreePanel.prototype.expand = function(expandAll) {
        var i, items = this.items.asArray();
        for(i = 0; i < items.length; i++) {
            items[i].expand(expandAll);
        }
        return this;
    };
    /**
     * Collapse the child nodes.
     * @param  {Boolean} [collapseAll=false] If the collapse function will be applied to all the sub child items or 
     * not. 
     * @chainable
     */
    TreePanel.prototype.collapse = function(collapseAll) {
        var i, items = this.items.asArray();
        for(i = 0; i < items.length; i++) {
            items[i].collapse(collapseAll);
        }
        return this;
    };
    /**
     * Hides the message to show when there are no items to display.
     * @chainable
     * @private
     */
    TreePanel.prototype.hideEmptyMessage = function() {
        jQuery(this.html).find('.pmui-treepanel-emptymessage').hide();
        return this;
    };
    /**
     * Shows an empty message when there are no items to display.
     * @chainable
     * @private
     */
    TreePanel.prototype.showEmptyMessage = function() {
        var node, message;
        if(!this.html) {
            return this;
        }
        node = jQuery(this.html).find('.pmui-treepanel-emptymessage').empty().get(0);
        if(!node) {
            node = PMUI.createHTMLElement('div');
            node.className = 'pmui-treepanel-emptymessage';
        }
        if(typeof this.emptyMessage === 'function') {
            message = this.emptyMessage(this, !!this.filterCriteria);
        } else if (typeof this.emptyMessage === 'string') {
            message = this.emptyMessage;
        } else {
            if(this.filterCriteria) {
                message = 'No matches found for \"' + this.filterCriteria + '\"';
            } else {
                message = 'No items found.';
            }
        }
        if(typeof message === 'string') {
            node.appendChild(document.createTextNode(message));
        } else if(PMUI.isHTMLElement(message)) {
            node.appendChild(message);
        }
        node.style.display = '';
        this.html.appendChild(node);
        return this;
    };
    /**
     * Clears the current filter.
     * @chainable
     */
    TreePanel.prototype.clearFilter = function() {
        return this.filter("");
    };
    /**
     * Filters the tree using a criteria.
     * @param  {String} filterCriteria The criteria.
     * @chainable
     */
    TreePanel.prototype.filter = function(filterCriteria) {
        var i = 0, childNodes = this.items.asArray(), partialRes, res = false;
        filterCriteria = filterCriteria.toString(10);
        this.filterCriteria = filterCriteria;
        jQuery(this.html).find('.pmui-treepanel-emptymessage').remove();
        for(i = 0; i < childNodes.length; i++) {
            childNodes[i].setVisible(partialRes = childNodes[i].filter(filterCriteria));
            res = res || partialRes;
        }
        if(!res) {
            this.showEmptyMessage();
        }
        return this;
    };
    /**
     * @inheritdoc
     */
    TreePanel.prototype.defineEvents = function() {
        TreePanel.superclass.prototype.defineEvents.call(this);
        this.filterControl.defineEvents();
        return this;
    };
    /**
     * @inheritdoc
     */
    TreePanel.prototype.createHTML = function() {
        var list, toolbar, title;
        if(this.html) {
            return this.html;
        }
        TreePanel.superclass.prototype.createHTML.call(this);
        title = PMUI.createHTMLElement('h2');
        title.className = 'pmui-treepanel-title';
        this.html.appendChild(title);
        this.dom.title = title;
        list = PMUI.createHTMLElement('ul');
        list.className = 'pmui-treepanel-list';
        this.dom.list = list;
        this.containmentArea = list;
        toolbar = PMUI.createHTMLElement('div');
        toolbar.className = 'pmui-treepanel-toolbar';
        toolbar.style.display = 'none';
        toolbar.appendChild(this.filterControl.getHTML());
        this.dom.toolbar = toolbar;
        this.html.appendChild(toolbar);
        this.html.appendChild(list);
        this.setItems(this.items.asArray().slice(0))
            .setTitle(this.title);
        if(this.filterable) {
            this.enableFiltering();
        } else {
            this.disableFiltering();
        }
        if(this.visibleTitle) {
            this.showTitle();
        } else {
            this.hideTitle();
        }
        return this.html;
    };

    PMUI.extendNamespace('PMUI.panel.TreePanel', TreePanel);

    if (typeof exports !== 'undefined'){
        module.exports === TreePanel;
    }
}());
(function(){
    /**
     * @class PMUI.panel.TabPanel
     * @extends PMUI.core.Panel
     * Is the container for every {@link PMUI.item.TabItem TabItem} class
     * 
     * Example:
     * 
     *     @example  
            var a,p,f,tree,w;

            //Form Panel*-*-*-*-*-*-*-**-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
            p = new PMUI.form.FormPanel({
                width: 604, 
                height: 130,
                fieldset: true,
                legend: "my fieldset panel",
                items: [
                    {
                        pmType: "text",
                        label: "Name",
                        id: "123",
                        value: "",
                        placeholder: "insert your name",
                        name: "name"
                    },{
                        pmType: "text",
                        label: "Last name",
                        value: "",
                        placeholder: "your lastname here asshole!",
                        name: "lastname"
                    }
                ],
                layout: "vbox"
            });
            //treePanel *-*-*-*-*-*-*-**-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
            tree = new PMUI.panel.TreePanel({
                items: [
                    {
                        label: "America",
                        items: [
                            {
                                label: "North America",
                                items: [
                                    {
                                        label: "Canada"
                                    },
                                    {
                                        label: "USA"
                                    },
                                    {
                                        label: "Mexico"
                                    }
                                ]
                            }
                        ]
                    }
                ]
            });
            //Layout Panel *-*-*-*-*-*-*-**-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
            f = new PMUI.form.Form({
                onSubmit: function() {
                    console.log("submitting...");
                },
                onChange: function(field, newValue, previousValue) {
                    console.log("The field " + field.getName() + " has changed from \"" + previousValue + "\" to \"" + newValue + "\"");
                },
                items: [
                    {
                        pmType: "dropdown",
                        label: "age",
                        options: [
                            {
                                label: "from 0 to 7 years old",
                                value: "0-7"
                            },
                            {
                                label: "from 8 to 13 years old",
                                value: "8-13"
                            }
                        ],
                        name: "age",
                        helper: "Select one of the options",
                        required: true
                    },
                    {
                        pmType: "radio",
                        label: "Gender",
                        value: "m",
                        name: "gender",
                        required: true,
                        options: [
                            {
                                label: "Male",
                                value: "m"
                            },
                            {
                                label: "Female",
                                value: "f"
                            }
                        ]
                    }
                ],
                buttons: [
                    {
                        text: "Submit",
                        handler: function() {
                            f.submit();
                            console.log("submitting form...");
                        }
                    }
                ]
            });                        
            //Tab Item*--**-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*
            a = new PMUI.panel.TabPanel({
                                            height : 400,
                                            width : 600,
                                            items:[
                                                {   
                                                    icon :'pmui-gridpanel-engranaje',
                                                    title:'Tree Panel',
                                                    panel:tree
                                                },
                                                {   
                                                    icon :'pmui-gridpanel-engranaje',
                                                    title:'Form Panel',
                                                    panel:p
                                                },
                                                {   icon :'pmui-gridpanel-engranaje',
                                                    title:'Form',
                                                    panel:f
                                                }
                                            ],
                                            itemsPosition : {position:"left",percentageWidth : 22}
                                            //itemsPosition : 'top'
                                    }); 
            //document.body.appendChild(a.getHTML());
            //a.defineEvents();
            w = new PMUI.ui.Window({
                        title: "TabPanel", 
                        width: 650, 
                        height: 460, 
                        footerHeight: 'auto', 
                        bodyHeight: 'auto', 
                        modal:true,
                        buttons: [ 
                            {
                                text: 'Saves',
                                handler: function() {
                                    alert('save');
                                }
                            },
                            {
                                text: 'Close',
                                handler: function() {
                                    alert('close');
                                }
                            },
                            {
                                text: "third button"
                            }
                        ],
                        buttonsPosition : 'center',
                        spaceButtons : 30
                                });
            w.open(); 
            w.addItem(a);
            w.defineEvents();

     * In the example, we use as a treePanel panels, FormPanel and layoutPanel, each TabItem can also use a className 
     * to put an icon, see {@link PMUI.item.TabItem}, {@link PMUI.item.TabItem#cfg-icon icon}. The pocision of items [TabItem]
     * can be "top", "left", "right" and "bottom", see the configuration options.
     * 
     * Note: used a window for a best visualisation.
     *
     * @cfg {Array} [items = []]
     * The items should be a Array, that Array should have Object JSON for config of the TabItems
     * for more detail see {@link PMUI.item.TabItem}
     * 
     * @cfg {String|Object} [itemsPosition = 'top']
     * This option config can be of type:
     * 
     *      Object, when position is "left" or "right", example: 
     *          itemsPosition: {position: "left", percentageWidth: 20} 
     *          itemsPosition: {position: "right", percentageWidth: 20} 
     *              JSON options:
     *              position: must be a string (can only be "left" or "right")
     *              percentageWidth: must be a number between 0 and 100
     *              
     *      String,  when the position is "top" or "bottom", example:
     *      itemsPosition: "top"
     *      itemsPosition: "bottom"
     *      percentageWidth is not require 
     */
    var TabPanel = function (settings) {
        TabPanel.superclass.call(this, settings);
        /**
         * Defines the tabpanel panelContainer HTML Element where is fix the content of the family Panel
         * @type {HTMLElement}
         */
        this.dom = {};
        /**
         * a property to assign events to see if the tabs is selected or not
         * @type {PMUI.ite.TabItem}
         */
        this.selectedTab = null;
        /**
         * a property that can be true or false values​​, when set to true indicates 
         * that the items will be collapsible
         * @type {boolean}
         */
        this.collapsible = null;
        /**
         * this posision items should be 'top', 'left', 'right' or 'bottom'
         * @type {String}
         */
        this.itemsPosition = null;
        /**
         * represents the space busy for the  containerItems HTMLElement,  default value  
         * this property is 0, when itemsPosition is "right" or "left " this value is different 0
         * @type {Number}
         */
        this.percentageWidth = null;
        this.onTabClick = null;
        TabPanel.prototype.init.call(this, settings);
    };

    PMUI.inheritFrom("PMUI.core.Panel", TabPanel);

    TabPanel.prototype.type = "tabpanel";
    TabPanel.prototype.family = "tabpanel";

    TabPanel.prototype.init = function (settings) {
        var defaults = {
            items : [],
            itemsPosition : 'top',
            onTabClick : null,
            collapsible : true
        };
        jQuery.extend(true, defaults, settings);
        this.setItems(defaults.items);
        this.setItemsPosition(defaults.itemsPosition);
        this.setOnTabClick(defaults.onTabClick);
        this.setCollapsible(defaults.collapsible);
    };

    TabPanel.prototype.setCollapsible  = function (collapsible){
        if(typeof collapsible !== 'boolean') {
            throw new Error ('setCollapsible(): the parameter collapsible should be type "boolean"');
        }
        this.collapsible = collapsible; 
        return this;
    };

    TabPanel.prototype.setOnTabClick = function (tabClick) {
        if(typeof tabClick !== 'function' && tabClick !== null) {
            throw new Error("setOnTabClick(): the parameter is not valid, should be a function.");
        }
        this.onTabClick = tabClick;
        return this;   
    };

    /**
     * set in this method and call addItem to create the TabItem
     * @param {Array} The JSON array with objects with parameters for creating TabItem 
     * @chainable
     */
    TabPanel.prototype.setItems = function (items) {
        if (jQuery.isArray(items)) {
            this.clearItems();
            for( i = 0 ; i < items.length ; i+=1 ) {
                this.addItem(items[i]);
            }
        } else {
            throw new Error("setTabs(): the parameter is not a valid, should to be as Array");
        }
        for(i = 0; i < this.items.getSize(); i+=1) {
                if(i==0){
                    this.items.get(0).select();
                }else{
                    this.items.get(i).deselect();
                }
            }
        return this;
    };
    /**
     * [onTabClickHandler description]
     * @return {[type]} [description]
     * @private
     */
    TabPanel.prototype.onTabClickHandler = function() {
        var that = this;
        return function(tabItem) {
            if(typeof that.onTabClick === 'function') {
                that.onTabClick(tabItem);
            }
        };
    };
    /**
     * Add an item to the panel.
     * @param {PMUI.item.TabItem} item
     * It can be a valid JSON object or an object that inherits from {@link PMUI.item.TabItem}.
     * @chainable
     */
    TabPanel.prototype.addItem = function (item) {
        var tabItem, i;
        item.onSelect = this.onTabSelectHandler();
        tabItem = new PMUI.item.TabItem(item);
        tabItem.deselect();
        tabItem.setOnClick(this.onTabClickHandler());
        this.items.insert(tabItem);
        if (this.html) {
            this.dom.listContainer.appendChild(tabItem.getHTML());
            this.updatePosition();
            if (this.eventsDefined) {
                tabItem.defineEvents();
            }
        }
        return this;
    };
    /**
     * @event
     * defines the function to be determined in each tabItem when it is selected, to 
     * call the event set
     * @chainable
     */ 
    TabPanel.prototype.onTabSelectHandler = function () {
        var that = this, i;
        return function () {
            if(that.selectedTab) {
                jQuery(that.selectedTab.panel.getHTML()).detach();    
            }
            that.selectedTab = this;
            for (i = 0; i < that.items.getSize(); i++) {
                if (that.items.get(i) !== that.selectedTab) {
                    that.items.get(i).deselect();
                }
            }
            //jQuery(that.dom.panelContainer).empty();
            jQuery(that.dom.panelContainer).append(this.getPanel().getHTML());
            if(!this.getPanel().eventsDefined){
                this.getPanel().defineEvents();
            }
        };
    };
    /**
     * @method  setItemsPosition
     * changes the position of the elements [PM UI.item.TabItem] within the sizes assigned to TabPanel.
     * @param {String|Object}
     * Position where items are positioned, the more info about input value see {@link PMUI.panel.TabPanel} 
     * config option: {@link PMUI.panel.TabPanel#cfg-itemsPosition itemsPosition}
     * @chainable
     */
    TabPanel.prototype.setItemsPosition = function (itemsPosition) {
        var validPosition, percentageWidth; 
        if(typeof itemsPosition == "string" || typeof itemsPosition == "object") {
            if (typeof itemsPosition == "object") {
                if(itemsPosition.position == "left" || itemsPosition.position == "right") {
                    this.itemsPosition = itemsPosition.position;
                    this.percentageWidth = itemsPosition.percentageWidth || 20; 
                } else {
                    throw new Error ("setItemsPosition(): JSON no valid, send position : 'left' or 'right' with percentage width");
                }
            } else {
                if(itemsPosition == "top" || itemsPosition == "bottom") {
                    this.itemsPosition = itemsPosition;
                    this.percentageWidth = 0; 
                } else {
                    throw new Error ("setItemsPosition(): no valid, send position : 'top' or 'bottom' ");
                }
            }

            this.style.removeClasses(['pmui-tabpanel-left','pmui-tabpanel-right','pmui-tabpanel-top','pmui-tabpanel-bottom']);
            this.style.addClasses(['pmui-tabpanel-' + this.itemsPosition]);

            if(this.html) {
                 this.updatePosition();
            }
        } else {
            throw new Error ("setItemsPosition(): the parameter is not a valid, should to be as string or Object JSON");
        }
        return this;
    };
    /**
     * @private
     * update the position for the HTMLElement.
     * exists the next elements HTMLElements:
     * 
     *     - containerItems
     *     - listContainer
     *     - barCollaspible
     *     - panelContainer
     *     
     * this elements should update your position in the DOM, also update dimensions respect to the property itemsPosition 
     * @chainable
     */
    TabPanel.prototype.updatePosition = function () {
        var i, itemsDisplay;
        switch (this.itemsPosition) {
            case 'top' : case 'left':
                $(this.dom.containerItems).prepend(this.dom.listContainer);
                $(this.html).prepend(this.dom.containerItems);


                this.dom.containerItems.style.display = this.itemsPosition === 'left' ? "inline-block" : "block";
                this.dom.listContainer.style.display = this.itemsPosition === 'left' ? "inline-block" : "block";
                this.dom.collapsibleBar.style.display = this.itemsPosition === 'left' ? "inline-block" : "none";
                this.dom.panelContainer.style.display = this.itemsPosition === 'left' ? "inline-block" : "block";
                $(this.dom.containerItems).css("float", this.itemsPosition === 'left' ? "left" : "");

                itemsDisplay = this.itemsPosition === 'left' ? "block" : "inline-block";

                for (var i = 0 ;i< this.items.getSize(); i+=1) {
                    this.items.get(i).setDisplay(itemsDisplay);
                }

                if (this.itemsPosition == 'left') {
                    this.dom.containerItems.style.width = this.getWidth() * (this.percentageWidth/100) + 'px';
                    this.dom.containerItems.style.height = this.getHeight()+"px";
                    //this.dom.containerItems.style.height = this.getHeight() + 'px';
                    this.dom.panelContainer.style.width = this.getWidth() -  (this.getWidth() * (this.percentageWidth/100)) + 'px';
                    this.dom.panelContainer.style.height = this.getHeight() + 'px';
                    this.dom.collapsibleBar.style.height = this.getHeight() +'px';
                    jQuery(this.dom.listContainer).css("float", 'left');
                    for (var i = 0 ;i< this.items.getSize(); i+=1) {
                        this.items.get(i).setWidth(this.getWidth() * (this.percentageWidth/100)-12);
                    }                    
                } else {
                     
                    this.dom.containerItems.style.width = this.getWidth() + 'px';
                    this.dom.containerItems.style.height = '40px';
                    //this.dom.containerItems.style.height = '52px';
                    this.dom.panelContainer.style.width = this.getWidth() + 'px';
                    this.dom.panelContainer.style.height = this.getHeight()- 40 + 'px';
                    this.dom.collapsibleBar.style.height = this.getHeight() +'px';
                    $(this.dom.listContainer).css("float", 'left');
                    for (var i = 0 ;i< this.items.getSize(); i+=1) {
                        this.items.get(i).setWidth('auto');
                    }                    
                }
                break;
            case 'right' : case 'bottom' :
                $(this.dom.containerItems).prepend(this.dom.collapsibleBar); 
                $(this.html).prepend(this.dom.panelContainer);
                                this.dom.containerItems.style.display = this.itemsPosition === 'right' ? "inline-block" : "block";
                this.dom.listContainer.style.display = this.itemsPosition === 'right' ? "inline-block" : "block";
                this.dom.collapsibleBar.style.display = this.itemsPosition === 'right' ? "inline-block" : "none";
                this.dom.panelContainer.style.display = this.itemsPosition === 'right' ? "inline-block" : "block";
                $(this.dom.containerItems).css("float", this.itemsPosition === 'right' ? "right" : "");

                itemsDisplay = this.itemsPosition === 'right' ? "block" : "inline-block";

                for (var i = 0 ;i< this.items.getSize(); i+=1) {
                    this.items.get(i).setDisplay(itemsDisplay);
                }

                if (this.itemsPosition == 'right') {
                    this.dom.containerItems.style.width = this.getWidth() * (this.percentageWidth/100) + 'px';
                    this.dom.containerItems.style.height = this.getHeight()+"px";
                    //this.dom.containerItems.style.height = this.getHeight() + 'px';
                    this.dom.panelContainer.style.width = this.getWidth() -  (this.getWidth() * (this.percentageWidth/100)) + 'px';
                    this.dom.panelContainer.style.height = this.getHeight() + 'px';
                    this.dom.collapsibleBar.style.height = this.getHeight() +'px';
                    $(this.dom.listContainer).css("float", 'right');
                    for (var i = 0 ;i< this.items.getSize(); i+=1) {
                        this.items.get(i).setWidth(this.getWidth() * (this.percentageWidth/100));
                    }                    
                } else { 
                    this.dom.containerItems.style.width = this.getWidth() + 'px';
                    this.dom.containerItems.style.height = '40px';
                    //this.dom.containerItems.style.height = '52px';
                    this.dom.panelContainer.style.width = this.getWidth() + 'px';
                    this.dom.panelContainer.style.height = this.getHeight() + 'px';
                    this.dom.collapsibleBar.style.height = this.getHeight() +'px';
                    $(this.dom.listContainer).css("float", 'none');
                    for (var i = 0 ;i< this.items.getSize(); i+=1) {
                        this.items.get(i).setWidth('auto');
                    }                    
                }               
                break;
        }
        return this;
    };
    /**
     * Create html for each item to container 
     * @return {Object} the current object html
     */
    TabPanel.prototype.createHTML = function () {
        var containerItems, panelContainer, collapsibleBar, listContainer,i;
        if(this.html) {
            return html;
        }
        this.html = PMUI.createHTMLElement('div');
        this.html.className = 'pmui-tabpanel';
        this.html.id = this.id;
        containerItems = PMUI.createHTMLElement('div');
        listContainer = PMUI.createHTMLElement('ul');
        collapsibleBar = PMUI.createHTMLElement('div');
        panelContainer = PMUI.createHTMLElement('div');

        containerItems.className = "pmui-tabpanel-tabs_container";
        listContainer.className = "pmui-tabpanel-tabs" ;
        collapsibleBar.className = "pmui-collapsibleBar";
        panelContainer.className = "pmui-tabpanel-container";

        this.dom.containerItems = containerItems ;
        this.dom.listContainer = listContainer;
        this.dom.collapsibleBar = collapsibleBar;
        this.dom.panelContainer = panelContainer;
        this.dom.panelContainer.style.overflow = 'auto';
        this.setItems(this.items.asArray().slice(0));

        this.dom.containerItems.appendChild(this.dom.listContainer);
        this.dom.containerItems.appendChild(this.dom.collapsibleBar);
        this.html.appendChild(this.dom.containerItems);
        this.html.appendChild(this.dom.panelContainer);
        
        if(this.itemsPosition == 'left' || this.itemsPosition == 'right') {
            this.setItemsPosition({position:this.itemsPosition, percentageWidth: this.percentageWidth});
        } else {
            this.setItemsPosition(this.itemsPosition);
        }
        this.applyStyle();
        if(this.eventsDefined) {
            this.defineEvents();
        }

        for ( i = 0 ; i < this.items.getSize(); i+=1) {
            if(!this.getItems()[i].visible){
                this.hideTab(i)
            }
        }
        return this.html;
    };
    /**
     * Executes children events defined
     * @chainable
     */
    TabPanel.prototype.defineEvents = function () {
        var that = this, i, auxPercentageWidth;
        this.removeEvents().eventsDefined = true;
        if(this.dom.collapsibleBar && this.collapsible) {
            this.addEvent('click').listen(this.dom.collapsibleBar, function(e){
                e.preventDefault();
                e.stopPropagation();
                auxPercentageWidth = that.percentageWidth;
                if(that.collapsible) {
                    that.setItemsPosition({position:that.itemsPosition, percentageWidth: 9});  
                    that.collapsible = false;
                    that.style.addClasses(['pmui-collipsable']);
                }else{
                    that.setItemsPosition({position:that.itemsPosition, percentageWidth: auxPercentageWidth});
                    that.style.removeClasses(['pmui-collipsable']);
                    that.collapsible = true;

                }
                    that.percentageWidth = auxPercentageWidth;

                if(typeof that.onTabClick == 'function') {

                }
            });
            for (i = 0; i < that.items.getSize() ; i+=1) {
                that.items.get(i).defineEvents();
            }
        }   
        return this;
    };

    TabPanel.prototype.setWidth = function (width) {
        TabPanel.superclass.prototype.setWidth.call(this,width);
        this.updatePosition();
        return this;
    };

    TabPanel.prototype.setHeight = function (height) {
        TabPanel.superclass.prototype.setHeight.call(this,height);
        this.updatePosition();
        return this;
    };

    /**
     * hides a TabItem, if hidden the last TabItem passes focus to the previous, 
     * otherwise the selection will move to the next always
     * @param  {Number} index  Is the selection of the TabItem that is concealed, 
     * must be within the range of the length of existing items
     * @param  {[type]} current [when hidden a tabItem, can fix focus in other tabItem]
     * @chainable
     */
    TabPanel.prototype.hideTab = function (index, current) {
        var item,currentTab,i,len;

        for (i = 0 ; i < this.getItems().length ; i++) {
            if (i===index){
                item = this.getItems()[i];
            }
        }
        if(!current) {
            if(item.selected) {
                len = index-1;            
                while (len > -1) {
                    if(this.getItems()[len].visible === true) {
                        currentTab =  this.getItems()[len];
                        len = -1;
                    }else{
                        len--;
                    }
                }
                if(!currentTab) {
                    i = index +1;
                    while ( i < this.items.getSize()) {
                        if(this.getItems()[i].visible === true) {
                            currentTab =  this.getItems()[i];
                            i = this.items.getSize();
                        }else{
                            i++;
                        }   
                    }
                }
            }
        } else {
            currentTab = this.getItems()[current];
        }

        item.setVisible(false);
        item.panel.setVisible(false);
        if(currentTab) {
            currentTab.select();
        }
        return this;
    }
    /**
     * tab shows a hidden item
     * @param  {Number} index Este valor es necesario para mostrar TabItem, debe estar dentro de un rango de la longitud de los elementos
     * @chainable
     */
    TabPanel.prototype.showTab = function (index) {

        if (typeof index == 'number') {
            if (!this.getItems()[index].visible){
                this.getItems()[index].setVisible(true);
                this.getItems()[index].panel.setVisible(true);
            }
        }   
        return this;
    };


    TabPanel.prototype.itemClick = function (index) {
        $($(this.dom.listContainer).find('a')[index]).trigger('click');
        return this;
    };

    if( typeof exports != 'undefined') {
        module.exports = TabPanel;
    }
    PMUI.extendNamespace("PMUI.panel.TabPanel", TabPanel);
}());
(function(){
    /**
     * @class PMUI.panel.AccordionPanel
     * @extends PMUI.core.Panel
     * 
     * Class to handle an Accordion component, this is a container for {@link PMUI.item.AccordionItem AccordionItem}.
     * 
     * The way of usage the class is showed on the example above
     *
     *              @example
     *               accordion   = new PMUI.panel.AccordionPanel({
     *                   width: 400,
     *                   height: 500,
     *                   hiddenTitle: true,
     *                   //multipleSelection : true,
     *                   title: "My accordion",
     *                   items: [
     *                       {
     *                           iconClass: "",
     *                           title: "First item",
     *                           selected: true
     *                       },
     *                       {
     *                           iconClass: "pmui-icon pmui-icon-warning",
     *                           title: "Second item",
     *                           body: "<a href=\"http://www.google.com\">Google Link</a>",
     *                           style: {
     *                               cssProperties: {
     *                                   "background-color": "#f2eaea"
     *                               }
     *                           }
     *                       },
     *                       {
     *                           title: "Third item <span class=\"classname-accordion\"></span>"
     *                       },
     *                       {
     *                           title: "Fourth"
     *                        }, new PMUI.item.AccordionItem({
     *                           title: "Five"
     *                        })
     *                   ],
     *                   listeners: {
     *                       select: function (obj, event) {
     *                           console.log("Selected...", obj);
     *                       }
     *                   }
     *               });
     *               document.body.appendChild(accordion.getHTML());
     *               accordion.defineEvents();
     *
     *
     * @constructor      
     * Creates a new component 
     * @param {Object} [settings] settings The configuration options may be specified as the follow sentence
     *      {
     *          items: [
     *              {
     *                  title: "First element",
     *                  body: item // This item can be any element that inherit from {@link PMUI.core.Panel Panel}
     *              },
     *              {
     *                  title: "Second element"
     *              }
     *          ]
     *      }
     * @cfg {String} name The name for the field.
     *
     * @cfg {Boolean} [hiddenTitle=true] Represents wether the title of the component will be showed, the
     * otherwise the panel will rendered without title.
     * @cfg {Array} [items=[]] The array contains the basic elements for {@PMUI.panel.AccordionPanel AccordionPanel class}
     * multipleSelection.
     * @cfg {String} [title=""] The title for the panel.
     * @cfg {Number} [heightItem = 'current height * 0.3'] Represents the height for every item
     * @cfg {Object} factory If it necessary is possible replace the current elements by default.
     *      factory: {
     *          products: {
     *              'accordionitem': PMUI.item.AccordionItem
     *          },
     *          defaultProduct: "accordionitem"
     *      }
     *
     */

    var AccordionPanel = function (settings) {
        AccordionPanel.superclass.call(this, settings);
        /**
         * @property {String} [title= ""] The text represents the title of the panel,
         * Whether the {PMUI.panel.AccordionPanel this.hiddenTitle} is false.
         */
        this.title = null;
        /**
         * @property {Boolean} [hiddenTitle=false] The property represents whether the title 
         * will be showed.
         */
        this.hiddenTitle = null;
        /**
         * @property {Boolean} [multipleSelection=false] If the property is enabled, the {@link PMUI.item.AccordionItem items} can
         * be selected more than one time
         */
        this.multipleSelection = null;
        /**
         * @property {Object} [listeners] A config object containing one or more handlers to be added 
         * to this object during initialization. by default the 'select' handler is enabled
         */
        this.listeners = {
            /**
             * @event select
             * Fires when a {@link PMUI.item.AccordionItem component} has been selected.
             * @param {Object} accordionItem {@link PMUI.item.AccordionItem AccordionItem}
             * @param {Object} event event
             */
            select: function(){}
        };
        /**
         * @property {HTMLElement} [header=null] 
         * The property encapsulate the HTMLElement header for the panel.
         */
        this.header = null;
        /**
         * @property {HTMLElement} [body=null] 
         * The property is related to HTMLElement body that will be used by
         * {@link PMUI.item.AccordionItem items}
         */
        this.body = null;
        /**
         * @property {HTMLElement} [footer=null] 
         * The property encapsulate the HTMLElement footer for the panel.
         */
        this.footer = null;
        /**
         * @property {String} [iconClass='pmui-accordion-panel-icon']
         * Represents the class name for the icon.
         * It can be replaced on the config JSON if it is necessary
         * @private
         */
        this.iconClass = null;
        /**
         * @property {String} [headerClass='pmui-accordion-panel-header']
         * Represents the class name for the header of the panel.
         * It can be replaced on the config JSON if it is necessary
         * @private
         */
        this.headerClass = null;
        /**
         * @property {String} [bodyClass='pmui-accordion-panel-body']
         * Represents the class name for the body of the panel.
         * It can be replaced on the config JSON if it is necessary
         * @private
         */
        this.bodyClass = null;
        /**
         * @property {String} [footerClass='pmui-accordion-panel-footer']
         * Represents the class name for footer of the panel.
         * It can be replaced on the config JSON if it is necessary
         * @private
         */
        this.footerClass = null;
        /**
         * @property {String} [containerClass='pmui-accordion-panel-container']
         * Represents the class name for the father container.
         * It can be replaced on the config JSON if it is necessary
         * @private
         */
        this.containerClass = null;
        /**
         * @property {Number} [heightItem = 'current height * 0.3']
         * Represents the size height for every {@link PMUI.item.AccordionItem item}
         */
        this.heightItem = null;
        /**
         * determines whether the items of the accordion panel is automatically set when adding a new item
         * @type {Boolean}
         */
        this.selfAdjusting = null; 
        AccordionPanel.prototype.init.call(this, settings);

    };

    PMUI.inheritFrom('PMUI.core.Panel', AccordionPanel);
    /**
     * Defines the object type of the element
     */
    AccordionPanel.prototype.type = 'Accordion';
    /**
     * Defines the object family of the element
     */
    AccordionPanel.prototype.family = 'Panel';

    AccordionPanel.prototype.init = function (settings) {
        var defaults = {
            multipleSelection: false,
            hiddenTitle: false,
            title: '',
            iconClass: 'pmui-accordion-panel-icon',
            headerClass: 'pmui-accordion-panel-header',
            bodyClass: 'pmui-accordion-panel-body',
            footerClass: 'pmui-accordion-panel-footer',
            containerClass: 'pmui-accordion-panel-container',
            heightItem: this.height * 0.3,
            items: [],
            factory: {
                products: {
                    'accordionitem': PMUI.item.AccordionItem
                },
                defaultProduct: "accordionitem"
            },
            listeners: {
                select: function(){}
            },
            selfAdjusting : false
        };

        jQuery.extend(true, defaults, settings);

        this.setFactory(defaults.factory)
            .setMultipleSelection(defaults.multipleSelection)
            .setHiddenTitle(defaults.hiddenTitle)
            .setTitle(defaults.title)
            .setIconClass(defaults.iconClass)
            .setHeaderClass(defaults.headerClass)
            .setBodyClass(defaults.bodyClass)
            .setFooterClass(defaults.footerClass)
            .setContainerClass(defaults.containerClass)
            .setHeightItem(defaults.heightItem)
            .setChildren(defaults.items)
            .setListeners(defaults.listeners)
            .setSelfAdjusting(defaults.selfAdjusting);
    };

    /**
     * Adds an child item to the object
     * @param {PMUI.core.Element|Object} item It can be one of the following data types:
     * - {PMUI.core.Element} the object to add
     * - {Object} a JSON object with the settings for the Container to be added
     * @chainable
     */
    AccordionPanel.prototype.addItem = function (item) {
        var itemToBeAdded;
        if (this.factory) {
            itemToBeAdded = this.factory.make(item);
        }
        if (itemToBeAdded && !this.isDirectParentOf(itemToBeAdded)) {
            itemToBeAdded.parent = this;
            this.items.insert(itemToBeAdded);
            if (this.body) {
                this.body.appendChild(itemToBeAdded.getHTML());
                if(this.selfAdjusting){
                    this.adjustHeightItems();
                }
                if(!itemToBeAdded.eventsDefined){
                    itemToBeAdded.defineEvents();
                    
                }
            }
        }
        return this;
    };
    /**
     * Adjust the height size for every {@link PMUI.item.AccordionItem item}.
     * Get the current size of the element and set the height for every child based
     * in the height of header, headerItems and footer.
     *
     */
    AccordionPanel.prototype.adjustHeightItems = function () {
        var maxHeight = this.height,
            headPanel = jQuery(this.header).outerHeight(),
            footerPanel =  jQuery(this.footer).outerHeight(),
            headItems = 0,
            footerItems = 0,
            bodyHeight = 0,
            items,
            i,j;

        items = this.getItems();

        for (i = 0; i < items.length; i+=1) {
            headItems += jQuery(items[i].header.html).outerHeight()+2;
            if (items[i].footer) {
                footerItems += jQuery(items[i].footer.html).outerHeight();    
            }
        } 
        
        bodyHeight = maxHeight - (headPanel+footerPanel+headItems+footerItems);
        for (j = 0; j < items.length; j+=1) {
            jQuery(items[j].body.html).height(bodyHeight);
        }
        return this;
    };
    /**
     * Clear all the object's current child items and add new ones
     * @param {Array} items An array where each element can be a {Element} object or a JSON object 
     * where is specified the setting for the child item to be added
     * @chainable
     */
    AccordionPanel.prototype.setChildren = function (items) {
        var i;
        if (jQuery.isArray(items)) {
            this.clearItems();
            for (i = 0; i < items.length; i += 1) {
                this.addItem(items[i]);
            }
        }
        return this;
    };
    /**
     * Defines and sets whether the {@link PMUI.item.AccordionItem items} from {@link PMUI.panel.AccordionPanel Accordion}
     * will be selectable
     * @param {Boolean} parameter
     * @return {PMUI.panel.AccordionPanel} this
     */
    AccordionPanel.prototype.setMultipleSelection = function (parameter) {
        if (typeof parameter === 'boolean') {
            this.multipleSelection = parameter;
        }
        return this;
    };
    /**
     * Sets whether the title will be rendered or not inside the header
     * @param {Boolean} enabled
     */
    AccordionPanel.prototype.setHiddenTitle = function (enabled) {
        if (typeof enabled === 'boolean') {
            this.hiddenTitle = enabled;

            if (this.html) {
                if (this.hiddenTitle === true) {
                    this.header.setAttribute("style","display:none")    
                } else {
                    this.header.removeAttribute("style");
                }
                this.adjustHeightItems();
            }
        }
        return this;  
    };
    /**
     * Sets the label for the Accordion title
     * @param {String} title
     */
    AccordionPanel.prototype.setTitle = function (title) {
        if (typeof title === "string") {
            this.title = title;
            
            if (this.html) {
                this.header.children[1].innerHTML = title;
            }
        }
        
        return this;
    };
    /**
     * Sets the class name for the header
     * @param {String} iconCls
     * @private
     */
    AccordionPanel.prototype.setIconClass = function (iconCls) {
        this.iconClass = iconCls;
        return this;
    };
    /**
     * Sets the class name for the Accordion's header
     * @param {String} class name
     * @private
     */
    AccordionPanel.prototype.setHeaderClass = function (className) {
        if (typeof className === 'string') {
            this.headerClass = className;
        }
        return this;
    };
    /**
     * Sets the class name for the Accordion's body
     * @param {String} class name
     * @private
     */
    AccordionPanel.prototype.setBodyClass = function (className) {
        if (typeof className === 'string') {
            this.bodyClass = className;
        }
        return this;
    };
    /**
     * Sets the class name for the Accordion's footer
     * @param {String} class name
     * @private
     */
    AccordionPanel.prototype.setFooterClass = function (className) {
        if (typeof className === 'string') {
            this.footerClass = className;
        }
        return this;
    };
    /**
     * Sets the class name for the object's container
     * @param {String} class name
     * @private
     */
    AccordionPanel.prototype.setContainerClass = function (className) {
        if (typeof className === 'string') {
            this.containerClass = className;
        }
        return this;
    };
    /**
     * Sets the height for every {@link PMUI.item.AccordionItem item}
     * @param {String} height
     */
    AccordionPanel.prototype.setHeightItem = function (height) {
        this.heightItem = height;
        return this;
    };
    /**
     * Sets the listeners for the object
     * @param {Object} events
     */
    AccordionPanel.prototype.setListeners = function (events) {
        if (typeof events === 'object') {
            this.listeners = events;
        }
        return this;
    };
    /**
     * Creates all the elements with their corresponding attributes
     * for make a container that encapsulate {@link PMUI.item.AccordionItem items}
     * @return  {HTMLElement} HTMLElement
     */
    AccordionPanel.prototype.createHTML = function () {
        var container,
            header,
            body,
            footer,
            title,
            icon,
            itemsLen,
            i;
        if (this.html) {
            return this.html;
        }
        container = AccordionPanel.superclass.prototype.createHTML.call(this);
        header = PMUI.createHTMLElement("div");
        header.setAttribute('class',this.headerClass);
        body = PMUI.createHTMLElement("div");
        body.setAttribute('class',this.bodyClass);
        footer = PMUI.createHTMLElement("div");
        footer.setAttribute('class',this.footerClass);

        title = PMUI.createHTMLElement("span");
        title.innerHTML = this.title;
        icon = PMUI.createHTMLElement("span");
        icon.setAttribute('class',this.iconClass);
        header.appendChild(icon);
        header.appendChild(title);
        if (this.hiddenTitle) {
            header.setAttribute("style", "display:none;");
        }
        itemsLen = this.items.getSize();
        if (typeof itemsLen === 'number') {
            for (i = 0; i < itemsLen; i += 1) {
                body.appendChild(this.getItems()[i].getHTML());
            }
        }

        this.header = header;
        this.body = body;
        this.footer = footer;
        container.appendChild(header);
        container.appendChild(body);
        container.appendChild(footer);
        container.className = this.containerClass;

        this.html = container;
        return this.html;
    };

    /**
     * Defines events related to the component and call to every DefineEvents method from 
     * {@link PMUI.item.AccordionItem items}
     * @return {Object} {@link PMUI.panel.AccordionPanel Accordion}
     */
    AccordionPanel.prototype.defineEvents = function () {
        var j,
            children,
            that = this;
        
        if (that.items.getSize() > 0 && this.eventsDefined !== true) {
            children = that.getItems();
            for (j = 0; j < children.length; j+=1) {
                children[j].defineEvents();
            }
            if(this.selfAdjusting){
                this.adjustHeightItems();
            }
        }

        return this;
    };

    AccordionPanel.prototype.setSelfAdjusting = function (value) {
        if (typeof value === "boolean") {
            this.selfAdjusting = value;
        }
        return this;
    };

    PMUI.extendNamespace('PMUI.panel.AccordionPanel', AccordionPanel);
    if (typeof exports !== "undefined") {
        module.exports = AccordionPanel;
    }
}());
(function() {
    /**
     * @class PMUI.panel.ButtonPanel
     * A panel that contains instances of {@link PMUI.ui.Button Button} and instances of {@link PMUI.ui.TextLabel Label}.
     * @extends {PMUI.core.Panel}
     *
     * Usage example:
     *
     *      @example
     *      var a = new PMUI.panel.ButtonPanel({
     *          items: [
     *              {
     *                  pmType: 'button',
     *                  text: 'Click Me'
     *              },
     *              {
     *                  pmType: 'label',
     *                  text: 'and then'
     *              },
     *              {
     *                  pmType: 'button',
     *                  text: 'Click Me'
     *              }
     *          ]
     *      });
     *      document.body.appendChild(a.getHTML());
     *      a.defineEvents();
     *
     * @constructor
     * Creates a new instance of the class.
     * @param {Object} [settings={}] An object literal with the config options for the class.
     *
     * @cfg {Array} [items=[]] An array in which each element can be:
     *
     * - An object literal, in this case it can have the config options for create a {@link PMUI.ui.Button Button} or a 
     * {@link PMUI.ui.TextLabel Label}, additionally it must include the respective pmType ('button' for Button and 'label'
     * for Label).
     *
     * -A PMUI object, in this case it must be an instance of {@link PMUI.ui.Button Button} or an instance of {@link 
     * PMUI.ui.TextLabel Label}.
     */
    var ButtonPanel = function(settings) {
        ButtonPanel.superclass.call(this, settings);
        this.alignment = null;
        ButtonPanel.prototype.init.call(this, settings);
    };

    PMUI.inheritFrom('PMUI.core.Panel', ButtonPanel);
    /**
     * The object's type.
     * @type {String}
     */
    ButtonPanel.prototype.type = 'ButtonPanel';
    /**
     * Initializes the object.
     * @param  {Object} [settings={}] An object literal with the config options the new object will be initialized 
     * with.
     * @chainable
     * @private
     */
    ButtonPanel.prototype.init = function(settings) {
        var defaults = {
            items: [],
            factory: {
                products: {
                    'button': PMUI.ui.Button,
                    'label': PMUI.ui.TextLabel
                },
                defaultProduct: 'button'
            },
            layout: 'box',
            alignment: 'center'
        };

        jQuery.extend(true, defaults, settings);

        this.setFactory(defaults.factory)
            .setLayout(defaults.layout)
            .setItems(defaults.items)
            .setAlignment(defaults.alignment);

        return this;
    };
    /**
     * Sets the alignment for the items in the panel.
     * @param {String} alignment A string which can take the values of 'center', 'left', 'right'.
     */
    ButtonPanel.prototype.setAlignment = function(alignment) {
        if(alignment === 'center' || alignment === 'left' || alignment === 'right') {
            this.alignment = alignment;
            this.style.addProperties({'text-align': alignment});
            return this;
        }
        throw new Error('setAlignment(): The parameter must be "center" or "left" or "right".');
    };

    PMUI.extendNamespace('PMUI.panel.ButtonPanel', ButtonPanel);

    // Declarations created to instantiate in NodeJS environment
    if (typeof exports !== "undefined") {
        module.exports = ButtonPanel;
    }
}());
(function() {
    /**
     * @class PMUI.panel.List
     */
    var ListPanel = function(settings){
        this.dom = {};
        ListPanel.superclass.call(this, settings);
        this.orientation = null;
        this.actions = null;
        this.dataItems = null;
        this.filterable = null;
        this.filterCriteria = null;
        this.filterControl = null;
        this.filterPlaceholder = null;
        this.statusBarMessage = null;
        this.visibleHeader = null;
        this.visibleStatusBar = null;
        this.listHeight = null;
        this.onItemClick = null;
        this.filteredItems = null;
        ListPanel.prototype.init.call(this, settings);
    };

    PMUI.inheritFrom('PMUI.core.Container', ListPanel);

    ListPanel.prototype.type = "ListPanel";

    ListPanel.prototype.init = function(settings) {
        var defaults = {
            orientation: 'vertical',
            actions: [],
            dataItems: null,
            itemTextDataBind: null,
            statusBarMessage: null,
            title: "[untitled list]",
            filterPlaceholder: "Search",
            visibleHeader: true,
            visibleStatusBar: true,
            filterable: true,
            listHeight: "auto",
            onItemClick: null
        };
        this.filteredItems = [];
        
        jQuery.extend(true, defaults, settings);

        this.filterControl = new PMUI.control.TextControl({
            onKeyUp: this.onFilterControlChangeHandler(),
            style: {
                cssClasses: [
                    "pmui-listpanel-search"
                ]
            }
        });

        this.setItemTextDataBind(defaults.itemTextDataBind)
            .setOrientation(defaults.orientation)
            .setElementTag(defaults.elementTag)
            .setActions(defaults.actions)
            .setStatusBarMessage(defaults.statusBarMessage)
            .setFilterPlaceholder(defaults.filterPlaceholder)
            .setTitle(defaults.title)
            .setListHeight(defaults.listHeight)
            .setOnItemClickHandler(defaults.onItemClick);

        if(defaults.filterable) {
            this.enableFiltering();
        } else {
            this.disableFiltering();
        }

        if(defaults.visibleHeader) {
            this.showHeader();
        } else {
            this.hideHeader();
        }

        if(defaults.visibleStatusBar) {
            this.showStatusBar();
        } else {
            this.hideStatusBar();
        }
    };

    ListPanel.prototype.onItemClickHandler = function(listItem) {
        if(typeof this.onItemClick === 'function') {
            this.onItemClick(this, listItem);
        }
        return this;
    };

    ListPanel.prototype.setOnItemClickHandler = function(handler) {
        if(!(handler === null || typeof handler === 'function')) {
            throw new Error("setOnItemClickHandler(): The parameter must be a function or a handler.");
        }
        this.onItemClick = handler;
        return this;
    };

    ListPanel.prototype.setListHeight = function(height) {
        if(typeof height === 'number') {
            this.listHeight = height;   
        } else if(/^\d+(\.\d+)?px$/.test(height)) {
            this.listHeight = parseInt(height, 10);
        } else if(/^\d+(\.\d+)?%$/.test(height)) {
            this.listHeight = height;
        } else if(/^\d+(\.\d+)?em$/.test(height)) {
            this.listHeight = height;
        } else if(height === 'auto' || height === 'inherit') {
            this.listHeight = height;
        } else {
            throw new Error('setHeight: height param is not a number');
        }
        if(this.dom.list) {
            this.dom.list.style.height = height + (typeof height === 'number' ? 'px': '');
        }
        return this;
    };

    ListPanel.prototype.hideStatusBar = function() {
        this.visibleStatusBar = false;
        if(this.dom.statusBar) {
            this.dom.statusBar.style.display = 'none';
        }
        return this;
    };

    ListPanel.prototype.showStatusBar = function() {
        this.visibleStatusBar = true;
        if(this.dom.statusBar) {
            this.dom.statusBar.style.display = '';
        }
        return this;
    };

    ListPanel.prototype.showHeader = function() {
        this.visibleHeader = true;
        if(this.dom.header) {
            this.dom.header.style.display = '';
        }
        return this;
    };

    ListPanel.prototype.hideHeader = function() {
        this.visibleHeader = false;
        if(this.dom.header) {
            this.dom.header.style.display = 'none';
        }
        return this;
    };

    ListPanel.prototype.enableFiltering = function() {
        this.filterable = true;
        if(this.dom.toolbar) {
            this.dom.toolbar.style.display = '';
        }
        return this;
    };

    ListPanel.prototype.disableFiltering = function() {
        this.filterable = false;
        if(this.dom.toolbar) {
            this.dom.toolbar.style.display = 'none';
        }
        this.clearFilter();
        return this;
    };

    ListPanel.prototype.setTitle = function(title) {
        if(typeof title !== 'string') {
            throw new Error("setTitle(): The parameter must be a string.");
        }
        this.title = title;
        if(this.html) {
            this.dom.header.textContent = title;
        }
        return this;
    };

    ListPanel.prototype.setFilterPlaceholder = function(filterPlaceholder) {
        this.filterControl.setPlaceholder(filterPlaceholder);
        return this;
    };

    ListPanel.prototype.onFilterControlChangeHandler = function() {
        var that = this;
        return function() {
            var nextFilter = this.getHTML().value;
            if (that.filterCriteria !== nextFilter) {
                that.filter(nextFilter);
            }
        };
    };

    ListPanel.prototype.setStatusBarMessage = function(statusBarMessage) {
        if(!(statusBarMessage === null || typeof statusBarMessage === 'function')) {
            throw new Error("setStatusBarMessage(): The parameter must be a function or null.")
        }
        this.statusBarMessage = statusBarMessage;
        return this;
    };

    ListPanel.prototype.setItemTextDataBind = function(itemTextDataBind) {
        var i, size;
        if(!(itemTextDataBind === null || typeof itemTextDataBind === 'string')) {
            throw new Error("setItemTextDataBind(): the parameter must be a string or null.");
        }
        this.itemTextDataBind = itemTextDataBind;
        size = this.items.getSize();
        for(i = 0; i < size; i++) {
            this.items.get(i).setTextDataBind(itemTextDataBind);
        }
        return this;
    };

    ListPanel.prototype.setActions = function(actions) {
        if(!jQuery.isArray(actions)) {
            throw new Error("setActions(): The parameter must be an array.");
        }
        this.actions = actions;
        return this;
    };

    ListPanel.prototype.addItem = function(item) {
        if(item instanceof PMUI.item.ListItem) {
            item.setActions(this.actions).setTextDataBind(this.itemTextDataBind);
        } else if(!item.actions) {
            item.actions =  this.actions;
            item.setTextDataBind = this.itemTextDataBind;
        }

        ListPanel.superclass.prototype.addItem.call(this, item);

        return this;
    };

    ListPanel.prototype.setItems = function(items) {
        if(this.actions !== null) {
            ListPanel.superclass.prototype.setItems.call(this, items);
            this.updateStatusBar();
        }
        return this;
    };

    ListPanel.prototype.updateStatusBar = function() {
        var msg, size;
        if(this.dom.statusBar) {
            size = this.filterCriteria ? jQuery(this.containmentArea).find('>li').length : this.items.getSize();
            if(typeof this.statusBarMessage === 'function') {
                msg = this.statusBarMessage(this, size, !!this.filterCriteria, this.filterCriteria);
            } else {
                msg = size + (this.filterCriteria ? " result(s) matching \"" + this.filterCriteria + "\"" : " item(s).");
            }
            this.dom.statusBar.textContent = msg;
        }
        return this;
    };
 
    ListPanel.prototype.setOrientation = function(orientation) {
        var items, displayBlock, i;
        if(orientation !== 'horizontal' && orientation !== 'vertical') {
            throw new Error("setOrientation(): The parameter must be \"horizontal\" or \"vertical\"");
        }
        this.orientation = orientation;
        displayBlock = orientation === 'vertical' ? 'block' : 'inline-block';
        items = this.items.asArray();
        for(i = 0; i < items.length; i++) {
            items[i].setDisplay(displayBlock);
        }
        return this;
    };

    ListPanel.prototype.setFactory = function() {
        this.factory = new PMUI.util.Factory({
            products: {
                'listitem': PMUI.item.ListItem
            },
            defaultProduct: 'listitem'
        });
        return this;
    };

    ListPanel.prototype.getData = function() {
        var i, items = this.items.asArray(), data = [];

        for(i = 0; i < items.length; i += 1) {
            data.push(items[i].getData());
        }

        return data;
    };

    ListPanel.prototype.clearFilter = function() {
        var i, items = this.items.asArray();
        this.filterCriteria = "";
        if(this.html) {
            for(i = 0; i < items.length; i += 1) {
                jQuery(items[i].html).detach();
            }
            for(i = 0; i < items.length; i += 1) {
                this.containmentArea.appendChild(items[i].getHTML());
            }   
        }
        this.updateStatusBar();
        return this;
    };

    ListPanel.prototype.getFilteredItems = function (){
            return this.filteredItems;
    };
    ListPanel.prototype.filter = function(criteria) {
        var i, regExp, itemsCopy;
        this.filteredItems = [];
        if (typeof criteria === 'string') {
            if (criteria === "") {
                this.clearFilter();
                return this;
            }
            this.filterCriteria = criteria;
        } else if (typeof criteria === 'number'){
            this.filterCriteria = criteria.toString(10);
        } else {
            throw new Error("filter(): The parameter must be a string or number");
        }

        if (!this.containmentArea) {
            return this;
        }

        regExp = new RegExp(this.filterCriteria.replace(/([\\\.\[\]\^\$\(\)\?\*\+\|\{\}])/g, "\\\$1"), "i");
        itemsCopy = this.items.asArray();

        for(i = 0; i < itemsCopy.length; i += 1) {
            jQuery(itemsCopy[i].html).detach();
        }
        
        for(i = 0; i < itemsCopy.length; i += 1) {
            if (regExp.test(itemsCopy[i].text)) {
                this.containmentArea.appendChild(itemsCopy[i].getHTML());
                this.filteredItems.push(itemsCopy[i]);
            }
        }

        this.updateStatusBar();
        return this;
    };

    ListPanel.prototype.defineEvents = function() {
        this.removeEvents().eventsDefined = true;
        this.filterControl.defineEvents();
        return this;
    };

    ListPanel.prototype.createHTML = function() {
        var list, toolbar, statusBar, header;
        if(this.html) {
            return this.html;
        }
        ListPanel.superclass.prototype.createHTML.call(this);

        list = PMUI.createHTMLElement('ul');
        list.className = 'pmui-listpanel-list';
        toolbar = PMUI.createHTMLElement('div');
        toolbar.className = 'pmui-listpanel-toolbar';
        statusBar = PMUI.createHTMLElement('div');
        statusBar.className = 'pmui-listpanel-statusbar';
        header = PMUI.createHTMLElement('div');
        header.className = 'pmui-listpanel-title';
        this.containmentArea = list;

        this.dom.list = list;
        this.dom.toolbar = toolbar;
        this.dom.statusBar = statusBar;
        this.dom.header = header;

        toolbar.appendChild(this.filterControl.getHTML());
        this.html.appendChild(header);
        this.html.appendChild(toolbar);
        this.html.appendChild(list);
        this.html.appendChild(statusBar);

        this.setTitle(this.title)
            .setListHeight(this.listHeight);

        if(this.filterable) {
            this.enableFiltering();
            this.filter(this.filterCriteria || "");
        } else {
            this.disableFiltering();
        }

        if(this.visibleHeader) {
            this.showHeader();
        } else {
            this.hideHeader();
        }

        if(this.visibleStatusBar) {
            this.showStatusBar();
        } else {
            this.hideStatusBar();
        }

        this.defineEvents();
        return this.html;
    };

    PMUI.extendNamespace('PMUI.panel.ListPanel', ListPanel);
}());
(function() {
    /**
     * @class PMUI.grid.GridPanel
     * Class to display data in a grid format.
     * @extends PMUI.core.Container
     *
     * 
     * Single grid example:
     * 
     *     @example
     *     var g;
     * 
     *     $(function(){
     *         g = new PMUI.grid.GridPanel({
     *             pageSize: 6,
     *             columns:[
     *                 {
     *                     title:'columna1',
     *                     dataType:'string',
     *                     columnData: "name"
     *                 },
     *                 {
     *                     title:'columna2',
     *                     dataType:'number',
     *                     width : 150,
     *                     columnData: "lastName",
     *                     sortable: true
     *                 },
     *                 {
     *                     title:'columna3',
     *                     dataType:'button',
     *                     width : "200px",
     *                     onButtonClick: function(row, grid) {
     *                         console.log(row, grid);
     *                     }
     *                 }
     *             ],
     *             dataItems: [
     *                 {
     *                     name: "Daniel",
     *                     lastName: "Canedo"
     *                 }, {
     *                     name: "John",
     *                     lastName: "McAllister"
     *                 }, {
     *                     name: "Ximena",
     *                     lastName: "Jimenez"
     *                 }, {
     *                     name: "Fernando",
     *                     lastName: "Fernandez"
     *                 }, {
     *                     name: "Rodrigo",
     *                     lastName: "Rodriguez"
     *                 }, {
     *                     name: "Andrea",
     *                     lastName: "Postigo"
     *                 }, {
     *                     name: "Hernando",
     *                     lastName: "Hernandez"
     *                 }, {
     *                     name: "Ramiro",
     *                     lastName: "Ramirez"
     *                 }, {
     *                     name: "Domingo",
     *                     lastName: "Dominguez"
     *                 }, {
     *                     name: "Gonzalo",
     *                     lastName: "Gonzales"
     *                 }
     *             ]                       
     *         });
     *         document.body.appendChild(g.getHTML());
     *         g.defineEvents();
     *     });
     *
     * Example with two grids with interchangeable rows.
     *
     *      @example
     *      var g, g2;
     *      
     *      $(function(){
     *          g2 = new PMUI.grid.GridPanel({
     *              behavior: 'dragdropsort',
     *              pageSize: 0,
     *              columns: [
     *                  {
     *                      title: 'lastName',
     *                      columnData: 'lastName',
     *                      sortable: true
     *                  }
     *              ],
     *              dataItems: [
     *                  {
     *                      name: "Ramiro",
     *                      lastName: "Ramirez"
     *                  }, {
     *                      name: "Pedro",
     *                      lastName: "Pedraza"
     *                  }, {
     *                      name: "Domingo",
     *                      lastName: "Dominguez"
     *                  }, {
     *                      name: "Oliva",
     *                      lastName: "Olivares"
     *                  }, {
     *                      name: "Gonzalo",
     *                      lastName: "Gonzales"
     *                  }, {
     *                      name: "Enrique",
     *                      lastName: "Enriquez"
     *                  }
     *              ],
     *             onDrop: function(container, draggableItem) {
     *                 var subject = draggableItem.getData();
     *                 subject = subject.name + " " + subject.lastName;
     *                 console.log(subject + " was dropped on grid 1");
     *             },
     *             onSort: function(container, item, index) {
     *                 var subject = item.getData();
     *                 subject = subject.name + " " + subject.lastName;
     *                 console.log(subject + "'s index has changed to " + index);
     *             }  
     *          });
     *          g = new PMUI.grid.GridPanel({
     *              behavior: 'dragdropsort',
     *              pageSize:6,
     *              columns:[
     *                  {
     *                      title: 'Initials',
     *                      columnData: function(data) {
     *                          return data.name.substr(0, 1) + " " + data.lastName.substr(0, 1);
     *                      }
     *                  },
     *                  {
     *                      title:'Name',
     *                      dataType:'string',
     *                      columnData: "name"
     *                  },
     *                  {
     *                      title:'LastName',
     *                      dataType:'number',
     *                      width : 150,
     *                      columnData: "lastName",
     *                      sortable: true
     *                  },
     *                  {
     *                      title:'columna3',
     *                      dataType:'button',
     *                      width : "200px",
     *                      onButtonClick: function(row, grid) {
     *                          console.log(row, grid);
     *                      }
     *                  }
     *              ],
     *              dataItems: [
     *                  {
     *                      name: "Daniel",
     *                      lastName: "Canedo"
     *                  }, {
     *                      name: "John",
     *                      lastName: "McAllister"
     *                  }, {
     *                      name: "Ximena",
     *                      lastName: "Jimenez"
     *                  }, {
     *                      name: "Fernando",
     *                      lastName: "Fernandez"
     *                  }, {
     *                      name: "Rodrigo",
     *                      lastName: "Rodriguez"
     *                  }, {
     *                      name: "Andrea",
     *                      lastName: "Postigo"
     *                  }, {
     *                      name: "Hernando",
     *                      lastName: "Hernandez"
     *                  }
     *              ],
     *             onDrop: function(container, draggableItem) {
     *                 var subject = draggableItem.getData();
     *                 subject = subject.name + " " + subject.lastName;
     *                 console.log(subject + " was dropped on grid 2");
     *             },
     *             onSort: function(container, item, index) {
     *                 var subject = item.getData();
     *                 subject = subject.name + " " + subject.lastName;
     *                 console.log(subject + "'s index has changed to " + index);
     *             }                     
     *          });
     *          document.body.appendChild(g.getHTML());
     *          document.body.appendChild(g2.getHTML());
     *          g.defineEvents();
     *          g2.defineEvents();
     *      });
     *
     * @constructor
     * Creates a new instance of GridPanel class.
     * @param {Object} [settings=null] A JSON object with the config options.
     *
     * @cfg {Array} [columns=[]] An Array where each element is an object literal or an instance of 
     * {@link PMUI.grid.GridPanelColumn}. In case to be an object literal it must have {@link PMUI.grid.GridPanelColumn
     * the config options for GridPanelColumn creation}.
     * @cfg {Array} [items=[]] An Array where each of its elements can be an instance of PMUI.grid.GridPanelRow or a 
     * JSON object with the config options to build a new one.
     * @cfg {Array} [dataItems=[]] An Array where each of its elements can be a JSON object with the data available for
     * the item (row) that will be created.
     * @cfg {Number} [pageSize=0] The size for every page in the table, if it is set to 0 (default) then there will no 
     * exists pagination.
     * @cfg {Boolean} [draggable=false] If the grid will apply the row dragging behavior.
     * @cfg {Boolean} [filterable=true] If the grid will have the filtering function enabled or not.
     * @cfg {Number} [initialPage=0] The page number to show initially.
     * @cfg {Function} [customStatusBar=null] The function to generate a custom message in the grid's status bar. This 
     * function must return a string with the text to be used as the message to be displayed in thegrid's status bar. 
     * When the function is called 5 parameters will be sent:
     *
     * - The current page (Number).
     * - The page size (Number).
     * - The number of items (Number).
     * - If a criteria is being applied to the grid (Boolean).
     * - The filter criteria applied to the grid (String).
     *
     * If you need access more grid's properties you can use 'this' to refer the grid since the function is called in 
     * the grid's context.
     * @cfg {Object|PMUI.util.Style|null} [rowStyle=null] The style to be applied to the grid's rows. It can be:
     *
     * - An object literal: in this case the object can have the properties:
     *     - cssClasses: an array of strings, each string is a css class that will be applied to each row's html.
     *     - cssProperties: an object literal in which each property is a valid css property and its value is a valid
     *     css value for the respective css property.
     *
     * - An instance of the {@link PMUI.util.Style Style} class.
     * - null, in this case no custom style will be applied to the html of the grid's rows.
     *
     * @cfg {String} [filterPlaceholder] Sets the placeholder for the filter textbox.
     * @cfg {Boolean} [visibleHeaders=true] Turns on/off the visibility of the column headers.
     * @cfg {String} [sortableItems='> tr'] Specifies which items inside the element should be sortable in case a 
     * sort behavior is applied. It must be a jQuery selector.
     * @cfg {Function|null} [onEmpty=null] A callback function to be called everytime the grid's last item is removed. 
     * The value for this config option can be a function or the null constant for no callback execution. For more 
     * info about the paramerters sent to the function please read the {@link #event-onEmpty onEmpty} event 
     * documentation.
     * @cfg {Function|null} [onRowClick=null] The callback function to be executed everytime the 
     * {@link #event-onRowClick onRowClick} event is fired. Read about this event to know about the parameters received
     * by the callback function.
     * @cfg {String|Function|null} [emptyMessage=null] The message to show when the grid is showing no items. It 
     * can be:
     *
     * - A String, in this case the string will be displayed in both cases, when the grid has no items and when no 
     * grid's item meet the filter.
     * - A Function, in this case the function must return a string or an HTML Element to be displayed. The function 
     * will receive two parameters: the {@link PMUI.grid.Grid grid}, a boolean If it is true it means the returned 
     * value will be used when a filter is applied, otherwise it means that the returned value will be used when 
     * there are no items in the grid.
     * - null, in this case a default message will be used for each situation.
     */
    var GridPanel = function (settings) {
        /**
         * A JSON object that contains the object's DOM components.
         * @type {Object}
         * @private
         */
        this.dom = {};
        /**
         * The grid's list of {@link PMUI.grid.GridPanelColumn column objects}.
         * @type {PMUI.ui.ArrayList}
         * @private
         */
        this.columns = new PMUI.util.ArrayList();
        GridPanel.superclass.call(this, jQuery.extend({
            factory: {
                products: {
                    "GridPanelRow": PMUI.grid.GridPanelRow
                },
                defaultProduct: "GridPanelRow"
            }
        }, settings, {
            elementTag: "div"
        }));
        /**
         * The number of rows to show in every page, 0 means no limit.
         * @type {Number}
         * @readonly
         */
        this.pageSize = null;
        /**
         * If the grid has the filtering functiionality enabled or not.
         * @type {Boolean}
         * @readonly
         */
        this.filterable = null;
        /**
         * The current filter criteria.
         * @type {String}
         * @readonly
         */
        this.filterCriteria = null;
        /**
         * The current page being displayed.
         * @type {Boolean}
         * @readonly
         */
        this.currentPage = null;
        /**
         * If the column sorting functionloty is enabled or not.
         * @type {Boolean}
         * @readonly
         */
        this.sortable = null;
        /**
         * The {@link PMUI.control.TextControl TextControl} object that is playing the filter text input control role.
         * @type {PMUI.control.TextControl}
         * @private
         */
        this.filterControl = null;
        /**
         * The {@link PMUI.ui.Button Button} to executes the filtering.
         * @type {[type]}
         */
        this.filterButton = null;
        /**
         * An {@link PMUI.util.ArrayList ArrayList} that contains the rows ordered by the 
         * {@link #property-filterCriteria filterCriteria}.
         * @type {PMUI.util.ArrayList}
         * @private
         */
        this.usableItemsList = null;
        /**
         * An {@link PMUI.util.ArrayList ArrayList} that contains only the rows that meet the filter criteria.
         * @type {PMUI.util.ArrayList}
         * @private
         */
        this.filteredItems = null;
        /**
         * @property {Function} statusBarHandler The function to generate a custom message in the grid's status bar.
         */
        this.customStatusBar = null;
        /**
         * The style to be applied to all the rows in the grid.
         * @type {PMUI.util.Style}
         * @private
         */
        this.rowStyle = null;
        /**
         * The placeholder to be used in the text control for filtering. It is set by the 
         * {@link #cfg-filterPlaceholder filterPlaceholder} config option and the 
         * {@link #method-setFilterPlaceholer setFilterPlaceholer()} method.
         * @type {String}
         */
        this.filterPlaceholder = null;
        /**
         * The data about the sorting that is currently applied to the grid.
         * @type {Object}
         */
        this.sortingData = null;
        /**
         * @event onRowClick
         * Fired everytime a grid's row is clicked.
         * @param {PMUI.grid.GridPanelRow} row The row object the click was performed on.
         * @param {Object} data The row's data
         */
        this.onRowClick = null;
        /**
         * @event onEmpty
         * Fired everytime the Grid removes its last item.
         * @param {PMUI.grid.GridPanel} grid The GridPanel object.
         */
        this.onEmpty = null;
        /**
         * If the column headers are visible or not, it is set by the {@link #cfg-visibleHeaders visibleHeaders} config 
         * option and the {@link #method-hideHeaders hideHeaders()} and {@link #method-showHeaders showHeaders()}.
         * @type {Boolean}
         */
        this.visibleHeaders = null;
        /**
         * [selectedRow description]
         * @type {[type]}
         */
        this.selectedRow = null;
        /**
         * [selectable description]
         * @type {[type]}
         */
        this.selectable = null;
        /**
         * If the footer is visible or not.
         * @type {Boolean}
         */
        this.visibleFooter = null;
        /**
         * The message to be displayed in the grid when there are no items to display. Set by the 
         * {@link #cfg-emptyMessage emptyMessage config option} and the 
         * {@link #method-setEmptyMessage setEmptyMessage()} method.
         * @type {String|Function}
         */
        this.emptyMessage = null;
        GridPanel.prototype.init.call(this, settings);
    };

    PMUI.inheritFrom("PMUI.core.Container", GridPanel);
    /**
     * The object type.
     * @type {String}
     */
    GridPanel.prototype.type = "GridPanel";
    /**
     * The object's family.
     * @type {String}
     */
    GridPanel.prototype.family = "GridPanel";
    /**
     * Initialize the object.
     * @param  {Object} settings A JSON object with the setting options.
     * @private
     */
    GridPanel.prototype.init = function (settings) {
        var defaults = {
            columns : [],
            items : [],
            dataItems: null,
            pageSize : 0,
            filterable : true,
            inititalPage : 0,
            customStatusBar: null,
            rowStyle: null,
            visibleHeaders: true,
            filterPlaceholder: "",
            onRowClick: null,
            sortableItems: '> tr:not(.pmui-nodrag)',
            onEmpty: null,
            emptyMessage: null,
            selectable : false,
            visibleFooter : true
        };

        jQuery.extend(true, defaults, settings);
        this.totalRows = 0;

        this.usableItemsList = new PMUI.util.ArrayList();
        this.sortingData = {
            criteria: null,
            type: null
        };
        this.rowStyle = new PMUI.util.Style();
        this.filterControl = new PMUI.control.TextControl({
            onKeyUp: this.onFilterControlChangeHandler(),
            style: {
                cssClasses: [
                    "pmui-gridpanel-search"
                ]
            }
        });
        this.filterButton = new PMUI.ui.Button({
            text: 'Search',
            style: {
                cssClasses: [
                    "pmui-gridpanel-buttonsearch"
                ]
            }
        });
        this.filteredItems = new PMUI.util.ArrayList();

        this.setSortableItems(defaults.sortableItems)
            .setSelectable(defaults.selectable)
            .setEmptyMessage(defaults.emptyMessage)
            .setOnEmptyHandler(defaults.onEmpty)
            .setRowStyle(defaults.rowStyle)
            .setColumns(defaults.columns)
            .setCustomStatusBar(defaults.customStatusBar)
            .setFilterPlaceholder(defaults.filterPlaceholder)
            .setOnRowClick(defaults.onRowClick)
            .setVisibleFooter(defaults.visibleFooter);

        if(jQuery.isArray(defaults.dataItems)) {
            this.setDataItems(defaults.dataItems);
        } else {
            this.setItems(defaults.items);
        }
        this.setPageSize(defaults.pageSize)
            .setCurrentPage(defaults.inititalPage);

        if(defaults.filterable) {
            this.enableFiltering();
        } else {
            this.disableFiltering();
        }
        if(defaults.visibleHeaders) {
            this.showHeaders();
        } else {
            this.hideHeaders();
        }
        this.updateUsableItemsList();
    };
    /**
     * @inherirdoc
     */
    GridPanel.prototype.setDataItems = function(dataItems) {
        if(this.usableItemsList) {
            GridPanel.superclass.prototype.setDataItems.call(this, dataItems);
        }
        return this;
    };
    /**
     * @inheritdoc
     */
    GridPanel.prototype.setItems = function(items) {
        if(this.usableItemsList) {
            GridPanel.superclass.prototype.setItems.call(this, items);
        }
        return this;
    };
    /**
     * Sets the message to display when there are no items to display in the grid.
     * @param {String|Function|null} emptyMessage It can be:
     *
     * - A String, in this case the string will be displayed in both cases, when the grid has no items and when no 
     * grid's item meet the filter.
     * - A Function, in this case the function must return a string or an HTML Element to be displayed. The function 
     * will receive two parameters: the {@link PMUI.grid.Grid grid}, a boolean If it is true it means the returned 
     * value will be used when a filter is applied, otherwise it means that the returned value will be used when 
     * there are no items in the grid.
     * - null, in this case a default message will be used for each situation.
     * @chainable
     */
    GridPanel.prototype.setEmptyMessage = function(emptyMessage) {
        if(!(emptyMessage === null || typeof emptyMessage === 'string' || typeof emptyMessage === 'function')) {
            throw new Error("setEmptyMessage(): the parameter must be a string, a function or null.");
        }
        this.emptyMessage = emptyMessage;
        return this;
    };
    /**
     * Disables the behaviors and column's buttons.
     * @chainable
     */
    GridPanel.prototype.disable = function() {
        this.disableColumn();
        return GridPanel.superclass.prototype.disable.call(this);
    };
    /**
     * Enables the behaviors and column's buttons.
     * @chainable
     */
    GridPanel.prototype.enable = function() {
        this.enableColumn();
        return GridPanel.superclass.prototype.enable.call(this);
    };
    /**
     * @inheritdoc
     */
    GridPanel.prototype.getBehavioralItems = function() {
        var items = [], initialIndex = this.currentPage * this.pageSize, finalIndex = initialIndex + this.pageSize, i;

        for(i = initialIndex; i < finalIndex && this.getItem(i); i++) {
            items.push(this.getItem(i));
        }
        return items;
    };
    /**
     * Enables one or all the columns.
     * @param {String|Number|PMUI.grid.GridPanelColumn} [column] A criteria to select the column that will be 
     * enabled. It supports the following data types:
     *
     * - String, in this case the parameter is used as the column's id.
     * - Number, in this case the parameter is used as the column's index.
     * - An instance of {@link PMUI.grid.GridPanelColumn GridPanelColumn}, in this case the supplied object must be 
     * a child of the grid.
     * - [no value], since this parameter is optional, you can skip it. In this case all the columns in the grid will
     * be enabled.
     * @chainable
     */
    GridPanel.prototype.enableColumn = function(column) {
        var i, targetColumn = this.getColumns(column);
        for(i = 0; i < targetColumn.length; i += 1) {
            if(targetColumn[i].dataType === 'button') {
                targetColumn[i].enable();
            }
        }
        this.disabled = false;
        this.style.removeClasses(['pmui-disabled']);
        return this;
    };
    /**
     * Disables one or all the columns.
     * @param {String|Number|PMUI.grid.GridPanelColumn} [column] A criteria to select the column that will be 
     * disabled. It supports the following data types:
     *
     * - String, in this case the parameter is used as the column's id.
     * - Number, in this case the parameter is used as the column's index.
     * - An instance of {@link PMUI.grid.GridPanelColumn GridPanelColumn}, in this case the supplied object must be 
     * a child of the grid.
     * - [no value], since this parameter is optional, you can skip it. In this case all the columns in the grid will 
     * be disabled.
     * @chainable
     */
    GridPanel.prototype.disableColumn = function(column) {
        var i, targetColumn = this.getColumns(column);
        for(i = 0; i < targetColumn.length; i += 1) {
            if(targetColumn[i].dataType === 'button') {
                targetColumn[i].disable();
            }
        }
        return this;
    };
    /**
     * Sets the width to one or all the columns.
     * @param {Number|String} width  The width to be applied to the column(s).
     * @param {String|Number|PMUI.grid.GridPanelColumn} [column] A criteria to select the column the width will be 
     * applied. It supports the following data types:
     *
     * - String, in this case the parameter is used as the column's id.
     * - Number, in this case the parameter is used as the column's index.
     * - An instance of {@link PMUI.grid.GridPanelColumn GridPanelColumn}, in this case the supplied object must be 
     * a child of the grid.
     * - [no value], since this parameter is optional, you can skip it. In this case the width will be applied to all 
     * the columns in the grid.
     * @chainable
     */
    GridPanel.prototype.setColumnWidth = function(width, column) {
        var targetColumn = [], aux, i;
        if(typeof column === 'string') {
            aux = this.columns.find("id", column);
        } else if(typeof column === 'number') {
            aux = this.columns.get(column);
        } else if(column instanceof PMUI.grid.GridPanelColumn && this.isDirectParentOf(column)) {
            targetColumn.push(column);
        } else if(column === undefined) {
            targetColumn = this.column.asArray();
        }
        if(aux) {
            targetColumn.push(aux);
        }
        for(i = 0; i < targetColumn.length; i += 1) {
            targetColumn[i].setWidth(width);
        }
        if(i > 0) {
            this.goToPage(this.currentPage);
        }
        return this;
    };
    /**
     * Sets the function callback to be called everytime the grid's last item is removed. 
     * @param {Function|null} handler The callback function or the null constant for no callback execution. For more 
     * info about the paramerters sent to the function please read the {@link #event-onEmpty onEmpty} event 
     * documentation.
     * @chainable
     */
    GridPanel.prototype.setOnEmptyHandler = function(handler) {
        if(!(typeof handler === 'function' || handler === null)) {
            throw new Error('setOnEmptyHandler(): The parameter must be a function or null.');
        }
        this.onEmpty = handler;
        return this;
    };
    /**
     * Shows the column headers.
     * @chainable
     */
    GridPanel.prototype.showHeaders = function() {
        this.visibleHeaders = true;
        if(this.html) {
            this.dom.thead.style.display = '';
        }
        return this;
    };
    /**
     * Hide the column headers.
     * @chainable
     */
    GridPanel.prototype.hideHeaders = function() {
        this.visibleHeaders = false;
        if(this.html) {
            this.dom.thead.style.display = 'none';
        }
        return this;
    };
    /**
     * Updates the usable items list.
     * @chainable
     * @private
     */
    GridPanel.prototype.updateUsableItemsList = function() {
        var i, itemsLength, items;
        if(this.usableItemsList) {
            this.usableItemsList.set(this.items.asArray());
        }
        return this;
    };
    /**
     * Sets the placeholder for the filter textbox.
     * @param {String} filterPlaceholder
     * @chainable
     */
    GridPanel.prototype.setFilterPlaceholder = function(filterPlaceholder) {
        if(typeof filterPlaceholder !== 'string') {
            throw new Error('setFilterPlaceholder(): The parameter must be a string.');
        }
        this.filterControl.setPlaceholder(this.filterPlaceholder = filterPlaceholder);
        return this;
    };
    /**
     * Sets the custom style to be applied to the grid's rows.
     * @param {Object|PMUI.util.Style|null} rowStyle It can be an object literal (with the properties 
     * cssProperties and/or cssClasses), an instance of PMUI.util.Style or null (in this case no custom style will be 
     * applied to the rows).
     */
    GridPanel.prototype.setRowStyle = function(rowStyle) {
        var i, rows, rowsLength;
        if(rowStyle !== null && typeof rowStyle !== 'object' && !(rowStyle instanceof PMUI.util.Style)) {
            throw new Error("setRowStyle(): The parameter must be null or an object literal or an instance of " 
                + "PMUI.grid.GridPanelRow.");
        }
        this.rowStyle.clear();
        if(rowStyle instanceof PMUI.util.Style) {
            this.rowStyle = rowStyle;
        } else if(rowStyle) {
            this.rowStyle.addProperties(rowStyle.cssProperties || {})
                .addClasses(rowStyle.cssClasses || []);
        }
        rows = this.items.asArray();
        rowsLength = rows.length;
        for(i = 0; i < rowsLength; i += 1) {
            rows[i].setStyle({
                cssProperties: this.rowStyle.cssProperties,
                cssClasses: this.rowStyle.cssClasses
            });
        }
        return this;
    };
    /**
     * Sets a function to generate a custom message in the grid's status bar.
     * @param {Function|NULL} handler The function for generate the message to be shown in the grid's status bar. It 
     * also can be NULL, in this case the message shown in the grid's status bar wiil be the default one.
     * For more info about the sent parameters to the function read the {@link #cfg-customStatusBar customStatusBar}
     * config option.
     */
    GridPanel.prototype.setCustomStatusBar = function(handler) {
        if(!(typeof handler === 'function' || handler === null)) {
            throw new Error('setCustomStatusBar(): this method only accepts as parameter a function or null.');   
        }
        this.customStatusBar = handler;
        return this;
    };
    /**
     * Enables the filter functionality.
     * @chainable
     */
    GridPanel.prototype.enableFiltering = function() {
        this.filterable = true;
        if(this.dom.toolbar) {
            this.dom.toolbar.style.display = '';
            this.goToPage(this.currentPage);
        }
        return this;
    };
    /**
     * Disables the filter functionality.
     * @chainable
     */
    GridPanel.prototype.disableFiltering = function() {
        this.filterable = false;
        if(this.dom.toolbar) {
            this.dom.toolbar.style.display = 'none';
            this.filterCriteria = "";
            this.filterControl.setValue("");
            this.goToPage(this.currentPage);
        }
        return this;
    };
    /**
     * Returns the index in page for the specified row.
     * @param  {PMUI.grid.GridPanelRow|Number|String} row It can be:
     *
     * - {@link PMUI.grid.GridPanelRow}.
     * - Number, in this case it is interpreted as the global index for the row.
     * - String, it will be interpreted as the row's id.
     * @return {Number}
     */
    GridPanel.prototype.indexInPage = function(row) {
        var globalIndex = this.getItemIndex(row);

        if (this.pageSize !== 0 && globalIndex >= 0) {
            globalIndex = globalIndex % this.pageSize;
        }

        return globalIndex;
    };
    /**
     * Returns the handler to be executed when the filter changes.
     * @return {Function}
     * @private
     */
    GridPanel.prototype.onFilterControlChangeHandler = function() {
        var that = this;
        return function() {
            var nextFilter = this.getHTML().value;
            if (that.filterCriteria !== nextFilter) {
                that.filter(nextFilter);
            }
        };
    };
    /**
     * Sets the columns.
     * @param {Array} columns An Array where each element is an object literal with the 
     * {@link PMUI.grid.GridPanelColumn the config options for GridPanelColumn creation} or a 
     * {@link PMUI.grid.GridPanelColumn GridPanelColumn object}.
     * @chainable
     */
    GridPanel.prototype.setColumns = function (columns) {
        var i;
        this.clearColumns();
        if (!jQuery.isArray(columns)) {
            throw new Error("setColumns(): The parameter must be an Array");
        }

        for (i = 0 ; i < columns.length; i += 1){
            this.addColumn(columns[i]);
        }
        return this;
    };
    /**
     * Returns one or all the grid's columns.
     * @param {String|Number|PMUI.grid.GridPanelColumn} [column] A criteria to select the column that will be 
     * returned. It supports the following data types:
     *
     * - String, in this case the parameter is used as the column's id.
     * - Number, in this case the parameter is used as the column's index.
     * - An instance of {@link PMUI.grid.GridPanelColumn GridPanelColumn}, in this case the supplied object must be 
     * a child of the grid.
     * - [no value], since this parameter is optional, you can skip it. In this case all the columns in the grid will
     * be returned.
     * @return {Array}
     */
    GridPanel.prototype.getColumns = function(column) {
        var targetColumn = [], aux;
        if(typeof column === 'string') {
            aux = this.columns.find("id", column);
        } else if(typeof column === 'number') {
            aux = this.columns.get(column);
        } else if(column instanceof PMUI.grid.GridPanelColumn && this.isDirectParentOf(column)) {
            targetColumn.push(column);
        } else if(column === undefined) {
            targetColumn = this.columns.asArray().slice(0);
        }
        if(aux) {
            targetColumn.push(aux);
        }
        return targetColumn;
    };
    /**
     * Paints a column into the grid's head.
     * @param  {PMUI.grid.GridPanelColumn} column
     * @chainable
     * @private
     */
    GridPanel.prototype.paintColumn = function(column) {
        if (this.dom.thead) {
            this.dom.thead.appendChild(column.getHTML());
        }
        return this;
    };
    GridPanel.prototype.paintColumns = function() {
        var i, columns = this.columns.asArray();
        for(i = 0; i < columns.length; i++) {
            this.paintColumn(columns[i]);
        }
        return this;
    };
    /**
     * Adds a column.
     * @param {PMUI.grid.GridPanelColumn|JSON} column A {@link PMUI.grid.GridPanelColumn} or a JSON Object with the
     * {@link PMUI.grid.GridPanelColumn#cfg config options} for the new column.
     */
    GridPanel.prototype.addColumn = function (column) {
        var that = this, columnToAdd, rows, i, defaults = {
            grid: this,
            title: '[untitled]',
            type: 'string',
            sortable: false,
            searchable: true,
            onSort: function(order) {
                var criteria = this.columnData;
                that.sort(criteria, order);
            }/*,
            align: "center"*/
        };
        if (column instanceof PMUI.grid.GridPanelColumn) {
            columnToAdd = column;
            columnToAdd.clearCells();
            columnToAdd.setGrid(this);
        } else if (typeof column === 'object') {
            jQuery.extend(true, defaults, column);
            columnToAdd = new PMUI.grid.GridPanelColumn(defaults);
        } else {
            throw new Error('addColumn(): The method only accepts an object or an instace of PMUI.grid.GridPanelColumn'
             + ' as parameter.');
        }

        //TODO control align and width
        //console.log("\tCOLUMN insert " + columnToAdd.title/*isd*/);
        this.columns.insert(columnToAdd);
        this.paintColumn(columnToAdd);
        rows = this.items.asArray();
        for(i = 0; i < rows.length; i += 1) {
            rows[i].addCell(columnToAdd);
        }

        return this;  
    };
    /**
     * Clears all the columns for the grid.
     * @chainable
     */
    GridPanel.prototype.clearColumns = function (){
        while(this.columns.getSize()){
            this.removeColumn(0);
        }       
        return this;
    };
    /**
     * Removes a column and all its cells from the grid.
     * @param  {Number|String|PMUI.grid.GridPanelColumn} column This value is used to determine the row to remove:
     *
     * - String, in this case the string must be the id of a column in the grid.
     * - Number, in this case the number is the index of the column to remove.
     * - a {@link PMUI.grid.GridPanelColumn GridPanelColumn object}, if it's a column that belongs to the grid then
     * this column will be the one to remove.
     * @chainable
     */
    GridPanel.prototype.removeColumn = function (column) {
        var columnToRemove, i, row, size, index;

        if (column instanceof PMUI.grid.GridPanelColumn) {
            columnToRemove = column;
        } else if (typeof column === 'number') {
            columnToRemove = this.columns.get(column);
        } else if (typeof column === 'string') {
            columnToRemove = this.columns.find('id', column);
        }

        if (columnToRemove) {
            index = this.columns.indexOf(columnToRemove);
            size = this.items.getSize();
            for(i = 0; i < size; i += 1) {
                row = this.items.get(i);
                row.removeCell(index);
            }
            columnToRemove.grid = null;
            columnToRemove.clearCells();
            this.columns.remove(columnToRemove);
            jQuery(columnToRemove.getHTML()).remove();
        }
        return this;
    };
    /**
     * Adds a new Item
     * @param {Object|PMUI.grid.GridPanelRow} item An instance of {@link PMUi.grid.GridPanelRow GridPanelRow} or an 
     * object with the properties to be used to create a new one.
     * @param {Number} [index] The insertion index position.
     */
    GridPanel.prototype.addItem = function (item, index) {
        var itemToBeAdded;

        if (typeof item !== 'object') {
            throw new Error('addItem(): The first parameter must be an object or an instance of PMUI.grid.GridPanel.');
        }
        item.onSelect = this.onRowSelectHandler();
        if (this.factory) {
            itemToBeAdded = this.factory.make(item);
        }
        if (itemToBeAdded && !this.isDirectParentOf(itemToBeAdded)) {
            itemToBeAdded.parent = this;
            itemToBeAdded.style.addClasses(['pmui-' + (this.items.getSize() % 2 ? 'odd' : 'even')]);
            if(itemToBeAdded.html || this.columns.asArray()){
                itemToBeAdded.setCells();
            }

            if (typeof index === 'number') {
                this.items.insertAt(itemToBeAdded, index);
                if(this.usableItemsList) {
                    this.usableItemsList.insertAt(itemToBeAdded, index);
                }
                if (this.filterCriteria) {
                    this.filter(this.filterCriteria);    
                } else {
                    this.goToPage(this.currentPage);
                }
            } else {
                //console.log("ROW insert " + /*isd +*/ itemToBeAdded.getData().lastName);
                this.items.insert(itemToBeAdded);
                this.usableItemsList.insert(itemToBeAdded);
                if (this.dom.tbody && !this.massiveAction) {
                    this.goToPage(this.currentPage);
                }    
            }
            if (this.eventsDefined) {
                itemToBeAdded.defineEvents();
            }
            //this.setBehavior(this.behavior);
        }
        return this;
    };
    /**
     * [onRowSelectHandler description]
     * @return {[type]} [description]
     */
    GridPanel.prototype.onRowSelectHandler = function () {
        var that = this;
        return function () {
            if(that.selectedRow) {
                that.selectedRow.deselectRow();
            }
            that.selectedRow = this;
        };
    };
    /**
     * Sets the callback function to be executed everytime the {@link #event-onRowClick onRowClick} event is fired. 
     * Read about this event to know about the parameters received by the callback function.
     * @param {Function|null} handler
     * @chainable
     */
    GridPanel.prototype.setOnRowClick = function (handler) {
        if(typeof handler !== 'function' && handler !== null) {
            throw new Error("sdfsdfgsd");
        }
        this.onRowClick = handler;
        return this;
    };
    /**
     * Shows an empty row in grid.
     * @chainable
     * @private
     */
    GridPanel.prototype.showEmptyCell = function() {
        var tr, td, message;
        if(this.dom.tbody) {
            tr = PMUI.createHTMLElement('tr'); 
            td = PMUI.createHTMLElement('td');
            tr.className = 'pmui-gridpanel-emptyrow pmui-nodrag';
            td.colSpan = this.columns.getSize();
            tr.appendChild(td);
            $(this.dom.tbody).find('.pmui-gridpanel-emptyrow').remove().end().append(tr);
            if(typeof this.emptyMessage === 'function') {
                message = this.emptyMessage(this, !!this.filterCriteria);
            } else if (typeof this.emptyMessage === 'string') {
                message = this.emptyMessage;
            } else {
                if(this.filterCriteria) {
                    message = 'No matches found for \"' + this.filterCriteria + '\"';
                } else {
                    message = 'No records found.';
                }
            }
            if(typeof message === 'string') {
                td.appendChild(document.createTextNode(message));
            } else if(PMUI.isHTMLElement(message)) {
                td.appendChild(message);
            }
            if(this.items.getSize() === 0 && typeof this.onEmpty === 'function') {
                this.onEmpty(this);
            }
        }
        return this;
    };
    /**
     * Removes a row from the object.
     * @param  {PMUI.core.Element|String|Number} item It can be a string (id of the child to remove), 
     a number (index of the child to remove) or a {Element} object
     * @chainable
     */
    GridPanel.prototype.removeItem = function(item) {
        var currentPage = this.currentPage;
        if(item.selected){
            item.deselectRow();
        }

        GridPanel.superclass.prototype.removeItem.call(this, item);

        if(!this.massiveAction) {
            this.updateUsableItemsList();
            if(!this.items.getSize()) {
                this.showEmptyCell();
            }
            if(currentPage >= this.getTotalPages()) {
                currentPage -= 1;
                this.currentPage = currentPage < 0 ? 0 : currentPage;
            }
            if (this.dom.tbody) {
                this.goToPage(this.currentPage);
            }
        }
        return this;
    };
    /**
     * @inheritdoc
     */
    GridPanel.prototype.clearItems = function() {
        var currentPage = this.currentPage;
        GridPanel.superclass.prototype.clearItems.call(this);
        this.updateUsableItemsList();
        if(!this.items.getSize()) {
            this.showEmptyCell();
        }
        if(currentPage >= this.getTotalPages()) {
            currentPage -= 1;
            this.currentPage = currentPage < 0 ? 0 : currentPage;
        }
        if (this.dom.tbody) {
            this.goToPage(this.currentPage);
        }
        return this;
    };
    /**
     * Moves a row from its current position to another.
     * @param  {PMUI.grid.GridPanelRow|Number|String} item  It can be:
     * 
     * - An instance of PMUI.grid.GridPanelRow, in this case this object must be a item of the object.
     * - A Number, in this case the number is interpereted as the index of the item to move.
     * - A String, in this case the string is interpreted as the id of the item to move.
     * @param  {Number} index The index in which the item will be moved to.
     * @chainable
     */
    GridPanel.prototype.moveItem = function(item, index) {
        var items = this.items, currentIndex,
            referenceObject;

        item = this.getItem(item);

        if (item instanceof PMUI.core.Element && this.isDirectParentOf(item)) {
            currentIndex = items.indexOf(item);
            items = items.asArray();
            referenceObject = this.items.get(index + (currentIndex < index ? 1 : 0));
            item = items.splice(currentIndex, 1)[0];
            this.items.insertAt(item, index);
            this.updateUsableItemsList();
            if (this.html) {
                if(!referenceObject) {
                    this.goToPage(this.currentPage);
                } else if(index === (this.currentPage * this.pageSize + this.pageSize - 1) 
                    || (this.getTotalPages() === (this.currentPage + 1) 
                        && index === (this.items.getSize() % this.pageSize) - 1)) {
                    this.containmentArea.appendChild(item.getHTML());
                } else {
                    this.containmentArea.insertBefore(item.getHTML(), referenceObject.getHTML());
                }
            }
        }
        return this;
    };
    /**
     * Returns the data for every rowe in the grid.
     * @return {Array}
     */
    GridPanel.prototype.getData = function() {
        var i, items = this.items.asArray(), data = [];

        for(i = 0; i < items.length; i += 1) {
            data.push(items[i].getData());
        }

        return data;
    };
    /**
     * @inheritdoc
     */
    GridPanel.prototype.paintItem = function(item, index) {
        var intitalItem = this.currentPage * this.pageSize, finalIndex = this.pageSize ? this.currentPage + this.pageSize - 1 : null;
        if(index === undefined) {
            index = this.items.getSize();
        }
        if(index >= initialIndex || (index <= finalIndex || finalIndex === null)) {
            this.goToPage(this.currentPage);
        }
        return this;
    };
    /**
     * @inheritdoc
     */
    GridPanel.prototype.paintItems = function() {
        return this.goToPage(this.currentPage);
    };
    /**
     * Creates the HTML for the Grid.
     * @return {HTMLElement}
     */
    GridPanel.prototype.createHTML = function () {
        var table, thead, tbody, toolbarDiv, footer, pager, statusBar,tableContainer, items;
        if (this.html) {
            return this.html;
        }

        GridPanel.superclass.superclass.prototype.createHTML.call(this);
        table = PMUI.createHTMLElement("table");
        table.className = 'pmui-gridpanel-table';
        table.cellSpacing = 0;
        table.cellPadding = 0;
        thead = PMUI.createHTMLElement("thead");
        tbody = PMUI.createHTMLElement("tbody");
        tableContainer = PMUI.createHTMLElement("div");
        tableContainer.className = "pmui-gridpanel-tableContainer";
        tbody.className = 'pmui-gridpanel-tbody';
        this.containmentArea = tbody;
        tableContainer.style.overflow = 'auto';

        toolbarDiv = PMUI.createHTMLElement("div");
        toolbarDiv.className = 'pmui-gridpanel-toolbar';
        toolbarDiv.appendChild(this.filterControl.getHTML());
        this.html.appendChild(toolbarDiv);

        this.dom.tableContainer = tableContainer;
        this.dom.thead = thead;
        this.dom.tbody = tbody;
        this.dom.table = table;
        this.dom.toolbar = toolbarDiv;
        
        table.appendChild(thead);
        table.appendChild(tbody);
        tableContainer.appendChild(table);
        this.html.appendChild(tableContainer);
        pager = PMUI.createHTMLElement('ul');
        pager.className = 'pmui-gridpanel-pager';

        statusBar = PMUI.createHTMLElement('div');
        statusBar.className = 'pmui-gridpanel-statusbar';

        footer = PMUI.createHTMLElement("div");
        footer.className = 'pmui-gridpanel-footer';
        footer.appendChild(pager);
        footer.appendChild(statusBar);

        this.html.appendChild(footer);

        this.dom.pager = pager;
        this.dom.statusBar = statusBar;
        this.dom.footer = footer;
       
        items = this.items.asArray();
        this.setVisibleFooter(this.visibleFooter);
        this.paintColumns()
            //.goToPage(this.currentPage)
            .setHeight(this.height);

        this.style.applyStyle();

        if(this.filterable) {
            this.enableFiltering();
        } else {
            this.disableFiltering();
        }
        if(this.visibleHeaders) {
            this.showHeaders();
        } else {
            this.hideHeaders();
        }
        if(!this.items.getSize()) {
            this.showEmptyCell();
        }
        if(this.eventsDefined) {
            this.defineEvents();
        }
        if(this.sortingData.sortingCriteria) {
            this.sort(this.sortingData.criteria, this.sortingData.type);
        }

        return this.html;
    };
    /**
     * [updateStatusBar description]
     * @return {[type]} [description]
     */
    GridPanel.prototype.updateStatusBar = function() {
        var itemsLength, filterCriteria, b, currentPage, filtered, totalPages;

        filterCriteria = this.filterCriteria;
        currentPage = this.currentPage;
        filtered = !!(filterCriteria && this.filterable);
        itemsLength = filtered ? this.filteredItems.getSize() : this.items.getSize();
        jQuery(this.dom.statusBar).empty();
        totalPages = this.getTotalPages(filtered);
        if(typeof this.customStatusBar === 'function') {
            this.dom.statusBar.textContent = 
                this.customStatusBar(currentPage, this.pageSize, itemsLength, filtered, filterCriteria || "");
        } else if(totalPages === 0) {
            b = PMUI.createHTMLElement('b');
            b.appendChild(document.createTextNode(""));
            this.dom.statusBar.appendChild(b);
        } else {
            b = PMUI.createHTMLElement('b');
            b.appendChild(document.createTextNode("PAGE " + (currentPage + 1)));
            this.dom.statusBar.appendChild(b);
            this.dom.statusBar.appendChild(document.createTextNode(" of " + totalPages 
                + (filtered ? " filtered pages" :  "")));
        }
        return this;
    };
    /**
     * Change the grid page.
     * @param  {Number} index The index of the page to go.
     * @param  {Boolean} [force] If its true you go to the specified page even if there's no data in it.
     * @chainable
     */
    GridPanel.prototype.goToPage =  function (index, force) {
        var i, initRow, lastRow, listTarget, size, numDisplayedItems = 0, totalPages;

        if (!((index < 0 || index >= this.getTotalPages(this.filterCriteria)) && !force)) {
            this.currentPage = index;

            if(!this.html) {
                return this;
            }

            if (this.filterCriteria && this.filterable) {
                listTarget = this.filteredItems.asArray();
            } else {
                listTarget = this.usableItemsList.asArray();
            }

            size = listTarget.length;

            totalPages = this.getTotalPages(!!this.filterCriteria);
            jQuery(this.dom.tbody).find('>tr').detach();        

            initRow =  this.pageSize !== 0 ? ((index+1) * this.pageSize) - this.pageSize : 0; 
            lastRow = this.pageSize !== 0 ? (initRow + this.pageSize): size;

            for (i = initRow; i < lastRow && i < size; i+=1) {
                this.dom.tbody.appendChild(listTarget[i].updateCellsWidth().getHTML());
                numDisplayedItems += 1;
            }    
        }

        if(!this.html) {
            return this;
        }

        if(numDisplayedItems === 0) {
            this.showEmptyCell();
        }

        this.setBehavior(this.behavior);

        return this.updatePager().updateStatusBar();
    };
    /**
     * Updates the pager.
     * @chainable
     * @private
     */
    GridPanel.prototype.updatePager = function() {
        var totalPages, page, i, link, textContent, additionalClass, currentPage = this.currentPage, dataIndex;
        if(this.dom.pager) {
            jQuery(this.dom.pager).empty();
            totalPages = this.getTotalPages(!!this.filterCriteria);
            if(!totalPages) {
                return this;
            }
            for(i = -1; i <= totalPages; i += 1) {
                additionalClass = textContent = dataIndex = "";
                if(i < 0) {
                    if(currentPage > 0) {
                        dataIndex = 'p';
                        additionalClass = 'pmui-gridpanel-previousbutton';
                    }
                } else if(i === totalPages) {
                    if(currentPage < totalPages - 1) {
                        additionalClass = 'pmui-gridpanel-nextbutton';
                        dataIndex = 'n';
                    }
                } else {
                    textContent = i + 1;
                    if(currentPage === i) {
                        additionalClass = 'pmui-active';
                    }
                    dataIndex = i;
                }
                if(dataIndex !== "") {
                    page = PMUI.createHTMLElement('li');
                    link = PMUI.createHTMLElement('a');
                    link.className = 'pmui-gridpanel-pagelink';
                    link.setAttribute("data-index", dataIndex);
                    if(textContent) {
                        link.textContent = textContent;
                    } else {
                        textContent = PMUI.createHTMLElement('div');
                        textContent.className = 'pmui-icon';
                        link.appendChild(textContent);
                    }
                    jQuery(link).addClass(additionalClass);
                    link.href = '#';
                    page.appendChild(link);
                    this.dom.pager.appendChild(page);  
                }
            }
        }
        return this;
    };
    /**
     * Sets the size for the grid's pages.
     * @param {Number} pageSize
     * @chainable
     */
    GridPanel.prototype.setPageSize =function(pageSize){
        var previousSize = this.pageSize;
        if (typeof pageSize !== 'number') {
            throw new Error("setPageSize(): The method only accepts a number as parameter.");
        }
        if (pageSize < 0) {
            throw new Error("setPageSize(): The method only accepts a number major or equal to zero (0).");   
        }
        this.pageSize = pageSize;
        if (this.html && previousSize !== pageSize) {
            if (pageSize === 0){
                jQuery(this.dom.pager).hide();
                this.currentPage = 0;
            }else{
                jQuery(this.dom.pager).show();
            }
            this.goToPage(this.currentPage);
        }
        return this;
    };
    /**
     * Returns the total number of pages on the grid.
     * @param  {Boolean} [filter=false] If the calculation will be done taking only the filtered results.
     * @return {Number}
     */
    GridPanel.prototype.getTotalPages = function(filter) {
        var targetListSize = filter ? this.filteredItems.getSize() : this.items.getSize();
        
        return !this.items.getSize() ? 0 : (this.pageSize === 0 ? 1 : Math.ceil(targetListSize / this.pageSize));
    };
    /**
     * It defines the events for the Grid.
     * @chainable
     */
    GridPanel.prototype.defineEvents = function() {
        var that = this, i, columnsNum = this.columns.getSize();

        this.removeEvents().eventsDefined = true;
        if(this.dom.pager) {
            this.filterControl.defineEvents();

            for(i = 0; i < columnsNum; i += 1) {
                this.columns.get(i).defineEvents();
            }

            /*for(i = 0; i < this.items.getSize(); i += 1) {
                this.items.get(i).defineEvents();
            }*/
            this.addEvent('click').listenWithDelegation(this.dom.pager, 'a', function(e) {
                var where;
                e.preventDefault();
                e.stopPropagation();
                where = jQuery(this).data("index");
                that.goToPage(where === 'n' ? that.currentPage + 1 : (where === 'p' ? that.currentPage - 1 : where));
            });
        }
        return this;
    };
    /**
     * Sets the current page to be displayed in the grid.
     * @param {Number} currentPage
     * @chainable
     */
    GridPanel.prototype.setCurrentPage = function (currentPage) {
        this.currentPage = currentPage;
        if (this.html) {
            this.goToPage(this.currentPage);
        }
        return this;
    };
    /**
     * Clears the filter.
     * @chainable
     */
    GridPanel.prototype.clearFilter = function() {
        this.filterCriteria = null;
        this.goToPage(this.currentPage);
        return this;
    };
    /**
     * Executes the sorting.
     * @param  {String} criteria The sorting criteria.
     * @param  {String} [sortType="asc"] The sorting order: "asc" or "desc".
     * @chainable
     */
    GridPanel.prototype.sort = function(criteria, sortType) {
        var items, theColumn, i, columns, evaluate;
        this.sortingData.criteria = criteria;
        this.sortingData.type = sortType;
        evaluate = function(a, b) {
            var dataA, dataB, res;
            if(typeof criteria === 'function') {
                dataA = criteria.call(a, a.getData()).toString();
                dataB = criteria.call(b, b.getData()).toString();
            } else {
                dataA = a.getData(); 
                dataB = b.getData();
                dataA = (dataA[criteria] && dataA[criteria].toString()) || "";
                dataB = (dataB[criteria] && dataB[criteria].toString()) || "";
            }
            dataA = dataA.toLowerCase();
            dataB = dataB.toLowerCase();
            if(dataA < dataB) {
                res = 1;
            } else if(dataA > dataB){
                res = -1;
            } else {
                res = 0;
            }
            if (sortType === 'asc') {
                res *= -1;
            }
            return res;
        };
        if(!this.html) {
            return this;
        }
        this.updateUsableItemsList();
        items = this.usableItemsList.asArray();
        
        items.sort(evaluate);

        columns = this.getColumns();
        for(i = 0; i < columns.length; i += 1) {
            columns[i].style.removeClasses(['pmui-sort-asc', 'pmui-sort-desc']);
            if(columns[i].columnData === criteria) {
                theColumn = columns[i];
            }
        }

        if(theColumn) {
            theColumn.style.addClasses(['pmui-sort-' + sortType]);
        }

        if(this.filterCriteria) {
            this.filter(this.filterCriteria);
        } else {
            this.goToPage(this.currentPage);
        }
        this.goToPage(this.currentPage);
        return this;
    };
    /**
     * Returns the function to be executed when an accepted draggedgable is dragged over the droppable area.
     * @return {Function}
     */
    GridPanel.prototype.onDragOver = function() {
        return function() {};
    };
    /**
     * Filters the rows using a criteria. The filter will be applied only in the columns in which its property 
     * "searchable" is true.
     * @param  {String} criteria The criteria.
     * @chainable
     */
    GridPanel.prototype.filter = function(criteria) {
        var i, columns, j, cell, regExp, rowsCopy, visibleRows, pmuiObject, 
            sortingCriteria = this.sortingData.criteria;
        if (typeof criteria === 'string') {
            if (criteria === "") {
                this.clearFilter();
                return this;
            }
            this.filterCriteria = criteria;
        } else if (typeof criteria === 'number'){
            this.filterCriteria = criteria.toString(10);
        }

        if (!this.dom.tbody) {
            return this;
        }

        columns = this.columns.asArray();
        regExp = new RegExp(this.filterCriteria.replace(/([\\\.\[\]\^\$\(\)\?\*\+\|\{\}])/g, "\\\$1"), "i");
        rowsCopy = this.usableItemsList.asArray().slice(0);

        visibleRows = jQuery(this.dom.tbody).find('tr');
        for(i = 0; i < visibleRows.length; i += 1) {
            pmuiObject = PMUI.getPMUIObject(visibleRows[i]);
            if(pmuiObject) {
                jQuery(pmuiObject.html).detach();
            }
        }
        this.filteredItems.clear();
        //if there is a sorting criteria set at the moment, first when use it to filter the rows
        //this is necessary to keep the current sorting order.
        if(sortingCriteria) {
            for(j = 0; j < rowsCopy.length; j += 1) {
                for(i = 0; i < columns.length; i += 1) { 
                    if (columns[i].getDataType() !== 'button' && columns[i].isSearchable()) {
                        if(!rowsCopy[j].html) {
                            rowsCopy[j].getHTML();
                            rowsCopy[j].setCells();
                        }
                        cell = rowsCopy[j].getCells()[i];
                        if (regExp.test(cell.getContent())) {
                            this.filteredItems.insert(rowsCopy.splice(j, 1)[0]);
                            j -= 1;
                            break;
                        }
                    }
                }
            }
        }
        if(rowsCopy.length) {
            for(i = 0; i < columns.length; i += 1) {
                if (columns[i].getDataType() !== 'button' && columns[i].isSearchable()) {
                    for(j = 0; j < rowsCopy.length; j += 1) {
                        if(!rowsCopy[j].html) {
                            rowsCopy[j].getHTML();
                            rowsCopy[j].setCells();
                        }
                        cell = rowsCopy[j].getCells()[i];
                        if (regExp.test(cell.getContent())) {
                            this.filteredItems.insert(rowsCopy.splice(j, 1)[0]);
                            j -= 1;
                        }
                    }
                }
            }
        }
        this.goToPage(0, true);
        return this;
    };
    /**
     * [setSelectable description]
     * @param {[type]} value [description]
     */
    GridPanel.prototype.setSelectable = function (value) {
        if (typeof value !== 'boolean' ) {
            throw new Error ('setSelectable(): the property is not valid, should be a boolean');
        }
        this.selectable = value;
        if(!value && this.selectedRow) {
            this.selectedRow.deselectRow();
            this.selectedRow = null;
        }
        return this;
    };
    /**
     * fixed the property visibleFooter, to hide or not the foot of GridPanel
     * @param {Boolean} value the property should be type 'boolean'
     */
    GridPanel.prototype.setVisibleFooter = function (value){
        if(typeof value !== 'boolean') {
            throw new Error ("setFooterHidden(): the value is no valid, should be a value type 'boolean'");
        } 
        this.visibleFooter = value;
        if(this.html) {
            if(this.visibleFooter){
                this.showFooter();
            }else{
                this.hiddenFooter();
            }
        }
        return this;
    };
    /**
     * hidden the footer of the GridPanel if is visible
     * @chainable 
     */
    GridPanel.prototype.hiddenFooter = function(){
        this.dom.footer.style.display = 'none';
        return this;
    };
    /**
     * show the footer of the GridPanel if it has been hidden
     * @chainable 
     */
    GridPanel.prototype.showFooter = function (){
        this.dom.footer.style.display = 'block';
        return this;
    };
    /**
     * Set the height for the HTML element
     * @param {Number|String} height it can be a number or a string.
     * In case of using a String you only can use 'auto' or 'inherit' or ##px or ##% or ##em when ## is a number.
     * also sets the height of the table being rendered, according to the high set in order gridPanel
     * @chainable
     */
    GridPanel.prototype.setHeight = function (height) {
        GridPanel.superclass.prototype.setHeight.call(this, height);
        
        if (this.dom.tableContainer) {
            if (typeof this.height !== "string") {
                this.dom.tableContainer.style.height = this.height - 74 + 'px';
            } else {
                this.dom.tableContainer.style.height = 'auto';
            }
        }
        return this;
    };

    PMUI.extendNamespace("PMUI.grid.GridPanel", GridPanel);

    if (typeof exports !== 'undefined'){
        module.exports === GridPanel;
    }
}());
(function(){
    /**
     * @class PMUI.grid.GridPanelColumn
     * Class that represents a column to be used in a {@link PMUI.grid.GridPanel GridPanel}.
     * Usually you don't need to instantiate this class since the {@link PMUI.grid.GridPanel GridPanel} do it on its 
     * own, in fact, this class can't be instantiate if a grid parent 
     * ({@link PMUI.grid.GridPanel#cfg-grid grid config option}) is not specified.
     * @extends {PMUI.core.Element}
     * 
     * We don't include any example for this class 'cause we don't recommend you to do it, let the GridPanelColumn do 
     * it. Anyway, if you need add new columns to a GridPanel use its 
     * {@link PMUI.grid.GridPanel#method-addColumn addColumn() method}.
     *
     * @constructor
     * Creates a new instance of the GridPanelColumn object.
     * @param {Object} settings A JSON object with the config options.
     *
     * @cfg {String} [title='[untitled]'] The text to display in the column header.
     * @cfg {String} [dataType='string'] The column data type. The accepted values are: "string", "number", "index" or 
     * "date".
     * @cfg {Boolean} [sortable=false] If the column will have the sorting function enabled.
     * @cfg {Boolean} [searchable=true] If the column must be filter when a filter criteria is set.
     * @cfg {Function} [onButtonClick=null] A callback function to be called when a column's button is clicked. For 
     * info about the callback parameters please read the 
     * {@link PMUI.grid.GridPanelColumn#event-onButtonClick onButtonClick event doc}.
     * @cfg {String} [buttonLabel='[button]'] The label for the button on a column for buttons.
     * @cfg {Array} [cells=[]] The cells that belongs to the column.
     * @cfg {PMUI.grid.GridPanel} grid The grid the column belongs to.
     * @cfg {String|Function} [columnData=null] 
     * - In case of a String: The data key to be used for the cells in this column.
     * - In case of a Function: A function that will return the content for the cells in column, the parameters 
     * received by the function are:
     *  - data {Object}: a JSON object with the data for the current row.
     *  The function will be called in the context of the current {@link PMUI.grid.GridPanelRow row object}.
     * @cfg {Function} [onSort=null] A callback function to be called every time the sorting is executed.
     * @cfg {String} [alignmentCell="center"] A string that specifies the alignmentCell for the content of cells that 
     * appertains to this column. The accepted values are: "center", "left", "right".
     * @cfg {Object|PMUI.util.Style|null} [cellStyle=null] The style to be applied to the cells that belongs to the 
     * column, if it is set to null (the default value) no custom styles will be applied to the cells.
     * @cfg {Object|PMUI.util.Style|Function|null} [buttonStyle=null] The style for the buttons in the column (only applicable
     * if the column has its {@link #property-dataType dataType property} set to "button"). It can be:
     * 
     *  - null then the buttons won't apply any custom style.
     *  - An object literal in that case it can have the properties: cssClasses (an array in which each element is 
     *  an string that is a class name) and cssProperties (another object literal that can contain css properties and
     *  its respective values).
     *  - An instance of {@link PMUI.util.Style}.
     *  - A function that must return any of the previous value types. The parameters received by the functions are:
     *      - the {@link PMUI.grid.GridPanelCell GridPanelCell} object in which the style is being applied.
     *      - data {Object}: a JSON object with the data for the current row.
     *      The function will be called in the context of the current {@link PMUI.grid.GridPanelRow row object}.
     * @cfg {Boolean} [disabledButtons=false] If the buttons will be disabled.
     * @cfg {String|Function|null} [buttonTooltip=null] Sets the tooltip message for the buttons (only applicable if 
     * the column has its {@link #property-dataType dataType property} set to "button"). It can be a string
     * or null or a function. In the latter case the function must return a string, it will receive two parameters:
     *
     * - The cell, a {@link PMUI.grid.GridPanelCell GridPanelCell} object.
     * - The data that belongs to the current row.
     * 
     * The function will be called in the context of the current 
     * {@link PMUI.grid.GridPanelCell GridPanelCell cell object}.
     */
    var GridPanelColumn = function (settings) {
        /**
         * The cells that belongs to the column.
         * @type {PMUI.util.ArrayList}
         */
        this.cells = new PMUI.util.ArrayList();
        GridPanelColumn.superclass.call(this, jQuery.extend(settings, {elementTag: "th"}));
        /**
         * The text to be shown in the column header.
         * @type {String}
         */
        this.title = null;
        /**
         * The data type for the cells in this column. It is set by the {@link #cfg-dataType dataType config option} or
         * the {@link #method-setDataType setDataType()} method.
         * @type {String}
         */
        this.dataType = null;
        /**
         * If the column has the sorting functionality enabled.
         * @type {Boolean}
         */
        this.sortable = null;
        /**
         * If the column has the searching functionality enabled.
         * @type {Boolean}
         */
        this.searchable = null;
        /**
         * @event onButtonClick
         * Fired when a button in this column is clicked. The event is called in the column context.
         * @param {PMUI.grid.GridPanelRow} row The row the button belongs to.
         * @param {PMUI.grid.GridPanel} grid The grid the button belongs to.
         */
        this.onButtonClick = null;
        /**
         * The label for the buttons in the column (only applicable when the datatType is set to "button").
         * @type {String|Function}
         */
        this.buttonLabel = null;
        /**
         * If the buttons are disabled or not.
         * @type {Boolean}
         */
        this.disabledButtons = null;
        /**
         * The grid the column belongs to.
         * @type {PMUI.grid.GridPanel}
         */
        this.grid = null;
        /**
         * The data key to be used for the cells in this column.
         * @type {String}
         */
        this.columnData = null;
        /**
         * The column sort order ("ASC" or "DESC").
         * @type {String}
         */
        this.sortOrder = null;
        /**
         * The alignmentCell to be applied to the content of the cells that appertain to this column.
         * @type {String}
         */
        this.alignmentCell = null;
        /**
         * The style object for the cells that belongs to the column.
         * @type {PMUI.util.Style}
         * @private
         */
        this.cellStyle = null;
        /**
         * @property {PMUI.util.Style} buttonStyle The style object to be applied in the buttons that belong to the 
         * column. Only applicable when the column's type is 'button'.
         */
        this.buttonStyle = null;
        /**
         * The HTML element in which the cell content will be appended.
         * @type {HTMLElement}
         * @private
         */
        this.contentTitle = null;
        /**
         * The tooltip to be shown for the buttons in the column. Only applicable when the column's type is 'button'.
         * @type {String|Function}
         */
        this.buttonTooltip = null;
        /**
         * The DOM element in which the sorting arrow is shown.
         * @type {HTMLElement}
         * @private
         */
        this.sortIcon = null;
        /**
         * Determines the alignment for the title in the column's header,
         * @type {String}
         */
        this.alignmentTitle = null; 
        GridPanelColumn.prototype.init.call(this, settings);
    };

    PMUI.inheritFrom("PMUI.core.Element", GridPanelColumn);
    
    GridPanelColumn.prototype.type = "GridPanelColumn";
    /**
     * Initialize the object
     * @param  {Object} [settings=null] A JSON object with the config options.
     * @private
     */
    GridPanelColumn.prototype.init = function (settings){
        var defaults = {
            title: '[untitled]',
            dataType: 'string',
            sortable: false,
            searchable: true,
            onButtonClick: function(){},
            buttonLabel: '[button]',
            cells: [],
            grid: null,
            columnData: null,
            onSort: null,
            alignmentCell: "center",
            cellStyle: null,
            buttonStyle: null,
            visible: true,
            disabledButtons: false,
            buttonTooltip: null,
            alignmentTitle : 'center',
            positionMode : 'inherit'
        };

        jQuery.extend(true, defaults, settings);
        
        this.buttonStyle = new PMUI.util.Style();
        this.cellStyle = new PMUI.util.Style();
        this.onSort = defaults.onSort;
        this.setPositionMode(defaults.positionMode);
        this.setTitle(defaults.title)
            .setButtonTooltip(defaults.buttonTooltip)
            .setCellStyle(defaults.cellStyle)
            .setButtonStyle(defaults.buttonStyle)
            .setAlignmentCell(defaults.alignmentCell)
            .setColumnData(defaults.columnData)
            .setDataType(defaults.dataType)
            .setOnButtonClick(defaults.onButtonClick)
            .setButtonLabel(defaults.buttonLabel)
            .setGrid(defaults.grid)
            .setSortable(defaults.sortable)
            .setCells(defaults.cells)
            .setVisible(defaults.visible)
            .setAlignmentTitle(defaults.alignmentTitle);

        if(defaults.searchable) {
            this.enableSearch();
        } else {
            this.disableSearch();
        }

        if(defaults.disabled) {
            this.disable();
        } else {
            this.enable();
        }
    };
    /**
     * Sets the button tooltip. Only applicable 
     * @param {String|Function|null} tooltip The tooltip to be used in the buttons in the column. It can be a string
     * or null or a function. In the latter case the function must return a string, it will receive two parameters:
     *
     * - The cell, a {@link PMUI.grid.GridPanelCell GridPanelCell} object.
     * - The data that belongs to the current row.
     * 
     * The function will be called in the context of the current 
     * {@link PMUI.grid.GridPanelCell GridPanelCell cell object}.
     */
    GridPanelColumn.prototype.setButtonTooltip = function(tooltip) {
        if(!(typeof tooltip === 'function' || typeof tooltip === 'string' || tooltip === null)) {
            throw new Error('setButtonTooltip(): the parameter must be a function or a string or null.');
        }
        this.buttonTooltip = tooltip;
        return this;
    };
    /**
     * Disables all the buttons in the column, this is applied only if the column's property 
     * {@link #property-dataType dataType} is set to "button".
     * @chainable
     */
    GridPanelColumn.prototype.disable = function() {
        var i, cells, cellsLength;
        this.disabledButtons = true;
        cellsLength = (cells = this.cells.asArray()).length;
        for(i = 0; i < cellsLength; i++) {
            cells[i].disable();
        }
        return this;
    };
    /**
     * Enables all the buttons in the column, this is applied only if the column's property 
     * {@link #property-dataType dataType} is set to "button".
     * @chainable
     */
    GridPanelColumn.prototype.enable = function() {
        var i, cells, cellsLength;
        if(this.dataType === 'button') {
            this.disabledButtons = false;
            cellsLength = (cells = this.cells.asArray()).length;
            for(i = 0; i < cellsLength; i++) {
                cells[i].enable();
            }
        }
        return this;
    };
    /**
     * [setVisible description]
     * @param {[type]} visible [description]
     */
    GridPanelColumn.prototype.setVisible = function(visible) {
        var cells, cellsLength, i;
        GridPanelColumn.superclass.prototype.setVisible.call(this, visible);
        if(this.cells) {
            cells = this.cells.asArray();
            cellsLength = cells.length;
            for(i = 0; i < cellsLength; i++) {
                cells[i].setVisible(this.visible);
            }
        }
        return this;
    };
    /**
     * [getUsableButtonStyle description]
     * @param  {[type]} cell [description]
     * @return {[type]}      [description]
     */
    GridPanelColumn.prototype.getUsableButtonStyle = function(cell) {
        var theStyle, buttonStyle = this.buttonStyle, theRow;
        if(typeof buttonStyle !== 'function') {
            return {
                    cssClasses: buttonStyle.cssClasses,
                    cssProperties: buttonStyle.cssProperties
                };
        } else {
            theRow = cell.row;
            theStyle = buttonStyle.call(theRow, cell, theRow.getData());
            return {
                cssClasses: (theStyle && theStyle.cssClasses) || [],
                cssProperties: (theStyle && theStyle.cssProperties) || {}
            };
        }
    };
    /**
     * Sets the style for the buttons in the column (only applicable if the column has its 
     * {@link #property-dataType dataType property} set to "button").
     * @param {PMUI.util.Style|Object|null|Function} cellStyle The style to be applied. It can be:
     * 
     *  - null then the buttons won't apply any custom style.
     *  - An object literal in that case it can have the properties: cssClasses (an array in which each element is 
     *  an string that is a class name) and cssProperties (another object literal that can contain css properties and
     *  its respective values).
     *  - An instance of {@link PMUI.util.Style}.
     *  - A function that must return any of the previous value types.
     *  @chainable
     */
    GridPanelColumn.prototype.setButtonStyle = function (buttonStyle) {
        var i, cells, cellsNum, theStyle, theRow;
        if(buttonStyle !== null && typeof buttonStyle !== 'object' && typeof buttonStyle !== 'function' && !(buttonStyle instanceof PMUI.util.Style)) {
            throw new Error("setbuttonStyle(): The parameter only can be an object, an instance of PMUI.util.Style or null.");
        }
        this.buttonStyle = this.buttonStyle.clear();
        if(buttonStyle instanceof PMUI.util.Style) {
            this.buttonStyle = buttonStyle;
        } else if(typeof buttonStyle === 'function') {
            this.buttonStyle = buttonStyle;
        } else if(buttonStyle) {
            this.buttonStyle.addProperties((buttonStyle && buttonStyle.cssProperties) || {})
                .addClasses((buttonStyle && buttonStyle.cssClasses) || []);
        }
        if(this.dataType === 'button') {
            cells = this.cells.asArray();
            cellsNum = cells.length;
            for(i = 0; i < cellsNum; i++) {
                cells[i].getContent().setStyle(this.getUsableButtonStyle(cells[i]));
            }
        }
        return this;
    };
    /**
     * Returns the style for the buttons that belongs to the column.
     * @return {Object} An object literal with the properties: cssProperties and cssClasses
     */
    GridPanelColumn.prototype.getButtonStyle = function() {
        return {
            cssClasses: this.buttonStyle.cssClasses,
            cssProperties: this.buttonStyle.cssProperties
        };
    };
    /**
     * Sets the style for the cells in the column.
     * @param {PMUI.util.Style|Object|null} cellStyle The style to be applied. If it is null then the cells won't apply 
     * any custom style.
     */
    GridPanelColumn.prototype.setCellStyle = function(cellStyle) {
        var i = 0, cells, cellsNum;
        if(cellStyle !== null && typeof cellStyle !== 'object' && !(cellStyle instanceof PMUI.util.Style)) {
            throw new Error("setCellStyle(): The parameter only can be an object, an instance of PMUI.util.Style or null.");
        }
        this.cellStyle = this.cellStyle.clear();
        if(cellStyle instanceof PMUI.util.Style) {
            this.cellStyle = cellStyle;
        } else if(cellStyle) {
            this.cellStyle.addProperties(cellStyle.cssProperties || {})
                .addClasses(cellStyle.cssClasses || []);
        }
        cells = this.cells.asArray();
        cellsNum = cells.length;
        for(i = 0; i < cellsNum; i++) {
            cells[i].setStyle({
                cssClasses: this.cellStyle.cssClasses,
                cssProperties: this.cellStyle.cssProperties
            });
        }
        return this;
    };
    /**
     * Returns the style for the cells that belongs to the column.
     * @return {Object} An object literal with the properties: cssProperties and cssClasses
     */
    GridPanelColumn.prototype.getCellStyle = function() {
        return {
            cssProperties: this.cellStyle.cssProperties,
            cssClasses: this.cellStyle.cssClasses
        };
    };
    /**
     * Sets the alignmentCell for the content in the cells that appertain to the column.
     * @param {String} alignmentCell The accepted values are: "center", "left", "right".
     * @chainable
     */
    GridPanelColumn.prototype.setAlignmentCell = function(alignmentCell) {
        var i, cells;
        if(alignmentCell === 'center' || alignmentCell === 'left' || alignmentCell === 'right') {
            this.alignmentCell = alignmentCell;
            cells = this.getCells();
            for(i = 0; i < cells.length; i++) {
                cells[i].setAlignment(this.alignmentCell);
            }
        } else {
            throw new Error('setAlignment(): This method only accepts one of the following values: ' 
                + '"center", "left", "right"');
        }
        return this;
    };

    GridPanelColumn.prototype.setAlignmentTitle = function(alignmentTitle) {
        var i, cells;
        if(alignmentTitle === 'center' || alignmentTitle === 'left' || alignmentTitle === 'right') {
            this.alignmentTitle = alignmentTitle;
            if(this.contentTitle) {
                this.contentTitle.style.textAlign = alignmentTitle;
            }
        } else {
            throw new Error('setAlignment(): This method only accepts one of the following values: ' 
                + '"center", "left", "right"');
        }
        return this;
    };
    /**
     * Returns the alignment for the content in the cells that appertain to the column.
     * @return {String}
     */
    GridPanelColumn.prototype.getAlignment = function() {
        return this.alignment;
    };
    /**
     * Returns true is the filtering function is enabled, otherwise returns false.
     * @return {Boolean} 
     */
    GridPanelColumn.prototype.isSearchable = function() {
        return this.searchable;
    };
    /**
     * Enables the filtering functionality.
     * @chainable
     */
    GridPanelColumn.prototype.enableSearch = function() {
        this.searchable = true;
        return this;
    };
    /**
     * Disables the filtering functionality.
     * @chainable
     */
    GridPanelColumn.prototype.disableSearch = function() {
        this.searchable = false;
        return this;
    };
    /**
     * Returns the data's field name that is used by the row to fill its cell for this column.
     * @return {String}
     */
    GridPanelColumn.prototype.getColumnData = function() {
        return this.columnData;
    };
    /**
     * Sets a function or a string that will represent the data's field name that is used by the row to fill its cell 
     * for this column.
     * @param {String|Function} columnData It can be a String or a Function
     * - In case of a String: The data key to be used for the cells in this column.
     * - In case of a Function: A function that will return the content for the cells in column, the parameters receive
     * by the function are:
     *  - data {Object}: a JSON object with the data for the current row.
     *  The function will be called in the context of the current {@link PMUI.grid.GridPanelRow row object}.
     * @chainable
     */
    GridPanelColumn.prototype.setColumnData = function(columnData) {
        if(columnData === null || typeof columnData === 'string' || typeof columnData === 'function') {
            this.columnData = columnData;
        } else {
            throw new Error('setColumnData(): The method only accepts a string or a function or NULL as parameter.');
        }

        return this;
    };
    /**
     * Sets the title to be shown in the column header.
     * @param {String} title
     * @chainable
     */
    GridPanelColumn.prototype.setTitle = function (title) {
        this.title = title;
        if(this.html){
            jQuery(this.contentTitle).empty();
            this.contentTitle.appendChild(document.createTextNode(title));
            this.contentTitle.appendChild(this.sortIcon);
        }
        return this;
    };
    /**
     * Returns the column's data type.
     * @return {String}
     */
    GridPanelColumn.prototype.getDataType = function() {
        return this.dataType;
    };
    /**
     * Sets the data type for the column.
     * @param {String} dataType The accepted values are:
     *
     * - "number" for cells with numeric values.
     * - "string" for cells with string values.
     * - "index" for cells with the row index.
     * - "date" for cells with date values.
     * - "button" for cells that contain a button.
     * @chainable
     */
    GridPanelColumn.prototype.setDataType = function (dataType) {
        if(dataType === 'string' || dataType === 'number' || dataType === 'index' || dataType === 'date' || dataType === 'button') {
            this.dataType = dataType;
        } else {
            throw new Error('setDataType(): It only accepts "string", "number", "index" or "date".');
        }
        return this;
    };
    /**
     * Enables or disables the sorting functionality.
     * @param {Boolean} sortable
     * @chainable
     */
    GridPanelColumn.prototype.setSortable = function (sortable){
        if(!(typeof sortable === 'boolean')) {
            throw new Error ("setSortable(): property 'sortable' should be a boolean");
        }
        //TODO enable event to listen the clicks on column for sorting
        this.sortable = sortable;
        if(this.html) {
            if(sortable) {
                this.style.addClasses(['pmui-sortable']);
            } else {
                this.style.removeClasses(['pmui-sortable']);
            }
        }
        return this;
    };
    /**
     * Returns the callback function to be executed when a column's button is clicked.
     * @return {Function}
     */
    GridPanelColumn.prototype.getOnButtonClick = function() {
        return this.onButtonClick;
    };
    /**
     * Sets the callback function to be executed when a column's button is clicked.
     * @param {Function} onButtonClick
     * @chainable
     */
    GridPanelColumn.prototype.setOnButtonClick = function (onButtonClick) {
        if(!(typeof onButtonClick == 'function')) {
            throw new Error ("setOnButtonClick(): property 'onButtonClick' should be a function");
        }
        this.onButtonClick = onButtonClick;
        //TODO update the listeners for all the buttons in the column
        return this; 
    };
    /**
     * Sets the text for the button labels in the column.
     * @param {String|Function} buttonLabel A string or a function that returns the string for the button. Three 
     * parameters function will be sent to the function:
     *
     * - The row the button belongs to ({@link PMUI.grid.GridPanelRow GridPanelRow}).
     * - The data from the row (object literal).
     *
     * It is important to mention that the function is called in the context of the column object.
     * @chainable
     */
    GridPanelColumn.prototype.setButtonLabel = function (buttonLabel) {
        if(typeof buttonLabel !== 'string' && typeof buttonLabel !== 'function') {
            throw new Error('setButtonLabel(): the parameter must be a string or a function.');
        }
        this.buttonLabel = buttonLabel;
        //TODO update the label for all the buttons in the column
        return this;
    };
    /**
     * Set the grid the column belongs to.
     * @param {PMUI.grid.Grid} grid
     * @chainable
     */
    GridPanelColumn.prototype.setGrid = function (grid){
        if(!(grid instanceof PMUI.grid.GridPanel)) {
            throw new Error ('setGrid(): the grid property should be instance of [PMUI.grid.GridPanel]');
        }
        this.grid = grid;
        return this;        
    };
    /**
     * Adds a cell to the column.
     * @param {PMUI.grid.GridPanelCell} cell
     * @chainable
     */
    GridPanelColumn.prototype.addCell = function(cell) {
        if(!(cell instanceof PMUI.grid.GridPanelCell)) {
            throw new Error("addCell(): this method only accepts instances of PMUI.grid.GridPanelCell.");
        }
        cell.setAlignment(this.alignmentCell);
        cell.setStyle({
            cssProperties: this.cellStyle.getProperties(),
            cssClasses: this.cellStyle.getClasses()
        });
        //console.log("\t\tCELL insert (in column)" + isd);
        this.cells.insert(cell);

        return this;
    };
    /**
     * Set the cells for the column.
     * @param {PMUI.util.ArrayList|Array} cells An array in which every element is an instance of the 
     * {@link PMUI.grid.GridPanelCell GridPanelCell class}.
     * @chainable
     */
    GridPanelColumn.prototype.setCells = function (cells) {
        var cellsAux = [], i, cellToAdd;

        this.clearCells();

        if(cells instanceof PMUI.util.ArrayList){
            cellsAux = cells.items.asArray();
        } else if(jQuery.isArray(cells)){
            cellsAux = cells;
        } else {
            throw new Error('setCells(): the cells property should be instanceof of PMUI.util.ArrayList or an Array');
        }
        for (i = 0; i < cellsAux.length; i += 1){
            this.addCell(cellsAux[i]);
        }
        return this;
    };
    /**
     * Return the column's cells.
     * @return {Array} An array in which each element is an instance of the 
     * {@link PMUI.grid.GridPanelCell GridPanelCell class}.
     */
    GridPanelColumn.prototype.getCells = function() {
        return this.cells.asArray().slice(0);
    };
    /**
     * Removes a cell from the column.
     * @param  {PMUI.grid.GridPanelCell|Number|String} cell The cell to remove, it can be:
     *
     * - An instance of {@link PMUI.grid.GridPanelCell GridPanelCell}, in this case this cell must belong to the 
     * column.
     * - A Number, in this case it is interpreted as the index of the cell to remove.
     * - A String, in this case it is interpreted as the id of the cell to remove.
     * @chainable
     */
    GridPanelColumn.prototype.removeCell = function(cell) {
        var cellToRemove;

        if(cell instanceof PMUI.grid.GridPanelCell) {
            cellToRemove = cell;
        } else if(typeof cell === 'number') {
            cellToRemove = this.cells.get(cell);
        } else if(typeof cell === 'string') {
            cellToRemove = this.cells.find("id", cell);
        }

        if(cellToRemove) {
            cellToRemove.column = null;
            this.cells.remove(cellToRemove);
        }

        return this;
    };
    /**
     * Clears all the cells of the column.
     * @chainable
     */
    GridPanelColumn.prototype.clearCells = function () {
        this.cells.clear();
        while(this.cells.getSize() > 0) {
            this.removeCell(0);
        }
        return this;
    };
    /**
     * Defines the event for the object.
     * @chainable
     */
    GridPanelColumn.prototype.defineEvents = function() {
        var that = this;
        this.removeEvents().eventsDefined = true;
        if(this.html) {
            this.addEvent('click').listen(this.html, function() {
                if(that.sortable) {
                    that.sortOrder = that.sortOrder === 'asc' ? 'desc' : 'asc';
                    if(typeof that.onSort === 'function') {
                        that.onSort(that.sortOrder);
                    }
                }
            });
        }
        return this;
    };
    /**
     * Creates the html for the object.
     * @return {HTMLElement}
     */
    GridPanelColumn.prototype.createHTML = function () {
        var contentTitle, sortIcon;
        if (this.html){
            return this.html;
        }
        GridPanelColumn.superclass.prototype.createHTML.call(this);
        sortIcon = PMUI.createHTMLElement('span');
        sortIcon.className = 'pmui-grid-sort-icon';
        contentTitle = PMUI.createHTMLElement('span');
        contentTitle.className = 'pmui-gridpanelColumn-title';
        this.contentTitle = contentTitle;
        this.sortIcon = sortIcon;
        this.html.appendChild(contentTitle);
        this.setTitle(this.title)
            .setSortable(this.sortable)
            .setWidth(this.width)
            .setAlignmentTitle(this.alignmentTitle);
        return this.html;
    };

    GridPanelColumn.prototype.setWidth = function(width) {
        var finalWidth, i, cellsLength, cells;
        if(typeof width === 'number') {
            this.width = width;
        } else if(/^\d+(\.\d+)?px$/.test(width)) {
            this.width = parseInt(width, 10);
        } else if(/^\d+(\.\d+)?%$/.test(width)) {
            this.width = width;
        } else if(/^\d+(\.\d+)?em$/.test(width)) {
            this.width = width;
        } else if(width === 'auto') {
            this.width = width;
        } else {
            throw new Error('setWidth: width param is not a number');
        }

        if (typeof this.width === 'string') {
            finalWidth = 'auto';
        } else {
            finalWidth = this.width + 'px';
        } 

        if(this.html){
            this.contentTitle.style.minWidth = finalWidth;  
        }
        cellsLength = (cells = this.cells.asArray()).length;
        for(i = 0; i < cellsLength; i++) {
            cells[i].setWidth(finalWidth);
        }
        return this;
    };

    GridPanelColumn.prototype.applyStyle = function() {
        if(this.html) {
            this.style.applyStyle();

            this.style.addProperties({
                display: this.visible ? this.display: "none",
                position: this.positionMode,
                left: this.x,
                top: this.y,
                width: (typeof this.width === 'string')? this.width : 'auto',
                height: this.height,
                zIndex: this.zOrder
            });
        }
        return this;
    };

    PMUI.extendNamespace("PMUI.grid.GridPanelColumn", GridPanelColumn);
    if(typeof exports!=='undefined'){
        module.exports = GridPanelColumn;
    }
}());
(function(){
    /**
     * @class PMUI.grid.GridPanelRow
     * Class that defines a row to be used in a {@link PMUI.grid.GridPanel GridPanel}.
     * The single class is not very useful alone, it should be used with a {@link PMUI.grid.GridPanel GridPanel}.
     * @extends {PMUI.core.Element}
     *
     * Usage example:
     *
     *      @example
     *      var row = new PMUI.grid.GridPanelRow({
     *              data: {
     *                  name: "John",
     *                  lastname: "McAllister"
     *              }
     *          }),
     *          grid = new PMUI.grid.GridPanel({
     *             pageSize: 6,
     *             columns:[
     *                 {
     *                     title:'columna1',
     *                     dataType:'string',
     *                     columnData: "name"
     *                 },
     *                 {
     *                     title:'columna2',
     *                     dataType:'number',
     *                     width : 150,
     *                     columnData: "lastName",
     *                     sortable: true
     *                 }
     *             ],
     *             items: [
     *                 row
     *             ]
     *          });
     *
     *      document.body.appendChild(grid.getHTML());
     *
     * @constructor
     * Creates a new instance of the GridPanelRow class.
     * @param {Object} settings A JSON object with the config options.
     *
     * @cfg {PMUI.data.DataField|Object} [data={}] A {@link PMUI.data.DataField DataField} or a JSON object with the 
     * data for the GridPanelRow.
     * @cfg {PMUI.grid.GridPanel} [parent=null] The {@link PMUI.grid.GridPanel GridPanel} the GridPanelRow belongs to.
     */
    var GridPanelRow = function (settings){
        GridPanelRow.superclass.call(this, jQuery.extend(settings, {elementTag: "tr"}));
        /**
         * An ArrayList that contains the row's cells ({@link PMUI.grid.GridPanelCell GridPanelCell}).
         * @type {PMUI.util.ArrayList}
         * @private
         */
        this.cells = null;
        /**
         * The parent GridPanel of the GridPanelRow.
         * @type {PMUI.grid.GridPanel GridPanel}
         * @readonly
         */
        this.parent = null;
        /**
         * The data that belongs to the GridPanelRow.
         * @type {PMUI.data.DataField}
         * @private
         */
        this.data = null;

        this.onClick = null;

        this.selected = null;
        this.onSelect = null;
        GridPanelRow.prototype.init.call(this, settings);
    };
    
    PMUI.inheritFrom('PMUI.core.Element', GridPanelRow);
    /**
     * The class family.
     * @type {String}
     */
    GridPanelRow.prototype.family = 'GridPanelRow';
    /**
     * The object's family.
     * @type {String}
     */
    GridPanelRow.prototype.type = 'GridPanelRow';
    /**
     * Initialize the object.
     * @param  {Object} settings A JSON object with the config options.
     * @private
     */
    GridPanelRow.prototype.init = function(settings) {
        var defaults = {
            data : {},
            parent: null,
            onClick: null,
            onSelect: function(){}
        };

        jQuery.extend(true, defaults, settings);

        this.cells = new PMUI.util.ArrayList();
        this.data = new PMUI.data.DataField();

        this.setData(defaults.data)
            .setParent(defaults.parent)
            .setOnClick(defaults.onClick)
            .setOnSelectCallback(defaults.onSelect);
    };
    /**
     * Set the data for the object.
     * @param {PMUI.data.DataField|Object} dataç
     * @chainable
     */
    GridPanelRow.prototype.setData = function(data) {
        var key;
        if(data instanceof PMUI.data.DataField) {
            this.data = data;
        } else if(typeof data === 'object') {
            this.data.clear();
            for(key in data) {
                this.data.addAttribute(key, data[key]);
            }
        } else {
            throw new Error("setData(): it only accepts a JSON object o an instance of PMUI.data.DataField as parameter.");
        }

        if(this.html) {
            this.setCells();
        }

        return this;
    };
    /**
     * Set the parent grid for the object.
     * @param {PMUI.grid.GridPanel} parent
     * @chainable
     */
    GridPanelRow.prototype.setParent = function (parent) {
        if(parent) {
            if(parent instanceof PMUI.grid.GridPanel) {
                this.parent = parent;
            } else {
                throw new Error("setParent() method only accepts a object [PMUI.panel.GridPanel] as parameter.");
            }
        }
        return this;
    };
    /**
     * Returns the object's parent grid.
     * @return {PMUI.grid.GridPanel}
     */
    GridPanelRow.prototype.getParent = function () {
        return this.parent;
    };
    /**
     * Updates the width for each row's cell using the respective column's width.
     * @chainable
     */
    GridPanelRow.prototype.updateCellsWidth = function() {
        var i, cells, cellsLength;
        cellsLength = (cells = this.cells.asArray()).length;
        for(i = 0; i < cellsLength; i++) {
            cells[i].setWidth(cells[i].column.width);
        }
        return this;
    };
    /**
     * Paints a cell in the row.
     * @param  {PMUI.grid.GridPanelCell} cell 
     * @chainable
     * @private
     */
    GridPanelRow.prototype.paintCell = function(cell) {
        if(this.html) {
            this.html.appendChild(cell.getHTML());
        }
        /*if(this.eventsDefined) {
            cell.defineEvents();
        }*/
        return this;
    };
    /**
     * Paints all cells in the row.
     * @chainable
     * @private
     */
    GridPanelRow.prototype.paintCells = function() {
        var i, cells = this.cells.asArray();
        for(i = 0; i < cells.length; i++) {
            this.paintCell(cells[i]);
        }
        return this;
    };
    /**
     * Add a single cell into the row.
     * @param {PMUI.grid.GridPanelColumn|null} [column=null] It can be:
     * 
     * - null, in this case an empty cell will be added at the end of the row.
     * - A {@link PMUI.grid.GridPanelColumn GridPanelColumn}, in this case the cell will be created using the settings 
     * the parameter has for its cells. Also the cell will be associated to the column.
     */
    GridPanelRow.prototype.addCell = function(column) {
        var data, columnData, cell, cellIndex, buttonLabel, buttonTooltip, content, that = this;
        if(!(column === null || column instanceof PMUI.grid.GridPanelColumn)) {
            throw new Error("addCell(): The parameter must be an instance of PMUI.grid.GridPanel o null.");
        }
        data = this.data.getRecord();
        cellIndex = this.parent.getColumns().length;
        cell = new PMUI.grid.GridPanelCell({
            columnIndex: cellIndex,
            row: this, 
            column: column || null,
            visible: (column && column.visible) && true,
            width: (column && column.width) || "auto"
        });
        if(column) {
            columnData = column.getColumnData();
            if(column.dataType === 'button') {
                buttonLabel = column.buttonLabel;
                buttonTooltip = column.buttonTooltip;
                content = new PMUI.ui.Button({
                    style: column.getUsableButtonStyle(),
                    text: typeof buttonLabel === 'function' ? buttonLabel(this, data) : (buttonLabel || ""),
                    messageTooltip: (typeof buttonTooltip === 'function' ? buttonTooltip.call(cell, cell, data) 
                        : (buttonTooltip || "")),
                    handler: (function(handler){
                        return function() {
                            handler(that, that.parent);
                        };
                    }(column.getOnButtonClick()))
                });
            } else if(column.dataType === 'index') {
                content = this.parent.getItemIndex(this) + 1;
            } else if(columnData) {
                if(typeof columnData === 'function') {
                    content = columnData.call(this, data);
                } else {
                    content =  this.data.getAttribute(columnData);
                }
            }
            
            cell.style.addClasses(["pmui-gridpanelcell-" + column.dataType]);
        } else {
            content = null;
        }
        cell.setContent(content);
        this.cells.insert(cell);
        column.addCell(cell);
        this.paintCell(cell);
        return this;
    };
    /**
     * @method setCells
     * Build the cells for the row using its parent {@link PMUI.grid.GridPanel#property-columns GridPanel's columns} to
     * retrieve the data from the odject's {@link PMUI.grid.GridPanelRow#property-data data}.
     * @chainable
     */
    GridPanelRow.prototype.setCells = function () {
        var i, columns;
        this.clearCells();
        if(this.parent) {
            columns = this.parent.getColumns();
            for(i = 0; i < columns.length; i++) {
                this.addCell(columns[i]);
            }
            //this.defineEvents();
        }

        return this;
    };
    /**
     * Returns the data that belongs to the object.
     * @return {Object} A JSON object with the data.
     */
    GridPanelRow.prototype.getData = function() {
        return this.data.getRecord();
    };
    /**
     * Returns the object's cells.
     * @return {Array} An array in which every item is a object's {@link PMUI.grid.GridPanelCell cell}.
     */
    GridPanelRow.prototype.getCells = function() {
        return this.cells.asArray().slice(0);
    };
    /**
     * Clear the cells from the object.
     * @chainable
     */
    GridPanelRow.prototype.clearCells = function (){
        var i;
        while(this.cells.getSize()){
            this.removeCell(0);
        }       
        return this;
    };
    /**
     * Remove a cell from the object.
     * @param  {PMUI.grid.GridPanelCell|Number|String} cell It can be:
     *
     * - A instance of {@link PMUI.grid.GridPanelCell GridPanelCell}, in this case it must be a cell that belongs to 
     * the object.
     * - A Number, in this case the cell with that index will be removed.
     * - A String, in this case the cell with that id will be removed.
     * @chainable
     */
    GridPanelRow.prototype.removeCell = function (cell) {
        var cellToRemove, column;

        if(cell instanceof PMUI.grid.GridPanelCell) {
            cellToRemove = cell;
        } else if(typeof cell === 'number') {
            cellToRemove = this.cells.get(cell);
        } else if(typeof cell === 'string') {
            cellToRemove = this.cells.find("id", cell)
        } else {
            throw new Error("removeCell(): This method only accepts a number or a string or a instance of " + 
                "PMUI.grid.GridPanelCell as parameter");
        }

        if(cellToRemove) {
            cellToRemove.setColumnIndex(null);
            cellToRemove.row = null;
            jQuery(cellToRemove.html).remove();
            column = cellToRemove.getColumn();
            if(column) {
                column.removeCell(cellToRemove);
            }
            cellToRemove.column = null;
            this.cells.remove(cellToRemove);
        }

        return this;
    };
    /**
     * Attach the listeners for the object and every of its children.
     * @chainable
     */
    GridPanelRow.prototype.defineEvents = function() {
        var i, that = this;
        
        this.removeEvents().eventsDefined = true;
        if(this.html) {
            /*for(i = 0; i < this.cells.getSize(); i++) {
                this.cells.get(i).defineEvents();
            }*/
            this.addEvent('click').listen(this.html,function(e) {
                e.preventDefault();
                if(typeof that.onClick === 'function') {
                    that.onClick(that, that.getData());
                }
                if(that.parent.selectable) {
                    if(!that.selected){
                        that.selectRow();
                    }
                }
                if(that.parent && typeof that.parent.onRowClick === 'function') {
                    that.parent.onRowClick(that, that.getData());
                } 
            });
        }
        return this;
    };
    /**
     * Creates the object's HTML.
     * @return {HTMLElement}
     */
    GridPanelRow.prototype.createHTML = function () {
        var td,type, i, text, number, date;
        if (this.html) {
            return this.html;
        }
        GridPanelRow.superclass.prototype.createHTML.call(this);
        this.paintCells()
            .defineEvents();
        return this.html;
    };
    /**
     * [setOnClick description]
     * @param {[type]} onClick [description]
     */
    GridPanelRow.prototype.setOnClick = function (onClick) {
        if (typeof  onClick !== 'function' && onClick !== null) {
            throw new Error ('setOnClick(): the parameter is no valid, should be a function');
        }
        this.onClick = onClick;
        return this;
    };
    /**
     * [selectRow description]
     * @return {[type]} [description]
     */
    GridPanelRow.prototype.selectRow = function() {
        this.selected = true;
        this.style.addClasses(['pmui-active-row']);
        if(typeof this.onSelect === 'function') {
            this.onSelect();   
        }
        return this;
    };
    /**
     * [isSelectedRow description]
     * @return {Boolean} [description]
     */
    GridPanelRow.prototype.isSelectedRow =function() {
        return this.selected;
    };
    /**
     * [deselectRow description]
     * @return {[type]} [description]
     */
    GridPanelRow.prototype.deselectRow = function() {
        if(this.onSelect) {
            this.selected = false;
            this.style.removeClasses(['pmui-active-row']);  
        }

        return this;
    };
    /**
     * [setOnSelectRowCallback description]
     * @param {Function} callback [description]
     */
    GridPanelRow.prototype.setOnSelectCallback = function(callback) {
        if(typeof callback === 'function' || callback == null) {
            this.onSelect = callback;
        } else {
            throw new Error("setOnSelectCallback(): The parameter is not a function.");
        }
        return this;
    };

    PMUI.extendNamespace('PMUI.grid.GridPanelRow',GridPanelRow);

    if (typeof exports!== 'undefined'){
        module.exports = GridPanelRow;
    }
}());
(function () {
    /**
     * @class PMUI.grid.GridPanelCell
     * Class that represent a cell to be used in instances of {@link PMUI.grid.GridPanelRow GridPanelRow}.
     * It's not very useful to instantiate this class to use it alone. Most of cases you won't need to instantiate it, 
     * the {@link PMUI.grid.GridPanelRow GridPanelRow} creates its own cells objects automatically.
     * {@link PMUI.grid.GridPanelRow GridPanelRow}.
     * @extends {PMUI.core.Element}
     *
     *  Usage example:
     *
     *      @example
     *      var cell = new PMUI.grid.GridPanelCell({
     *          content: "This is the cell content"
     *      });
     *
     *      document.body.appendChild(cell.getHTML());
     *
     * @constructor
     * Creates a new instance of the GridPanelCell class.
     * @param {Object} settings A JSON object with the config options.
     *
     * @cfg {String|Number|PMUI.core.Element|null} [content=null] The content to be displayed in the cell, in case to 
     * be null the cells has no content.
     * @cfg {Number} [columnIndex=null] The column index the cell belongs to.
     * @cfg {PMUI.grid.GridPanelRow} [row=null] The row the cell belongs to.
     * @cfg {PMUI.grid.GridPanelColumn} [column=null] The column the cell belongs to.
     * @cfg {String} [alignment="center"] A string that specifies the alignment for the cell content. The accepted 
     * values are: "center", "left", "right".
     * @cfg {Boolean} [disabled=false] If the cell will be disabled or not.
     */
    var GridPanelCell = function (settings) {
        GridPanelCell.superclass.call(this, jQuery.extend(settings, {elementTag: "td"}));   
        /**
         * The content for the cell.
         * @type {PMUI.core.Element|String|Number}
         */
        this.content = null;
        /**
         * The position index for the cell, relative to its parent row.
         * @type {Number}
         */
        this.columnIndex = null;
        /**
         * The row the cell belongs to.
         * @type {PMUI.grid.GridPanelRow}
         */
        this.row = null;
        /**
         * The column the cell appertains to.
         * @type {PMUI.grid.GridPanelColumn}
         */
        this.column = null;
        /**
         * The alignment to be applied to the cell content.
         * @type {String}
         */
        this.alignment = null;
        this.contentSpan = null;
        /**
         * If the cell content is disabled or not.
         * @type {Boolean}
         */
        this.disabled = null;
        GridPanelCell.prototype.init.call(this, settings);
    };
    PMUI.inheritFrom("PMUI.core.Element", GridPanelCell);
    /**
     * The object's type.
     * @type {String}
     */
    GridPanelCell.prototype.type = "GridPanelCell";
    /**
     * The class family.
     * @type {String}
     */
    GridPanelCell.prototype.family = "grid"; 
    /**
     * Initializes the object.
     * @param  {Object} settings A JSON object with the config options.
     * @private
     */
    GridPanelCell.prototype.init = function (settings) {
        var defaults = {
            content : null,
            columnIndex : null,
            row : null,
            column : null,
            alignment: "center",
            disabled: false
        };
        jQuery.extend(true, defaults, settings);

        this.row = defaults.row;
        this.column = defaults.column;
        this.eventsDefined = false;

        if(defaults.disabled) {
            this.disable();
        } else {
            this.enable();
        }

        this.setAlignment(defaults.alignment)
            .setContent(defaults.content)
            .setColumnIndex(defaults.columnIndex);
    };
    /**
     * Sets the alignment for the content in the cell.
     * @param {String} alignment The accepted values are: "center", "left", "right".
     * @chainable
     */
    GridPanelCell.prototype.setAlignment = function(alignment) {
        if(alignment === 'center' || alignment === 'left' || alignment === 'right') {
            this.alignment = alignment;
            if(this.html && this.contentSpan) {
                this.contentSpan.style.textAlign = this.alignment;
                this.style.addProperties({'text-align':this.alignment});
            }
        } else {
            throw new Error('setAlignment(): This method only accepts one of the following values: ' 
                + '"canter", "left", "right"');
        }
        return this;
    };
    /**
     * Returns the cell's content alignment.
     * @return {String}
     */
    GridPanelCell.prototype.getAlignment = function() {
        return this.alignment;
    };
    /**
     * Sets the content for the cell.
     * @param {String|Number|PMUI.core.Element|null} content The content to be displayed in the cell, in case to
     * be null the cells has no content.
     * @chainable
     */
    GridPanelCell.prototype.setContent = function (content) {
        var type, visibleContent;   

        this.content = content;

        if(content instanceof PMUI.core.Element) {
            visibleContent = content.getHTML();
        } else if(typeof content === 'number' || typeof content === 'string') {
            visibleContent = document.createTextNode(content);
        } else if(content === null || content === undefined) {
            visibleContent = undefined;
        } else {
            throw new Error("setContent(): invalid parameter, it should be a Number, String or an instance of PMUI.core.Element.");
        }

        if(this.html) {
            jQuery(this.contentSpan).empty();
            if(visibleContent) {
                this.contentSpan.appendChild(visibleContent);
            } else {
                this.contentSpan.innerHTML = '&nbsp;';
            }
            if(this.disabled) {
                this.disable();
            } else {
                this.enable();
            }
        }

        return this;
    };
    /**
     * Returns the column the cell belongs to.
     * @return {PMUI.grid.GridPanelColumn}
     */
    GridPanelCell.prototype.getColumn = function() {
        return this.column;
    };
    /**
     * Returns the row the cell belongs to.
     * @return {PMUI.grid.GridPanelRow}
     */
    GridPanelCell.prototype.getRow = function() {
        return this.row;
    };
    /**
     * Returns the cell's content.
     * @return {String|Number|PMUI.core.Element|null}
     */
    GridPanelCell.prototype.getContent = function () {
        return this.content;
    };
    /**
     * Sets the index of the column the cell belongs to.
     * @param {Number} columnIndex
     */
    GridPanelCell.prototype.setColumnIndex = function (columnIndex) {
        this.columnIndex = columnIndex
        return this;
    };
    /**
     * Returns the index of the column the cell belongs to.
     * @return {Number}
     */
    GridPanelCell.prototype.getColumnIndex = function () {
        return this.columnIndex;
    };
    /**
     * Defines the events for the object.
     * @chainable
     */
    GridPanelCell.prototype.defineEvents = function() {
        this.eventsDefined = true;
        if(this.content instanceof PMUI.core.Element) {
            this.content.defineEvents();
        }
        return this;
    };
    /**
     * Creates the object's html.
     * @return {HTMLElement}
     */
    GridPanelCell.prototype.createHTML = function () {
        var type, contentSpan;

        if (this.html) {
            return this.html;
        }

        GridPanelCell.superclass.prototype.createHTML.call(this);   
        contentSpan = PMUI.createHTMLElement('span');
        contentSpan.className = 'pmui-gridpanelcell-content';
        this.contentSpan = contentSpan;
        //this.contentSpan.setAttribute('title',typeof this.content === 'string' ? this.content : '');
        this.html.appendChild(this.contentSpan);
        this.setAlignment(this.alignment)
            .setContent(this.content)
            .setWidth(this.width)
            .defineEvents();
        return this.html;       
    };

    GridPanelCell.prototype.setWidth = function(width) {
        if(typeof width === 'number') {
            this.width = width;
        } else if(/^\d+(\.\d+)?px$/.test(width)) {
            this.width = parseInt(width, 10);
        } else if(/^\d+(\.\d+)?%$/.test(width)) {
            this.width = width;
        } else if(/^\d+(\.\d+)?em$/.test(width)) {
            this.width = width;
        } else if(width === 'auto') {
            this.width = width;
        } else {
            throw new Error('setWidth: width param is not a number');
        }

        if(this.html){
            if (typeof this.width === 'string') {
                this.contentSpan.style.width = 'auto';
            } else {
                this.contentSpan.style.width = this.width + 'px';
            }           
        }
        return this;
    };

    GridPanelCell.prototype.applyStyle = function() {
        if(this.html) {
            this.style.applyStyle();

            this.style.addProperties({
                display: this.visible ? this.display: "none",
                position: this.positionMode,
                left: this.x,
                top: this.y,
                width: (typeof this.width === 'string')? this.width : 'auto',
                height: this.height,
                zIndex: this.zOrder
            });
        }
        return this;
    };
    /**
     * Enables the cell's content.
     * @chainable
     */
    GridPanelCell.prototype.enable = function() {
        this.disabled = false;
        if(this.content && this.content.enable) {
            this.content.enable();
        }
        return this;
    };
    /**
     * Disables the cell's content.
     * @chainable
     */
    GridPanelCell.prototype.disable = function() {
        this.disabled = true;
        if(this.content && this.content.disable) {
            this.content.disable();
        }
        return this;
    };
    PMUI.extendNamespace("PMUI.grid.GridPanelCell", GridPanelCell);

    if(typeof exports !== 'undefined'){
        module.exports = GridPanelCell;
    }

}());
(function() {
    var ControlGrid = function(settings) {
        ControlGrid.superclass.call(this, jQuery.extend(true, settings, {
            factory: {
                products: {
                    "GridControlRow": PMUI.grid.GridControlRow
                },
                defaultProduct: "GridControlRow"
            }
        }));
        this.columnsFactory = new PMUI.util.Factory();
        this.dependencies = {};
        ControlGrid.prototype.init.call(this, settings);
    };

    PMUI.inheritFrom('PMUI.grid.GridPanel', ControlGrid);

    ControlGrid.prototype.init = function(settings) {
        var defaults = {
            allowedCells: {
                'text': PMUI.grid.TextControlColumn,
                'dropdownlist': PMUI.grid.DropDownListControlColumn,
                'checkboxgroup': PMUI.grid.CheckBoxGroupColumn,
                'number' : PMUI.grid.NumberColumn
            }
        };

        jQuery.extend(true, defaults, settings);

        this.setAllowedCells(defaults.allowedCells)
            .setColumns(defaults.columns);
    };

    ControlGrid.prototype.setAllowedCells = function(allowedCells) {
        this.columnsFactory.clearProducts()
            .setProducts(allowedCells);
        return this;
    };

    ControlGrid.prototype.getColumn = function(column) {
        var the_column = null;
        if(typeof column === 'string') {
            the_column = this.columns.find('name', column);
        } else if(typeof column === 'number')  {
            the_column = this.columns.get(column);
        }
        return the_column;
    };

    ControlGrid.prototype.setColumns = function(columns) {
        if(this.columnsFactory) {
            ControlGrid.superclass.prototype.setColumns.call(this, columns);
        }
        return this;
    };

    ControlGrid.prototype.removeColumn = function(column) {
        ControlGrid.superclass.prototype.removeColumn.call(this, column);
        return this.updateDependencies();
    };

    ControlGrid.prototype.updateDependencies = function() {
        var dependencies = {}, i, j, columns = this.columns.asArray(), dependent, dependents;

        for(i = 0; i < columns.length; i += 1) {
            dependents = columns[i].dependentCols;
            for(j = 0; j < dependents.length; j += 1) {
                if(this.getColumn(dependents[j])) {
                    if(!dependencies[dependents[j]]) {
                        dependencies[dependents[j]] = [];
                    }
                    dependencies[dependents[j]].push(columns[i]);
                }
            }
        }
        this.dependencies = dependencies;
        return this;
    };

    ControlGrid.prototype.addColumn = function (column) {
        var newColumn;
        if(typeof column === 'object') {
            column.grid = this;
            newColumn = this.columnsFactory.make(column);
        } else {
            throw new Error('addColumn(): invalid column to add');
        }

        if(newColumn) {
            this.columns.insert(newColumn);
            if (this.dom.thead) {
                this.dom.thead.appendChild(newColumn.getHTML());
            }
        }
        this.updateDependencies();
        return this;  
    };

    /**
     * Disables one or all the columns.
     * @param {String|Number|PMUI.grid.GridPanelColumn} [column] A criteria to select the column that will be 
     * disabled. It supports the following data types:
     *
     * - String, in this case the parameter is used as the column's id.
     * - Number, in this case the parameter is used as the column's index.
     * - An instance of {@link PMUI.grid.GridPanelColumn GridPanelColumn}, in this case the supplied object must be 
     * a child of the grid.
     * - [no value], since this parameter is optional, you can skip it. In this case all the columns in the grid will 
     * be disabled.
     * @chainable
     */
    ControlGrid.prototype.disableColumn = function(column) {
        var i, targetColumn = this.getColumns(column);
        for(i = 0; i < targetColumn.length; i += 1) {
            targetColumn[i].disable();
        }
        return this;
    };
    /**
     * [disableColumn description]
     * @param  {[type]} column [description]
     * @return {[type]}        [description]
     */
    ControlGrid.prototype.enableColumn = function(column) {
        var i, targetColumn = this.getColumns(column);
        for(i = 0; i < targetColumn.length; i += 1) {
            targetColumn[i].enable();
        }
        return this;
    };

    ControlGrid.prototype.isValid = function() {
        var i, rows, valid = true; 

        rows = this.getItems();
        for(i = 0; i < rows.length; i++) {
            valid = valid && rows[i].isValid();
            if(!valid) {
                return valid;
            }
        }
        return valid;
    };

    ControlGrid.prototype.fireDependency = function() {
        //  this.updateDependencies
    };

    PMUI.extendNamespace('PMUI.grid.ControlGrid', ControlGrid);
}());
(function() {
    var GridControlCell = function(settings) {
        this.controls = [];
        GridControlCell.superclass.call(this, settings);
        /**
         * @property {Boolean} [validAtChange=true] 
         * If the validation must be executed every time the field's value changes.
         */
        this.validAtChange = null;
        this.value = null;
        this.onChange = null;
        this.validAtChange = null;
        this.disable = null;
        GridControlCell.prototype.init.call(this, settings);
    };

    PMUI.inheritFrom('PMUI.grid.GridPanelCell', GridControlCell);

    GridControlCell.prototype.init = function(settings) {
        var defaults = {
            onChange: null,
            validAtChange: true,
            disabled: false,
            value: ""
        };

        jQuery.extend(true, defaults, settings);
        this.initControls();

        this.setOnChangeHandler(defaults.onChange)
            .setValidAtChange(defaults.validAtChange)
            .setValue(defaults.value);

        if(defaults.disabled) {
            this.disable();
        } else {
            this.enable();
        }
    };
    /**
     * [setValueToControls description]
     * @param {[type]} value [description]
     */
    GridControlCell.prototype.setValueToControls = function(value) {
        var i;
        for(i = 0; i < this.controls.length; i += 1) {
            this.controls[i].setValue(value);
        }
        return this;
    };
    /**
     * [setValue description]
     * @param {[type]} value [description]
     */
    GridControlCell.prototype.setValue = function(value) {
        if(typeof value === 'number') {
            value = value.toString();
        }
        if(typeof value === 'string') {
            this.value = value;
        } else {
            throw new Error("The setValue() method only accepts string values!");
        }
        this.setValueToControls(value);
        return this;
    };
    /**
     * Turns on/off the validation when the field's value changes.
     * @param {Boolean} validAtChange
     * @chainable
     */
    GridControlCell.prototype.setValidAtChange = function(validAtChange) {
        this.validAtChange = !!validAtChange;
        return this;
    };
    /**
     * Sets the callback function to be called when the field's value changes.
     * @param {Function} handler
     * @chainable
     */
    GridControlCell.prototype.setOnChangeHandler = function(handler) {
        if(typeof handler === 'function' || handler === null) {
            this.onChange = handler;
        }

        return this;
    };

    GridControlCell.prototype.initControls = function() {
        return this;
    };

    GridControlCell.prototype.setValuesToControls = function(values) {
        return this;
    };

    GridControlCell.prototype.setContent = function(content) {
        return this.setValuesToControls(content);
    };

    /**
     * Update the field's value property from the controls
     * @protected
     * @chainable
     */
    GridControlCell.prototype.updateContentFromControls = function() {
        var value = '', i;

        for(i = 0; i < this.controls.length; i++) {
            value += ' ' + this.controls[i].getValue();
        }

        this.value = this.content = value.substr(1);

        return this;
    };
    /**
     * The onChange event handler constructor.
     * @private
     * @return {Function} The handler.
     */
    GridControlCell.prototype.onChangeHandler = function() {
        var that = this;

        return function() {
            var previousValue = that.content;
            that.updateContentFromControls();
            if(that.validAtChange) {
                that.isValid();
            }

            if(typeof that.onChange === 'function') {
                that.onChange(that.getContent(), previousValue);
            }

            if(that.column) {
                that.column.onChangeHandler(that, that.content, previousValue);
            }
        };
    };
    GridControlCell.prototype.hideMessage = function() {
        this.style.removeClasses(['pmui-error']);
        this.html.removeAttribute("title");
        try {
            jQuery(this.html).tooltip("destroy");
        } catch(e) {}
        return this;
    };
    GridControlCell.prototype.showMessage = function(message) {
        this.style.addClasses(['pmui-error']);
        this.html.setAttribute("title", "error");
        jQuery(this.html).tooltip({
            content: message,
            tooltipClass: 'pmui-cell-tooltip-error',
            position: {
                my: "left",
                at: "left bottom"
            }
        });
        jQuery(this.html).tooltip('open');
        return this;
    };
    /**
     * [evalRequired description]
     * @return {[type]} [description]
     */
    GridControlCell.prototype.evalRequired = function() {
        var valid = true, column = this.column;
        if(column) {
            if(column.required && this.value === "") {
                valid = false;
            }
        }
        if(valid) {
            this.hideMessage();
        } else {
            this.showMessage(column.requiredMessage);
        }
        return valid;
    };
    /**
     * [isValid description]
     * @return {Boolean} [description]
     */
    GridControlCell.prototype.isValid = function() {
        var valid = true, validator, validators;

        valid = valid && this.evalRequired();
        if(!valid) {
            return valid;
        }
        if(this.column) {
            validators = this.column.validators;
            for(validator in validators) {
                if(validators.hasOwnProperty(validator)) {
                    validators[validator].setParent(this);
                    valid = validators[validator].isValid();
                    if(!valid) {
                        return valid;
                    }
                }
            }
        }
        return valid;
    };
    /**
     * Attach the event listeners to the HTML element
     * @private
     * @chainable
     */
    GridControlCell.prototype.defineEvents = function() {
        var i;
        if (this.eventsDefined) {
            return this;
        };
        for(i = 0; i < this.controls.length; i++) {
            this.controls[i].setOnChangeHandler(this.onChangeHandler())
                .defineEvents();
        }
        this.eventsDefined = true;

        return this;
    };
    /**
     * @inheritdoc
     */
    GridControlCell.prototype.enable = function() {
        var i;
        for(i = 0; i < this.controls.length; i +=1) {
            this.controls[i].disable(false);
        }
        this.disabled = false;
        return this;
    };
    /**
     * @inheritdoc
     */
    GridControlCell.prototype.disable = function() {
        var i;
        for(i = 0; i < this.controls.length; i +=1) {
            this.controls[i].disable(true);
        }
        this.disabled = true;
        return this;
    };
    /**
     * [createHTML description]
     * @return {[type]} [description]
     */
    GridControlCell.prototype.createHTML = function() {
        var i;
        if(this.html) {
            return this.html;
        }
        GridControlCell.superclass.prototype.createHTML.call(this);
        for(i = 0; i < this.controls.length; i++) {
            this.contentSpan.appendChild(this.controls[i].getHTML());
        }

        this.defineEvents();
        return this.html;
    };

    PMUI.extendNamespace('PMUI.grid.GridControlCell', GridControlCell);
}());
(function() {
    var NumberCell = function(settings) {
        NumberCell.superclass.call(this, settings);

        NumberCell.prototype.init.call(this, settings);
    };

    PMUI.inheritFrom('PMUI.grid.GridControlCell', NumberCell);

    NumberCell.prototype.init = function(settings) {
        var defaults = {
            placeholder: "",
            maxLength: 524288,
            mask :["###,###.###"],
            onKeyUp : null,
            content : ''
        };

        jQuery.extend(true, defaults, settings);
        this.setPlaceholder(defaults.placeholder)
            .setMaxLength(defaults.maxLength)
            .setOnKeyUpHandler(defaults.onKeyUp)
            .setContent(defaults.content)
            .setMask(defaults.mask);
    };

    NumberCell.prototype.setOnKeyUpHandler = function(handler) {
        this.controls[0].setOnKeyUpHandler(handler);
        return this;
    };

    NumberCell.prototype.setPlaceholder = function(placeholder) {
        this.controls[0].setPlaceholder(placeholder);
        return this;
    };

    NumberCell.prototype.setMask = function (mask) {
        this.controls[0].setMask(mask);
        return this;
    };

    NumberCell.prototype.setMaxLength = function (maxLength){
        this.controls[0].setMaxLength(maxLength);
        return this;
    };

    NumberCell.prototype.setReadOnly = function (value) {
        this.controls[0].setReadOnly(value);
        return this;
    };

    NumberCell.prototype.initControls = function() {
        this.controls[0] = new PMUI.control.NumberControl();
        return this;
    };

    NumberCell.prototype.setContent = function(content) {
        return this.setValuesToControls(content);
    };

    NumberCell.prototype.setValuesToControls = function(values) {
        if(this.controls.length){
            this.controls[0].setValue(values); 
            this.content = this.controls[0].getValue();   
        }
        return this;
    };

    NumberCell.prototype.getPlaceholder = function() {
        return this.controls[0].getPlaceholder();
    };
    /**
     * [isReadOnly description]
     * @return {Boolean} [description]
     */
    NumberCell.prototype.isReadOnly = function() {
        return this.controls[0].isReadOnly();
    };
    /**
     * Switches on/off the value trimming for the field's value when it loses focus.
     * @param {Boolean} trimOnBlur Send true for switch on or false to switch off.
     * @chainable
     */
    PMUI.extendNamespace('PMUI.grid.NumberCell', NumberCell);

    if (typeof exports!== 'undefined'){
        module.exports = NumberCell;
    }
}());
(function() {
    var GridTextControlCell = function(settings) {
        GridTextControlCell.superclass.call(this, settings);
        GridTextControlCell.prototype.init.call(this, settings);
    };

    PMUI.inheritFrom('PMUI.grid.GridControlCell', GridTextControlCell);

    GridTextControlCell.prototype.init = function(settings) {
        var defaults = {
            placeholder: "",
            maxLength: 0,
            trimOnBlur: true,
            readOnly : false
        };

        jQuery.extend(true, defaults, settings);

        this.setPlaceholder(defaults.placeholder)
            .setMaxLength(defaults.maxLength)
            .setTrimOnBlur(defaults.trimOnBlur)
            .setReadOnly(defaults.readOnly);
    };

    GridTextControlCell.prototype.initControls = function() {
        this.controls[0] = new PMUI.control.TextControl();
        return this;
    };

    GridTextControlCell.prototype.setValuesToControls = function(values) {
        return this;
    };

    GridTextControlCell.prototype.setContent = function(content) {
        return this.setValuesToControls(content);
    };
    /**
     * Sets the placeholder for the control. Note that this feature is only supported 
     * by browsers which support the "placeholder" for input elements.
     * @param {String} placeholder
     * @chainable
     */
    GridTextControlCell.prototype.setReadOnly = function(readonly) {
        this.controls[0].setReadOnly(readonly);
        return this;
    };
    /**
     * [setPlaceholder description]
     * @param {[type]} placeholder [description]
     */
    GridTextControlCell.prototype.setPlaceholder = function(placeholder) {
        this.controls[0].setPlaceholder(placeholder);
        return this;
    };
    /**
     * [getPlaceholder description]
     * @return {[type]} [description]
     */
    GridTextControlCell.prototype.getPlaceholder = function() {
        return this.controls[0].getPlaceholder();
    };
    /**
     * [setMaxLength description]
     * @param {[type]} maxLength [description]
     */
    GridTextControlCell.prototype.setMaxLength = function(maxLength) {
        this.controls[0].setMaxLength(maxLength);
        return this;
    };
    /**
     * [getMaxLength description]
     * @return {[type]} [description]
     */
    GridTextControlCell.prototype.getMaxLength = function() {
        return this.controls[0].getMaxLength();
    };
    /**
     * [isReadOnly description]
     * @return {Boolean} [description]
     */
    GridTextControlCell.prototype.isReadOnly = function() {
        return this.controls[0].isReadOnly();
    };
    /**
     * Switches on/off the value trimming for the field's value when it loses focus.
     * @param {Boolean} trimOnBlur Send true for switch on or false to switch off.
     * @chainable
     */
    GridTextControlCell.prototype.setTrimOnBlur = function(trimOnBlur) {
        this.trimOnBlur = !! trimOnBlur;
        return this;
    };

    PMUI.extendNamespace('PMUI.grid.GridTextControlCell', GridTextControlCell);

    if (typeof exports!== 'undefined'){
        module.exports = GridTextControlCell;
    }
}());
(function() {
    var DropDownListControlCell = function(settings) {
        DropDownListControlCell.superclass.call(this, settings);
        DropDownListControlCell.prototype.init.call(this, settings);
    };

    PMUI.inheritFrom('PMUI.grid.GridControlCell', DropDownListControlCell);

    DropDownListControlCell.prototype.init = function(settings) {
        var defaults = {
            options: [],
            value: null
        };

        jQuery.extend(true, defaults, settings);

        this.setOptions(defaults.options);

        if(defaults.value !== null) {
            this.setValue(defaults.value);
        }
    };

    DropDownListControlCell.prototype.initControls = function() {
        this.controls[0] = new PMUI.control.DropDownListControl();
        return this;
    };

    DropDownListControlCell.prototype.clearOptions = function() {
        this.controls[0].clearOptions();
        this.content = this.controls[0].getValue();
        return this;
    };

    DropDownListControlCell.prototype.enableDisableOption = function(disabled, option, group) {
        this.controls[0].enableDisableOption(disabled, option, group);
        return this;
    };

    DropDownListControlCell.prototype.disableOption = function(option, group) {
        return this.controls[0].enableDisableOption(true, option, group);
    };

    DropDownListControlCell.prototype.enableOption = function(option, group) {
        return this.controls[0].enableDisableOption(false, option, group);
    };

    DropDownListControlCell.prototype.removeOption = function(option, group) {
        this.controls[0].removeOption(option,group);
        return this;
    };

    DropDownListControlCell.prototype.addOptionGroup = function(optionGroup) {
        this.controls[0].addOptionGroup(optiongroup);
        return this;
    };

    DropDownListControlCell.prototype.addOption = function(option, group) {
       this.controls[0].addOption(option, group);
        return this;
    };

    DropDownListControlCell.prototype.getSelectedLabel = function() {
        return this.controls[0].getSelectedLabel();
    };

    DropDownListControlCell.prototype.valueExistsInOptions = function(value) {
        return this.controls[0].valueExistsInOptions(value);
    };

    DropDownListControlCell.prototype.getFirstAvailableOption = function() {
        return this.controls[0].getFirstAvailableOption(); 
    }

    DropDownListControlCell.prototype.setOptions = function(options) {
        this.controls[0].setOptions(options);
        return this;
    };
  
    DropDownListControlCell.prototype.getOptions = function(includeGroups) {
        return this.controls[0].getOptions(includeGroups);
    };

    DropDownListControlCell.prototype.setValuesToControls = function(values) {
        if(this.controls.length){
            this.controls[0].setValue(values); 
            this.content = this.controls[0].getValue();   
        }
        return this;
    };

    DropDownListControlCell.prototype.getContent = function (){
        return this.controls[0].getValue();
    };

    PMUI.extendNamespace('PMUI.grid.DropDownListControlCell', DropDownListControlCell);
}());
(function() {
    var CheckBoxGroupControlCell = function(settings) {
        CheckBoxGroupControlCell.superclass.call(this, settings);
              /**
         * @property {String} controlPositioning The direction for the options to be added in the field.
         * @readonly
         */
        this.controlPositioning = null;
        /**
         * @property {Number} maxDirectionOptions The max number of options that can be in the current options 
         * direction.
         * @readonly
         */
        this.maxDirectionOptions = null;
        /**
         * The status if the controls to be saved when the field is switch between enabled/disabled.
         * @type {Object}
         * @private
         */
        this.auxControlStates = {};
        CheckBoxGroupControlCell.prototype.init.call(this, settings);
    };

    PMUI.inheritFrom('PMUI.grid.GridControlCell', CheckBoxGroupControlCell);

    CheckBoxGroupControlCell.prototype.init = function(settings) {
        var defaults = {
            options:[], 
            controlPositioning: "vertical",
            maxDirectionOptions: 1,
            value: '[]'
        };

        jQuery.extend(true, defaults, settings);
        this.setOptions(defaults.options);
        this.setMaxDirectionOptions(defaults.maxDirectionOptions);
        this.setControlPositioning(defaults.controlPositioning);
    };


    CheckBoxGroupControlCell.prototype.enable = function() {
        var key, i, controlsLength = this.controls.length, controls = this.controls;
        CheckBoxGroupControlCell.superclass.prototype.enable.call(this);
        this.disabled = false;
        for(i = 0; i < controlsLength; i++) {
                controls[i].disable(this.auxControlStates[controls[i].id]);
        }
        return this;
    };

    CheckBoxGroupControlCell.prototype.setMaxDirectionOptions = function(max) {
        if(typeof max === 'number') {
            this.maxDirectionOptions = Math.floor(max);
            if(this.html) {
                this.setControlPositioning(this.controlPositioning);
            }
        } else {
            throw new Error("setMaxDirectionOptions(): it only accepts number values.");
        }

        return this;
    };

    CheckBoxGroupControlCell.prototype.setControlPositioning = function(positioning) {
        var errorMessage = "The setControlPositioning() method only accepts \"horizontal\" or \"vertical\" as value.",
            table, tbody, cell, row, i, column, rowIndex;
        if(typeof positioning === 'string') {
            if(!(positioning === 'horizontal' || positioning === 'vertical')) {
                return this;
            }
            this.controlPositioning = positioning;
            if(this.html && this.controls) {
                for(i = 0; i < this.controls.length; i++) {
                    jQuery(this.controls[i].getHTML()).detach();
                }
                $(this.contentSpan).empty();
                table = PMUI.createHTMLElement("table");
                table.className = 'pmui-field-control-table';
                tbody = PMUI.createHTMLElement("tbody");
                if(positioning === 'horizontal') {
                    row = PMUI.createHTMLElement("tr");
                    for(i = 0; i < this.controls.length; i++) {
                        cell = PMUI.createHTMLElement('td');
                        this.controls[i].getHTML();
                        this.controls[i].control.tabIndex = i;
                        cell.appendChild(this.controls[i].getHTML());
                        row.appendChild(cell);
                        if(this.maxDirectionOptions > 0 && (i + 1) % this.maxDirectionOptions === 0) {
                            tbody.appendChild(row);
                            row = PMUI.createHTMLElement("tr");
                        }
                    }
                    cell.style.textAlign = "start";
                    tbody.appendChild(row);
                } else {
                    column = 0;
                    for(i = 0; i < this.controls.length; i++) {
                        cell = PMUI.createHTMLElement('td');
                        this.controls[i].getHTML();
                        this.controls[i].control.tabIndex = i;
                        cell.appendChild(this.controls[i].getHTML());
                        rowIndex = this.maxDirectionOptions === 0 ? i : i % this.maxDirectionOptions;

                        row = jQuery(tbody).find('tr').eq(rowIndex).get(0);
                        if(!row) {
                            row = PMUI.createHTMLElement('tr');
                            tbody.appendChild(row);
                        }
                        cell.style.textAlign = "start";
                        row.appendChild(cell);
                    }
                }
                table.appendChild(tbody);
                this.contentSpan.appendChild(table);
            }
        }

        return this;
    };

    CheckBoxGroupControlCell.prototype.setOptions = function (options){
        var i = 0;

        this.content = [];
        this.content = JSON.stringify(this.content);

        if(jQuery.isArray(options)){
            for(i = 0; i < options.length; i++) {
                this.addOption(options[i]);          
            }
        } else {
            throw new Error("setOptions(): the supplied argument must be an array.");
        } 
        
        return this;
    };

    CheckBoxGroupControlCell.prototype.addOption = function (option) {
       var newOption;

        newOption = new PMUI.control.SelectableControl(jQuery.extend(true, option, {
            mode: 'checkbox', 
            name: this.controls.length + 1, 
            selected: option.selected
        }));
        
        if(newOption.isSelected()) {
            this.content = JSON.parse(this.content);
            this.content.push(newOption.getValue());
            this.content = JSON.stringify(this.content);
        }
        if(this.eventsDefined) {
            newOption.setOnChangeHandler(this.onChangeHandler()).getHTML();
            newOption.defineEvents();
        }
        this.auxControlStates[newOption.id] = newOption.disabled;
        this.controls.push(newOption);
        this.setControlPositioning(this.controlPositioning);
 
        return this;
    };

    CheckBoxGroupControlCell.prototype.removeOption = function(item) {
        var itemToRemove, i;
        if (item instanceof PMUI.control.SelectableControl) {
            for(i = 0; i < this.controls.length; i++){
                if(this.controls[i] === item){
                    itemToRemove = i;
                    break;
                }
            }
        } else if (typeof item === 'string') {
            for(i = 0; i < this.controls.length; i++){
                if(this.controls[i].id === item){
                    itemToRemove = this.controls[i];
                    break;
                }
            }
        } else {
            itemToRemove = item;
        }
        if (typeof itemToRemove === 'number') {
            itemToRemove = this.controls[itemToRemove];
            delete this.auxControlStates[itemToRemove.id];
            jQuery(itemToRemove.html).detach();
            this.controls.splice(itemToRemove, 1);
            this.setControlPositioning(this.controlPositioning);
        }
        return this;
    };             

    CheckBoxGroupControlCell.prototype.clearOptions = function() {
        while(this.controls.length) {
            this.removeOption(0);
        }

        return this;
    };

    CheckBoxGroupControlCell.prototype.updateContentFromControls = function() {
        var value = [], i;

        for(i = 0; i < this.controls.length; i++) {
            if(this.controls[i].isSelected()) {
                value.push(this.controls[i].getValue());
            }
        }

        this.content = JSON.stringify(value);
        return this;
    };

    CheckBoxGroupControlCell.prototype.evalRequired = function() {
        var valid = true, value;
        value = JSON.parse(this.getContent());
        if(this.required && value.length === 0) {
            this.showMessage(this.requiredMessage, "error");
            valid = false;
        } else {
            this.hideMessage();
        }

        return valid;
    };

 
    CheckBoxGroupControlCell.prototype.disableOption = function (value){

        var index=-1, i;
        if(this.html){
            if(typeof value === 'string'){

                for(i=0 ;i<this.controls.length; i++){
                    if(this.controls[i].label===value){
                        index=i;
                    }
                }
            }else{
                    index=value;
            }
            if(index===-1){
                throw new Error ('the value send is not part of group CheckBox');
            }else{
                this.controls[index].disable(true);
            }
        }
        return this;
    };
   
    CheckBoxGroupControlCell.prototype.enableOption = function (value){
        var index=-1, i;
        if(this.html){
            if(typeof value === 'string'){

                for(i=0 ;i<this.controls.length; i++){
                    if(this.controls[i].label===value){
                        index=i;
                    }
                }
            }else{
                    index=value;
            }
            if(index===-1){
                throw new Error ('the value send is not part from group CheckBox');
            }else{
     
                    this.controls[index].disable(false);
            }
        }
        return this;
    };

    CheckBoxGroupControlCell.prototype.createHTML = function () {
        if(this.html){
            return this.html;
        }

        CheckBoxGroupControlCell.superclass.prototype.createHTML.call(this)

        this.setControlPositioning(this.controlPositioning);

        return this.html;
    };

    CheckBoxGroupControlCell.prototype.setValuesToControls = function(value) {
        var i, j, values, controls, controlsLength;
        if(value){
            try {
                 values = JSON.parse(value);
            } catch(e) {
                values = '[]';
            }

            controlsLength = (controls = this.controls.slice(0)).length;
            for(i = 0; i < controlsLength; i += 1){
                controls[i].deselect();
            }
            for(i = 0; i < values.length; i++) {
                for(j = 0; j < controlsLength; j++) {
                    if(controls[j].getValue() === values[i]) {
                        controls[j].select();
                        controls.splice(j, 1);
                        controlsLength -= 1;
                        j -= 1;
                    }   
                }
            }
            this.updateContentFromControls();
        }

        return this;
    };

    PMUI.extendNamespace('PMUI.grid.CheckBoxGroupControlCell', CheckBoxGroupControlCell);
}());
(function() {
    var GridControlColumn = function(settings) {
        this.cells = new PMUI.util.ArrayList();
        GridControlColumn.superclass.call(this, jQuery.extend(settings, {elementTag: "th"}));
        /**
         * The name for the column.
         * @type {String}
         */
        this.name = null;
        /**
         * The text to be shown in the column header.
         * @type {String}
         */
        this.title = null;
        /**
         * The grid the column belongs to.
         * @type {PMUI.grid.GridPanel}
         */
        this.grid = null;
        /**
         * The HTML element in which the cell content will be appended.
         * @type {HTMLElement}
         * @private
         */
        this.contentTitle = null;
        /**
         * [dependentCols description]
         * @type {Array}
         */
        this.dependentCols = [];
        /**
         * [onChange description]
         * @type {[type]}
         */
        this.onChange = null;
        /**
         * [dependencyHandler description]
         * @type {[type]}
         */
        this.dependencyHandler = null;
        /**
         * [disabled description]
         * @type {[type]}
         */
        this.disabled = null;
        /**
         * [validatorFactory description]
         * @type {[type]}
         */
        this.validatorFactory = null;
        /**
         * [validators description]
         * @type {Object}
         */
        this.validators = {};
        /**
         * [required description]
         * @type {[type]}
         */
        this.required = null;
        /**
         * [requiredMessage description]
         * @type {[type]}
         */
        this.requiredMessage = null;
        GridControlColumn.prototype.init.call(this, settings);
    };

    PMUI.inheritFrom('PMUI.core.Element', GridControlColumn);

    GridControlColumn.prototype.CELL = null;

    GridControlColumn.prototype.init = function(settings) {
        var defaults = {
            name: this.id,
            title: '[untitled]',
            grid: null,
            cells: [],
            dependentCols: [],
            onChange: null,
            dependencyHandler: null,
            disabled: false,
            validators: [], 
            required: false,
            requiredMessage: 'This cell is required.',
            validatorFactory: new PMUI.form.ValidatorFactory()
        };

        jQuery.extend(true, defaults, settings);

        this.setName(defaults.name)
            .setTitle(defaults.title)
            .setGrid(defaults.grid)
            .setRequired(defaults.required)
            //.setCells(defaults.cells)
            .setDependentCols(defaults.dependentCols)
            .setOnChangeHandler(defaults.onChange)
            .setDependencyHandler(defaults.dependencyHandler)
            .setValidatorFactory(defaults.validatorFactory)
            .setValidators(defaults.validators)
            .setRequiredMessage(defaults.requiredMessage);

        if(defaults.disabled) {
            this.disable();
        } else {
            this.enable();
        }
    };

    GridControlColumn.prototype.setRequiredMessage = function(requiredMessage) {
        if(typeof requiredMessage !== 'string') {
            throw new Error('setRequiredMessage(): The parameter must be a strng.');
        }
        this.requiredMessage = requiredMessage;
        return this;
    };

    GridControlColumn.prototype.setRequired = function(required) {
        this.required = !!required;
        return this;
    };

    GridControlColumn.prototype.setValidatorFactory = function(factory) {
        if(factory instanceof PMUI.util.Factory) {
            this.validatorFactory = factory;
        } else {
            this.validatorFactory  = new PMUI.form.ValidatorFactory(factory);
        }
        return this;
    };

    GridControlColumn.prototype.addValidator = function(validator) {
        var newValidator;
        if(this.validatorFactory) {
            if(this.validatorFactory.isValidClass(validator) || this.validatorFactory.isValidName(validator.pmType)) {
                newValidator = this.validatorFactory.make(validator);
            } else {
                throw new Error('Invalid validator to add.');
            }
        }

        if(newValidator && newValidator instanceof PMUI.form.Validator) {
            this.validators[newValidator.type] = newValidator;
        }

        return this;
    };

    GridControlColumn.prototype.clearValidators = function() {
        var key;
        for(key in this.validators) {
            if(this.validators.hasOwnProperty(key)) {
                this.validators[key] = null;
                delete this.validators[key];   
            }
        }

        return this;
    };

    GridControlColumn.prototype.setValidators = function(validators) {
        var i = 0;
        if(jQuery.isArray(validators)) {
            this.clearValidators();
            for(i = 0; i < validators.length; i++) {
                this.addValidator(validators[i]);
            }
        }
        return this;
    };

    GridControlColumn.prototype.getCells = function() {
        return this.cells.asArray().slice(0);
    };

    GridControlColumn.prototype.setName = function(name) {
        if(typeof name !== 'string') {
            throw new Error("setName(): The parameter must be a string.");
        }
        this.name = name;
        return this;
    };

    GridControlColumn.prototype.setDependencyHandler = function(handler) {
        if(!(typeof handler === 'function' || handler === null)) {
            throw new Error("setDependencyHandler(): The parameter must be a function or null.");
        }
        this.dependencyHandler = handler;
        return this;
    }; 

    GridControlColumn.prototype.onChangeHandler = function(cell, newValue, oldValue) {
        var i, dependentCols = this.dependentCols, grid = this.grid, dependentCell, col, row;
        cell.row.updateData(cell);
        if(typeof this.onChange === 'function') {
            this.onChange(cell, newValue, oldValue);
        }
        if(grid) {
            for(i = 0; i < dependentCols.length; i += 1) {
                col = grid.getColumn(dependentCols[i]);
                if(!col) {
                    this.removeDependentCol(dependentCols[i]);
                    continue;
                }
                dependentCell = cell.row.getCell(col);
                col.fireDependencyHandler(dependentCell);
            }
        }
        return this;
    };

    GridControlColumn.prototype.fireDependencyHandler = function(cell) {
        var grid = this.grid, dependsOf, obj = {}, i, targetRow, targetCells, j;
        if(grid) {
            targetCells = cell ? [cell] : this.cells.asArray();
            for(j = 0; j < targetCells.length; j += 1) {
                cell = targetCells[j];
                targetRow = cell.row;
                dependsOf = grid.dependencies[this.name];
                for(i = 0; i < dependsOf.length; i++) {
                    obj[dependsOf[i].name] = targetRow.getCell(dependsOf[i]);
                }
                if(typeof this.dependencyHandler === 'function') {
                    this.dependencyHandler.call(cell, cell, obj);
                }
            }
        }
        return this;
    };

    GridControlColumn.prototype.enable = function() {
        var i, cells = this.getCells();
        for(i = 0; i < cells.length; i += 1) {
            cells[i].enable();
        }
        this.disabled = false;
        return this;
    };

    GridControlColumn.prototype.disable = function() {
        var i, cells = this.getCells();
        for(i = 0; i < cells.length; i += 1) {
            cells[i].disable();
        }
        this.disabled = true;
        return this;
    };

    GridControlColumn.prototype.setOnChangeHandler = function(handler) {
        if(!(typeof handler === 'function' || handler === null)) {
            throw new Error("setOnChangeHandler(): the parameter must be a function or null.");
        }
        this.onChange = handler;
        return this;
    };

    GridControlColumn.prototype.setDependentCols = function(dependentCols) {
        if(!jQuery.isArray(dependentCols)) {
            throw new Error("setDependentCols(): The parameter must be an array.");
        }
        this.dependentCols = dependentCols;
        if(this.grid) {
            this.grid.updateDependencies();
        }
        return this;
    };

    GridControlColumn.prototype.removeDependentCol = function(columnName) {
        var i;
        for(i = 0; this.dependentCols.length; i++) {
            if(this.dependentCols[i] === columnName) {
                this.dependentCols.splice(i, 1);
                i--;
            }
        }
        return this;
    };

    GridControlColumn.prototype.removeCell = function(cell){
        var cellToRemove;

        if(cell instanceof PMUI.grid.GridControlCell) {
            cellToRemove = cell;
        } else if(typeof cell === 'number') {
            cellToRemove = this.cells.get(cell);
        } else if(typeof cell === 'string') {
            cellToRemove = this.cells.find("id", cell);
        }

        if(cellToRemove) {
            cellToRemove.column = null;
            this.cells.remove(cellToRemove);
        }

        return this;
    };

    GridControlColumn.prototype.clearCells = function() {
        this.cells.clear();
        while(this.cells.getSize() > 0) {
            this.removeCell(0);
        }
        return this;
    };

    GridControlColumn.prototype.addCell = function(cell) {
        if(!(cell instanceof PMUI.grid.GridControlCell)) {
            throw new Error("addCell(): The parameter must be an instance of PMUI.grid.GridControlCell.");
        }
        this.cells.insert(cell);
        return this;
    };

    /*GridControlColumn.prototype.setCells = function(cells) {
        var cellsAux = [], i, cellToAdd;

        this.clearCells();

        if(cells instanceof PMUI.util.ArrayList){
            cellsAux = cells.items.asArray();
        } else if(jQuery.isArray(cells)){
            cellsAux = cells;
        } else {
            throw new Error('setCells(): the cells property should be instanceof of PMUI.util.ArrayList or an Array');
        }
        for (i = 0; i < cellsAux.length; i += 1){
            this.addCell(cellsAux[i]);
        }
        return this;
    };*/

    GridControlColumn.prototype.setTitle = function(title) {
        this.title = title;
        if(this.html){
            jQuery(this.contentTitle).empty();
            this.contentTitle.appendChild(document.createTextNode(title));
        }
        return this;
    };

    GridControlColumn.prototype.setGrid = function(grid) {
        if(!(grid instanceof PMUI.grid.ControlGrid)) {
            throw new Error ('setGrid(): the grid property should be instance of [PMUI.grid.GridControl]');
        }
        this.grid = grid;
        this.grid.updateDependencies();
        return this;
    };

    GridControlColumn.prototype.getCellDefaultCfg = function() {
        return {
            disabled: this.disabled
        };
    };

    GridControlColumn.prototype.createCell = function() {
        var cell;
        if(!this.CELL instanceof PMUI.grid.GridControlCell) {
            throw new Error("The cell associated to the " + this.type + " is not valid!");
        }
        cell = new this.CELL(this.getCellDefaultCfg());
        this.addCell(cell);

        return cell;
    };

    GridControlColumn.prototype.createHTML = function() {
        var contentTitle, sortIcon;
        if (this.html){
            return this.html;
        }
        GridControlColumn.superclass.prototype.createHTML.call(this);
        contentTitle = PMUI.createHTMLElement('span');
        contentTitle.className = 'pmui-gridpanelColumn-title';
        this.contentTitle = contentTitle;
        this.html.appendChild(contentTitle);
        this.setTitle(this.title)
            /*.setCells(this.cells.asArray().slice(0))*/;

        return this.html;
    };

    PMUI.extendNamespace('PMUI.grid.GridControlColumn', GridControlColumn);
}());
(function() {
    var NumberColumn = function(settings) {
        NumberColumn.superclass.call(this, settings);
            this.placeholder= null;
            this.maxLength= null;
            this.mask = null;
        NumberColumn.prototype.init.call(this, settings);
    };

    PMUI.inheritFrom('PMUI.grid.GridControlColumn', NumberColumn);

    NumberColumn.prototype.CELL = PMUI.grid.NumberCell;

    NumberColumn.prototype.init = function(settings) {
        var defaults = {
            placeholder: "",
            maxLength: 524288,
            mask :"##############.###############",
            content : '',
            readOnly : false
        };

        jQuery.extend(true, defaults, settings);

        this.setMaxLength(defaults.maxLength)
            .setPlaceHolder(defaults.placeholder)
            .setReadOnly(defaults.readOnly)
            .setMask(defaults.mask);
    };

    NumberColumn.prototype.setMaxLength = function(input) {
        this.maxLength = input
        return this;
    };

    NumberColumn.prototype.setPlaceHolder = function(input) {
        this.placeholder = input;
        return this;
    };

    NumberColumn.prototype.setReadOnly = function(input) {
        this.readOnly = input;
        return this;
    };

    NumberColumn.prototype.setMask = function (mask) {
        this.mask = mask;
        return this;
    };

    NumberColumn.prototype.getCellDefaultCfg = function() {
        return jQuery.extend(true, NumberColumn.superclass.prototype.getCellDefaultCfg.call(this), {
            maxLength: this.maxLength,
            mask : this.mask,
            placeholder : this.placeholder,
            readOnly: this.readOnly
        });
    };

    PMUI.extendNamespace('PMUI.grid.NumberColumn', NumberColumn);
}());
(function() {
    var TextControlColumn = function(settings) {
        TextControlColumn.superclass.call(this, settings);
        this.maxLength = null;
        this.placeholder = null;
        this.trimOnBlur = null;
        this.readOnly = null;
        TextControlColumn.prototype.init.call(this, settings);
    };

    PMUI.inheritFrom('PMUI.grid.GridControlColumn', TextControlColumn);

    TextControlColumn.prototype.CELL = PMUI.grid.GridTextControlCell;

    TextControlColumn.prototype.init = function(settings) {
        var defaults = {
            maxLength: null,
            placeholder: null,
            trimOnBlur: false,
            readOnly: false
        };

        jQuery.extend(true, defaults, settings);

        this.setMaxLength(defaults.maxLength)
            .setPlaceHolder(defaults.placeholder)
            .setTrimOnBlur(defaults.trimOnBlur)
            .setReadOnly(defaults.readOnly);
    };

    TextControlColumn.prototype.setMaxLength = function(input) {
        var i, cells, cellsLength;
        if(typeof input !== 'number') {
            throw new Error("setMaxLength(): the parameter must be a number.");
        }
        this.maxLength = input;
        cells = this.cells.asArray();
        cellsLength = cells.length;
        for(i = 0; i < cellsLength; i++) {
            cells[i].setMaxLength(this.maxLength);
        }
        return this;
    };
    TextControlColumn.prototype.setPlaceHolder = function(input) {
        var i, cells, cellsLength;
        if(typeof input !== 'string') {
            throw new Error("setPlaceHolder(): the parameter must be a string.");
        }
        this.placeholder = input;
        cells = this.cells.asArray();
        cellsLength = cells.length;
        for(i = 0; i < cellsLength; i++) {
            cells[i].setPlaceHolder(this.placeholder);
        }
        return this;
    };
    TextControlColumn.prototype.setTrimOnBlur = function(input) {
        var i, cells, cellsLength;
        this.trimOnBlur = !!input;
        cells = this.cells.asArray();
        cellsLength = cells.length;
        for(i = 0; i < cellsLength; i++) {
            cells[i].setTrimOnBlur(this.trimOnBlur);
        }
        return this;
    };
    TextControlColumn.prototype.setReadOnly = function(input) {
        var i, cells, cellsLength;
        this.readOnly = !!input;
        cells = this.cells.asArray();
        cellsLength = cells.length;
        for(i = 0; i < cellsLength; i++) {
            cells[i].setReadOnly(this.readOnly);
        }
        return this;
    };
    TextControlColumn.prototype.getCellDefaultCfg = function() {
        return jQuery.extend(true, TextControlColumn.superclass.prototype.getCellDefaultCfg.call(this), {
            maxLength: this.maxLength,
            trimOnBlur: this.trimOnBlur,
            placeholder: this.placeholder,
            readOnly: this.readOnly
        });
    };

    PMUI.extendNamespace('PMUI.grid.TextControlColumn', TextControlColumn);
}());
(function() {
    var DropDownListControlColumn = function(settings) {
        DropDownListControlColumn.superclass.call(this, settings);
        this.options = [];
        DropDownListControlColumn.prototype.init.call(this, settings);
    };

    PMUI.inheritFrom('PMUI.grid.GridControlColumn', DropDownListControlColumn);

    DropDownListControlColumn.prototype.CELL = PMUI.grid.DropDownListControlCell;

    DropDownListControlColumn.prototype.init = function(settings) {
        var defaults = {
            options: []
        };

        jQuery.extend(true, defaults, settings);

        this.setOptions(defaults.options);
    };

    DropDownListControlColumn.prototype.setOptions = function(options) {
        var i, cells;
        if(!jQuery.isArray(options)) {
            throw new Error("setOptions(): the parameter must be an array.");
        }
        this.options = options;
        cells = this.cells.asArray();
        for(i = 0; i < cells.length; i++) {
            cells[i].setOptions(this.options);
        }
        return this;
    };

    DropDownListControlColumn.prototype.getCellDefaultCfg = function() {
        return jQuery.extend(true, DropDownListControlColumn.superclass.prototype.getCellDefaultCfg.call(this), {
            options: this.options
        });
    };

    PMUI.extendNamespace('PMUI.grid.DropDownListControlColumn', DropDownListControlColumn);
}());
(function() {
    var CheckBoxGroupColumn = function(settings) {
        CheckBoxGroupColumn.superclass.call(this, settings);
        this.controlPositioning = null;
        this.maxDirectionOptions = null;
        this.options = [];
        CheckBoxGroupColumn.prototype.init.call(this, settings);
    };

    PMUI.inheritFrom('PMUI.grid.GridControlColumn', CheckBoxGroupColumn);

    CheckBoxGroupColumn.prototype.CELL = PMUI.grid.CheckBoxGroupControlCell;

    CheckBoxGroupColumn.prototype.init = function(settings) {
        var defaults = {
            options:[], 
            controlPositioning: "vertical",
            maxDirectionOptions: 1
        };
        jQuery.extend(true, defaults, settings);

        this.setOptions(defaults.options);
        this.setMaxDirectionOptions(defaults.maxDirectionOptions);
        this.setControlPositioning(defaults.controlPositioning);
    };

    CheckBoxGroupColumn.prototype.setOptions = function (options){
        var i, cells;
        if(!jQuery.isArray(options)) {
            throw new Error("setOptions(): the parameter must be an array.");
        }
        this.options = options;
        cells = this.cells.asArray();
        for(i = 0; i < cells.length; i++) {
            cells[i].setOptions(this.options);
        }
        return this;
    };
    CheckBoxGroupColumn.prototype.setControlPositioning = function(positioning) {
        this.controlPositioning = positioning;
        return this;
    };

    CheckBoxGroupColumn.prototype.setMaxDirectionOptions = function (maxDirectionOptions)  {
        this.maxDirectionOptions = maxDirectionOptions;
        return this;
    };

    CheckBoxGroupColumn.prototype.getCellDefaultCfg = function() {
        return jQuery.extend(true, CheckBoxGroupColumn.superclass.prototype.getCellDefaultCfg.call(this), {
            controlPositioning: this.controlPositioning,
            maxDirectionOptions: this.maxDirectionOptions,
            options: this.options
        });
    };

    PMUI.extendNamespace('PMUI.grid.CheckBoxGroupColumn', CheckBoxGroupColumn);
}());
(function() {
    var GridControlRow = function(settings) {
        GridControlRow.superclass.call(this, settings);
        this.removeButton = null;
    };

    PMUI.inheritFrom('PMUI.grid.GridPanelRow', GridControlRow);

    GridControlRow.prototype.getCell = function(column) {
        var cell = null;
        if(column instanceof PMUI.grid.GridControlColumn) {
            cell = this.cells.find("column", column);
        } else if(typeof column === 'number') {
            cell = this.cells.get(column);
        }
        return cell;
    };

    GridControlRow.prototype.setCells = function() {
        var i, columns, cellsFactory, cell;
        this.clearCells();
        if(this.parent) {
            columns = this.parent.columns.asArray();
            cellsFactory = this.parent.cellsFactory;
            for(i = 0; i < columns.length; i++) {
                cell = columns[i].createCell();
                cell.column = columns[i];
                cell.row = this;
                this.cells.insert(cell);
                if(this.html) {
                    this.html.appendChild(cell.getHTML());
                }
                this.data.addAttribute(cell.column.name, cell.getContent());
            }
        }
        return this;
    };

    GridControlRow.prototype.updateData = function(cell) {
        if(this.cells.contains(cell)) {
            this.data.addAttribute(cell.column.name, cell.getContent());    
        }
        return this;
    };

    GridControlRow.prototype.defineEvents = function() {
        var that = this;
        if(!this.removeButton) {
            return this;
        }
        //GridControlRow.superclass.prototype.defineEvents.call(this);
        jQuery(this.html).on('mouseenter', function() {
            var cell = that.getCell(that.cells.getSize() - 1);
            cell.html.appendChild(that.removeButton);
        }).on('mouseleave', function() {
            jQuery(that.removeButton).detach();
        });
        jQuery(this.removeButton).on('click', '.pmui-gridpanelrow-deleteButton-link', function() {
            if(that.parent) {
                that.parent.removeItem(that);
            }
        });
        return this;
    };

    GridControlRow.prototype.createHTML = function() {
        var deleteButton, deleteAnchor;
        if(this.html) {
            return this.html;
        }
        GridControlRow.superclass.prototype.createHTML.call(this);
        deleteButton = PMUI.createHTMLElement('div');
        deleteAnchor = PMUI.createHTMLElement('a');
        deleteButton.className = 'pmui-gridpanelrow-deleteButton';
        deleteAnchor.className = 'pmui-gridpanelrow-deleteButton-link';
        deleteAnchor.href = '#';
        deleteButton.appendChild(deleteAnchor);
        this.removeButton = deleteButton;
        this.defineEvents();
        return this.html;
    };

    GridControlRow.prototype.isValid = function() {
        var i, cells = this.cells.asArray(), valid = true;

        for(i = 0; i < cells.length; i++) {
            valid = valid && cells[i].isValid();
            if(!valid) {
                return valid;
            }
        }
        return valid;
    };

    PMUI.extendNamespace('PMUI.grid.GridControlRow', GridControlRow);
}());
(function() {
    var ContainableItem = function(settings) {
        ContainableItem.superclass.call(this, settings);
        /**
         * The parent container for the object.
         * @type {PMUI.core.Container}
         * @internal
         */
        this.parent = null;
        /**
         * The data abstraction for the object.
         * @type {PMUI.data.DataField}
         */
        this.data = null;
        ContainableItem.prototype.init.call(this, settings);
    };

    PMUI.inheritFrom('PMUI.core.Element', ContainableItem);

    ContainableItem.prototype.init = function(settings) {
        var defaults = {
            parent: null,
            data: new PMUI.data.DataField()
        };

        $.extend(true, defaults, settings);

        this.setParent(defaults.parent)
            .setData(defaults.data);
    };

    ContainableItem.prototype.setData = function(data) {
        if(data instanceof PMUI.data.DataField) {
            this.data = data;
        }

        return this;
    };

    ContainableItem.prototype.getData = function() {
        return this.data;
    };

    ContainableItem.prototype.setParent = function(parent) {
        if(parent !== undefined) {
            if(parent instanceof PMUI.core.Container || parent === null) {
                this.parent = parent;
            } else {
                throw new Error("setParent(): The parameter must be instance of PMUI.core.Container or null.");
            }
        }

        return this;
    };

    PMUI.extendNamespace('PMUI.item.ContainableItem', ContainableItem);

    if(typeof exports!== 'undefined'){
        module.exports == ContainableItem;
    }
}());
(function() {
/**
 * @class PMUI.item.TabItem
 * Creates an TabItem class, this is a basic element
 * @extends {PMUI.core.Element}
 *
 *  Usage example:
 *
 *      @example
 *          var ti,p;
 *          p = new PMUI.form.FormPanel({
 *          width: 604, 
 *                    height: 130,
 *                    fieldset: true,
 *                    legend: "my fieldset panel",
 *                    items: [
 *                        {
 *                            pmType: "text",
 *                            label: "Name",
 *                            id: "123",
 *                            value: "",
 *                            placeholder: "insert your name",
 *                            name: "name"
 *                        },{
 *                            pmType: "text",
 *                            label: "Last name",
 *                            value: "",
 *                            placeholder: "your lastname here asshole!",
 *                            name: "lastname"
 *                        }, {
 *                            pmType: "panel",
 *                            layout: 'hbox',
 *                            items: [
 *                                {
 *                                    pmType: "text",
 *                                    label: "E-mail",
 *                                    value: "",
 *                                    name: "email"
 *                                },{
 *                                    pmType: "text",
 *                                    label: "Phone",
 *                                    value: "555",
 *                                    name: "phone"
 *                                }
 *                            ]
 *                        }
 *                    ],
 *                    layout: "vbox"
 *                });
 *               ti = new PMUI.item.TabItem({
 *                       title : 'titulo1',
 *                       panel : p,
 *                       icon : 'pmui-gridpanel-engranaje'
 *                   });
 *               jQuery('body').append(ti.getHTML());
 *               ti.defineEvents();
 *
 * create a new instance of the TabItem, according to the example we set the following properties
 * @cfg {String} [title='[untitled]'], Title The title for the Window
 * @cfg {PMUI.core.Panel|Object} panel, this property is to set TabItems that are sent from the JSON
 * @cfg {String} [icon = ""], this property serves to define one TabItem icon with css with putting a className for this icon 
 * parameter to family {panel}
 * @cfg {Function} [onSelect= function() {}], this property is a function handler to from execute the
 * function when is callback define events
 */
    var TabItem = function (settings){
        TabItem.superclass.call(this, settings);
        /** 
         * Defines the Title of the TabItem
         * @type {String}
         */
        this.title = null;
        /** 
         * Element defines the panel that concentrates tab
         * @type {PMUI.core.Panel}
         */
        this.panel = null;
        /** 
         * Defines the property is a function handler to from execute the
         * function when is callback define events
         * @type {function}
         */
        this.onSelect = null;
        /** 
         * The property define the TabItem is selected for triggering the
         * @type {boolean}
         */
        this.selected = null;
        /** 
         * The property define the TabItem is selected for triggering 
         * @type {boolean}
         */
        this.onClose = null;
        /** 
         * property is defined when a TabItem is selected or event triggers a reference
         * @type {HTMLElement}
         */
        this.link = null;
        /** 
         * Defines the fatory  for childrend [PMUI.core.Panel]
         * @type {PMUI.util.factory}
         */
        this.factory = null;
        /**
         * property defined Tag for icon 
         * @type {HTMLElement}
         */
        this.classNameIcon = null;
        /**
         * property defined Tag for spanTitle 
         * @type {HTMLElement}
         */
        this.spanTitle = null;
        /**
         * TabItem title can be hidden or viewed by the user if you only want an icon mostart
         * @type {Boolean}
         */
        this.titleVisible = null;
        /**
         * [onClick description]
         * @type {[type]}
         * @private
         */
        this.onClick = null;
        this.icon = null;
        TabItem.prototype.init.call(this, settings);
    }; 
    PMUI.inheritFrom('PMUI.core.Element',TabItem);

    TabItem.prototype.family = 'TabItem';
    TabItem.prototype.type = 'TabItem';

    TabItem.prototype.init = function (settings) {
        var defaults = {
            title : '[untitled]',
            panel : null,
            onSelect : function() {},
            elementTag: 'li',
            onClick : null
        };

        jQuery.extend(true, defaults, settings);

        this.factory = new PMUI.util.Factory({
            products: {
                'treepanel': PMUI.panel.TreePanel,
                'panel': PMUI.core.Panel,
                'grid' : PMUI.grid.GridPanel
            },
            defaultProduct: 'panel'
        });
        this.setElementTag(defaults.elementTag)
            .setTitle(defaults.title)
            .setPanel(defaults.panel)
            .setOnSelectCallback(defaults.onSelect)
            .setOnClick(defaults.onClick);
    };

    TabItem.prototype.setOnClick = function (onClick){
        if(typeof onClick !== 'function' && onClick !== null) {
            throw new Error ('setOnClick(): the parameter is no valid, should be a function');
        }
        this.onClick = onClick;

        return this;
    }

    /**
     * Add an item to the panel.
     * @param {PMUI.item.TabItem} item
     * It can be a valid JSON object or an object that inherits from {@link PMUI.item.TabItem}.
     * @chainable
     */
    TabItem.prototype.setOnSelectCallback = function(callback) {
        if(typeof callback === 'function') {
            this.onSelect = callback;
        } else {
            throw new Error("setOnSelectCallback(): The parameter is not a function.");
        }
        return this;
    };
    /**
     * @event 
     * Define the event listeners for the TabPanel.
     * @chainable
     */
    TabItem.prototype.defineEvents = function() {
        var that = this;
        this.removeEvents().eventsDefined = true;
        if(this.link) {
            this.addEvent('click').listen(this.link, function(e) {
                if(typeof that.onSelect === 'function') {
                    e.preventDefault(); 
                    e.stopPropagation();
                    if (!that.selected){
                        that.select();
                    }
                }
                if(typeof that.onClick === 'function') {
                    that.onClick(that);
                }
            });
            that.panel.defineEvents();
        }
        return this;
    };
    /**
     * Sets the tabItem title
     * Set the proportion of the html element
     * @param {string} title
     * @chainable
     */ 
    TabItem.prototype.setTitle = function (title) {
        this.title = title;
        if (this.html) {
            this.spanTitle.textContent = this.title;
            //this.link.setAttribute("title",this.title);
        }
        return this;
    };

    /**
    * @method setIcon
    * className icon sets to be used
    * @param {String} icon must be a string indicating the icon to use css
    * @chainable
    */
    /*
    TabItem.prototype.setIcon = function (icon) {

        if (!(typeof icon == "string")) {
            throw new Error ("setIcon(), is not valid");
        }
        this.icon = icon;
        if(this.html){
            if(!this.classNameIcon) {
                this.classNameIcon = PMUI.createHTMLElement('i');
                this.classNameIcon.className = this.icon;

                jQuery(this.link).prepend(this.classNameIcon);
            } else {
                this.classNameIcon.className = this.icon;
            }
        }
        return this;
    };*/
    /**
     * Returns title the tabItem.
     * @return {string}
     */
    TabItem.prototype.getTitle = function () {
        return this.title;
    };
    /**
     * Sets the tabPanel [PMUI.core.panel] and full children except Window
     * @chainable
     */
    TabItem.prototype.setPanel = function (panel){
        var newPanel, errorMessage = "setPanel(): the object supplied as a parameter is not a valid type.";
        if(!panel) {
            throw new Error("setPanel(): you must specified a valid panel.");
        }
        if (panel instanceof PMUI.ui.Window){
            throw new Error(errorMessage);
        }
        if(this.factory) {
            if(this.factory.isValidClass(panel) || this.factory.isValidName(panel.pmType) || (panel instanceof PMUI.grid.GridPanel)) {
                newPanel = this.factory.make(panel);
            } else {
                throw new Error(errorMessage);
            }
        }
        if(newPanel) {
            this.panel = newPanel;
        }
        return this;
    };
    /**
     * Returns panel the tabItem.
     * @return {PMUI.core.Panel}
     */
    TabItem.prototype.getPanel = function (){
        return this.panel;
    };
    /**
     * if the tab item is selected, the selected property changes
     * @return {boolean}
     */
    TabItem.prototype.isSelected =function() {
        return this.selected;
    };
    /**
     * if the tab item is selected, the selected property changes to true and calls 
     * your function onSelect
     * @chainable
     */
    TabItem.prototype.select = function() {
        this.selected = true;
        this.style.addClasses(['pmui-active']);
        if(this.html) {
            this.onSelect();   
        }
        return this;
    };
    /**
     * if the tab item is selected, the selected property changes to false
     * @chainable
     */
    TabItem.prototype.deselect = function() {
        this.selected = false;
        this.style.removeClasses(['pmui-active']);
        return this;
    };
    /**
     * @method hideTitle
     * hide the title of TabItem
     * @chainable
     */
    TabItem.prototype.hideTitle = function () {
        this.titleVisible = false;
        if (this.html){
            this.spanTitle.style.display = "none";
        }
        return this;
    };
    /**
     * @method hideTitle
     * shows the title of the TabItem
     * @chainable
     */
    TabItem.prototype.showTitle = function () {
        this.titleVisible = true;
        if (this.html){
            this.spanTitle.style.display = "inline-block";
        }
        return this;
    };

    TabItem.prototype.createHTML = function() {
        var tab, ref, spanTitle, title, icon;
        if(this.html) {
            return this.html;
        }
        TabItem.superclass.prototype.createHTML.call(this);
        ref = PMUI.createHTMLElement('a');
        spanTitle = PMUI.createHTMLElement('span');
        icon = PMUI.createHTMLElement('i');
        icon.className = 'pmui-tab-icon';
        ref.className = 'pmui-tab-ref';
        ref.href= '#';
        spanTitle.className = 'pmui-tab_title';
        spanTitle.textContent = this.title;
        ref.appendChild(icon);
        ref.appendChild(spanTitle);
        this.icon = icon;
        this.html.appendChild(ref);
        this.spanTitle = spanTitle;
        this.link = ref;
        this.setWidth(this.width);

        if(this.eventsDefined) {
            this.defineEvents();
        }
        return this.html;
    };

    TabItem.prototype.setWidth = function (width) {
        TabItem.superclass.prototype.setWidth.call(this, width);
        if(this.html){
            if(width=="auto"){
                this.link.setAttribute("title","");
                this.spanTitle.style.width = "auto";
            } else {
                this.link.setAttribute("title",this.title);
                if(this.width >= 65) {
                    this.spanTitle.style.width = this.width - 46 + 'px';  
                }else{
                    this.spanTitle.style.width = '0px';
                }
  
            }
            
        }
        return this;
    };

    PMUI.extendNamespace('PMUI.item.TabItem', TabItem);

    if(typeof exports!== "undefined"){
        module.exports = TabItem;
    }

}());
(function(){
    /**
     * @class PMUI.item.AccordionItem
     * @extends PMUI.core.Item
     * 
     * Class to handle Accordion items, this is the basic element for make an {@link PMUI.panel.AccordionPanel Accordion}
     *
     * @constructor
     * For creates a new component the JSON config can have the following sentences
     * @param {Object} settings The configuration options contain:
     *
     *      ... {
     *          title: "The title",
     *          body: panel // This item can be any element that inherit from {@link PMUI.core.Panel Panel}
     *          iconClass: "the-class",
     *          style: {
     *              cssProperties: {
     *                  "background-color": "#f2eaea"
     *              }
     *          }
     *      } ...
     * @cfg {String} [title=""] Shows the title for the container
     * @cfg {Object} [body=null] Defines the element that will be inside the element
     * @cfg {String} [iconClass='pmui-accordion-item-icon'] Defines the class name for the header of the item
     * 
     */
    AccordionItem = function (settings) {
        AccordionItem.superclass.call(this, settings);
        /**
         * @property {String} [iconClass='pmui-accordion-item-icon'] 
         * Defines the class name for the header of the item
         */
        this.iconClass = null;
        /**
         * @property {String} [headerClass='pmui-accordion-item-header'] 
         * Defines the class name for the header
         * @private
         */
        this.headerClass = null,
        /**
         * @property {String} [bodyClass='pmui-accordion-item-body'] 
         * Defines the class name for the body
         * @private
         */
        this.bodyClass = null,
        /**
         * @property {String} [footerClass='pmui-accordion-item-footer'] 
         * Defines the class name for the footer
         * @private
         */
        this.footerClass = null;
        /**
         * @property {String} [containerClass='pmui-accordion-item-container'] 
         * Defines the class name for the global container
         * @private
         */
        this.containerClass = null;
        /**
         * @property {String} [iconExpanded='pmui-accordion-item-closed'] 
         * Defines the class name that represents the icon expanded for the AccordionItem
         * @private
         */
        this.iconExpanded = null;
        /**
         * @property {String} [iconClosed='pmui-accordion-item-expanded'] 
         * Defines the class name that represents the icon closed for the AccordionItem
         * @private
         */
        this.iconClosed = null;

        /**
         * @property {Boolean} [collapsed=true]
         * Defines whether the object is collapsed or not
         * @private
         */
        this.collapsed = null;
        /** 
         * @property {Object} [container=null] 
         * container Defines the container object for the class xxxxxxx
         * @private
         */
        this.container = null;
        /**
         * @property {Object} [header=null]
         * Defines the header object for the class
         * @private
         */
        this.header = null;
        /**
         * @property {Object} [body={}]
         * Defines the body object for the item
         * @private
         */
        this.body = {};
        /**
         * @property {Object} [footer=null]
         * Defines the footer object for the class
         * @private
         */
        this.footer = null;
        /**
         * @property {Boolean} [selected=false]
         * Defines whether the item is selected by default when the component is ready to render
         * on the browser
         */
        this.selected = null;
        /**
         * @property {Boolean} [propName] 
         * Represents a flag for control the rendered of the item.
         * If the property is True the item was rendered the otherwise if it
         * False the item still was not rendered
         * @private
         */
        this.hiddenBody = null;
        AccordionItem.prototype.init.call(this, settings);

    };

    PMUI.inheritFrom('PMUI.core.Item', AccordionItem);
    AccordionItem.prototype.type = 'Accordion';
    AccordionItem.prototype.family = 'Item';
    AccordionItem.prototype.init = function (settings) {
        var defaults = {
            title: '',
            body: '',
            collapsed: true,
            iconClass: 'pmui-accordion-item-icon',
            headerClass: 'pmui-accordion-item-header',
            bodyClass: 'pmui-accordion-item-body',
            footerClass: 'pmui-accordion-item-footer',
            containerClass: 'pmui-accordion-item-container',
            iconClosed: "pmui-accordion-item-closed",
            iconExpanded: "pmui-accordion-item-expanded",
            selected: false,
            factory: {
                products: {
                    'panel': PMUI.core.Item
                }
            }
        };

        jQuery.extend(true, defaults, settings);

        this.setFactory(defaults.products)
            .setTitle(defaults.title)
            .setBody(defaults.body)
            .setIconClass(defaults.iconClass)
            .setHeaderClass(defaults.headerClass)
            .setBodyClass(defaults.bodyClass)
            .setFooterClass(defaults.footerClass)
            .setContainerClass(defaults.containerClass)
            .setIconExpanded(defaults.iconExpanded)
            .setIconClosed(defaults.iconClosed)
            .setCollapsed(defaults.collapsed)
            .setSelected(defaults.selected);
    };

    /**
     * Sets the string that represent the title. It could be an simple string
     * or an string that contains html tags. If the string contains html code
     * it will be rendered
     * @param {String} title
     */
    AccordionItem.prototype.setTitle = function (title) {
        if (typeof title  === "string") {
            this.title = title;
            if (this.header) {
                this.header.items[2].title.html.innerHTML = title;
            }    
        }
        
        return this;
    };
    /**
     * Sets the class name for the AccordionItem's title
     * @param {String} class name
     */
    AccordionItem.prototype.setIconClass = function (className) {
        this.iconClass = (typeof className === 'string') ? className: '';
        if (this.header) {
            this.header.items[1].iconTitle.html.className = 'pmui-accordion-item-iconbase '+className;
        }
        return this;
    };
    /**
     * Sets the class name for the AccordionItem's header
     * @param {String} class name
     * @private
     */
    AccordionItem.prototype.setHeaderClass = function (className) {
        this.headerClass = (typeof className === 'string') ? className: '';
        if (this.header) {
            this.header.html.className = className;
        }
        return this;
    };
    /**
     * Sets the class name for the AccordionItem's body
     * @param {String} class name
     * @private
     */
    AccordionItem.prototype.setBodyClass = function (className) {
        this.bodyClass = (typeof className === 'string') ? className: '';
        if (this.html) {
            this.container[1].className = className;
        }
        return this;
    };
    /**
     * Sets the class name for the AccordionItem's footer
     * @param {String} class name
     * @private
     */
    AccordionItem.prototype.setFooterClass = function (className) {
        this.footerClass = (typeof className === 'string') ? className: '';
        if (this.html){
            this.container[2].className = className;
        }
        return this;
    };
    /**
     * Sets the class name for the AccordionItem's container
     * @param {String} class name
     * @private
     */
    AccordionItem.prototype.setContainerClass = function (className) {
        this.containerClass = (typeof className === 'string') ? className: '';
        if (this.html) {
            this.html.className = className;
        }
        return this;    
    };
    /**
     * Sets the class name that represents the icon in its expanded status
     * @param {String} class name
     * @private
     */
    AccordionItem.prototype.setIconExpanded = function (className) {
        this.iconExpanded = (typeof className === 'string') ? className: '';
        if (this.header) {
            this.header.items[0].iconNode.html.className = 'pmui-accordion-item-iconbase '+className;
        }
        return this;    
    };
    /**
     * Sets the class name that represents the icon in its closed status
     * @param {String} class name
     * @private
     */
    AccordionItem.prototype.setIconClosed = function (className) {
        this.iconClosed = (typeof className === 'string') ? className: '';
        if (this.header) {
            this.header.items[0].iconNode.html.className = 'pmui-accordion-item-iconbase '+className;
        }
        return this;
    };
    /**
     * Sets the status for the item, whether it collapsed or not
     * @param {Boolean} option
     * @private
     */
    AccordionItem.prototype.setCollapsed = function (option) {
        this.collapsed = option;
        return this;
    };
    /**
     * Sets the status collapsed. If it true the container will be opened
     * and if it false will be closed
     * @param {Boolean} status
     */
    AccordionItem.prototype.setSelected = function (status) {
        if (typeof status === 'boolean') {
            this.selected = status;
            this.setCollapsed(!status);
        }   
        return this;
    };
    /**
     * Sets the body for the item as a new child for the container.
     * The parameter can be:
     *
     *  - {@link PMUI.core.Panel panel}: In this case the item can be any class that inherit from
     * {@link PMUI.core.Panel Panel}. For example {@link PMUI.form.Form Form}, {@link PMUI.panel.TreePanel TreePanel}, etc.
     *  - String: In this case the string must have contain html code o can be a simple string like to exaple below:
     *
     *      "<a href=\"http://www.colosa.com\">Colosa</a>"
     *       
     * @param {PMUI.core.Panel|String} body
     */
    AccordionItem.prototype.setBody = function (body) {
        var currentPanel = this.body.html;
        if (body instanceof PMUI.core.Container) {
            body.getHTML();
            this.body.items =   [
                                    {
                                        panel: {
                                            html: body.html
                                        }
                                    }
                                ];
        } else if (typeof body === 'string'){
            this.body.items =   [
                                    {
                                        panel: {
                                            html: body
                                        }
                                    }
                                ];
        }
        try {
            this.body.html.removeChild(this.body.items[0].panel.html);
        } catch (e) {
            //throw new Error ("Error, removing html from "+ this.body.html);
        }
            
        this.addItem(body);
        return this;
    };
    /**
     * Adds an child item to the object.
     * @param {PMUI.core.Element|Object} item It can be one of the following data types:
     * - {PMUI.core.Element} the object to add.
     * - {Object} a JSON object with the settings for the Container to be added.
     * @param {Number} [index] An index in which the item will be added.
     * @chainable
     */
    AccordionItem.prototype.addItem = function (item, index) {
        var itemToBeAdded;
        if (this.factory) {
            itemToBeAdded = this.factory.make(item);
        }
        if (itemToBeAdded) {
            itemToBeAdded.parent = this;
            this.clearItems();
            this.items.insert(itemToBeAdded);
            if (this.body.html) {
                this.body.html.appendChild(itemToBeAdded.getHTML());

                if(this.eventsDefined){
                    itemToBeAdded.defineEvents();
                }
            }
        }
        return this;
    };
    /**
     * Handler the collapse funtionality
     * @return {PMUI.item.AccordionItem}
     */
    AccordionItem.prototype.toggleCollapse = function() {
        if (this.collapsed) {
            this.expand();
        } else {
            this.collapse();
        }
        return this;
    };
    /**
     * Collapses the item and sets the collapsed property in false
     */
    AccordionItem.prototype.collapse = function() {
        var i, 
            children,
            contentItem,
            items,
            otherElements;
        items = this.container;
        jQuery(items[1]).slideUp(300);
        jQuery(items[2]).slideUp(300);
        this.header.items[0].iconNode.html.className = 'pmui-accordion-item-iconbase '+this.iconClosed;
        this.collapsed = true;
        
        return this;
    };    
    /**
     * Expands the item selected. If the multipleSelection is enabled, It is posible select
     * one or more items the otherwise is only possible select one item.
     * @return {PMUI.item.AccordionItem}
     */
    AccordionItem.prototype.expand = function() {
        var i = 0, 
            children,
            contentItem;
        items = this.container;
        jQuery(items[1]).slideDown(300);
        jQuery(items[2]).slideDown(300);
        this.header.items[0].iconNode.html.className = 'pmui-accordion-item-iconbase '+this.iconExpanded;
        this.collapsed = false;

        //For other items
        if (this.parent.multipleSelection === false) {
            otherElements = this.parent.getItems();
            for (i = 0; i < otherElements.length; i+=1) {
                if (this.id != otherElements[i].id) {
                    otherElements[i].collapsed = false;
                    otherElements[i].toggleCollapse();
                }
            }
        }
        return this;
    };

    AccordionItem.prototype.createHTML = function () {
        var container, header, iconNode, iconTitle, title, body, footer;
        if (this.html) {
            return this.html;
        }
        container = AccordionItem.superclass.prototype.createHTML.call(this);
        container.setAttribute('class',this.containerClass);
        /*=============================================
                        Header accordion
        =============================================*/
        header = PMUI.createHTMLElement("div");
        header.setAttribute('class', this.headerClass);
        iconNode = PMUI.createHTMLElement('span');
        if (this.collapsed) {
            iconNode.setAttribute("class", 'pmui-accordion-item-iconbase '+this.iconClosed);
        } else if(!this.collapsed) {
            iconNode.setAttribute("class", 'pmui-accordion-item-iconbase '+this.iconExpanded);
        }
        iconTitle = PMUI.createHTMLElement('span');
        iconTitle.setAttribute('class', 'pmui-accordion-item-iconbase '+this.iconClass);
        title = PMUI.createHTMLElement('span');
        title.innerHTML = this.title;
        header.appendChild(iconNode);
        header.appendChild(iconTitle);
        header.appendChild(title);
        this.header = {
            html: header,
            items: [
                {
                    iconNode: {
                        html: iconNode
                    }
                },
                {
                    iconTitle: {
                        html: iconTitle
                    }
                },
                {   
                    title: {
                        html: title
                    }
                }
            ]
        };
        /*=============================================
                        body accordion
        =============================================*/
        body = PMUI.createHTMLElement("div");
        body.setAttribute('class',this.bodyClass);

        if(typeof this.parent.heightItem === "string" ) {
            body.style.height = this.parent.heightItem;
        } else {
            body.style.height = this.parent.heightItem + 'px';   
        }
        if (this.collapsed) {
            body.style.display = 'none';
        }
        
        this.body.html = body;
        jQuery(body).html(this.body.items[0].panel.html);

        /*=============================================
                        footer accordion
        =============================================*/
        footer = PMUI.createHTMLElement("div");
        footer.setAttribute('class',this.footerClass);
        this.container = new Array();
        this.container.push(header, body, footer);
        /*=============================================
                        Container accordion
        =============================================*/
        container.appendChild(header);
        container.appendChild(body);
        container.appendChild(footer);
        this.html = container;
        
        return this.html;
    };
    /**
     * Defines events related to the current panel and the calls to DefineEvents from 
     * every {@link PMUI.core.Item item}
     * return {this}
     */
    AccordionItem.prototype.defineEvents = function () {
        var j,
            fnSelect,
            children,
            that = this;
        this.removeEvents().eventsDefined = true;
        if(that.container[0]) {
            that.addEvent('click').listen(that.container[0], function(e){
                e.preventDefault();

                that.toggleCollapse();
                fnSelect = that.parent.listeners.select;
                fnSelect(that, e);
                
                e.stopPropagation();
            });   
        }

        if (that.items.getSize() > 0) {
            children = that.getItems();
            for (j = 0; j < children.length; j+=1) {
                children[j].defineEvents();
            }
        }
        return this;
    };

    PMUI.extendNamespace('PMUI.item.AccordionItem', AccordionItem);
    if (typeof exports !== "undefined") {
        module.exports = AccordionItem;
    }
}());
(function() {
    /**
     * @class PMUI.item.TreeNode
     * Each node in a TreePanel. It is able to apply data binding, that means that it can use its 
     * {@link PMUI.item.TreeNode#property-data data property} for set another object's property.
     * @extends PMUI.core.Item
     *
     * The following example shows a TreeNode object whose child nodes are created using the 
     * {@link PMUI.item.TreeNode#cfg-items items config option}:
     *
     *      @example
     *      var t;
     *      $(function() {
     *          t = new PMUI.item.TreeNode({
     *              label: 'America',
     *              childrenDefaultSettings: {
     *                  label: '[untitled node]'
     *              },
     *              items: [
     *                  {
     *                      label: "North America",
     *                      items: [
     *                          {
     *                              label: 'Canada',
     *                              data: {
     *                                  label: 'Canada',
     *                                  value: 'cnd'
     *                              }
     *                          },
     *                          {
     *                              label: 'Mexico'
     *                          }
     *                      ]
     *                  },
     *                  {
     *                      label: "Central America and Caribbean",
     *                      items: [
     *                          {
     *                              label: 'Cuba'
     *                          },
     *                          {
     *                              label: 'Guatemala'
     *                          } 
     *                      ]
     *                  },
     *                  {
     *                      label: "South America",
     *                      items: [
     *                          {
     *                              label: 'Argentina'
     *                          }, 
     *                          {
     *                              label: 'Bolivia'
     *                          }
     *                      ]
     *                  }
     *              ]
     *          });
     *          document.body.appendChild(t.getHTML());
     *          t.defineEvents();
     *      });
     *
     * The following example shows a TreeNode object whose child nodes are created using the 
     * {@link PMUI.item.TreeNode#cfg-dataItems dataItems config option}:
     *
     *      @example
     *      var t2;
     *      $(function() {
     *          t2 = new PMUI.item.TreeNode({
     *              data: {
     *                  name: 'America'
     *              },
     *              labelDataBind: 'name',
     *              autoBind: true,
     *              childrenDefaultSettings: {
     *                  labelDataBind: 'name',
     *                  autoBind: true
     *              },
     *              dataItems: [
     *                  {
     *                      name: "Argentina"
     *                  },
     *                  {
     *                      name: "Bolivia"
     *                  },
     *                  {
     *                      name: "Brasil"
     *                  }
     *              ]
     *          });
     *          document.body.appendChild(t2.getHTML());
     *          t2.defineEvents();
     *      });
     *
     * @cfg {Boolean} [sterile=false] If the node will be able to have children nodes.
     * @cfg {String} [itemsDataBind=null] The key from the object's data to be used as the array for create the child 
     * nodes of the current node when the data binding be applied. If the data binding is applied then the 
     * {@link PMUI.item.TreeNode#cfg-items items config option} will be ignored.
     * @cfg {String} [labelDataBind=null] The key from the object's data to be used as the label for the current node 
     * when the data binding be applied. If the data binding is applied then the 
     * {@link PMUI.item.TreeNode#cfg-label label config option} will be ignored.
     * @cfg {String} [nodeClassDataBind=null] The key from the object's data to be used as the class for the node's 
     * html when the data binding be applied. If the data binding is applied then the 
     * {@link PMUI.item.TreeNode#cfg-nodeClass nodeClass config option} will be ignored.
     * @cfg {Object} [data={}] The data for the node.
     * @cfg {Boolean} [collapsed=true] If the node will be collapsed or not from the beginning.
     * @cfg {String} [label='[item]'] The label text for the node.
     * @cfg {String} [nodeClass=''] The css class for the node's html.
     * @cfg {Function} [onCollapse=null] The callback function to be executed everytime the current node is collpased.
     * Please read the {@link PMUI.item.TreeNode#event-onCollapse onCollapse event documentation}.
     * @cfg {Function} [onExpand=null] The callback function yto be executed everytime the current node is expanded.
     * Please read the {@link PMUI.item.TreeNode#event-onExpanded onExpanded event documentation}.
     * @cfg {Object} [childrenDefaultSettings={}] The default config options to be applied to the new child nodes for 
     * the current node. This only will be applied when the appended item is an object literal.
     * @cfg {Boolean} [autobind] If the data binding will be applied automatically on instantiation and everytime the 
     * data is set. This will cause that some others config options be ignored (like 
     * {@link PMUI.item.TreeNode#cfg-label label}, {@link PMUI.item.TreeNode#cfg-nodeClass nodeClass} and/or 
     * {@link PMUI.item.TreeNode#cfg-items items}).
     * @cfg {Boolean} [recursiveChildrenDefaultSettings=false] If the current 
     * {@link PMUI.item.TreeNode#cfg-childrenDefaultSettings childrenDefaultSettings config option} will be inherited 
     * to the same config option of its child nodes.
     * @cfg {Function|null} [onBeforeAppend=null] A callback function to be called everytime the 
     * {@link #event-onBeforeAppend onBeforeAppend} event is fired. For info about the parameters used for this 
     * callback please read the documentation for this event.
     * @cfg {Function|null} [onAppend=null] A callback function to be called everytime the 
     * {@link #event-onAppend onAppend} event is fired. For info about the parameters used for this callback please 
     * read the documentation for this event.
     */
    var TreeNode = function(settings) {
        TreeNode.superclass.call(this, settings);
        /**
         * If the node can have child nodes.
         * @type {Boolean}
         */
        this.sterile = null;
        /**
         * If the node is collapsed.
         * @type {Boolean}
         * @readonly
         */
        this.collapsed = null;
        /**
         * The current label being used by the node.
         * @type {String}
         * @readonly
         */
        this.label = null;
        /**
         * A literal object that will contain all the dom elements that compose the node's html structure.
         * @type {Object}
         * @readonly
         */
        this.dom = {};
        /**
         * The css class for the node's html.
         * @type {String}
         * @readonly
         */
        this.nodeClass = null;
        /**
         * @event onClick
         * Fired when the current node is clicked
         */
        this.onClick = null;
        /**
         * @event onCollapse
         * Fired when the current node is collapsed.
         */
        this.onCollapse = null;
        /**
         * @event onExpand
         * Fired when the current node is expanded.
         */
        this.onExpand = null;
        /**
         * The key of the object's data that will be used as the array to build the child nodes for the current one 
         * when the data binding is applied.
         * @type {String}
         * @readonly
         */
        this.itemsDataBind = null;
        /**
         * The key of the object's data that will be used as the label for the current node when the data binding is 
         * applied.
         * @type {String}
         * @readonly
         */
        this.labelDataBind = null;
        /**
         * The key of the object's data that will be used as the css class for the current node's html when the data
         * binding is applied.
         * @type {String}
         * @readonly
         */
        this.nodeClassDataBind = null;
        /**
         * If the data binding will be applied on initialization and everytime the data is set.
         * @type {Boolean}
         */
        this.autoBind = null;
        /**
         * The default config options to be applied to the child nodes to be add to the current node, but only when an
         * object literal is used.
         * @type {Object}
         */
        this.childrenDefaultSettings = {};
        /**
         * The data for the current node. This data can be used to modify another node's properties through data 
         * binding.
         * @type {PMUI.data.DataField}
         * @private
         */
        this.data = null;
        /**
         * If the settings defined in the {@link #property-childrenDefaultSettings childrenDefaultSettings} property will
         * be inherited to all the child nodes of the current one.
         * @type {Boolean}
         */
        this.recursiveChildrenDefaultSettings = null;
        /**
         * @event onBeforeAppend
         * Fired before a child node is appended to the current node or to any of its children nodes.
         * @param {PMUI.item.TreeNode} node The node in which the event is being fired.
         * @param {PMUI.item.TreeNode} newNode The new node to be appended.
         */
        this.onBeforeAppend = null;
        /**
         * @event onAppend
         * Fired everytime a child node is appended to the current node or to any of its children nodes.
         * @param {PMUI.item.TreeNode} node The node in which the event is being fired.
         * @param {PMUI.item.TreeNode} newNode The appended node.
         */
        this.onAppend = null;
        TreeNode.prototype.init.call(this, settings);
    };

    PMUI.inheritFrom('PMUI.core.Item', TreeNode);
    /**
     * @property {String} type=TreeNode The type of the class.
     * @readonly
     */
    TreeNode.prototype.type = 'TreeNode';
    /**
     * @property {String} family=Tree The class family.
     * @readonly
     */
    TreeNode.prototype.family = 'Tree';
    /**
     * Initializes the object.
     * @param  {Object} settings The object literal with the config options.
     * @private
     */
    TreeNode.prototype.init = function(settings) {
        var defaults = {
            itemsDataBind: null,
            labelDataBind: null,
            nodeClassDataBind: null,
            data: {},
            sterile: false,
            collapsed: true,
            label: '[item]',
            nodeClass: "",
            onCollapse: null, 
            onExpand: null,
            onClick: null,
            childrenDefaultSettings: {},
            autoBind: false,
            recursiveChildrenDefaultSettings: false,
            onAppend: null,
            onBeforeAppend: null
        };

        jQuery.extend(true, defaults, settings);

        this.data = new PMUI.data.DataField();

        this.setNodeClass(defaults.nodeClass)
        if(defaults.autoBind) {
            this.enableAutoBind();
        } else {
            this.disableAutoBind();
        }
        this.setRecursiveChildrenDefaultSettings(defaults.recursiveChildrenDefaultSettings)
            .setChildrenDefaultSettings(defaults.childrenDefaultSettings)
            .setSterility(defaults.sterile)
            .setLabel(defaults.label)
            .setParent(defaults.parent)
            .setOnCollapseHandler(defaults.onCollapse)
            .setOnExpandHandler(defaults.onExpand)
            .setOnClickHandler(defaults.onClick)
            .setItemsDataBind(defaults.itemsDataBind)
            .setLabelDataBind(defaults.labelDataBind)
            .setNodeClassDataBind(defaults.nodeClassDataBind)
            .setOnAppendHandler(defaults.onAppend)
            .setOnBeforeAppendHandler(defaults.onBeforeAppend);

        if(defaults.dataItems && jQuery.isArray(defaults.dataItems)) {
            this.setDataItems(defaults.dataItems);
        } else {
            this.setItems(defaults.items);
        }
        this.setData(defaults.data);

        if(defaults.collapsed) {
            this.collapse();
        } else {
            this.expand();
        }
        this.elementTag = 'li';
    };
    /**
     * Sets the callback function to be called everytime the {@link #event-onBeforeAppend onBeforeAppend} event is 
     * fired. For info about the parameters used for this callback please read the documentation for this event.
     * @param {Function|null} handler
     */
    TreeNode.prototype.setOnBeforeAppendHandler = function(handler) {
        if(!(handler === null || typeof handler === 'function')) {
            throw new Error("setOnBeforeAppendHandler(): The parameter must be a function or null.");
        }
        this.onBeforeAppend = handler;
        return this;
    };
    /**
     * Sets the callback function to be called everytime the 
     * {@link #event-onAppend onAppend} event is fired. For info about the parameters used for this callback please 
     * read the documentation for this event.
     * @param {Function|null} handler
     */
    TreeNode.prototype.setOnAppendHandler = function(handler) {
        if(!(handler === null || typeof handler === 'function')) {
            throw new Error("setOnAppendHandler(): The parameter must be a function or null.");
        }
        this.onAppend = handler;
        return this;
    };
    /**
     * Returns the level of the node relative to its root node.
     * @return {Number}
     */
    TreeNode.prototype.getDepth = function() {
        if(!this.parent || !(this.parent instanceof PMUI.item.TreeNode)) {
            return 0;
        } else {
            return this.parent.getDepth() + 1;
        }
    };
    /**
     * Returns the root node of the current one.
     * @return {PMUI.item.TreeNode}
     */
    TreeNode.prototype.getRoot = function() {
        if(this.getDepth() === 0) {
            return this;
        } else {
            return this.parent.getRoot();
        }
    };
    /**
     * Turns on the auto binding functionality.
     * @chainable
     */
    TreeNode.prototype.enableAutoBind = function() {
        this.autoBind = true;
        return this;
    };
    /**
     * Turns off the auto binding functionality.
     * @chainable
     */
    TreeNode.prototype.disableAutoBind = function() {
        this.autoBind = false;
        return this;
    };
    /**
     * Returns true if the current {@link PMUI.item.TreeNode#property-childrenDfaultSettings childrenDfaultSettings} 
     * will be inherited to the next child nodes that will be added.
     * @return {Boolean} [description]
     */
    TreeNode.prototype.isRecursiveChildrenDefaultSettings = function() {
        return this.recursiveChildrenDefaultSettings;
    };
    /**
     * Establish if the current object's childrenDefaultSettings property values will be set it to the same property
     * of its future child nodes.
     * @param {Boolean} recursive
     */
    TreeNode.prototype.setRecursiveChildrenDefaultSettings = function(recursive) {
        this.recursiveChildrenDefaultSettings = !!recursive;
        return this;
    };
    /**
     * Returns the default config options for the node's child nodes.
     * @return {Object} 
     */
    TreeNode.prototype.getChildrenDefaultSettings = function() {
        return this.childrenDefaultSettings;
    };
    /**
     * Sets the default config options for the child nodes to be added. Note that if you set this, the effect will 
     * take effect only in new nodes added from the moment of the new set. The default settings will have effect only
     * when you add an item using anything but an instance of {@link PMUI.core.Element} (or ant subclass of it).
     * @param {Object} settings A object literal with some or all of the config options for a TreeNode instance.
     */
    TreeNode.prototype.setChildrenDefaultSettings = function(settings) {
        if(typeof settings !== 'object') {
            throw new Error('setChildrenDefaultSettings(): The parameter must be an object.');
        }
        this.childrenDefaultSettings = settings;

        return this;
    };
    /**
     * Sets the key of the current node's data to be used as the css class for the node's html.
     * @param {String} nodeClassDataBind
     */
    TreeNode.prototype.setNodeClassDataBind = function(nodeClassDataBind) {
        if(nodeClassDataBind !== null && typeof nodeClassDataBind !== 'string') {
            throw new Error("setNodeClassDataBind(): the parameter must be a string or null.");
        }
        this.nodeClassDataBind = nodeClassDataBind;
        return this;
    };
    /**
     * Sets the key of the current node's data to be used as the array of items to be add to the current node.
     * Note that every item in the array will be added as a data item 
     * (see the {@link PMUI.core.Container#method-addDataItem addDataItem()} method).
     * @param {String} itemsDataBind
     * @chainable
     */
    TreeNode.prototype.setItemsDataBind = function(itemsDataBind) {
        if(itemsDataBind !== null && typeof itemsDataBind !== 'string') {
            throw new Error("setItemsDataBind(): the parameter must be a string or null.");
        }
        this.itemsDataBind = itemsDataBind;
        return this;
    };
    /**
     * Sets the key of the current node's data to be used as the label for the current node.
     * @param {String} labelDataBind
     * @chainable
     */
    TreeNode.prototype.setLabelDataBind = function(labelDataBind) {
        if(labelDataBind !== null && typeof labelDataBind !== 'string') {
            throw new Error("setLabelDataBind(): the parameter must be a string or null.");
        }
        this.labelDataBind = labelDataBind;
        return this;
    };
    /**
     * Bind the node's data to its properties using:
     *
     * - The {@link PMUI.item.TreeNode#property-labelDataBind labelDataBind property} to set the 
     * {@link PMUI.item.TreeNode#property-label label property}.
     * - The {@link PMUI.item.TreeNode#property-itemsDataBind labelDataBind property} for set the node's child items.
     * - The {@link PMUI.item.TreeNode#property-nodeClassDataBind nodeClassDataBind property} for set the css class for
     * the node's html.
     * @chainable
     */
    TreeNode.prototype.bindData = function() {
        var data = this.data.getRecord();
        this.setLabel(data[this.labelDataBind] || this.label);
        if(this.itemsDataBind && jQuery.isArray(data[this.itemsDataBind])) {
            this.setDataItems(data[this.itemsDataBind]);
        }
        this.setNodeClass(data[this.nodeClassDataBind] || this.nodeClass);

        return this;
    };
    /**
     * Sets the data for the node.
     * @param {Object|PMUI.data.DataField} data
     */
    TreeNode.prototype.setData = function (data) {
        var key, items;
        if(data instanceof PMUI.data.DataField) {
            this.data = data;
        } else if(typeof data ==='object') {
            this.data.clear();
            for(key in data) {
                this.data.addAttribute(key, data[key]);
            }
        } else {
            throw new Error("setData(): it only accepts a JSON object o an instance of PMUI.data.DataField as parameter.");
        }

        if(this.autoBind) {
            this.bindData();
        }

        return this;
    };
    /**
     * Return the node's data.
     * @return {Object} A object literal containing the node's data.
     */
    TreeNode.prototype.getData = function() {
        return this.data.getRecord();
    };
    /**
     * Sets the callback function for the {@link PMUI.item.TreeNode#event-onClickHandler onClickHandler event}.
     * @param {Function} handler 
     */
    TreeNode.prototype.setOnClickHandler = function(handler) {
        if(handler === null || typeof handler === 'function') {
            this.onClick = handler;
        }
        return this;
    };
    /**
     * Sets the callback function for the {@link PMUI.item.TreeNode#event-onCollapse onCollapse event}.
     * @param {Function} handler
     */
    TreeNode.prototype.setOnCollapseHandler = function(handler) {
        if(handler === null || typeof handler === 'function') {
            this.onCollapse = handler;
        }

        return this;
    };
    /**
     * Sets the callback function for the {@link PMUI.item.TreeNode#event-onExpand onExpand event}.
     * @param {Function} handler
     */
    TreeNode.prototype.setOnExpandHandler = function(handler) {
        if(handler === null || typeof handler === 'function') {
            this.onExpand = handler;
        }

        return this;
    };
    /**
     * Sets the factory for the curren node.
     * @private
     * @chainable
     */
    TreeNode.prototype.setFactory = function() {
        this.factory = new PMUI.util.Factory({
            products: {
                'treeNode': PMUI.item.TreeNode
            },
            defaultProduct: 'treeNode'
        });

        return this;
    };
    /**
     * Sets the parent node for the current one.
     * @param {null|PMUI.item.TreeNode|PMUI.panel.TreePanel} parent.
     * @chainable
     */
    TreeNode.prototype.setParent = function(parent) {
        if(parent === null || parent instanceof PMUI.item.TreeNode || parent instanceof PMUI.panel.TreePanel) {
            TreeNode.superclass.prototype.setParent.call(this, parent);
        }
        return this;
    };
    /**
     * Returns true if the node is sterile (it can have more child nodes).
     * @return {Boolean}
     */
    TreeNode.prototype.isSterile = function() {
        return this.sterile;
    };
    /**
     * Sets a custom css class for the node's html.
     * @param {String} nodeClass
     */
    TreeNode.prototype.setNodeClass = function(nodeClass) {
        if(typeof nodeClass !== 'string') {
            throw new Error('setNodeClass(): the parameter must be a string.');
        }
        nodeClass = jQuery.trim(nodeClass);
        if(this.html) {
            jQuery (this.dom.nodeElement).removeClass(this.nodeClass).addClass('pmui-treepanel-node').addClass(nodeClass);
        }
        this.nodeClass = nodeClass;
        return this;
    };
    /**
     * Returns the custom css class of the node's html.
     * @return {String}
     */
    TreeNode.prototype.getNodeClass = function() {
        return this.nodeClass;
    };
    /**
     * Sets if the node is sterile or not (if can have more child nodes).
     * @param {Boolean} sterile
     * @chainable
     */
    TreeNode.prototype.setSterility = function(sterile) {
        this.sterile = !!sterile;
        return this;
    };
    /**
     * Sets the label for the node.
     * @param {String} label
     */
    TreeNode.prototype.setLabel = function(label) {
        if(typeof label !== 'string') {
            throw new Error('setLabel(): This method only accepts parameter of type String.');
        }
        this.label = label;
        if(this.html) {
            this.dom.label.textContent = label;
        }

        return this;
    };
    /**
     * Applies the collapsing to its child nodes list.
     * @param  {Boolean} applyToChildren If the child nodes will apply the collapsing too.
     * @chainable
     */
    TreeNode.prototype.collapse = function(applyToChildren) {
        var i, childNodes;
        this.collapsed = true;
        if(applyToChildren) {
            childNodes = this.items.asArray();
            for(i = 0; i < childNodes.length; i++) {
                childNodes[i].collapse(true);
            }
        }
        if(this.html) {
            jQuery(this.containmentArea).hide();
            jQuery(this.dom.nodeElement).addClass('pmui-treepanel-node-collapsed');
            if(typeof this.onCollapse === 'function') {
                this.onCollapse();
            }
        }

        return this;
    };
    /**
     * Applies the expanding to its child nodes list.
     * @param  {Boolean} applyToChildren If the child nodes will apply the expanding too.
     * @chainable
     */
    TreeNode.prototype.expand = function(applyToChildren) {
        var i, childNodes;
        this.collapsed = false;
        if(applyToChildren) {
            childNodes = this.items.asArray();
            for(i = 0; i < childNodes.length; i++) {
                childNodes[i].expand(true);
            }
        }
        if(this.html) {
            jQuery(this.containmentArea).show();
            jQuery(this.dom.nodeElement).removeClass('pmui-treepanel-node-collapsed');
            if(typeof this.onExpand === 'function') {
                this.onExpand();
            }
        }

        return this;
    };
    /**
     * Returns true if the node's child list is collapsed, otherwise it returns false.
     * @return {Boolean}
     */
    TreeNode.prototype.isCollapsed = function() {
        return this.collapsed;
    };
    /**
     * Returns true if the node has no child nodes, otherwise it returns false.
     * @return {Boolean}
     */
    TreeNode.prototype.isLeaf = function() {
        return !this.items.getSize();
    };
    /**
     * Returns true id the node is root, otherwise it returns false.
     * @return {Boolean}
     */
    TreeNode.prototype.isRoot = function() {
        return this.parent instanceof PMUI.panel.TreePanel || !this.parent;
    };
    /**
     * Updates the node's css class that sepcifies if the node is leaf or not.
     * @private
     * @chainable
     */
    TreeNode.prototype.updateNodeTypeClass = function() {
        if(this.items.getSize() === 0) {
            jQuery(this.dom.nodeElement).removeClass('pmui-treepanel-node-father').addClass('pmui-treepanel-node-leaf');
        } else {
            jQuery(this.dom.nodeElement).removeClass('pmui-treepanel-node-leaf').addClass('pmui-treepanel-node-father');
        }
        return this;
    };
    /**
     * The private handler fot the {@link #event-onBeforeAppend onBeforeAppend} event.
     * @private
     * @chainable
     */
    TreeNode.prototype.onBeforeAppendHandler = function(treeNode) {
        if(typeof this.onBeforeAppend === 'function') {
            this.onBeforeAppend(this, treeNode);
        }
        this.parent && this.parent.onBeforeAppendHandler(treeNode);
        return this;
    };
    /**
     * The private handler for the {@link #event-onAppend} event.
     * @private
     * @chainable
     */
    TreeNode.prototype.onAppendHandler = function(treeNode) {
        if(typeof this.onAppend === 'function') {
            this.onAppend(this, treeNode);
        }
        this.parent && this.parent.onAppendHandler(treeNode);
        return this;
    };
     /**
     * Adds a new child node.
     * @param {Object|PMUI.item.TreeNode} item  The item to add, it can be:
     * - A object literal with the config options to create the {@link PMUI.item.TreeNode TreeNode} to be added. If
     * you have set the {@link PMUI.item.TreeNode#property-childDefaultSettings childDefaultSettings property} the 
     * contents of the object literal will be merge with (and into) the contens of this property. The resulting object
     * literal will be used to create the {@link PMUI.item.TreeNode TreeNode} to be added.
     * - An instance of {@link PMUI.item.TreeNode TreeNode}.
     * @param {Number} index The index position for the node insertion.
     */
    TreeNode.prototype.addItem = function(item, index) {
        var settings = {}, toAdd;
        if(this.isSterile()) {
            throw new Error('addItem(): The tree node is sterile, it can\'t have children.');
        }
        if(typeof item !== 'object' && !(item instanceof PMUI.item.TreeNode)) {
            throw new Error('addItem(): The parameter must be an instance of PMUI.item.TreeNode or an object.');
        }/*
        if(this.inheritNodeClass) {
            if(typeof item === 'object') {
                item.nodeClass = item.nodeClass === undefined || item.nodeClass === null ? this.nodeClass : item.nodeClass;
            } else if(item instanceof PMUI.item.TreeNode && !item.nodeClass) {
                item.setNodeClass(this.nodeClass, true);
            }
        }*/
        if(item instanceof PMUI.item.TreeNode) {
            item.setParent(this);
            toAdd = item;
        } else if(typeof item === 'object') {
            if(this.recursiveChildrenDefaultSettings) {
                item.childrenDefaultSettings = this.childrenDefaultSettings;
                item.recursiveChildrenDefaultSettings = true;
            }
            jQuery.extend(true, settings, this.childrenDefaultSettings, item);
            settings.parent = this;
            toAdd = this.factory.make(settings);
        }

        if(toAdd) {
            if(typeof this.onBeforeAppend === 'function') {
                this.onBeforeAppend(this, toAdd);
            }
            this.onBeforeAppendHandler(toAdd);
            TreeNode.superclass.prototype.addItem.call(this, toAdd, index);
            this.onAppendHandler(toAdd);
        }

        if(this.html) {
            this.updateNodeTypeClass();
        }

        return this;
    };
    /**
     * @inheritdoc
     */
    TreeNode.prototype.setItems = function(items) {
        if(this.onAppend !== undefined) {
            TreeNode.superclass.prototype.setItems.call(this, items);
        }
        return this;
    };
    /**
     * Removes a direct child node.
     * @param  {Number|PMUI.item.TreeNode|String} item It can be a string (id of the child to remove), 
     * a number (index of the child to remove) or a TreeNode object.
     * @chainable
     */
    TreeNode.prototype.removeItem = function(item) {
        TreeNode.superclass.prototype.removeItem.call(this, item);
        this.updateNodeTypeClass();
        return this;
    };
    /**
     * Toggles the collapsing/expanding actions.
     * @chainable
     */
    TreeNode.prototype.toggleCollapsing = function() {
        if(this.isCollapsed()) {
            this.expand();
        } else {
            this.collapse();
        }
        return this;
    };
    /**
     * Clears the current filter criteria.
     * @chainable
     */
    TreeNode.prototype.clearFilter = function() {
        this.filter("");
        return this;
    };
    /**
     * Filter the current node and all its children.
     * @param  {String} filterCriteria The criteria for filtering, the filtering will be perform in the node's data.
     * @return {Boolean} True if any of its nodes meets the filtering criteria.
     */
    TreeNode.prototype.filter = function(filterCriteria) {
        var res = false, childNodes = this.items.asArray(), i, key, regExp, partialRes;

        for(i = 0; i < childNodes.length; i += 1) {
            childNodes[i].setVisible(partialRes = childNodes[i].filter(filterCriteria));
            res = res || partialRes;
        }
        if(!res) {
            regExp = new RegExp(filterCriteria, "i");
            if(regExp.test(this.label)) {
                return true;
            }
        }
        return res;
    };
    /**
     * Define then events for the object.
     * @chainable
     */
    TreeNode.prototype.defineEvents = function() {
        var that, i, items;
        TreeNode.superclass.prototype.defineEvents.call(this);
        if(this.dom.title) {
            that = this;
            this.addEvent('click').listen(this.dom.title, function(e) {
                var root = that.getRoot(), panel = root.parent;
                e.preventDefault();
                if(typeof that.onClick === 'function') {
                    that.onClick();
                }
                if(panel instanceof PMUI.panel.TreePanel && typeof panel.onNodeClick === 'function') {
                    panel.onNodeClick(panel, that);
                }
                that.toggleCollapsing();
            });
        }

        return this;
    };
    /**
     * Creates the node's html.
     * @return {HTMLELement}
     */
    TreeNode.prototype.createHTML = function() {
        var nodeElement, title, label, icon, childrenList, elbow;
        if(this.html) {
            return this.html;
        }

        TreeNode.superclass.superclass.prototype.createHTML.call(this);
        nodeElement = this.html;
        nodeElement.className = 'pmui-treepanel-node';
        nodeElement.id = this.id;
        title = PMUI.createHTMLElement('a');
        title.className = 'pmui-treepanel-node-title';
        title.href = '#';
        elbow = PMUI.createHTMLElement('b');
        elbow.className = 'pmui-treepanel-node-elbow';
        elbow.innerHTML = '&nbsp;';
        label = PMUI.createHTMLElement('span');
        label.className = 'pmui-treepanel-node-label';
        icon = PMUI.createHTMLElement('i');
        icon.className = 'pmui-treepanel-node-icon';
        icon.innerHTML = '&nbsp;';
        childrenList = PMUI.createHTMLElement('ul');
        childrenList.className = 'pmui-treepanel-list';

        this.actionHTML = title;

        title.appendChild(elbow);
        title.appendChild(icon);
        title.appendChild(label);
        nodeElement.appendChild(title);
        nodeElement.appendChild(childrenList);

        this.dom.nodeElement = nodeElement;
        this.dom.title = title;
        this.dom.label = label;
        this.dom.icon = icon;
        this.containmentArea = childrenList;

        this.html = nodeElement;

        this.updateNodeTypeClass();

        this.setParent(this.parent)
            .setSterility(this.sterile)
            .setLabel(this.label)
            .setNodeClass(this.nodeClass)
            .setItems(this.items.asArray().slice(0));

        if(this.collapsed) {
            this.collapse();
        } else {
            this.expand();
        }
        
        this.style.applyStyle();
        this.setBehavior(this.behavior);

        return this.html;
    };

    PMUI.extendNamespace('PMUI.item.TreeNode', TreeNode);
}());
(function() {
    /**
     * @class PMUI.item.ListItem
     */
    var ListItem = function(settings) {
        ListItem.superclass.call(this, settings);
        this.text = null;
        this.iconClass = null;
        this.toolbar = null;
        this.dom = {};
        this.actions = [];
        this.visibleIcon = null;
        this.data = null; 
        this.textDataBind = null;
        this.onClick = null;
        ListItem.prototype.init.call(this, settings);
    };

    PMUI.inheritFrom('PMUI.core.Item', ListItem);

    ListItem.prototype.type = "ListItem";

    ListItem.prototype.init = function(settings) {
        var defaults = {
            elementTag: 'li',
            text: '[list-item]',
            actions: [],
            iconClass: null,
            visibleIcon: false,
            data: {},
            textDataBind: null,
            onClick: null
        };

        this.toolbar = new PMUI.toolbar.Toolbar({
            style: {
                cssClasses: ['pmui-listitem-actions']
            }
        });

        this.data = new PMUI.data.DataField();

        jQuery.extend(true, defaults, settings);

        this.setTextDataBind(defaults.textDataBind)
            .setElementTag(defaults.elementTag)
            .setText(defaults.text)
            .setIconClass(defaults.iconClass)
            .setActions(defaults.actions)
            .setData(defaults.data)
            .setOnClickHandler(defaults.onClick);

        if(defaults.visibleIcon) {
            this.showIcon();
        } else {
            this.hideIcon();
        }
    };

    ListItem.prototype.setOnClickHandler = function(handler) {
        if(!(handler === null || typeof handler === 'function')) {
            throw new Error("setOnClickHandler(): The parameter must be or null.");
        }
        this.onClick = handler;
        return this;
    };

    ListItem.prototype.onClickHandler = function() {
        var that = this;
        return function() {
            if(typeof that.onClick === 'function') {
                that.onClick(that);
            }
        };
    };

    ListItem.prototype.setTextDataBind = function(textDataBind) {
        if(textDataBind !== null && typeof textDataBind !== 'string') {
            throw new Error("setTextDataBind(): the parameter must be a string or null.");
        }
        this.textDataBind = textDataBind;
        return this.bindData();
    };

    ListItem.prototype.bindData = function() {
        var data = this.data.getRecord();
        this.setText(data[this.textDataBind] || this.text || "");

        return this;
    };

    ListItem.prototype.setData = function(data) {
        var key;
        if(data instanceof PMUI.data.DataField) {
            this.data = data;
        } else if(typeof data === 'object') {
            this.data.clear();
            for(key in data) {
                this.data.addAttribute(key, data[key]);
            }
        } else {
            throw new Error("setData(): it only accepts a JSON object o an instance of PMUI.data.DataField as parameter.");
        }
        this.bindData();

        return this;
    };

    ListItem.prototype.clearActions = function() {
        this.toolbar.clearItems();
        return this;
    };

    ListItem.prototype.addAction = function(action) {
        var that = this, aux, handler;

        aux = action.handler || null;
        if(!(aux === null || typeof aux === 'function')) {
            throw new Error("addAction(): The parameter's \"'handler\" property must be a function or null or not be specified.");
        }

        handler = aux ? function() {
            if(aux) {
                aux.call(that, that, this); 
            }
        } : null;

        action.action = handler;

        if(action instanceof PMUI.toolbar.ToolbarAction) {
            action.setAction(handler);
        } else if(typeof action === 'object') {
            action.action = handler;
        }       

        this.toolbar.addItem(action);

        return this;
    };

    ListItem.prototype.setActions = function(actions) {
        var i;
        if(!jQuery.isArray(actions)) {
            throw new Error("setActions(): The parameter must be an array.");
        }
        this.clearActions();
        for(i = 0; i < actions.length; i++) {
            this.addAction(actions[i]);
        }
        return this;
    };

    ListItem.prototype.setIconClass = function(iconClass) {
        if(!(iconClass === null || typeof iconClass === 'string')) {
            throw new Error("setIconClass(): the parameter must be a string or null.");
        }
        this.iconClass = iconClass;
        if(this.dom.iconContainer) {
            this.dom.iconContainer.className = 'pmui-listitem-icon ' + (iconClass || "");
        }
        return this;
    };

    ListItem.prototype.hideIcon = function() {
        this.visibleIcon = false;
        if(this.dom.iconContainer) {
            this.dom.iconContainer.style.display = 'none';
        }
        return this;
    };

    ListItem.prototype.showIcon = function() {
        this.visibleIcon = true;
        if(this.dom.iconContainer) {
            this.dom.iconContainer.style.display = '';
        }
        return this;
    }

    ListItem.prototype.setText = function(text) {
        if(typeof text !== 'string') {
            throw new Error("setText(): the parameter must be a string.");
        }
        this.text = text;
        if(this.dom.textContainer) {
            this.dom.textContainer.textContent = text;
        }
        return this;
    };

    ListItem.prototype.getData = function() {
        return this.data.getRecord();
    };

    ListItem.prototype.defineEvents = function() {
        return this;
    };

    ListItem.prototype.createHTML = function() {
        var textContainer, iconContainer, that = this;
        if(this.html) {
            return this.html;
        }
        ListItem.superclass.prototype.createHTML.call(this);
        textContainer = PMUI.createHTMLElement('div');
        textContainer.className = 'pmui-listitem-text';
        iconContainer = PMUI.createHTMLElement('i');
        iconContainer.className = 'pmui-listitem-icon';

        this.dom.textContainer = textContainer;
        this.dom.iconContainer = iconContainer;

        this.html.appendChild(iconContainer);
        this.html.appendChild(textContainer);
        this.html.appendChild(this.toolbar.getHTML());

        this.setText(this.text);

        if(this.visibleIcon) {
            this.showIcon();
        } else {
            this.hideIcon();
        }

        this.addEvent('click').listen(this.html, function(e) {
            var parent = that.getParent();
            e.stopPropagation();
            if(typeof that.onClick === 'function') {
                that.onClick(that);
            }
            if(parent) {
                parent.onItemClickHandler(that);
            }
        });

        return this.html;
    };

    PMUI.extendNamespace('PMUI.item.ListItem', ListItem);
}());
(function () {
    /**
     * @class PMUI.proxy.Proxy
     * Defines the proxy functionality to persist data
     * @abstract
     *
     * @constructor
     * Creates a new instance of this class
     * @param {Object} options
     *
     * @cfg {Object} data Object that will be sent through the proxy 
     */
    var Proxy = function (options){
        /**
         * Data object used to send/receive through proxy
         * @type {Object}
         */
        this.data = null;
        Proxy.prototype.init.call(this, options);
    };

    /**
     * Defines the object's type
     * @type {String}
     */
    Proxy.prototype.type = "Proxy";

    /**
     * Defines the object's family
     * @type {String}
     */
    Proxy.prototype.family = "Proxy";

    /**
     * @private
     * Initilizes the object with the default values
     * @param {Object} options
     */
    Proxy.prototype.init = function (options) {
        var defaults = {
            data: null
        };
        jQuery.extend(true, defaults, options);
        this.setData(defaults.data);
    };

    /**
     * Sets the data
     * @param {Object} data Object to be sent 
     */
    Proxy.prototype.setData = function (data) {
        this.data = data;
        return this;
    };

    /**
     * Returns the data related to this class
     * @return {Object} 
     */
    Proxy.prototype.getData = function() {
        return this.data;
    };

    /**
     * @abstract
     * Sends the data to the target
     */
    Proxy.prototype.send = function () {
    };

    /**
     * @abstract
     * Receives the data from the target
     */
    Proxy.prototype.receive = function () {
    };

    PMUI.extendNamespace("PMUI.proxy.Proxy",Proxy);

    // Declarations created to instantiate in NodeJS environment
    if (typeof exports !== "undefined") {
        module.exports = Proxy;
    }

}());
(function (){
    /**
     * @class PMUI.proxy.RestProxy
     * @extend PMUI.proxy.Proxy
     *
     * How used the class
     * The {@link PMUI.proxy.RestClient Restclient} class encapsulate all the funcionality about the communicate 
     * with the server.
     * 
     *          @example
     *          restclient = new PMUI.proxy.RestProxy({
     *              url: "url.php",
     *              method: "GET",
     *              data: {},
     *              dataType: "json",
     *              success: function () {
     *                  ...
     *              },
     *              failure: function () {
     *                  ...
     *              },
     *              complete: function() {
     *                  ...
     *              }
     *          });
     *
     * 
     * @constructor      
     * Creates a new component 
     * @param {Object} [settings] settings The configuration options are specified on the config options:
     *      
     * @cfg {String} [url=""] The url property defines the target for the RestClient.
     * @cfg {String} method 
     * The available methods to consume are the follow:
     * 
     *          'create', 'read', 'update', 'delete'
     *      
     * These have a directly relation to:
     * 
     *          'create': 'POST',
     *          'read'  : 'GET',
     *          'update': 'PUT',
     *          'delete': 'DELETE'
     *      
     * Based in the {@link RestClient RestClient} library
     * @cfg {Object} data Object that will be sent through the proxy.
     * @cfg {String} [dataType='json'] The available options are the following sentences:
     * 
     *      json: 'application/json',
     *      plain: 'text/plain',
     *      form: 'application/x-www-form-urlencoded',
     *      html: 'text/html'
     *      
     * @cfg {String} [authorizationType= "none"] The options availables for the property:
     * 
     *      "none"
     *      "basic"
     *      "oauth2"
     * 
     * @cfg {Boolean} [authorizationOAuth=false] For enable the communicate between client-server 
     * through Oauth protocol the property must to be true
     *
     */
    var RestProxy = function (options) {
        RestProxy.superclass.call(this, options);
        this.url = null;
        this.method = null;
        this.rc = null;
        this.data = null;
        RestProxy.prototype.init.call(this, options);
    };

    PMUI.inheritFrom('PMUI.proxy.Proxy', RestProxy);
    /**
     * @property {String} 
     * @readonly
     */
    RestProxy.prototype.type = "RestProxy";
    RestProxy.prototype.init = function (options) {
        var defaults = {
            url: null,
            method: 'GET',
            data: {},
            dataType: 'json',
            authorizationType: 'none',
            authorizationOAuth: false,
            success: function(){},
            failure: function(){},
            complete: function(){}
        };
        jQuery.extend(true, defaults, options);
        this.setRestClient()
            .setUrl(defaults.url)
            .setAuthorizationOAuth(defaults.authorizationOAuth)
            .setMethod(defaults.method)
            .setData(defaults.data)
            .setDataType(defaults.dataType)
            .setSuccessAction(defaults.success)
            .setFailureAction(defaults.failure)
            .setCompleteAction(defaults.complete);
    };

    /**
     * Creates an instance of the {@link RestClient ResClient} and 
     * stores the object inside of {@link RestProxy#rc rc property}
     */
    RestProxy.prototype.setRestClient = function () {
        if (this.rc instanceof RestClient === false) {
            this.rc = new RestClient();
        }
        return this;
    };
    /**
     * Sets the url that will be used for send or get the data
     * from server, It's a basic element for {@link RestClient RestClient}
     * @param {String} url A string that represents the connection between
     * Client and server
     *
     *
     */
    RestProxy.prototype.setUrl = function (url) {
        this.url = url;
        return this;
    };
    /**
     * Sets the OAuth protocol to 'false' or 'true' for enable or disable the property.
     *
     *      //Example
     *      obj = new PMUI.proxy.RestProxy();
     *      obj.setAuthorizationOAuth(true);
     * 
     * @param {Boolean} option Defines the boolean option
     */
    RestProxy.prototype.setAuthorizationOAuth = function (option) {
        if (typeof option === 'boolean') {
            this.rc.setSendBearerAuthorization(option);
        }
        return this;
    };
    /**
     * This method is useful for update the current action.
     * The available methods to consume are the follow:
     *      'create', 'read', 'update', 'delete'
     * These have a directly relation to:
     *      'create': 'POST',
     *      'read'  : 'GET',
     *      'update': 'PUT',
     *      'delete': 'DELETE'
     * Based in the {@link RestClient RestClient} library
     * @param {String} A string that reporesents
     */
    RestProxy.prototype.setMethod = function (method) {
        this.method = method;
        return this;
    };
    RestProxy.prototype.setSuccessAction = function (action) {
        RestProxy.prototype.success = action;
        return this;
    };
    RestProxy.prototype.setFailureAction = function (action) {
        RestProxy.prototype.failure = action;
        return this;
    };
    RestProxy.prototype.setCompleteAction = function (action) {
        RestProxy.prototype.complete = action;
        return this;
    };
    /**
     *
     * @param {Object} data Data for send to server
     * @return {PMUI.proxy.RestProxy} this The current component
     *
     */
    RestProxy.prototype.setData = function (data) {
        this.data = data;
        return this;
    };
    /**
     * Sets the Data type used for request data.
     * the types availables for the methods are the following:
     *   
     *      json: 'application/json',
     *      plain: 'text/plain',
     *      form: 'application/x-www-form-urlencoded',
     *      html: 'text/html'
     *      
     *      //For example
     *      restclient = new PMUI,proxy.RestProxy();
     *      restclient.setDataType("json");
     *      ...
     * 
     * @param {String} dataType Type 
     */
    RestProxy.prototype.setDataType = function (dataType) {
        this.rc.setDataType(dataType);
        return this;
    };
    RestProxy.prototype.setCredentials = function (usr, pass) {
        this.rc.setBasicCredentials(usr, pass);
        return this;
    };
    RestProxy.prototype.setContentType = function () {
        this.rc.setContentType();
        return this;
    };
    /**
     * Sets the authorization type and credentials for the RestClient.
     *
     * The options availables about authorization types are:
     *
     *      'none'
     *      'basic'
     *      'oauth2'
     *      
     *      The credentials must have two keys: The username and password
     *      
     *      //For example
     *      restclient = new PMUI.proxy.RestProxy();
     *      restclient.setAuthorizationType('oauth2', credentials);
     *
     *      Is possible sets the property while instances the class like to sentence below:
     *
     *      restclient = new PMUI.proxy.RestProxy({
     *          ...
     *          authorizationOAuth: true
     *          ...
     *      });
     *
     *      
     * @param {Object} type Authorization type
     * @param {Object} credentials These are the keys that represent the username and password
     */
    RestProxy.prototype.setAuthorizationType = function (type, credentials) {
        this.rc.setAuthorizationType(type); 
        switch(type) {
            case 'none':
                break;
            case 'basic':
                    this.rc.setBasicCredentials(credentials.client, credentials.secret);
                break;
            case 'oauth2':
                    this.rc.setAccessToken(credentials);
                break;
        }
        
        return this;
    };
    /**
     * Send data to server using POST method from RestClient
     * @param {Object} settings Configuration parameters
     *
     *      //For example
     *      this.post({
     *          url: "url.php",
     *          data: {
     *              grant_type: "authorization_code",
     *              code: keys.authorizedCode
     *          },
     *          success: function (xhr, response) {
     *              console.log(response);
     *          },
     *          failure: function (xhr, response) {
     *              console.log(response);
     *          },
     *          complete: function (xhr, response) {
     *              console.error(response);
     *          }
     *      });
     *      
     */
    RestProxy.prototype.post = function (settings) {
        var that = this;
        if (settings !== undefined) {
            that.init(settings);
        }
        if (this.rc) {
            that.rc.postCall({
                url: that.url,
                id: that.uid,
                data: that.data,
                success : function (xhr, response) {
                    that.success.call(that, xhr, response);
                },
                failure: function (xhr, response) {
                    that.failure.call(that, xhr, response);
                },
                complete: function (xhr, response) {
                    that.complete.call(that, xhr, response);
                }
            });
            that.rc.setSendBearerAuthorization(false);

        } else {
            throw new Error("the RestClient was not defined, please verify the property 'rc' for continue.");
        }
    };
    /**
     * Updates data from the server using UPDATE method from RestClient
     * @param  {Object} settings
     * If it necessary is possible overwrite all the data instantiated on the RestClient
     *
     *      //For example
     *      this.update({
     *          url: "url.php",
     *          authorizationOAuth: true,
     *          success: function (xhr, response) {
     *              console.log(response);
     *          },
     *          failure: function (xhr, response) {
     *              console.error(response);
     *          },
     *          complete: function (xhr, response) {
     *              console.error(response);
     *          }
     *      });
     *      
     */
    RestProxy.prototype.update = function (settings) {
        var that = this;
        if (settings !== undefined) {
            that.init(settings);
        }
        if (this.rc) {
            this.rc.putCall({
                url: this.url,
                id: this.uid,
                data: this.data,
                success : function (xhr, response) {
                    that.success.call(this, xhr, response);
                },
                failure: function (xhr, response) {
                    that.failure.call(this, xhr, response);
                },
                complete: function (xhr, response) {
                    that.complete.call(that, xhr, response);
                }
            });
        } else {
            throw new Error("the RestClient was not defined, please verify the property 'rc' for continue.");
        }
    };
    /**
     * Gets data from the server using GET method from RestClient
     * @param  {Object} settings
     * If it necessary is possible overwrite all the data instantiated on the RestClient
     *
     *      //For example
     *      this.get({
     *          url: "url.php",
     *          authorizationOAuth: true,
     *          success: function (xhr, response) {
     *              console.log(response);
     *          },
     *          failure: function (xhr, response) {
     *              console.error(response);
     *          },
     *          complete: function (xhr, response) {
     *              console.error(response);
     *          }
     *      });
     *      
     */
    RestProxy.prototype.get = function (settings) {
        var that = this;
        if (settings !== undefined) {
            that.init(settings);
        }
        if (this.rc) {
            that.rc.getCall({
                url: that.url,
                id: that.uid,
                data: that.data,
                success : function (xhr, response) {
                    that.success.call(that, xhr, response);
                },
                failure: function (xhr, response) {
                    that.failure.call(that, xhr, response);
                },
                complete: function (xhr, response) {
                    that.complete.call(that, xhr, response);
                }
            });
            that.rc.setSendBearerAuthorization(false);

        } else {
            throw new Error("the RestClient was not defined, please verify the property 'rc' for continue.");
        }
    };
    /**
     * Removes data using DELETE method from RestClient
     * @param  {Object} settings
     * If it necessary is possible overwrite all the data instantiated on the RestClient
     *     
     *      //For example
     *      this.remove({
     *          url: "url.php",
     *          authorizationOAuth: true,
     *          success: function (xhr, response) {
     *              console.log(response);
     *          },
     *          failure: function (xhr, response) {
     *              console.error(response);
     *          },
     *          complete: function (xhr, response) {
     *              console.error(response);
     *          }
     *      });
     *      
     */
    RestProxy.prototype.remove = function (settings) {
        var that = this;
        if (settings !== undefined) {
            that.init(settings);
        }
        if (this.rc) {
            this.rc.deleteCall({
                url: this.url,
                id: this.uid,
                data: this.data,
                success : function (xhr, response) {
                    that.success.call(this, xhr, response);
                },
                failure: function (xhr, response) {
                    that.failure.call(this, xhr, response);
                },
                complete: function (xhr, response) {
                    that.complete.call(that, xhr, response);
                }
            });
        } else {
            throw new Error("the RestClient was not defined, please verify the property for continue.");
        }
    };
    /**
     * This is the callback for {@link RestClient RestClient} and it will be actived
     * and executed when the {@link RestClient RestClient} petition was success.
     * @param {Object} xhr
     * @param {Object} response
     */
    RestProxy.prototype.success = function (xhr, response){        
    };
    /**
     * If something happend with the connection or maybe the server is
     * down, this callback will be executed
     * @param {Object} xhr
     * @param {Object} response
     */
    RestProxy.prototype.failure = function (xhr, response){
    };
    /**
     * This is the callback for {@link RestClient RestClient} and it will be actived
     * and executed when the {@link RestClient RestClient} petition finishes.
     * @param {Object} xhr
     * @param {Object} response
     */
    RestProxy.prototype.complete = function (xhr, response){
    };
    
    PMUI.extendNamespace('PMUI.proxy.RestProxy', RestProxy);

    if (typeof exports !== 'undefined') {
        module.exports = RestProxy;
    }

}());
(function(){
    /**
     * @class PMUI.event.Event
     * Handles the events generated in the PMUI library
     * @abstract
     *
     * @constructor
     * Creates a new instance of class
     * @param {Object} options 
     *
     * @cfg {Object} element Defines the HTMLElement
     * @cfg {Function} handler Defines the callback Function to be executed
     * @cfg {PMUI.event.Action} action Defines an Action to be used to handle the callback
     * @cfg {String} name Event name
     */
    var Event = function (options) {
        /**
         * Stores the HTMLElement associated
         * @type {Object}
         */
        this.element = null;
        /**
         * Stores the callback function to be executed
         * @type {Function}
         */
        this.handler = null;
        /**
         * Stores the selector for delegated event listeners.
         * @type {String}
         */
        this.selector = null;
        /**
         * Event name
         * @type {String}
         */
        this.eventName = null;
        Event.prototype.init.call(this, options);
    };

    /**
     * Defines the object's type
     * @type {String}
     */
    Event.prototype.type = "Event";

    /**
     * Defines the object's family
     * @type {String}
     */
    Event.prototype.family = "Event";

    /**
     * @private
     * Initializes the object with default options
     * @param  {Object} options 
     */
    Event.prototype.init = function (options){
        var defaults = {
            handler: function(scope) {}
        };
        jQuery.extend(true, defaults, options);
        if (defaults.action && defaults.action instanceof PMUI.event.Action) {
            this.setHandler(defaults.action.handler);
        } else {    
            this.setHandler(defaults.handler);  
        }
        this.setElement(defaults.element)
            .setEventName(defaults.name);
        return this;
    };

    /**
     * Sets the HTML Element
     * @param {Object} element [description]
     */
    Event.prototype.setElement = function (element){
        this.element = element;
        return this;
    };

    /**
     * Sets the callback function
     * @param {Function} fn 
     */
    Event.prototype.setHandler = function (fn) {
        if (typeof fn === 'function') {
            this.handler = fn;
        }
        return this;
    };
    /**
     * Sets the selector to be used for delegated event handlers.
     * @param {String} selector
     * @chainable
     */
    Event.prototype.setSelector = function(selector) {
        if(typeof selector === 'string') {
            this.selector = selector;
        }
        return this;
    };
    /**
     * Sets the event name
     * @param {String} name
     */
    Event.prototype.setEventName = function (name) {
        this.eventName = name;
        return this;
    };
    /**
     * Add a handler to listen for a event with delegation.
     * @param  {HTMLElement} parentElement The parent element whose child elements the event will be delegate to.
     * @param  {String} selector      A valid jQuery selector to filter the descendants of the selected 
     * elements that trigger the event.
     * @param  {Function} handler       A function to execute when the event is triggered.
     * @chainable
     */
    Event.prototype.listenWithDelegation = function(parentElement, selector, handler) {
        var handlerFunction;
        if (handler instanceof PMUI.event.Action){
            handlerFunction = handler.handler;
        } else {
            handlerFunction = handler;
        }
        $(parentElement).on(this.eventName, selector, handler);
        this.setHandler(handler).setElement(parentElement).setSelector(selector);
        return this;
    };

    /**
     * @abstract
     * Defines the way to listen the event (jquery)
     * @param  {HTMLElement} element [description]
     * @param  {Function} handler [description]
     */
    Event.prototype.listen = function (element, handler){
        var handlerFunction;
        if (handler instanceof PMUI.event.Action){
            handlerFunction = handler.handler;
        } else {
            handlerFunction = handler;
        }
        jQuery(element).on(this.eventName, handlerFunction);
        this.setHandler(handlerFunction).setElement(element);
        return this;
    };
    /**
     * Removes the event listener.
     * @chainable
     */
    Event.prototype.unlisten = function() {
        if(this.selector) {
            jQuery(this.element).off(this.eventName, this.selector, this.handler);
        } else {
            jQuery(this.element).off(this.eventName, this.handler);
        }        
        return this;
    };

    PMUI.extendNamespace('PMUI.event.Event', Event);

    if (typeof exports !== 'undefined') {
        module.exports = Event;
    }

}());
(function () {
    /**
     * @class  PMUI.event.EventFactory
     * @extend PMUI.util.Factory
     * Extends the factory class to produce Events instances
     *
     * @constructor
     * Creates a new instance od the class
     */
    var EventFactory = function () {
        EventFactory.superclass.call(this);
        EventFactory.prototype.init.call(this);
    };

    PMUI.inheritFrom('PMUI.util.Factory', EventFactory);

    /**
     * Defines the object's type
     * @type {String}
     */
    EventFactory.prototype.type = 'EventFactory';

    /**
     * @private
     * Define the event types supported
     * @type {Object}
     */
    EventFactory.prototype.eventTypes = {
        'click' : 'mouse',
        'mousedown' : 'mouse',
        'mouseup' : 'mouse',
        'mousemove' : 'mouse',
        'mouseover' : 'mouse',
        'mouseout' : "mouse",
        'mouseenter': "mouse",
        'mouseleave': "mouse",
        'dblclick' : "mouse",
        'drag': 'mouse',
        'drop' : 'mouse',
        'resize': 'mouse',
        'rightclick' : 'mouse',
        'contextmenu' : 'mouse',
        'blur': 'form',
        'change': 'form',
        'focus': 'form',
        'select': 'form',
        'submit': 'form',
        'keyup' : 'keyboard',
        'keydown': 'keyboard',
        'keypress': 'keyboard'
    };

    /**
     * @private
     * Initializes the object with default values
     */
    EventFactory.prototype.init = function () {
        var defaults = {
            products: {
                "mouse" : PMUI.event.MouseEvent,
                "form" : PMUI.event.FormEvent,
                "keyboard" : PMUI.event.KeyboardEvent,
                "event": PMUI.event.Event               
            },
            defaultProduct: "event"
        };
        this.setProducts(defaults.products)
            .setDefaultProduct(defaults.defaultProduct);
    };

    /**
     * Overwrite the make function to accept strings
     * @param  {Object/String} type 
     * @return {PMUI.event.Event} 
     */
    EventFactory.prototype.make = function (type) {
        var eventInstance,
            eventType;
        if (this.isValidClass(type)){
            eventInstance = type;
        } else {
            eventType = this.eventTypes[type] || 'event';
            eventInstance = this.build(eventType, {name: type});
        }
        return eventInstance;
    };

    PMUI.extendNamespace('PMUI.event.EventFactory', EventFactory);

    if (typeof exports !== 'undefined') {
        module.exports = EventFactory;
    }

}());
(function () {
    /**
     * @class PMUI.event.MouseEvent
     * @extend PMUI.event.Event
     * Handles the events generated by the mouse
     *
     * @constructor
     * Create a new instance
     * @param {Object} options Constructor options
     *
     * @cfg {Object} element Defines the HTMLElement
     * @cfg {Function} handler Defines the callback Function to be executed
     * @cfg {PMUI.event.Action} action Defines an Action to be used to handle the callback
     * @cfg {String} name Event name
     */
    var MouseEvent = function (options) {
        MouseEvent.superclass.call(this, options);
    };

    PMUI.inheritFrom('PMUI.event.Event', MouseEvent);

    /**
     * Defines the object's type
     * @type {String}
     */
    MouseEvent.prototype.type = "MouseEvent";

    PMUI.extendNamespace('PMUI.event.MouseEvent', MouseEvent);

    if (typeof exports !== 'undefined') {
        module.exports = MouseEvent;
    }

}());
(function () {
    /**
     * @class PMUI.event.KeyboardEvent
     * @extend PMUI.event.Event
     * Handles the events generated by the mouse
     *
     * @constructor
     * Create a new instance
     * @param {Object} options Constructor options
     *
     * @cfg {Object} element Defines the HTMLElement
     * @cfg {Function} handler Defines the callback Function to be executed
     * @cfg {PMUI.event.Action} action Defines an Action to be used to handle the callback
     * @cfg {String} name Event name
     */
    var KeyboardEvent = function (options) {
        KeyboardEvent.superclass.call(this, options);
    };

    PMUI.inheritFrom('PMUI.event